#!/usr/bin/python
# -*- coding: utf-8 -*-
#
# Copyright (C) 2013-2022:
# This file is part of Shinken Enterprise, all rights reserved.

import os
import threading
import time

from shinken.log import PartLogger
from shinken.misc.type_hint import TYPE_CHECKING

if TYPE_CHECKING:
    from shinken.misc.type_hint import Optional

LOG_SHINKEN_LOCK_SCHEDULER_FLAG_NAME = 'SHINKEN_LOG_SHINKEN_LOCK_SCHEDULER_FLAG'
LOG_SHINKEN_LOCK_SCHEDULER_FLAG = os.environ.get(LOG_SHINKEN_LOCK_SCHEDULER_FLAG_NAME, '0') == '1'


# An OrdonnancementGroup is like a Lock that will protect against another Group:
# * you cannot run in the same time as a thread of another OrdonnancementGroup.
# * BUT you can run against another thread of the same OrdonnancementGroup
class OrdonnancementGroup(object):
    def __init__(self, recursive_lock, name, max_switch_time, logger):
        # type: (threading.RLock, unicode, int, PartLogger) -> None
        self.cond = threading.Condition(recursive_lock)
        self.name = name
        if max_switch_time is None or not str(max_switch_time).isdigit() or max_switch_time < 0:
            raise Exception(u'Bad parameter for max_switch_time')
        self._max_switch_time = max_switch_time
        self.asking_task = 0
        self.working_task = 0
        self.logger = logger
        
        self._start_sleep_time = None
    
    
    def __str__(self):
        # type: () -> unicode
        return u'%s(working_task:%s, asking_task:%s, switch limit:%s)' % (str(self.name), str(self.working_task), str(self.asking_task), str(self._max_switch_time))
    
    
    def get_name(self):
        # type: () -> unicode
        return self.name
    
    
    def get_nb_asking_task(self):
        # type: () -> int
        return self.asking_task
    
    
    def get_nb_working_task(self):
        # type: () -> int
        return self.working_task
    
    
    def get_max_switch_time(self):
        # type: () -> int
        return self._max_switch_time
    
    
    def log_debug(self, s):
        # type: (unicode) -> None
        if LOG_SHINKEN_LOCK_SCHEDULER_FLAG:
            self.logger.debug(s)
    
    
    # Another Group is allowing us to run
    def allow_to_run(self):
        # type: () -> None
        with self.cond:
            # If we are based on time, we allow everyone to start
            self.log_debug('%s => waking up all workers' % self.name)
            self.cond.notify_all()
            return
    
    
    # A new task was allowed to start
    def add_working_task(self):
        # type: () -> None
        # we are working, so no more wait
        self._start_sleep_time = None
        self.working_task += 1
    
    
    # A task did finish
    def remove_working_task(self):
        # type: () -> None
        self.working_task -= 1
    
    
    # Are we just void, allowing another Group to run freely
    def is_idle(self):
        # type: () -> bool
        return self.asking_task == 0 and self.working_task == 0
    
    
    def have_working_task(self):
        # type: () -> bool
        return self.working_task != 0
    
    
    # Do we have any task that is still waiting?
    def is_asking_for_lock(self):
        # type: () -> bool
        self.log_debug('%s number of pending tasks: %s' % (self.name, self.asking_task))
        return self.asking_task > 0
    
    
    # We will go wait for a START message from another Group for max sleep_time
    def go_wait(self, sleep_time):
        # type: (Optional[int]) -> None
        self.asking_task += 1
        
        if self._start_sleep_time is None:
            self._start_sleep_time = time.time()
        
        time_before_wait = time.time()
        # NOTE: wait() give a bool about (timeout or not in python 3.2, so need to manage it manually)
        _only_in_python_3_2 = self.cond.wait(sleep_time)
        time_after_wait = time.time()
        
        diff_time = time_after_wait - time_before_wait
        self.log_debug(u'%s was off during %s (log sleep =%s)' % (self.name, diff_time, sleep_time))
        
        self.asking_task -= 1
    
    
    def is_asking_since_too_long(self):
        # type: () -> bool
        if self._start_sleep_time is not None:
            too_long = self._start_sleep_time < time.time() - self._max_switch_time
            self.log_debug(u'%s too long? %s' % (self.name, too_long))
            return too_long
        return False


class FairLockGroupOrdonnancer(object):
    def __init__(self, name, consumer_name, producer_name, logger, consumer_max_wish_switch_time=1, producer_max_wish_switch_time=1, error_log_time=None):
        # type: (unicode, unicode, unicode, PartLogger, int, int, Optional[int] ) -> None
        self._error_log_time = error_log_time
        
        self.logger = logger.get_sub_part(name)
        
        lock = threading.RLock()
        self._consumer = OrdonnancementGroup(lock, consumer_name, consumer_max_wish_switch_time, self.logger)
        self._producer = OrdonnancementGroup(lock, producer_name, producer_max_wish_switch_time, self.logger)
        
        self._mode_producer = True
        self._cur_mode_consecutive_tasks = 0
        self._active_task = {}
    
    
    def _log_too_long_wait(self, elapsed_time, me, other, i_am_producer):
        # type: (float, OrdonnancementGroup, OrdonnancementGroup, bool) -> None
        
        # NOTE: pending +1 because we log outside the wait() so thread is not count
        if self._mode_producer == i_am_producer:
            self.logger.error(
                u'[ LONG LOCK ] Still have %(me_active_nb)d running task%(me_active_s)s ongoing (%(me_name)s). => ( %(other_pending_nb)d ) %(other_name)s and then ( %(me_pending_nb)s ) %(me_name)s are waiting since %(elapsed_time)ds (>= log error limit:%(limit_time)ds)' %
                {
                    'me_active_nb'    : me.get_nb_working_task(),
                    'me_active_s'     : 's' if me.get_nb_working_task() > 1 else '',
                    'me_name'         : me.get_name(),
                    'other_pending_nb': other.get_nb_asking_task(),
                    'other_name'      : other.get_name(),
                    'me_pending_nb'   : me.get_nb_asking_task() + 1,
                    'elapsed_time'    : elapsed_time,
                    'limit_time'      : self._error_log_time
                })
        else:
            self.logger.error(
                u'[ LONG LOCK ] %s are waiting (%s thread%s) since %ds (>= log error limit=%ds) because %s (%s thread%s) have the LOCK' % (
                    me.get_name(),
                    me.get_nb_asking_task() + 1,
                    's' if me.get_nb_asking_task() > 0 else '',
                    elapsed_time,
                    self._error_log_time,
                    other.get_name(),
                    other.get_nb_working_task(),
                    's' if other.get_nb_working_task() > 1 else ''
                ))
    
    
    def log_debug(self, message):
        # type: (unicode) -> None
        if LOG_SHINKEN_LOCK_SCHEDULER_FLAG:
            self.logger.debug(message)
    
    
    def _is_wait_time_too_long(self, elapsed_time):
        # type: (float) -> bool
        return self._error_log_time is not None and elapsed_time > self._error_log_time
    
    
    def _switch_to_mode(self, mode):
        # type: (bool) -> None
        self._mode_producer = mode
        self._cur_mode_consecutive_tasks = 0
    
    
    def _my_group_have_the_lock(self, mode):
        # type: (bool) -> bool
        return self._mode_producer == mode
    
    
    def _must_wait_to_acquire_lock(self, me, other, i_am_producer):
        # type: (OrdonnancementGroup, OrdonnancementGroup, bool) -> bool
        return not self._my_group_have_the_lock(i_am_producer) or (other.is_asking_for_lock() and other.is_asking_since_too_long() and me.have_working_task())
    
    
    def _wait_allowed_to_acquire_lock(self, me, other, i_am_producer):
        # type: (OrdonnancementGroup, OrdonnancementGroup, bool) -> None
        start_time = time.time()
        log_time = start_time
        
        while self._must_wait_to_acquire_lock(me, other, i_am_producer):
            # Maybe the other did stop, it's time to switch
            if not self._my_group_have_the_lock(i_am_producer) and other.is_idle():
                # my group take the lock
                self._switch_to_mode(i_am_producer)
                self.log_debug(u'%s acquiring the lock because %s is idle' % (me.name, other.name))
                return
            
            # Log if the wait is starting to be too long
            elapsed_time = abs(time.time() - log_time)  # abs: manage time get back
            if self._is_wait_time_too_long(elapsed_time):
                self._log_too_long_wait(elapsed_time, me, other, i_am_producer)
            
            me.go_wait(self._error_log_time)  # Maybe it won't be finish, and we raised the timeout
        self.log_debug(u'%s acquiring the lock' % me.name)
        # We can run now
    
    
    def _acquire_for_ordonnancement_group(self, me, other, i_am_producer):
        # type: (OrdonnancementGroup, OrdonnancementGroup, bool) -> None
        with me.cond:
            
            my_id = threading.current_thread().ident
            if my_id in self._active_task:
                self.logger.debug(u'_acquire_for_ordonnancement_group %s thread[%s] already own the lock' % (me.name, my_id))
                return
            
            # Wait until we can run
            self._wait_allowed_to_acquire_lock(me, other, i_am_producer)
            
            # Now we can run
            me.add_working_task()
            self._cur_mode_consecutive_tasks += 1
            
            self._active_task.update({my_id: 1})
    
    
    def _release_for_ordonnancement_group(self, me, other, i_am_producer):
        # type: (OrdonnancementGroup, OrdonnancementGroup, bool) -> None
        with me.cond:
            
            my_id = threading.current_thread().ident
            if my_id not in self._active_task:
                self.logger.debug(u'_release_for_ordonnancement_group %s thread[%s] already released the lock' % (me.name, my_id))
                return
            
            me.remove_working_task()
            if not me.have_working_task() and other.is_asking_for_lock():
                self._switch_to_mode(not i_am_producer)
                self.log_debug(u'SWITCH: %s is now allowed to run' % other.name)
                other.allow_to_run()
                # My own waiting tasks are still waiting until the other allow us to run too
            
            del (self._active_task[my_id])
    
    
    def producer_acquire(self):
        # type: () -> None
        self._acquire_for_ordonnancement_group(self._producer, self._consumer, True)
    
    
    def producer_release(self):
        # type: () -> None
        self._release_for_ordonnancement_group(self._producer, self._consumer, True)
    
    
    def consumer_acquire(self):
        # type: () -> None
        self._acquire_for_ordonnancement_group(self._consumer, self._producer, False)
    
    
    def consumer_release(self):
        # type: () -> None
        self._release_for_ordonnancement_group(self._consumer, self._producer, False)
