#!/usr/bin/python
# -*- coding: utf-8 -*-
#
# Copyright (C) 2013-2022
# This file is part of Shinken Enterprise, all rights reserved.

# ################ IMPORTANT #################
# THIS FILE IS LOADED IN APACHE WORKER PROCESS
# ############################################

import time
import os

from graphite.logger import log
from pymongo.errors import AutoReconnect


class InApacheMongodbInventoryReader(object):
    def __init__(self, elements_mapping, elements_mapping_lock, log_about_f, graphite_banner):
        self._elements_mapping = elements_mapping
        self._elements_mapping_lock = elements_mapping_lock
        self._log_about = log_about_f
        self._graphite_banner = graphite_banner
        
        self._most_recent_update = -2  # we get all inventory data, but each time we only ask newer
        
        self._mongodb_connection = None
        self._mongodb_database_name = u'shinken'
        self._mongodb_collection_name = u'metrology_inventory'
        self._mongodb_uri = u'mongodb://localhost/?w=1&fsync=false'
        self._mongodb_use_ssh = False
        self._mongodb_ssh_user = u'shinken'
        self._mongodb_ssh_keyfile = u'/opt/graphite/conf/id_rsa'
        self._mongodb_ssh_tunnel_timeout = 5
        self._mongodb_configuration_file = u'/opt/graphite/conf/mongodb.conf'
        
        self._mongodb_retry_nb = 5  # Number of tries
        self._mongodb_retry_delay = 1  # In sec
        
        self._fake_host_check = {u'uuid': u'__HOST__'}
        self._inventory_update_chapter = u'%s [ INVENTORY CACHE UPDATE ] [ MONGO ]' % self._graphite_banner
    
    
    def _get_mongodb_collection(self):
        # This function must be called with elements_mapping_lock taken (done by InApacheGraphiteMappingInventory.update_hosts_checks_mapping)
        if self._mongodb_connection is not None:
            try:
                self._mongodb_connection.admin.command(u'ping')
                return getattr(getattr(self._mongodb_connection, self._mongodb_database_name), self._mongodb_collection_name)
            except:
                log.info(u'%s Renewing MongoDB connection' % self._inventory_update_chapter)
            self._mongodb_connection = None
        from shinkensolutions.ssh_mongodb.sshtunnelmongomgr import mongo_by_ssh_mgr
        if os.path.isfile(self._mongodb_configuration_file):
            from shinken.compat import ConfigParser
            
            parser = ConfigParser.ConfigParser()
            parser.read(self._mongodb_configuration_file)
            
            self._mongodb_uri = parser.get('mongodb', 'URI')
            self._mongodb_database_name = parser.get('mongodb', 'DATABASE')
            # NOTE: these properties are only after 2.06.01, so can be missing in the file
            try:
                self._mongodb_use_ssh = (parser.get('mongodb', 'USE_SSH_TUNNEL') == '1')
            except ConfigParser.NoOptionError:
                pass
            try:
                self._mongodb_ssh_user = parser.get('mongodb', 'SSH_USER')
            except ConfigParser.NoOptionError:
                pass
            try:
                self._mongodb_ssh_keyfile = parser.get('mongodb', 'SSH_KEYFILE')
            except ConfigParser.NoOptionError:
                pass
            try:
                self._mongodb_ssh_tunnel_timeout = int(parser.get('mongodb', 'SSH_TUNNEL_TIMEOUT'))
            except ConfigParser.NoOptionError:
                pass
            
            try:
                self._mongodb_collection_name = parser.get('mongodb', 'COLLECTION')
            except ConfigParser.NoOptionError:
                self._mongodb_collection_name = u'metrology_inventory'
            
            log.info(u'%s Reading mongodb parameters %s from %s' % (self._inventory_update_chapter, parser.items('mongodb'), self._mongodb_configuration_file))
        
        mongo_by_ssh_mgr.disable_data_hub()
        con_result = mongo_by_ssh_mgr.get_connection(
            self._mongodb_uri,
            use_ssh=self._mongodb_use_ssh,
            ssh_keyfile=self._mongodb_ssh_keyfile,
            ssh_user=self._mongodb_ssh_user,
            ssh_tunnel_timeout=self._mongodb_ssh_tunnel_timeout,
            requestor=u'graphite',
        )
        self._mongodb_connection = con_result.get_connection()
        return getattr(getattr(self._mongodb_connection, self._mongodb_database_name), self._mongodb_collection_name)
    
    
    def _read_host_name(self, element):
        last_update = element[u'last_update']
        # log.info('MONITORING START TIME: %s %s' % (monitoring_start_time, MOST_RECENT_UPDATE))
        if last_update > self._most_recent_update:
            self._most_recent_update = last_update
            # log.info('%s Update host/check mapping:: updating most recent monitoring start to reduce query size to %s' % (graphite_banner, MOST_RECENT_UPDATE))
        return element[u'host_name'].replace(u'.', u'_').replace(u' ', u'_').replace(u'.', u'_')
    
    
    def _assert_host(self, host_name, host_uuid):
        host_entry = self._elements_mapping.get(host_name, None)
        if host_entry is None:  # New element?
            # NOTE: we always have the __HOST__ check entry, that do not exist in the SLA database, but in the graphite data
            host_entry = {
                u'uuid'  : host_uuid,
                u'checks': {
                    u'__HOST__': self._fake_host_check,
                },
            }
            self._elements_mapping[host_name] = host_entry
            self._log_about(host_name, u'%s Update host/check mapping:: new host detected: %s, uuid=%s' % (self._inventory_update_chapter, host_name, host_uuid))
        return host_entry
    
    
    # noinspection PyUnusedLocal
    
    def inventory_data_from_shinken_lookup(self, cache_invalidation_date):  # cache_invalidation_date unused here
        # This function must be called with elements_mapping_lock taken (done by InApacheGraphiteMappingInventory.update_hosts_checks_mapping)
        #
        # Failed to get inventory data from Shinken, fallback to MongoDB
        
        t0 = time.time()
        # Ok we can update now, let's grok objects since the MOST_RECENT_UPDATE
        # to do not dump all every time
        attempt = 1
        new_elements = []
        while attempt <= self._mongodb_retry_nb:
            try:
                col = self._get_mongodb_collection()
                new_elements = list(col.find({u'last_update': {u'$gt': self._most_recent_update}}))
                break
            except Exception as error:
                self._mongodb_connection = None
                log.info(u'%s Could not fetch data from MongoDB, attempt [%s/%s] ( %s )' % (self._inventory_update_chapter, attempt, self._mongodb_retry_nb, error))
                if attempt >= self._mongodb_retry_nb:
                    raise
            time.sleep(self._mongodb_retry_delay)
            attempt = attempt + 1
        
        log.info(u'%s Update host/check mapping:: got %d new elements from date %s (and in %3fs)' % (self._inventory_update_chapter, len(new_elements), self._most_recent_update, time.time() - t0))
        
        new_hosts = [element for element in new_elements if element[u'service_description'] == u'']
        new_checks = [element for element in new_elements if element[u'service_description'] != u'']
        
        # SLA INFO entries looks like (with unicode strings):
        # {'monitoring_start_time': 1527631200,
        #  'service_description': 'Disks Stats',
        #  '_id': 'd41ed7d463e011e88ecd080027f6d105-c29735965ad911e58cc5080027f08538',
        #  'host_name': 'serveur linux shinken',
        #  'check_interval': 5}
        t0 = time.time()
        # Beware: the hosts are all migrated, but some old checks can stay, so first we need to be sure about the hosts
        
        for element in new_hosts:
            host_name = self._read_host_name(element)
            host_uuid = element[u'_id']
            host_entry = self._assert_host(host_name, host_uuid)
            # Be sure to always update the uuid
            host_entry[u'uuid'] = host_uuid
        
        for element in new_checks:
            host_name = self._read_host_name(element)
            
            host_uuid = element[u'_id']
            service_description = element.get(u'service_description', u'').replace(u'.', u'_').replace(u' ', u'_').replace(u'.', u'_')
            
            host_uuid, service_uuid = host_uuid.split('-', 1)  # DO NOT TAKE host_uuid as maybe this is an old check
            # log.info('LOOKING AT HOST %s in cache: %s' % (host_name, ELEMENTS_MAPPING.keys()))
            host_entry = self._assert_host(host_name, host_uuid)
            if host_uuid != host_entry[u'uuid']:  # seems like an old check entry, do not use it
                self._log_about(host_name, u'%s Update host/check mapping:: Old check name is detected, skipping it: %s/%s %s %s' % (self._inventory_update_chapter, host_name, service_description, host_uuid, service_uuid))
                continue
            
            service_entry = host_entry[u'checks'].get(service_description, None)
            if service_entry is None:  # new service
                service_entry = {u'uuid': service_uuid}
            host_entry[u'checks'][service_description] = service_entry
            self._log_about(host_name, u'%s Update host/check mapping:: New check add: %s/%s, uuid=%s-%s' % (self._inventory_update_chapter, host_name, service_description, host_uuid, service_uuid))
        
        nb_hosts = len(self._elements_mapping)
        nb_checks = 0
        for host_entry in self._elements_mapping.values():  # PY3: NOT itervalues
            nb_checks += (len(host_entry[u'checks']) - 1)  # remove the __host__ entry
        log.info(u'%s Update host/check mapping:: Total number of hosts/checks in mapping cache from database: %s/%s' % (self._inventory_update_chapter, nb_hosts, nb_checks))
        log.info(u'%s Update host/check mapping:: Cache update took: %.3fs' % (self._inventory_update_chapter, time.time() - t0))
    
    
    def cache_was_reset(self):
        self._most_recent_update = -2
