#!/usr/bin/python
# -*- coding: utf-8 -*-
#
# Copyright (C) 2013-2022
# This file is part of Shinken Enterprise, all rights reserved.

import time

from shinkensolutions.ssh_mongodb.mongo_error import ShinkenMongoException


def retry_on_auto_reconnect(retry_connection_on_error=True):
    def decorator(func):
        def wrapper(self, *args, **kwargs):
            left_round = self._auto_reconnect_max_try
            collection_name = getattr(self, '_collection_name', '')
            if collection_name:
                operation_name = u'%s on %s' % (func.__name__, collection_name)
            else:
                operation_name = u'%s' % func.__name__
            while True:
                try:
                    return func(self, *args, **kwargs)
                except (ShinkenMongoException, StopIteration):
                    # White-listed exceptions, just re-raise them
                    raise
                except Exception as e:
                    if left_round <= 1:
                        error_msg = u'Mongo raised %s( %s ) on the operation %s. Operation failed : %s/%s. We tried %s times but it kept failing.' % (
                            type(e).__name__, e, operation_name, self._auto_reconnect_max_try - left_round + 1, self._auto_reconnect_max_try, self._auto_reconnect_max_try)
                        self.logger.error(error_msg)
                        raise ShinkenMongoException(error_msg)
                    else:
                        self.logger.info(u'Mongo raised %s( %s ) on the operation %s. Operation failed : %s/%s' % (type(e).__name__, e, operation_name, self._auto_reconnect_max_try - left_round + 1, self._auto_reconnect_max_try))
                    time.sleep(self._auto_reconnect_sleep_between_try)
                    if retry_connection_on_error:
                        self._retry_connection()
                    left_round -= 1
        
        
        return wrapper
    
    
    return decorator
