#!/usr/bin/env python
# -*- coding: utf-8 -*-
#
# Copyright (C) 2009-2022:
#     Gabes Jean, naparuba@gmail.com
#     Gerhard Lausser, Gerhard.Lausser@consol.de
#     Gregory Starck, g.starck@gmail.com
#     Hartmut Goebel, h.goebel@goebel-consult.de
#
# This file is part of Shinken.
#
# Shinken is free software: you can redistribute it and/or modify
# it under the terms of the GNU Affero General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# Shinken is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Affero General Public License for more details.
#
# You should have received a copy of the GNU Affero General Public License
# along with Shinken.  If not, see <http://www.gnu.org/licenses/>.

import math
import time
from collections import deque


# From advanced load average's code (another of my projects :) )
# def calc_load_load(load, exp,n):
#        load = n + exp*(load - n)
#        return (load, exp)

class Load:
    """This class is for having a easy Load calculation
    without having to send value at regular interval
    (but it's more efficient if you do this :) ) and without
    having a list or other stuff. It's just an object, an update and a get
    You can define m: the average for m minutes. The val is
    the initial value. It's better if it's 0 but you can choose.

    """
    
    
    def __init__(self, m=1, initial_value=0):
        self.exp = 0  # first exp
        self.m = m  # Number of minute of the avg
        self.last_update = 0  # last update of the value
        self.val = initial_value  # first value
    
    
    # 
    def update_load(self, new_val, forced_interval=None):
        # The first call do not change the value, just tag
        # the beginning of last_update
        # IF  we force : bail out all time thing
        if not forced_interval and self.last_update == 0:
            self.last_update = time.time()
            return
        now = time.time()
        try:
            if forced_interval:
                diff = forced_interval
            else:
                diff = now - self.last_update
            self.exp = 1 / math.exp(diff / (self.m * 60.0))
            self.val = new_val + self.exp * (self.val - new_val)
            self.last_update = now
        except OverflowError:  # if the time change without notice, we overflow :(
            pass
        except ZeroDivisionError:  # do not care
            pass
    
    
    def get_load(self):
        return self.val


class AvgInRange:
    def __init__(self, t_range, initial_value=None):
        self.index = 0
        self.values = [None] * t_range
        self.t_range = t_range
        self.update_avg(initial_value)
    
    
    def update_avg(self, value):
        if value is None:
            return
        self.values[self.index] = (int(time.time()), value)  # save time too, as a loop can be longer than 1s
        self.index = (self.index + 1) % len(self.values)
    
    
    # Get average on the period, and only the period (exclude non period values)
    # with 2 modes:
    # flatten: [default] flatten the average on the time period (so on average manage X by sec)
    # not flatten: take a classic average, by the number of elements (so by on average manage X by loop turn)
    def get_avg(self, default_value=0, flatten=True):
        now = int(time.time())
        sum_val = 0
        
        # valid values must be no more than now - t_range (60s) because such are useless
        valid_values = (v for v in self.values if v is not None)
        valid_values = [(t, v) for (t, v) in valid_values if t >= now - self.t_range]
        
        # if no value, give default value
        if len(valid_values) == 0:
            return default_value
        
        # By default the time range will be the one defined
        range_size = self.t_range
        
        # If we have at least 2 values, we can compute a real range
        if len(valid_values) >= 2:
            # if flatten means flatten by the time period
            if flatten:
                lower_t = 9999999999999999
                higher_t = 0
                for (t, _) in valid_values:
                    if t < lower_t:
                        lower_t = t
                    if t > higher_t:
                        higher_t = t
                range_size = higher_t - lower_t
            else:  # not flatten means only take number of elements
                range_size = len(valid_values)
        
        if range_size == 0:
            return default_value
        
        for (t, v) in valid_values:
            sum_val += v
        avg = sum_val / float(range_size)
        
        return avg


class AvgForFixSizeCall:
    def __init__(self, size_limit=None, time_limit=None, initial_value=None):
        if not (size_limit or time_limit):
            raise Exception('you must set size_limit or time_limit or both')
        
        self.index = 0
        self.values = [None] * size_limit if size_limit else []
        self.size_limit = size_limit
        self.time_limit = time_limit
        self.update_avg(initial_value)
        self._last_clean = 0.0
    
    
    def update_avg(self, value):
        if value is None:
            return
        if self.size_limit:
            self.values[self.index] = (int(time.time()), value)
            self.index = (self.index + 1) % len(self.values)
        else:
            now = int(time.time())
            self.values.append((int(time.time()), value))
            # We will only clean once a minute, because clean each time is just too time consuming
            if now > self._last_clean + 60:
                self.values = [(t, v) for (t, v) in self.values if now - t < self.time_limit]
                self._last_clean = now
    
    
    def _get_valid_values(self, time_reference=None):
        now = time.time()
        valid_values = [v for v in self.values if v is not None]
        if time_reference:
            valid_values = [(t, v) for (t, v) in valid_values if t >= (now - time_reference)]
        return valid_values
    
    
    def get_avg(self, default_value=0, avg_on_time=True, with_range_size=True):
        
        # valid values must be no more than now - t_range (60s) because such are useless
        valid_values = self._get_valid_values(self.time_limit)
        
        range_size = 0
        if avg_on_time:
            if len(valid_values) >= 2:
                all_time = [t for (t, _) in valid_values]
                lower_t = min(all_time)
                higher_t = max(all_time)
                range_size = higher_t - lower_t
        else:
            range_size = len(valid_values)
        
        if range_size == 0:
            if with_range_size:
                return default_value, 0
            else:
                return default_value
        
        sum_val = sum((v for (_, v) in valid_values))
        avg = sum_val / float(range_size)
        
        if with_range_size:
            return avg, range_size
        else:
            return avg
    
    
    def get_sum_in_range(self, default_value=0, avg_on_time=True, with_range_size=True, time_limit_overload=None):
        
        if time_limit_overload is None:
            time_limit_overload = self.time_limit
        
        # valid values must be no more than now - t_range (60s) because such are useless
        valid_values = self._get_valid_values(time_limit_overload)
        
        range_size = 0
        if avg_on_time:
            if len(valid_values) >= 2:
                all_time = [t for (t, _) in valid_values]
                lower_t = min(all_time)
                higher_t = max(all_time)
                range_size = higher_t - lower_t
        else:
            range_size = len(valid_values)
        
        if range_size == 0:
            if with_range_size:
                return default_value, 0
            else:
                return default_value
        
        sum_val = sum((v for (_, v) in valid_values))
        
        if with_range_size:
            return sum_val, range_size
        else:
            return sum_val
    
    
    # this will return the time with the most value in X seconds
    def max_call_in_period(self, period=1.0):
        best_one_second_request = {'time': '', 'nb': 0}
        valid_values = self._get_valid_values(self.time_limit)
        if not valid_values or not valid_values[0] or len(valid_values) < 2:
            return best_one_second_request
        tmp = valid_values[0]
        counter = 1
        best_one_second_request = {'time': tmp[0], 'nb': counter}
        for value in valid_values:
            if value and abs(tmp[0] - value[0]) >= period:
                if counter >= best_one_second_request['nb']:
                    best_one_second_request = {'time': tmp[0], 'nb': counter}
                    tmp = value
                    counter = 0
            counter += 1
        return best_one_second_request
    
    
    def get_max_in_period(self):
        return max(self._get_valid_values(self.time_limit))
    
    
    def get_min_in_period(self):
        return min(self._get_valid_values(self.time_limit))


# This class can be use to make a top.
class TopList:
    def __init__(self, size, descending=True, get_value=lambda x: x, duration=0, get_time_entry=None, max_buffer_size=0):
        self.size = size
        self.descending = descending
        self.bound_value = None
        self.items = []
        self.get_value = get_value
        self.duration = duration
        self.get_time_entry = get_time_entry
        self.max_buffer_size = max_buffer_size if max_buffer_size else size * 3
    
    
    def get_items(self, nb_element=0):
        self._compute_list()
        
        if nb_element:
            return self.items[:nb_element]
        else:
            return self.items
    
    
    def _compute_list(self):
        if self.duration:
            self.items = [i for i in self.items if time.time() - self.get_time_entry(i) < self.duration]
        self.items = sorted(self.items, key=self.get_value, reverse=self.descending)
        self.items = self.items[:self.size]
    
    
    def add_item(self, item):
        self.items.append(item)
        if len(self.items) > self.max_buffer_size:
            self._compute_list()


class WindowsValue(object):
    def __init__(self, scale=1, values_ttl=300):
        # type: (int, int) -> None
        self.scale = scale
        self.values_ttl = values_ttl
        self.values = deque()
    
    
    def push_value(self, value):
        # type:(float) -> None
        now = int(time.time())
        at = now - now % self.scale
        last_values = self.values[-1] if self.values else [None]
        if last_values[0] == at:
            last_values[1] += value
        else:
            self.values.append([at, value])
        self.clean_values(now)
    
    
    def get_sum_value(self, duration=None):
        # type:(int) -> float
        if duration is None:
            duration = self.values_ttl
        sum_value = 0
        now = int(time.time())
        min_value = now - duration
        for v in self.values:
            if v[0] >= min_value:
                sum_value += v[1]
        self.clean_values(now)
        return sum_value
    
    
    def clean_values(self, now):
        # type:(int) -> None
        if not self.values:
            return
        min_value = now - self.values_ttl
        v = self.values[0]
        while v[0] < min_value:
            self.values.pop()
            if not self.values:
                return
            v = self.values[0]


if __name__ == '__main__':
    average = AvgInRange(10, 0)
    t = time.time()
    for i in xrange(1, 300):
        average.update_avg(1)
        print '[', int(time.time() - t), ']', average.get_avg()
        time.sleep(1)
    load = Load()
    t = time.time()
    for i in xrange(1, 300):
        load.update_load(1)
        print '[', int(time.time() - t), ']', load.get_load(), load.exp
        time.sleep(5)
