#!/usr/bin/python

# -*- coding: utf-8 -*-

# Copyright (C) 2009-2012:
#    Gabes Jean, naparuba@gmail.com
#    Gerhard Lausser, Gerhard.Lausser@consol.de
#    Gregory Starck, g.starck@gmail.com
#    Hartmut Goebel, h.goebel@goebel-consult.de
#
# This file is part of Shinken.
#
# Shinken is free software: you can redistribute it and/or modify
# it under the terms of the GNU Affero General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# Shinken is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Affero General Public License for more details.
#
# You should have received a copy of the GNU Affero General Public License
# along with Shinken.  If not, see <http://www.gnu.org/licenses/>.

"""
Helper functions for some filtering, like for user based
"""
import re
import threading
import time
from weakref import proxy as weak_ref, ProxyType as WeakRef

from shinken.log import logger
from shinken.misc.type_hint import TYPE_CHECKING
from shinken.objects.host import VIEW_CONTACTS_DEFAULT_VALUE

if TYPE_CHECKING:
    from shinken.objects.host import Host
    from shinken.objects.contact import Contact
    from shinken.objects.service import Service
    from shinken.misc.type_hint import Union, List, Dict
    from shinken.misc.monitoring_item_manager.monitoring_item_manager import MonitoringItemManager

item_access = {}
user_items = {}
user_item_uuid = {}
count_items_by_user = {}


# Return true if user have access to element.
def check_user_item_access(user, item_uuid):
    if user.is_admin:
        return True
    
    if item_uuid is None:
        return False
    
    if '-' in item_uuid:
        item_uuid = item_uuid.split('-', 1)[0]
    
    users_with_access = item_access.get(item_uuid, set())
    if users_with_access == VIEW_CONTACTS_DEFAULT_VALUE.EVERYONE:
        return True
    
    return user.uuid in users_with_access


def clear_user_access_cache():
    global item_access, user_items, user_item_uuid, count_items_by_user
    item_access = {}
    user_items = {}
    user_item_uuid = {}
    count_items_by_user = {}


def init_user_item_access_cache(rg):
    global item_access, user_items, user_item_uuid
    
    _def_access = rg.get_default_property_value('host', 'view_contacts', missing=VIEW_CONTACTS_DEFAULT_VALUE.NOBODY)
    _item_access = {}
    user_items = {}
    user_item_uuid = {}
    
    for host in rg.hosts:
        if host.view_contacts:
            _item_access[host.uuid] = set([c.uuid for c in host.view_contacts])
        else:
            _item_access[host.uuid] = _def_access
    
    item_access = _item_access


def is_authorized(element, user, app):
    if user.is_admin:
        return True
    
    if user.uuid not in user_item_uuid:
        if hasattr(app, u'datamgr'):
            _compute_cache(user, app.datamgr)
        else:
            raise AttributeError(u'The app object [ class: %s ] does not have a datamanager [ attribute "datamgr" ]' % app.__class__.__name__)
    
    return element.uuid in user_item_uuid[user.uuid]


def get_all_my_visible_items(user, monitoring_item_manager):
    # type: (Contact, MonitoringItemManager) -> List
    if user.is_admin:
        return monitoring_item_manager.get_all_hosts_and_services()
    
    if user.uuid not in user_item_uuid:
        _compute_cache(user, monitoring_item_manager)
    
    return user_items[user.uuid]


def get_count_of_my_visible_items(user, monitoring_item_manager):
    # type: (Contact, MonitoringItemManager) -> Dict[unicode, int]
    if user.is_admin:
        return monitoring_item_manager.get_count_items()
    
    if user.uuid not in user_item_uuid:
        _compute_cache(user, monitoring_item_manager)
    
    return count_items_by_user[user.uuid]


def only_related_to(items, user, app):
    if user.is_admin:
        return items
    
    return [item for item in items if is_authorized(item, user, app)]


def _compute_cache(user, monitoring_item_manager):
    # type: (Contact, MonitoringItemManager) -> None
    global user_items, user_item_uuid
    
    if user and not isinstance(user, WeakRef):
        user = weak_ref(user)
    
    user_uuid = user.uuid
    if user_uuid not in count_items_by_user:
        count_items_by_user[user_uuid] = {
            u'all'     : 0,
            u'hosts'   : 0,
            u'clusters': 0,
            u'checks'  : 0,
        }
    
    start_time = time.time()
    filter_item = []
    filter_item_uuid = set()
    for host in monitoring_item_manager.get_hosts():
        # Maybe the user is a direct contact
        if user in host.view_contacts:
            _add_host(user_uuid, host, filter_item, filter_item_uuid)
            continue
        
        # Maybe view_contacts is void, and we are allowed to set access to everyone
        if not host.view_contacts:
            _def_access = monitoring_item_manager.rg.get_default_property_value('host', 'view_contacts', missing=VIEW_CONTACTS_DEFAULT_VALUE.NOBODY)
            if _def_access == VIEW_CONTACTS_DEFAULT_VALUE.EVERYONE:  # ok match a void contacts and anyone accesses, access is granted
                _add_host(user_uuid, host, filter_item, filter_item_uuid)
                continue
    
    other_uuid_with_same_list = next((user_uuid for user_uuid, _user_items_uuids in user_item_uuid.iteritems() if filter_item_uuid == _user_items_uuids), None)
    if other_uuid_with_same_list:
        user_items[user_uuid] = user_items[other_uuid_with_same_list]
        user_item_uuid[user_uuid] = user_item_uuid[other_uuid_with_same_list]
    else:
        filter_item.sort(key=get_element_full_name)
        user_items[user_uuid] = filter_item
        user_item_uuid[user_uuid] = filter_item_uuid
    thread_id = '%s' % threading.current_thread().name
    logger.info('[ %s ] [ %s ] generated elements list cache for user in %.3fs, with %d elements' % (thread_id, user.contact_name, time.time() - start_time, len(filter_item)))


def _add_host(user_uuid, host, filter_item, filter_item_uuid):
    if not isinstance(host, WeakRef):
        host = weak_ref(host)
    
    count_items_for_one_user = count_items_by_user[user_uuid]
    count_items_for_one_user[u'all'] += 1
    count_items_for_one_user[u'clusters' if host.got_business_rule else u'hosts'] += 1
    filter_item.append(host)
    filter_item_uuid.add(host.uuid)
    for service in host.services:
        if not isinstance(service, WeakRef):
            service = weak_ref(service)
        filter_item.append(service)
        filter_item_uuid.add(service.uuid)
        count_items_for_one_user[u'checks'] += 1


def _natural_key(string_):
    natural_key = []
    for s in re.split(r'(\d+)', string_):
        if s.isdigit():
            natural_key.append(int(s))
        else:
            natural_key.append(s.lower())
    return natural_key


def _stringify(_input):
    return _input.encode('utf-8', 'ignore') if callable(getattr(_input, 'encode', None)) else str(_input)


def get_element_full_name(elt):
    # type: (Union[Service,Host]) -> List
    return _natural_key(_stringify(elt.get_full_name()))
