#!/usr/bin/python
# -*- coding: utf-8 -*-

# Copyright (C) 2009-2012:
#    Gabes Jean, naparuba@gmail.com
#    Gerhard Lausser, Gerhard.Lausser@consol.de
#    Gregory Starck, g.starck@gmail.com
#    Hartmut Goebel, h.goebel@goebel-consult.de
#
# This file is part of Shinken.
#
# Shinken is free software: you can redistribute it and/or modify
# it under the terms of the GNU Affero General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# Shinken is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Affero General Public License for more details.
#
# You should have received a copy of the GNU Affero General Public License
# along with Shinken.  If not, see <http://www.gnu.org/licenses/>.


# Calendar date
# -------------
#  '(\d{4})-(\d{2})-(\d{2}) - (\d{4})-(\d{2})-(\d{2}) / (\d+) ([0-9:, -]+)'
#   => len = 8  => CALENDAR_DATE
#
#  '(\d{4})-(\d{2})-(\d{2}) / (\d+) ([0-9:, -]+)'
#   => len = 5 => CALENDAR_DATE
#
#  '(\d{4})-(\d{2})-(\d{2}) - (\d{4})-(\d{2})-(\d{2}) ([0-9:, -]+)'
#   => len = 7 => CALENDAR_DATE
#
#  '(\d{4})-(\d{2})-(\d{2}) ([0-9:, -]+)'
#   => len = 4 => CALENDAR_DATE
#
# Month week day
# --------------
#  '([a-z]*) (\d+) ([a-z]*) - ([a-z]*) (\d+) ([a-z]*) / (\d+) ([0-9:, -]+)'
#  => len = 8 => MONTH WEEK DAY
#  e.g.: wednesday 1 january - thursday 2 july / 3
#
#  '([a-z]*) (\d+) - ([a-z]*) (\d+) / (\d+) ([0-9:, -]+)' => len = 6
#  e.g.: february 1 - march 15 / 3 => MONTH DATE
#  e.g.: monday 2 - thusday 3 / 2 => WEEK DAY
#  e.g.: day 2 - day 6 / 3 => MONTH DAY
#
#  '([a-z]*) (\d+) - (\d+) / (\d+) ([0-9:, -]+)' => len = 6
#  e.g.: february 1 - 15 / 3 => MONTH DATE
#  e.g.: thursday 2 - 4 => WEEK DAY
#  e.g.: day 1 - 4 => MONTH DAY
#
#  '([a-z]*) (\d+) ([a-z]*) - ([a-z]*) (\d+) ([a-z]*) ([0-9:, -]+)' => len = 7
#  e.g.: wednesday 1 january - thursday 2 july => MONTH WEEK DAY
#
#  '([a-z]*) (\d+) - (\d+) ([0-9:, -]+)' => len = 7
#  e.g.: thursday 2 - 4 => WEEK DAY
#  e.g.: february 1 - 15 / 3 => MONTH DATE
#  e.g.: day 1 - 4 => MONTH DAY
#
#  '([a-z]*) (\d+) - ([a-z]*) (\d+) ([0-9:, -]+)' => len = 5
#  e.g.: february 1 - march 15  => MONTH DATE
#  e.g.: monday 2 - thusday 3  => WEEK DAY
#  e.g.: day 2 - day 6  => MONTH DAY
#
#  '([a-z]*) (\d+) ([0-9:, -]+)' => len = 3
#  e.g.: february 3 => MONTH DATE
#  e.g.: thursday 2 => WEEK DAY
#  e.g.: day 3 => MONTH DAY
#
#  '([a-z]*) (\d+) ([a-z]*) ([0-9:, -]+)' => len = 4
#  e.g.: thusday 3 february => MONTH WEEK DAY
#
#  '([a-z]*) ([0-9:, -]+)' => len = 6
#  e.g.: thusday => normal values
#
# Types: CALENDAR_DATE
#        MONTH WEEK DAY
#        WEEK DAY
#        MONTH DATE
#        MONTH DAY
#

import re
import time

from item import Item, Items
from shinken.brok import Brok
from shinken.daterange import Daterange, CalendarDaterange
from shinken.daterange import MonthDateDaterange, WeekDayDaterange
from shinken.daterange import MonthDayDaterange
from shinken.daterange import StandardDaterange, MonthWeekDayDaterange
from shinken.log import logger, naglog_result
from shinken.misc.configuration_error_log import log_configuration_error, log_configuration_warning
from shinken.misc.type_hint import TYPE_CHECKING
from shinken.property import IntegerProp, StringProp, ListProp, BoolProp
from shinkensolutions.time_period_parser import TimePeriodParser

if TYPE_CHECKING:
    from shinken.misc.type_hint import List, Dict

TIMEPERIOD_VERSION = 1
TIMEPERIOD_24X7 = set([u'monday 00:00-24:00', u'tuesday 00:00-24:00', u'friday 00:00-24:00', u'wednesday 00:00-24:00', u'thursday 00:00-24:00', u'sunday 00:00-24:00', u'saturday 00:00-24:00'])
THIRTY_DAY_AS_SECONDS = 3600 * 24 * 30
ONE_YEAR_AS_SECONDS = 3600 * 24 * 366 + 1


class Timeperiod(Item):
    id = 1
    my_type = 'timeperiod'
    
    properties = Item.properties.copy()
    properties.update({
        'timeperiod_name': StringProp(fill_brok=['full_status']),
        'alias'          : StringProp(default='', fill_brok=['full_status']),
        'use'            : StringProp(default=''),
        'register'       : IntegerProp(default='1'),
        
        # These are needed if a broker module calls methods on timeperiod objects
        'dateranges'     : ListProp(fill_brok=['full_status'], default=''),
        'exclude'        : ListProp(fill_brok=['full_status'], default='', merging='join'),
        'is_active'      : BoolProp(default='0')
    })
    running_properties = Item.running_properties.copy()
    
    
    def __init__(self, params=None, skip_useless_in_configuration=False):
        if params is None:
            params = {}
        self.id = Timeperiod.id
        Timeperiod.id = Timeperiod.id + 1
        self.unresolved = []
        self.dateranges = []  # type: List[Daterange]
        self.exclude = []  # type: List[Timeperiod]
        self.customs = {}
        self.plus = {}
        self.invalid_entries = []
        for key in params:
            # timeperiod objects are too complicated to support multi valued
            # attributes. we do as usual, last set value wins.
            if isinstance(params[key], list):
                if params[key]:
                    params[key] = params[key][-1]
                else:
                    params[key] = ''
            if key in ('pack', 'presence_protection'):
                continue
            if key in ['name', 'alias', 'timeperiod_name', 'exclude', 'use', 'register', 'imported_from', 'is_active', 'dateranges', 'sources', 'uuid', 'enabled']:
                setattr(self, key, params[key])
            elif key.startswith('_'):
                self.customs[key.upper()] = params[key]
            elif key == 'advanced_time_periods':
                self.unresolved.extend(TimePeriodParser.explode_advanced_time_periods_into_multiple_definitions(key, params))
            else:
                self.unresolved.extend(TimePeriodParser.explode_simple_time_period_into_multiple_definitions(key, params))
        
        self.cache = {}  # For tunning purpose only
        self.invalid_cache = {}  # same but for invalid search
        self._is_time_valid_cache = {}  # type: Dict[int, bool]
        
        self.configuration_errors = []
        self.configuration_warnings = []
        # By default, the tp is None, so we know we just start
        self.is_active = None
        self.tags = list()
        self.time_is_always_valid = False
    
    
    def get_name(self):
        return getattr(self, 'timeperiod_name', 'unknown_timeperiod')
    
    
    # We fillfull properties with template ones if need
    # for the unresolved values (like sunday ETCETC)
    def get_unresolved_properties_by_inheritance(self, items):
        # Ok, I do not have prop, Maybe my templates do?
        # Same story for plus
        for i in self.templates:
            self.unresolved.extend(i.unresolved)
    
    
    # Ok timeperiods are a bit different from classic items, because we do not have a real list
    # of our raw properties, like if we got february 1 - 15 / 3 for example
    def get_raw_import_values(self):
        properties = ['timeperiod_name', 'alias', 'use', 'register']
        r = {}
        for prop in properties:
            if hasattr(self, prop):
                v = getattr(self, prop)
                r[prop] = v
        # Now the unresolved one. The only way to get ride of same key things is to put
        # directly the full value as the key
        for other in self.unresolved:
            r[other] = ''
        return r
    
    
    def is_time_valid(self, time_to_check):
        time_to_check = int(time_to_check)  # be sure to have an int, because all daterange code is done for compare int and not float
        # note: float can be just a few ms AFTER the last second, as this last one is a INT
        
        if self.time_is_always_valid:
            return True
        
        res_from_cache = self.find_is_time_valid_from_cache(time_to_check)
        if res_from_cache is not None:
            return res_from_cache
        
        if self.has('exclude'):
            for tp in self.exclude:
                if tp.is_time_valid(time_to_check):
                    self._is_time_valid_cache[time_to_check] = False
                    return False
        
        for dr in self.dateranges:
            if dr.is_time_valid(time_to_check):
                self._is_time_valid_cache[time_to_check] = True
                return True
        
        self._is_time_valid_cache[time_to_check] = False
        return False
    
    
    # will give the first time > t which is valid
    def get_min_from_t(self, t):
        mins_incl = []
        for dr in self.dateranges:
            mins_incl.append(dr.get_min_from_t(t))
        return min(mins_incl)
    
    
    # will give the first time > t which is not valid
    def get_not_in_min_from_t(self, f):
        pass
    
    
    def find_next_valid_time_from_cache(self, t):
        return self.cache.get(t, None)
    
    
    def find_next_invalid_time_from_cache(self, t):
        return self.invalid_cache.get(t)
    
    
    def find_is_time_valid_from_cache(self, t):
        # type: (int) -> bool
        
        # SEE SEF-8941 : The cache is set in the __init__ method and called by the Arbiter. If the Arbiter version is older, the item haven't the cache, so ensure to get it here
        # This check can be removed in a future version, when we are sure that the arbiter has already been updated (02.09.00)
        if not hasattr(self, u'_is_time_valid_cache'):
            self._is_time_valid_cache = {}
        return self._is_time_valid_cache.get(t, None)
    
    
    # will look for active/un-active change. And log it
    # [1327392000] TIMEPERIOD TRANSITION: <name>;<from>;<to>
    # from is -1 on startup.  to is 1 if the timeperiod starts
    # and 0 if it ends.
    def check_and_log_activation_change(self):
        now = int(time.time())
        
        was_active = self.is_active
        self.is_active = self.is_time_valid(now)
        
        # If we got a change, log it!
        if self.is_active != was_active:
            _from = 0
            _to = 0
            # If it's the start, get a special value for was
            if was_active is None:
                _from = -1
            if was_active:
                _from = 1
            if self.is_active:
                _to = 1
            
            # Now raise the log
            naglog_result('info', 'TIMEPERIOD TRANSITION: %s;%d;%d' % (self.get_name(), _from, _to))
    
    
    # clean the get_next_valid_time_from_t cache
    # The entries are a dict on t. t < now are useless
    # Because we do not care about past anymore.
    # If not, it's not important, it's just a cache after all :)
    def clean_cache(self):
        now = int(time.time())
        
        for cache_name in (u'cache', u'invalid_cache', u'_is_time_valid_cache'):
            cache = getattr(self, cache_name, {})
            
            time_to_del = []
            for time_in_cache in cache:
                if time_in_cache < now:
                    time_to_del.append(time_in_cache)
            for time_in_cache in time_to_del:
                del cache[time_in_cache]
    
    
    def get_next_valid_time_from_t(self, t, max_t=ONE_YEAR_AS_SECONDS):
        # first find from cache
        t = int(t)
        original_t = t
        
        # logger.debug("[%s] Check valid time for %s" % ( self.get_name(), time.asctime(time.localtime(t)))
        if self.time_is_always_valid:
            return t
        
        res_from_cache = self.find_next_valid_time_from_cache(t)
        if res_from_cache is not None:
            return res_from_cache
        
        local_min = None
        
        first_valid_time_in_daterange = []
        
        for dr in self.dateranges:
            first_valid_time_in_daterange.append(dr.get_next_valid_time_from_t(t))
        
        sorted_first_valid_time_in_daterange = sorted([d for d in first_valid_time_in_daterange if d is not None])
        if sorted_first_valid_time_in_daterange:
            local_min = sorted_first_valid_time_in_daterange[0]
            if self.exclude:
                for exclude_timeperiod in self.exclude:
                    if exclude_timeperiod.is_time_valid(local_min):
                        max_lookup_time = max_t - (local_min - t)
                        if max_lookup_time <= 0:
                            local_min = None
                        exclude_end = exclude_timeperiod.get_next_invalid_time_from_t(local_min, max_period_to_search=max_lookup_time)
                        if exclude_end:
                            max_lookup_time = max_t - (exclude_end - t)
                            if max_lookup_time <= 0:
                                local_min = None
                            else:
                                local_min = self.get_next_valid_time_from_t(exclude_end, max_t=max_lookup_time)
                        else:
                            local_min = None
        
        if local_min and local_min > (original_t + max_t):
            # No loop more than one year
            local_min = None
        
        # We update the cache if the max_period_to_search is far in future because if the max_period_to_search if too small we will not find the real next invalid
        if max_t >= ONE_YEAR_AS_SECONDS:
            self.cache[original_t] = local_min
        return local_min
    
    
    def get_next_invalid_time_from_t(self, start_time, max_period_to_search=ONE_YEAR_AS_SECONDS):
        start_time = int(start_time)
        original_start_time = start_time
        still_loop = True
        
        if self.time_is_always_valid:
            return start_time + max_period_to_search
        
        # First try to find in cache
        res_from_cache = self.find_next_invalid_time_from_cache(start_time)
        if res_from_cache is not None:
            return res_from_cache
        
        # Then look, maybe start_time is already invalid
        if not self.is_time_valid(start_time):
            return start_time
        
        local_min = start_time
        time_padding = 60
        
        # Loop for all minutes...
        while still_loop:
            by_daterange_next_invalid_times = []
            
            # But maybe we can find a better solution with next invalid of standard date ranges
            for daterange in self.dateranges:
                by_daterange_next_invalid_time = daterange.get_next_invalid_time_from_t(local_min)
                if by_daterange_next_invalid_time is not None:
                    by_daterange_next_invalid_times.append(by_daterange_next_invalid_time)
            
            if by_daterange_next_invalid_times:
                local_min = min(by_daterange_next_invalid_times)
            
            # Now check if local_min is valid
            if not self.is_time_valid(local_min):
                # We can stop we found the next invalid because self is invalid
                still_loop = False
                break
            
            # Now check exclude. The next valid from exclude will be our next invalid
            for exclude_tp in self.exclude:
                if exclude_tp.is_time_valid(local_min):
                    # local_min is in valid period of an excluded, so we can stop
                    still_loop = False
                    
                    # We go back in time to found the start of exclude period
                    next_invalid_from_tp = exclude_tp.get_next_valid_time_from_t(local_min - time_padding, time_padding + 1)
                    if next_invalid_from_tp is not None and next_invalid_from_tp < local_min:
                        local_min = next_invalid_from_tp
                        if local_min < original_start_time:
                            local_min = original_start_time
                            
                            # After one month, go quicker...
            if local_min > original_start_time + THIRTY_DAY_AS_SECONDS:
                time_padding = 3600
            
            if still_loop:
                local_min += time_padding
            
            # after max_time_to_search, stop.
            if local_min > (original_start_time + max_period_to_search):
                still_loop = False
                local_min = None
        
        # We update the cache if the max_period_to_search is far in future because if the max_period_to_search if too small we will not found the real next invalid
        if max_period_to_search >= ONE_YEAR_AS_SECONDS:
            self.invalid_cache[original_start_time] = local_min
        return local_min
    
    
    def has(self, prop):
        return hasattr(self, prop)
    
    
    # We are correct only if our daterange are
    # and if we have no unmatch entries
    def is_correct(self):
        # type: () -> bool
        state = True
        
        for tp in self.exclude:
            if tp.get_uuid() == self.get_uuid():
                log_configuration_error(self, u'invalid exclude value, it cannot exclude itself')
                state = False
        
        for daterange in self.dateranges:
            if not daterange.is_correct():
                log_configuration_error(self, u'invalid daterange')
                state = False
        
        # Warn about non-correct entries
        for _entry in self.invalid_entries:
            log_configuration_warning(self, u'invalid entry : %s', _entry)
        
        if self.configuration_errors:
            for error in self.configuration_errors:
                log_configuration_error(self, error)
                state = False
        
        return state
    
    
    def __str__(self):
        s = ''
        s += str(self.__dict__) + '\n'
        for elt in self.dateranges:
            s += str(elt)
            (start, end) = elt.get_start_and_end_time()
            if (start, end) == Daterange.INVALID_PERIOD:
                s += "\nInvalid Period"
            else:
                start = time.asctime(time.localtime(start))
                end = time.asctime(time.localtime(end))
                s += "\nStart and end:" + str((start, end))
        s += '\nExclude'
        for elt in self.exclude:
            s += str(elt)
        
        return s
    
    
    def resolve_daterange(self, dateranges, entry):
        # print "Trying to resolve ", entry
        
        res = re.search(r'(\d{4})-(\d{2})-(\d{2}) - (\d{4})-(\d{2})-(\d{2}) / (\d+)[\s\t]*([0-9:, -]+)', entry)
        if res is not None:
            # print "Good catch 1"
            (syear, smon, smday, eyear, emon, emday, skip_interval, other) = res.groups()
            dateranges.append(CalendarDaterange(syear, smon, smday, 0, 0, eyear, emon, emday, 0, 0, skip_interval, other))
            return
        
        res = re.search(r'(\d{4})-(\d{2})-(\d{2}) / (\d+)[\s\t]*([0-9:, -]+)', entry)
        if res is not None:
            # print "Good catch 2"
            (syear, smon, smday, skip_interval, other) = res.groups()
            eyear = syear
            emon = smon
            emday = smday
            dateranges.append(CalendarDaterange(syear, smon, smday, 0, 0, eyear, emon, emday, 0, 0, skip_interval, other))
            return
        
        res = re.search(r'(\d{4})-(\d{2})-(\d{2}) - (\d{4})-(\d{2})-(\d{2})[\s\t]*([0-9:, -]+)', entry)
        if res is not None:
            # print "Good catch 3"
            (syear, smon, smday, eyear, emon, emday, other) = res.groups()
            dateranges.append(CalendarDaterange(syear, smon, smday, 0, 0, eyear, emon, emday, 0, 0, 0, other))
            return
        
        res = re.search(r'(\d{4})-(\d{2})-(\d{2})[\s\t]*([0-9:, -]+)', entry)
        if res is not None:
            # print "Good catch 4"
            (syear, smon, smday, other) = res.groups()
            eyear = syear
            emon = smon
            emday = smday
            dateranges.append(CalendarDaterange(syear, smon, smday, 0, 0, eyear, emon, emday, 0, 0, 0, other))
            return
        
        res = re.search(r'([a-z]*) ([\d-]+) ([a-z]*) - ([a-z]*) ([\d-]+) ([a-z]*) / (\d+)[\s\t]*([0-9:, -]+)', entry)
        if res is not None:
            # print "Good catch 5"
            (swday, swday_offset, smon, ewday, ewday_offset, emon, skip_interval, other) = res.groups()
            dateranges.append(MonthWeekDayDaterange(0, smon, 0, swday, swday_offset, 0, emon, 0, ewday, ewday_offset, skip_interval, other))
            return
        
        res = re.search(r'([a-z]*) ([\d-]+) - ([a-z]*) ([\d-]+) / (\d+)[\s\t]*([0-9:, -]+)', entry)
        if res is not None:
            # print "Good catch 6"
            (t0, smday, t1, emday, skip_interval, other) = res.groups()
            if t0 in Daterange.weekdays and t1 in Daterange.weekdays:
                swday = t0
                ewday = t1
                swday_offset = smday
                ewday_offset = emday
                dateranges.append(WeekDayDaterange(0, 0, 0, swday, swday_offset, 0, 0, 0, ewday, ewday_offset, skip_interval, other))
                return
            elif t0 in Daterange.months and t1 in Daterange.months:
                smon = t0
                emon = t1
                dateranges.append(MonthDateDaterange(0, smon, smday, 0, 0, 0, emon, emday, 0, 0, skip_interval, other))
                return
            elif t0 == 'day' and t1 == 'day':
                dateranges.append(MonthDayDaterange(0, 0, smday, 0, 0, 0, 0, emday, 0, 0, skip_interval, other))
                return
        
        res = re.search(r'([a-z]*) ([\d-]+) - ([\d-]+) / (\d+)[\s\t]*([0-9:, -]+)', entry)
        if res is not None:
            # print "Good catch 7"
            (t0, smday, emday, skip_interval, other) = res.groups()
            if t0 in Daterange.weekdays:
                swday = t0
                swday_offset = smday
                ewday = swday
                ewday_offset = emday
                dateranges.append(WeekDayDaterange(0, 0, 0, swday, swday_offset, 0, 0, 0, ewday, ewday_offset, skip_interval, other))
                return
            elif t0 in Daterange.months:
                smon = t0
                emon = smon
                dateranges.append(MonthDateDaterange(0, smon, smday, 0, 0, 0, emon, emday, 0, 0, skip_interval, other))
                return
            elif t0 == 'day':
                dateranges.append(MonthDayDaterange(0, 0, smday, 0, 0, 0, 0, emday, 0, 0, skip_interval, other))
                return
        
        res = re.search(r'([a-z]*) ([\d-]+) ([a-z]*) - ([a-z]*) ([\d-]+) ([a-z]*) [\s\t]*([0-9:, -]+)', entry)
        if res is not None:
            # print "Good catch 8"
            (swday, swday_offset, smon, ewday, ewday_offset, emon, other) = res.groups()
            # print "Debug:", (swday, swday_offset, smon, ewday, ewday_offset, emon, other)
            dateranges.append(MonthWeekDayDaterange(0, smon, 0, swday, swday_offset, 0, emon, 0, ewday, ewday_offset, 0, other))
            return
        
        res = re.search(r'([a-z]*) ([\d-]+) - ([\d-]+)[\s\t]*([0-9:, -]+)', entry)
        if res is not None:
            # print "Good catch 9"
            (t0, smday, emday, other) = res.groups()
            if t0 in Daterange.weekdays:
                swday = t0
                swday_offset = smday
                ewday = swday
                ewday_offset = emday
                dateranges.append(WeekDayDaterange(0, 0, 0, swday, swday_offset, 0, 0, 0, ewday, ewday_offset, 0, other))
                return
            elif t0 in Daterange.months:
                smon = t0
                emon = smon
                dateranges.append(MonthDateDaterange(0, smon, smday, 0, 0, 0, emon, emday, 0, 0, 0, other))
                return
            elif t0 == 'day':
                dateranges.append(MonthDayDaterange(0, 0, smday, 0, 0, 0, 0, emday, 0, 0, 0, other))
                return
        
        res = re.search(r'([a-z]*) ([\d-]+) - ([a-z]*) ([\d-]+)[\s\t]*([0-9:, -]+)', entry)
        if res is not None:
            # print "Good catch 10"
            (t0, smday, t1, emday, other) = res.groups()
            if t0 in Daterange.weekdays and t1 in Daterange.weekdays:
                swday = t0
                ewday = t1
                swday_offset = smday
                ewday_offset = emday
                dateranges.append(WeekDayDaterange(0, 0, 0, swday, swday_offset, 0, 0, 0, ewday, ewday_offset, 0, other))
                return
            elif t0 in Daterange.months and t1 in Daterange.months:
                smon = t0
                emon = t1
                dateranges.append(MonthDateDaterange(0, smon, smday, 0, 0, 0, emon, emday, 0, 0, 0, other))
                return
            elif t0 == 'day' and t1 == 'day':
                dateranges.append(MonthDayDaterange(0, 0, smday, 0, 0, 0, 0, emday, 0, 0, 0, other))
                return
        
        res = re.search(r'([a-z]*) ([\d-]+) ([a-z]*)[\s\t]*([0-9:, -]+)', entry)
        if res is not None:
            # print "Good catch 11"
            (t0, swday_offset, t1, other) = res.groups()
            if t0 in Daterange.weekdays and t1 in Daterange.months:
                swday = t0
                smon = t1
                emon = smon
                ewday = swday
                ewday_offset = swday_offset
                dateranges.append(MonthWeekDayDaterange(0, smon, 0, swday, swday_offset, 0, emon, 0, ewday, ewday_offset, 0, other))
                return
        
        res = re.search(r'([a-z]*) ([\d-]+)[\s\t]+([0-9:, -]+)', entry)
        if res is not None:
            # print "Good catch 12"
            (t0, smday, other) = res.groups()
            if t0 in Daterange.weekdays:
                swday = t0
                swday_offset = smday
                ewday = swday
                ewday_offset = swday_offset
                dateranges.append(WeekDayDaterange(0, 0, 0, swday, swday_offset, 0, 0, 0, ewday, ewday_offset, 0, other))
                return
            if t0 in Daterange.months:
                smon = t0
                emon = smon
                emday = smday
                dateranges.append(MonthDateDaterange(0, smon, smday, 0, 0, 0, emon, emday, 0, 0, 0, other))
                return
            if t0 == 'day':
                emday = smday
                dateranges.append(MonthDayDaterange(0, 0, smday, 0, 0, 0, 0, emday, 0, 0, 0, other))
                return
        
        res = re.search(r'([a-z]*)[\s\t]+([0-9:, -]+)', entry)
        if res is not None:
            # print "Good catch 13"
            (t0, other) = res.groups()
            if t0 in Daterange.weekdays:
                day = t0
                dateranges.append(StandardDaterange(day, other))
                return
        logger.info("[timeentry::%s] no match for %s" % (self.get_name(), entry))
        self.invalid_entries.append(entry)
    
    
    def apply_inheritance(self):
        pass
    
    
    # create daterange from unresolved param
    def explode(self, timeperiods):
        # SEF-10098: all elements have a time period, defaulting with 24x7
        # Adding a simple / fast test to manage elements using 24x7 like if they had no time period (in Scheduler)
        self.time_is_always_valid = not self.exclude and TIMEPERIOD_24X7 == set(self.unresolved)
        for entry in self.unresolved:
            # print "Revolving entry", entry
            self.resolve_daterange(self.dateranges, entry)
        self.unresolved = []
    
    
    # Will make tp in exclude with id of the timeperiods
    def linkify(self, timeperiods):
        new_exclude = []
        if self.has('exclude') and self.exclude != []:
            logger.debug("[timeentry::%s] have excluded %s" % (self.get_name(), self.exclude))
            excluded_tps = self.exclude.split(',')
            # print "I will exclude from:", excluded_tps
            for tp_name in excluded_tps:
                tp = timeperiods.find_by_name(tp_name.strip())
                if tp is not None:
                    new_exclude.append(tp)
                else:
                    logger.error("[timeentry::%s] unknown %s timeperiod" % (self.get_name(), tp_name))
        self.exclude = new_exclude
    
    
    def check_exclude_rec(self):
        if self.rec_tag:
            logger.error("[timeentry::%s] is in a loop in exclude parameter" % self.get_name())
            return False
        self.rec_tag = True
        for tp in self.exclude:
            tp.check_exclude_rec()
        return True
    
    
    def fill_data_brok_from(self, data, brok_type):
        cls = self.__class__
        # Now config properties
        for prop, entry in cls.properties.iteritems():
            # Is this property intended for broking?
            # if 'fill_brok' in entry:
            if brok_type in entry.fill_brok:
                if hasattr(self, prop):
                    data[prop] = getattr(self, prop)
                elif entry.has_default:
                    data[prop] = entry.default
    
    
    # Get a brok with initial status
    def get_initial_status_brok(self):
        cls = self.__class__
        my_type = cls.my_type
        data = {'id': self.id}
        
        self.fill_data_brok_from(data, 'full_status')
        b = Brok('initial_' + my_type + '_status', data)
        return b


class Timeperiods(Items):
    name_property = "timeperiod_name"
    inner_class = Timeperiod
    
    
    def explode(self):
        for id in self.items:
            tp = self.items[id]
            tp.explode(self)
    
    
    def linkify(self):
        for id in self.items:
            tp = self.items[id]
            tp.linkify(self)
    
    
    def apply_inheritance(self):
        # The only interesting property to inherit is exclude
        self.apply_partial_inheritance('exclude')
        for i in self:
            i.get_customs_properties_by_inheritance(self)
        
        # And now apply inheritance for unresolved properties
        # like the dateranges in fact
        for tp in self:
            tp.get_unresolved_properties_by_inheritance(self.items)
    
    
    # check for loop in definition
    def is_correct(self):
        # type: () -> bool
        r = True
        # We do not want a same hg to be explode again and again
        # so we tag it
        for tp in self.items.values():
            tp.rec_tag = False
        
        for tp in self.items.values():
            for tmp_tp in self.items.values():
                tmp_tp.rec_tag = False
            r &= tp.check_exclude_rec()
        
        # We clean the tags
        for tp in self.items.values():
            del tp.rec_tag
        
        # And check all timeperiods for correct (sunday is false)
        for tp in self:
            r &= tp.is_correct()
        return r


if __name__ == '__main__':
    t = Timeperiod()
    test = ['1999-01-28  00:00-24:00',
            'monday 3                   00:00-24:00             ',
            'day 2                      00:00-24:00',
            'february 10                00:00-24:00',
            'february -1 00:00-24:00',
            'friday -2                  00:00-24:00',
            'thursday -1 november 00:00-24:00',
            '2007-01-01 - 2008-02-01    00:00-24:00',
            'monday 3 - thursday 4      00:00-24:00',
            'day 1 - 15         00:00-24:00',
            'day 20 - -1                00:00-24:00',
            'july -10 - -1              00:00-24:00',
            'april 10 - may 15          00:00-24:00',
            'tuesday 1 april - friday 2 may 00:00-24:00',
            '2007-01-01 - 2008-02-01 / 3 00:00-24:00',
            '2008-04-01 / 7             00:00-24:00',
            'day 1 - 15 / 5             00:00-24:00',
            'july 10 - 15 / 2 00:00-24:00',
            'tuesday 1 april - friday 2 may / 6 00:00-24:00',
            'tuesday 1 october - friday 2 may / 6 00:00-24:00',
            'monday 3 - thursday 4 / 2 00:00-24:00',
            'monday 4 - thursday 3 / 2 00:00-24:00',
            'day -1 - 15 / 5            01:00-24:00,00:30-05:60',
            'tuesday 00:00-24:00',
            'sunday 00:00-24:00',
            'saturday 03:00-24:00,00:32-01:02',
            'wednesday 09:00-15:46,00:00-21:00',
            'may 7 - february 2 00:00-10:00',
            'day -1 - 5 00:00-10:00',
            'tuesday 1 february - friday 1 may 01:00-24:00,00:30-05:60',
            'december 2 - may -15               00:00-24:00',
            ]
    for entry in test:
        print("**********************")
        print(entry)
        t = Timeperiod()
        t.timeperiod_name = ''
        t.resolve_daterange(t.dateranges, entry)
        # t.exclude = []
        # t.resolve_daterange(t.exclude, 'monday 00:00-19:00')
        # t.check_valid_for_today()
        now = time.time()
        # print "Is valid NOW?", t.is_time_valid(now)
        t_next = t.get_next_valid_time_from_t(now + 5 * 60)
        if t_next is not None:
            print("Get next valid for now + 5 min ==>", time.asctime(time.localtime(t_next)), "<==")
        else:
            print("===> No future time!!!")
        # print "End date:", t.get_end_time()
        # print "Next valid", time.asctime(time.localtime(t.get_next_valid_time()))
        print(str(t) + '\n\n')
    
    print("*************************************************************")
    t3 = Timeperiod()
    t3.timeperiod_name = 't3'
    t3.resolve_daterange(t3.dateranges, 'day 1 - 10 10:30-15:00')
    t3.exclude = []
    
    t2 = Timeperiod()
    t2.timeperiod_name = 't2'
    t2.resolve_daterange(t2.dateranges, 'day 1 - 10 12:00-17:00')
    t2.exclude = [t3]
    
    t = Timeperiod()
    t.timeperiod_name = 't'
    t.resolve_daterange(t.dateranges, 'day 1 - 10 14:00-15:00')
    t.exclude = [t2]
    
    print("Mon T", str(t) + '\n\n')
    t_next = t.get_next_valid_time_from_t(now)
    t_no_next = t.get_next_invalid_time_from_t(now)
    print("Get next valid for now ==>", time.asctime(time.localtime(t_next)), "<==")
    print("Get next invalid for now ==>", time.asctime(time.localtime(t_no_next)), "<==")
