#!/usr/bin/python
# -*- coding: utf-8 -*-
#
# Copyright (C) 2013-2020
# This file is part of Shinken Enterprise, all rights reserved.

import operator
import threading
import time
import uuid
from collections import Iterable
from collections import OrderedDict

import shinkensolutions.shinkenjson as json
from shinken.log import LoggerFactory
from shinken.misc.type_hint import TYPE_CHECKING
from shinken.webui import bottlewebui as bottle
from shinken.webui.bottlewebui import request, response, parse_auth, HTTPResponse
from shinken.webui.cherrypybackend import CherryPyServerHTTP
from shinkensolutions.api.synchronizer import ITEM_STATE, ITEM_TYPE, NOT_TO_LOOK, DEF_ITEMS, METADATA, DataProviderMongo, get_type_item_from_class, get_name_from_type
from shinkensolutions.api.synchronizer.source.abstract_module.listener_module import ListenerModule
from shinkensolutions.system_tools import get_linux_local_addresses

try:
    from shinken.synchronizer.component.component_manager_synchronizer import ComponentManagerSynchronizer
except ImportError:
    from synchronizer.component.component_manager_synchronizer import ComponentManagerSynchronizer

if TYPE_CHECKING:
    from shinken.misc.type_hint import Optional, Dict, Any, NoReturn


# This module will open an HTTP service, where a user can send an host (for now)
class BaseRESTListener(ListenerModule):
    url_base_path = '/shinken/listener-rest'
    
    _configuration_fields = None
    
    
    def __init__(self, modconf):
        super(BaseRESTListener, self).__init__(modconf)
        self.srv = None
        self.srv_lock = threading.RLock()
        self.provider_mongo = None
        self.thread = None
        self.serveropts = {}
        self.logger = LoggerFactory.get_logger().get_sub_part(self.get_name()[7:])
        self.logger_creation = self.logger.get_sub_part(u'CREATION REQUEST')
        self.logger_update = self.logger.get_sub_part(u'UPDATE REQUEST')
        self.logger_delete = self.logger.get_sub_part(u'DELETE REQUEST')
        self.logger_get = self.logger.get_sub_part(u'GET REQUEST')
        self.logger_list = self.logger.get_sub_part(u'LIST REQUEST')
        self.logger_listening = self.logger.get_sub_part(u'LISTENING')
        self.logger_init = self.logger.get_sub_part(u'INITIALISATION')
        self.logger_authentication = self.logger.get_sub_part(u'AUTHENTICATION')
        
        try:
            self.logger_init.debug(u'Configuration starting ...')
            self.port = int(getattr(modconf, u'port', u'7761'))
            self.host = getattr(modconf, u'host', u'0.0.0.0')
            self.module_name = getattr(modconf, u'module_name', u'module-listener-rest')
            if self.module_name.startswith(u'module-'):
                self.module_name = self.module_name[7:]
            self.logger_init.debug(u'Configuration done, the configuration summary will be logged as soon as the source_controller is initialized, and the module can access its stocked configuration in mongo')
        except AttributeError:
            if modconf.imported_from != u'internal_source':
                self.logger_init.error(u'The module is missing a property, check module declaration in %s' % modconf.imported_from)
            else:
                self.logger_init.error(u'The module is missing a property, please contact Shinken Solutions support')
            
            raise
        except Exception, e:
            self.logger_init.error(u'Exception : %s' % unicode(e))
            raise
        
        self.hosts_lock = threading.RLock()
    
    
    def source_start(self, source_is_enabled):
        # type: (bool) -> None
        if not self._is_auth_configuration_valid():
            self.logger_authentication.warning(u'Authentication is activated, but username or password parameters are not defined. Please check their values or deactivate authentication.')
        self.logger_init.info(u'Configuration is OK :')
        self.log_configuration(self.logger_init)
        listening_message = u'The source is enabled, it will start listen.' if source_is_enabled else u'The source is disabled, it will not listen until it is enabled.'
        
        self.logger_init.info(listening_message)
        
        if source_is_enabled:
            self.start_listener()
    
    
    def _is_auth_configuration_valid(self):
        # type: () -> bool
        conf = self.get_my_configuration()
        if not conf.get('configuration', {}).get('authentication'):
            # This means the authentication is not activated, and it's ok
            return True
        auth_conf = conf.get('configuration', {})
        username = auth_conf.get('login', '')
        password = auth_conf.get('password', '')
        # If the authentication is ON, both username and password needs to be activated
        return username and password
    
    
    def log_configuration(self, logger=None):
        if logger is None:
            logger = self.logger.get_sub_part(u'CONFIGURATION')
        conf = self.get_my_configuration().get(u'configuration', {})
        logger.info(u'   - Host --------------------------------------- : %s' % self.host)
        logger.info(u'   - Port --------------------------------------- : %s' % self.port)
        logger.info(u'   - Authentication required ( login/password ) - : %s' % (u'Yes' if conf.get(u'authentication') else u'No'))
        logger.info(u'      - login ----------------------------------- : %s' % (u'********' if conf.get(u'login') else u''))
        logger.info(u'      - password -------------------------------- : %s' % (u'********' if conf.get(u'password') else u''))
        logger.info(u'   - SSL required ( HTTPS ) --------------------- : %s' % (u'Yes' if conf.get(u'use_ssl') else u'No'))
        logger.info(u'      - ssl_cert -------------------------------- : %s' % conf.get(u'ssl_cert'))
        logger.info(u'      - ssl_key --------------------------------- : %s' % conf.get(u'ssl_key'))
        
        listen_on_all_interfaces = self.host == u'0.0.0.0'
        
        url = u'%(protocol)s://%(host)s:%(port)s/shinken/%(source_name)s/v%(listener_version)s/hosts' % {
            u'protocol'        : u'https' if conf.get(u'use_ssl') else u'http',
            u'host'            : u'SYNCHRONIZER_IP' if listen_on_all_interfaces else self.host,
            u'port'            : self.port,
            u'source_name'     : self.get_name(),  # The source_name is hardcoded in self.url_base_path, must be self.get_name() in future
            u'listener_version': u'1',  # the listener version is hardcoded in _init_routes :(
        }
        logger.info(u'The listener URL is : %s' % url)
        if listen_on_all_interfaces:
            all_interfaces = get_linux_local_addresses(logger)
            logger.info(u'The source listen an all interfaces, SYNCHRONIZER_IP can be one of these address found on this system :')
            for interface in all_interfaces:
                logger.info(u'   - %s' % interface)
            logger.info(u'   - 127.0.0.1')  # The local interface is not in the list of address but it can be called when listen on 0.0.0.0
    
    
    def get_configuration_fields(self):
        if self._configuration_fields is None:
            self._configuration_fields = OrderedDict([
                ('configuration', OrderedDict([
                    ('authentication', {
                        'display_name': self._('analyzer.conf_authentication'),
                        'default'     : False,
                        'protected'   : False,
                        'help'        : '',
                        'type'        : 'checkbox',
                        'display_bind': ('login', 'password')
                    }),
                    ('login', {
                        'display_name': self._('analyzer.conf_login'),
                        'default'     : '',
                        'protected'   : False,
                        'help'        : '',
                        'type'        : 'text',
                    }),
                    ('password', {
                        'display_name': self._('analyzer.conf_password'),
                        'default'     : '',
                        'protected'   : True,
                        'help'        : '',
                        'type'        : 'text',
                    }),
                    ('use_ssl', {
                        'display_name': self._('analyzer.conf_use_ssl'),
                        'default'     : False,
                        'protected'   : False,
                        'help'        : '',
                        'type'        : 'checkbox',
                        'display_bind': ('ssl_key', 'ssl_cert')
                    }),
                    ('ssl_key', {
                        'display_name': self._('analyzer.conf_ssl_key'),
                        'default'     : '',
                        'protected'   : False,
                        'help'        : '',
                        'type'        : 'text',
                    }),
                    ('ssl_cert', {
                        'display_name': self._('analyzer.conf_ssl_cert'),
                        'default'     : '',
                        'protected'   : False,
                        'help'        : '',
                        'type'        : 'text',
                    }),
                
                ])
                 ),
            ])
        return self._configuration_fields
    
    
    def _init_http(self):
        if self.srv:
            # we need to stop it to restart it right after that that’s why we don’t launch it in a thread
            self.srv.stop()
        try:
            conf = self.get_my_configuration()
        except Exception:
            self.logger.print_stack()
            raise
        
        ssl_conf = conf.get('configuration', {})
        use_ssl = ssl_conf.get('use_ssl', '')
        ssl_key = ssl_conf.get('ssl_key', '')
        ssl_cert = ssl_conf.get('ssl_cert', '')
        try:
            # instantiate a new Bottle object, don't use the default one otherwise all module will share the same
            app = bottle.Bottle()
            app = self._init_routes(app)
            self.srv = app.run(host=self.host, port=self.port, server=CherryPyServerHTTP, use_ssl=use_ssl, ssl_key=ssl_key, ssl_cert=ssl_cert, quiet=True, **self.serveropts)
        except Exception, e:
            self.logger.error(u'Exception : %s' % unicode(e))
            raise
        self.logger.debug(u'Server loaded')
    
    
    def _init_routes(self, app):
        
        def base_rest_items():
            response.content_type = 'application/json'
            self._query_check_auth()
            if request.method == 'PUT':
                return self.create_item()
            elif request.method == 'GET':
                return self.list_items()
        
        
        def specific_rest_item(item_uuid):
            response.content_type = 'application/json'
            self._query_check_auth(item_uuid)
            import_needed = bottle.request.GET.get('import_needed', True)
            if request.method == 'DELETE':
                return self.delete_item(item_uuid, import_needed)
            elif request.method == 'GET':
                return self.get_item(item_uuid)
            
            elif request.method in ('POST', 'PUT', 'PATCH'):
                return self.update_item(item_uuid)
        
        
        app.route('%s/v1/hosts/' % self.url_base_path, callback=base_rest_items, method=('PUT', 'GET',))
        app.route('%s/v1/hosts' % self.url_base_path, callback=base_rest_items, method=('PUT', 'GET'))
        app.route('%s/v1/hosts/:item_uuid' % self.url_base_path, callback=specific_rest_item, method=('DELETE', 'GET', 'POST', 'PUT', 'PATCH'))
        return app
    
    
    def _init_data_provider_mongo(self):
        # type: () -> None
        mongo_component = ComponentManagerSynchronizer.get_mongo_component()
        self.provider_mongo = DataProviderMongo(mongo_component, self.syncdaemon.database_cipher)
    
    
    def get_dataprovider(self):
        # type: () -> DataProviderMongo
        if self.provider_mongo is None:
            self._init_data_provider_mongo()
        return self.provider_mongo
    
    
    def start_listener(self, reason=None):
        # type: (Optional[unicode]) -> None
        self.get_dataprovider()
        # We must protect against a user that spam the enable/disable button
        with self.srv_lock:
            # We already did start, skip it
            if self.thread is not None:
                if self.thread.is_alive():
                    return
                
                self.thread.join()
                self.thread = None
            
            self._init_http()
            self._start_thread(reason)
    
    
    def stop_listener(self, reason=None):
        # type: (Optional[unicode]) -> None
        with self.srv_lock:
            if self.thread is None:
                return
            self.srv.stop()
            self.thread.join(1)
            self.thread = None
            self.logger_listening.info(u'Closing connections on %s:%s %s' % (self.host, self.port, reason if reason else u''))
    
    
    def _start_thread(self, reason):
        # type: (Optional[unicode]) -> NoReturn
        self.thread = threading.Thread(None, target=self._http_start, name='Listener-REST', args=(reason,))
        self.thread.daemon = True
        self.thread.start()
    
    
    def _http_start(self, reason=None):
        # type: (Optional[unicode]) -> NoReturn
        self.logger_listening.info(u'Opening connections on %s:%s %s' % (self.host, self.port, reason if reason else u''))
        self.srv.start()
    
    
    # WARNING: do not call it check_auth or it will be used for Configuration UI auth!
    def _query_check_auth(self, item_uuid=None):
        # type: (Optional[unicode]) -> None
        conf = self.get_my_configuration()
        if conf.get('configuration', {}).get('authentication'):
            auth_conf = conf.get('configuration', {})
            username = auth_conf.get('login', '')
            password = auth_conf.get('password', '')
            basic = parse_auth(request.environ.get('HTTP_AUTHORIZATION', ''))
            # Maybe the user not even ask for user/pass. If so, bail out
            if not basic:
                error_message = [u'Refuse unauthenticated request']
                if item_uuid:
                    error_message.append(u'for item se_uuid : "%s"' % item_uuid)
                error_message.append(self._get_request_informations())
                
                self.logger_authentication.error(u' '.join(error_message))
                raise HTTPResponse(self._(u'listener.error_auth_required'), 401)
            # Maybe he do not give the good credential?
            bad_user_name = basic[0] != username
            if bad_user_name or basic[1] != password:
                error_message = [u'Refuse request']
                if item_uuid:
                    error_message.append(u'for item se_uuid : "%s"' % item_uuid)
                error_message.append(u'with bad credentials')
                if bad_user_name:
                    error_message.append(u'( username used was "%s")' % basic[0])
                error_message.append(self._get_request_informations())
                self.logger_authentication.error(u' '.join(error_message))
                raise HTTPResponse(self._(u'listener.error_auth_denied'), 403)
    
    
    def _return_in_400(self, err, logger):
        logger.error(u'Bad request object received %s : %s' % (self._get_request_informations(), err))
        raise HTTPResponse(err, 400)
    
    
    @staticmethod
    def _disallow_duplicate_keys(ordered_pairs):
        d = {}
        duplicate = []
        for k, v in ordered_pairs:
            if k in d:
                duplicate.append(k)
            else:
                d[k] = v
        if duplicate:
            raise ValueError('Duplicate keys:%s' % ', '.join(duplicate))
        return d
    
    
    def _get_data(self, logger):
        # Getting lists of information for the commands
        data = request.body.readline()
        if not data:
            self._return_in_400(self._(u'listener.error_no_data'), logger)
        try:
            try:
                decoded_data = json.loads(data, object_pairs_hook=self._disallow_duplicate_keys)
            except:
                # In python 2.6 object_pairs_hook does not exist
                decoded_data = json.loads(data)
            if '' in decoded_data:
                del decoded_data['']
            for key, value in decoded_data.iteritems():
                if not isinstance(value, basestring):
                    if isinstance(value, Iterable):
                        decoded_data[key] = ','.join(value)
                    else:
                        decoded_data[key] = str(value)
            return decoded_data
        except Exception as exp:
            if 'Duplicate keys:' in exp.message:
                duplicate_keys = exp.message.split(':')[1]
                if len(duplicate_keys.split(',')) == 1:
                    return self._return_in_400(self._(u'listener.error_duplicate_key') % duplicate_keys, logger)
                else:
                    return self._return_in_400(self._(u'listener.error_duplicate_keys') % duplicate_keys, logger)
            self._return_in_400(self._(u'listener.error_bad_json') % exp, logger)
    
    
    def delete_item(self, item_uuid, import_needed=True):
        item_type = ITEM_TYPE.HOSTS
        
        item_found = self.get_dataprovider().find_items(item_type, item_state=ITEM_STATE.RAW_SOURCES, item_source=self.get_name(), where={u'_SE_UUID': item_uuid})
        if any(item_found):
            item_found = item_found[0]
        else:
            self.logger_delete.error(u'item_type : %s, se_uuid : "%s" => There is no %s with this se_uuid %s' % (item_type, item_uuid, item_type.capitalize(), self._get_request_informations()))
            raise HTTPResponse(self._(u'listener.error_id') % (self._(u'type.host').lower(), item_uuid), 404)
        
        item_name = get_name_from_type(item_type, item_found)
        
        self.logger_delete.info(u'item_type : %s, name "%s", se_uuid : "%s" => %s' % (item_type, item_name, item_uuid, self._get_request_informations()))
        with self.hosts_lock:
            self.get_dataprovider().delete_item(item_found, item_type, item_state=ITEM_STATE.RAW_SOURCES, item_source=self.get_name())
        response.status = 200
        self.callback_synchronizer_about_delete_elements(items_type=item_type, data={'_id': item_uuid}, import_needed=import_needed)
        self.logger_delete.info(u'item_type : %s, name "%s", se_uuid : "%s" => %s is DELETED' % (item_type, item_name, item_uuid, item_type.capitalize()))
        return 'done'
    
    
    def update_item(self, item_uuid):
        hosts = self.get_dataprovider().find_items(ITEM_TYPE.HOSTS, item_state=ITEM_STATE.RAW_SOURCES, item_source=self.get_name(), where={'_SE_UUID': item_uuid})
        new_data = self._get_data(self.logger_update)
        item_type = get_type_item_from_class(ITEM_TYPE.HOSTS, new_data)
        
        self.logger_update.info(u'item_type : %s, se_uuid : "%s" => %s' % (item_type, item_uuid, self._get_request_informations()))
        if item_type != ITEM_TYPE.HOSTS:
            return self._return_in_400(self._(u'listener.error_only_host_allowed') % item_type, self.logger_update)
        
        host = None  # type: Optional[Dict[unicode, Any]]
        if any(hosts):
            raw_source_host = hosts[0]
            name_key = DEF_ITEMS[item_type]['key_name']
            host = {
                name_key  : raw_source_host[name_key],
                '_SE_UUID': 'core-%s-%s' % (item_type, raw_source_host['_id']),
                '_id'     : raw_source_host['_id']
            }
        else:
            # if no host in listeners, try to fund it in work_area or stagging
            where = {'_id': item_uuid}
            if item_uuid.startswith('core'):
                where = {'_SE_UUID': item_uuid}
            # try to find an object with the same id in stagging or work_area
            stagging_hosts = self.get_dataprovider().find_merge_state_items(item_type, item_states=[ITEM_STATE.WORKING_AREA, ITEM_STATE.STAGGING], where=where)
            if any(stagging_hosts):
                # now re-use the object _SE_UUID or one from the stagging object
                stagging_host = stagging_hosts[0]
                name_key = DEF_ITEMS[item_type]['key_name']
                host = {
                    name_key  : stagging_host[name_key],
                    '_SE_UUID': 'core-%s-%s' % (item_type, stagging_host['_id']),
                    '_id'     : stagging_host['_id']
                }
        
        # make sure the update host have the same id as the old one
        if not host:
            self.logger_update.error(u'item_type : %s, se_uuid : "%s" => There is no %s with this se_uuid' % (item_type, item_uuid, item_type))
            raise HTTPResponse(self._(u'listener.error_se_uuid') % self._(u'type.host').lower(), item_uuid, 404)
        
        key_name = DEF_ITEMS[item_type]['key_name']
        new_name = new_data.get(key_name, None)
        if new_name:
            self.logger_update.info(u'item_type : %s, name : "%s", se_uuid : "%s" => the request will change the name to "%s"' % (item_type, get_name_from_type(item_type, host), item_uuid, new_name))
        
        # use the NOT_TO_LOOK as list of forbidden keys
        forbidden_keys = set(map(lambda key: key.upper().strip(), new_data.keys())).intersection(set(map(lambda name: name.upper().strip().decode('utf8', 'ignore'), NOT_TO_LOOK)))
        if forbidden_keys:
            return self._return_in_400(self._(u'listener.error_forbidden_keys') % ', '.join(forbidden_keys), self.logger_update)
        
        # add the '_id' and host_name in sent data so we can keep track of this object (in the sources history)
        sent_data = new_data.copy()
        sent_data['_id'] = item_uuid
        
        if not key_name in sent_data and key_name in host:
            sent_data[key_name] = host[key_name]
        elif key_name in sent_data and sent_data[key_name] != host[key_name]:
            # host name have been updated
            # we have to update the synk_keys
            sync_keys = list(host.get('_SYNC_KEYS', ()))
            # remove the old host name
            if host[key_name] in sync_keys:
                sync_keys.remove(host[key_name])
            if not sent_data[key_name] in sync_keys:
                sync_keys.append(sent_data[key_name])
            host['_SYNC_KEYS'] = map(operator.methodcaller('lower'), sync_keys)
        
        host.update(new_data)
        host['update_date'] = time.time()
        
        if not host.get('_SYNC_KEYS', None):
            host['_SYNC_KEYS'] = '%s,%s' % (host['_SE_UUID'], host[key_name])
        if isinstance(host.get('_SYNC_KEYS', None), unicode):
            host['_SYNC_KEYS'] = host['_SYNC_KEYS'].encode('ascii')
        with self.hosts_lock:
            ciphered_host = self.syncdaemon.database_cipher.cipher(host, item_type=item_type)
            self.get_dataprovider().save_item(ciphered_host, item_type=item_type, item_state=ITEM_STATE.RAW_SOURCES, item_source=self.get_name())
        response.status = 200
        self.callback_synchronizer_about_update_elements(items_type=item_type, data=sent_data)
        self.logger_update.info(u'item_type : %s, name : "%s", se_uuid : "%s" => %s is valid and UPDATED' % (item_type, get_name_from_type(item_type, host), item_uuid, item_type.capitalize()))
        return 'done'
    
    
    def get_item(self, item_uuid):
        item_type = ITEM_TYPE.HOSTS
        ciphered_items = self.get_dataprovider().find_items(item_type, item_state=ITEM_STATE.RAW_SOURCES, item_source=self.get_name(), where={'_SE_UUID': item_uuid})
        if not any(ciphered_items):
            self.logger_get.error(u'item_type : %s, se_uuid : "%s" => There is no %s with this se_uuid %s' % (item_type, item_uuid, item_type.capitalize(), self._get_request_informations()))
            raise HTTPResponse(self._(u'listener.error_se_uuid') % (self._(u'type.host').lower(), item_uuid), 404)
        
        item = ciphered_items[0]
        self.logger_get.info(u'item_type : %s, name : "%s", se_uuid : "%s" => %s' % (item_type, get_name_from_type(item_type, item), item_uuid, self._get_request_informations()))
        host = self.syncdaemon.database_cipher.uncipher(item, item_type=item_type)
        host.pop('@metadata', None)
        return json.dumps(host)
    
    
    def list_items(self):
        item_type = ITEM_TYPE.HOSTS
        self.logger_list.info(u'item_type : %s => %s' % (item_type, self._get_request_informations()))
        raw_hosts = self.get_dataprovider().find_items(ITEM_TYPE.HOSTS, item_state=ITEM_STATE.RAW_SOURCES, item_source=self.get_name())
        hosts = []
        for host in raw_hosts:
            self.syncdaemon.database_cipher.uncipher(host, item_type=ITEM_TYPE.HOSTS)
            host.pop('@metadata', None)
            hosts.append(host)
        
        return json.dumps(hosts)
    
    
    def create_item(self):
        in_creation = True
        item = self._get_data(self.logger_creation)
        sent_data = item.copy()
        item_type = get_type_item_from_class(ITEM_TYPE.HOSTS, sent_data)
        item_name = get_name_from_type(item_type, sent_data)
        key_for_name = DEF_ITEMS[item_type][u'key_name']
        
        self.logger_creation.info(u'item_type : %s, name : "%s" => %s' % (item_type, item_name, self._get_request_informations()))
        
        if item_type != ITEM_TYPE.HOSTS:
            return self._return_in_400(self._(u'listener.error_only_host_allowed') % item_type, self.logger_creation)
        
        if not item_name:
            return self._return_in_400(self._(u'listener.error_missing_name_field') % key_for_name, self.logger_creation)
        
        if not isinstance(item_name, basestring):
            return self._return_in_400(self._(u'listener.error_name_not_string') % key_for_name, self.logger_creation)
        
        # use the NOT_TO_LOOK as list of forbidden keys
        forbidden_keys = set(map(lambda key: key.upper().strip(), item.keys())).intersection(set((map(lambda name: name.upper().strip().decode(u'utf8', u'ignore'), NOT_TO_LOOK))))
        if forbidden_keys:
            return self._return_in_400(self._(u'listener.error_forbidden_keys') % u', '.join(forbidden_keys), self.logger_creation)
        
        duplicate_keys = []
        for try_key in [k[:-7] for k in item if k.endswith(u'[FORCE]')]:
            if item.get(try_key, u''):
                duplicate_keys.append(try_key)
        
        if duplicate_keys:
            if len(duplicate_keys) == 1:
                return self._return_in_400(self._(u'listener.error_duplicate_key') % u', '.join(duplicate_keys), self.logger_creation)
            else:
                return self._return_in_400(self._(u'listener.error_duplicate_keys') % u', '.join(duplicate_keys), self.logger_creation)
        
        # try to find if an already object exists with this name
        if not u'_id' in item:
            ciphered_item_in_listener = self.get_dataprovider().find_items(item_type, item_state=ITEM_STATE.RAW_SOURCES, item_source=self.get_name(), where={key_for_name: item_name})
            if any(ciphered_item_in_listener):
                in_creation = False
                # an old item have been found, update the item and return the id
                item_in_listener = self.syncdaemon.database_cipher.uncipher(ciphered_item_in_listener[0], item_type=item_type)
                self.logger_creation.info(u'item_type : %s, name : "%s" => %s already exists in listener with the same name ( se_uuid "%s" ). This request will UPDATE the item instead of creating it' % (
                    item_type, item_name, item_type.capitalize(), item_in_listener[u'_SE_UUID']))
                # remove the _SYNC_KEYS, in any case they will be override
                item_in_listener.pop(u'_SYNC_KEYS', None)
                key_to_remove = []
                for key in item_in_listener:
                    try_key = key[:-7] if key.endswith(u'[FORCE]') else u'%s[FORCE]' % key
                    if item.get(try_key, u''):
                        key_to_remove.append(key)
                for key in key_to_remove:
                    item_in_listener.pop(key)
                item_in_listener.update(item)
                item = item_in_listener
                # the _id is an Object, and mongo will refuse to save it again, cast it in unicode
                item[u'_id'] = unicode(item[u'_id'])
            # else try to find if an already object exists with this name in stagging and workarea
            else:
                # try to find an object with the same name in raw_source
                # for them just reuse their _id and/or _SE_UUID if available
                existing_items = self.get_dataprovider().find_merge_state_items(item_type, item_states=[ITEM_STATE.WORKING_AREA, ITEM_STATE.STAGGING], where={key_for_name: item_name})
                if any(existing_items):
                    existing_item = existing_items[0]
                    item[u'_SE_UUID'] = u'core-%s-%s' % (item_type, existing_item[u'_id'])
                    item[u'_id'] = existing_item[u'_id']
                    self.logger_creation.info(
                        u'item_type : %(item_type)s, name : "%(item_name)s" => %(pretty_item_type)s already exists in %(item_state)s with the same name ( se_uuid "%(item_se_uuid)s" ). This request will CREATE item in listener with same SE_UUID' %
                        {
                            u'item_type'       : item_type,
                            u'item_name'       : item_name,
                            u'pretty_item_type': item_type.capitalize(),
                            u'item_state'      : METADATA.get_metadata(existing_item, METADATA.STATE),
                            u'item_se_uuid'    : item[u'_SE_UUID']
                        })
        
        if '_SE_UUID' in item:
            item_se_uuid = item['_SE_UUID']
        else:
            _id = uuid.uuid4().hex
            item['_id'] = _id
            item_se_uuid = 'core-%s-%s' % (item_type, _id)
            item['_SE_UUID'] = item_se_uuid
        
        for (k, v) in item.iteritems():
            if not k.startswith('_'):
                continue
            if not isinstance(k, basestring):
                return self._return_in_400(self._(u'listener.error_data_keys_not_strings') % (type(k), k), self.logger_creation)
            if k != '_id' and k != k.upper():
                return self._return_in_400(self._(u'listener.error_data_keys_not_uppercase'), self.logger_creation)
            if not isinstance(v, basestring):
                return self._return_in_400(self._(u'listener.error_data_values_not_strings') % (type(v), k, v), self.logger_creation)
        
        address = item.get('address', '')
        if address and not isinstance(address, basestring):
            return self._return_in_400(self._(u'listener.error_address_not_string'), self.logger_creation)
        
        item['_SYNC_KEYS'] = '%s,%s' % (item_se_uuid.lower(), item_name.lower())
        item['update_date'] = time.time()
        # request.environ.get('HTTP_USER_AGENT', '') # if needed
        item['imported_from'] = '%s %s' % (self.get_my_source().get_name(), self._('listener.sent_from') % request.environ.get('REMOTE_ADDR', 'UNSET'))
        
        if isinstance(item['_SYNC_KEYS'], unicode):
            item['_SYNC_KEYS'] = item['_SYNC_KEYS'].encode('ascii')
        with self.hosts_lock:
            ciphered_host = self.syncdaemon.database_cipher.cipher(item, item_type=item_type)
            self.get_dataprovider().save_item(ciphered_host, item_type=item_type, item_state=ITEM_STATE.RAW_SOURCES, item_source=self.get_name())
        
        self.callback_synchronizer_about_new_elements(items_type=item_type, data=sent_data)
        
        self.logger_creation.info(u'item_type : %s, name : "%s" => %s is valid and %s' % (item_type, item_name, item_type.capitalize(), u'CREATED' if in_creation else u'UPDATED'))
        # HTTP code 201 is OK CREATED => REST COMPLIANCE
        response.status = 201
        return json.dumps(item_se_uuid)
    
    
    def get_all_discovery_elements(self):
        raw_objects = {'host': []}
        # Get a copy of the values as we don't know how much time they will be keep outside this code, and so outside the lock
        with self.hosts_lock:
            my_hosts = self.get_dataprovider().find_items(ITEM_TYPE.HOSTS, item_state=ITEM_STATE.RAW_SOURCES, item_source=self.get_name())
        
        # remove unnecessary property for object after merge and set them as metadata
        for host in my_hosts:
            update_date = host.pop('update_date', None)
            host = self.syncdaemon.database_cipher.uncipher(host, item_type=ITEM_TYPE.HOSTS)
            if update_date:
                METADATA.update_metadata(host, METADATA.UPDATE_DATE, int(update_date))
        
        output = self.syncdaemon._('listener.output_successful') % len(my_hosts)
        raw_objects['host'] = my_hosts
        res = {'state': 'OK', 'output': output, 'objects': raw_objects, 'errors': [], 'warnings': []}
        return res
    
    
    def remove_source_item(self, item_type, source_item):
        with self.hosts_lock:
            self.delete_item(source_item['_SE_UUID'], import_needed=False)
    
    
    @staticmethod
    def _get_request_informations():
        # type: () -> unicode
        if getattr(request, u'environ', None) is None:
            return u'( request from Configuration UI )'
        else:
            return u'( HTTP %s request from %s )' % (request.method, request.remote_addr)
