#!/usr/bin/python
# -*- coding: utf-8 -*-

# Copyright (C) 2013-2020:
# This file is part of Shinken Enterprise, all rights reserved

from shinken.basesubprocess import EventHandler
from shinken.log import PartLogger
from shinken.misc.type_hint import List, Optional

from shinken.basemodule import SOURCE_STATE
from shinkensolutions.api.synchronizer import SourceTranslatePart, ItemType, ComponentManagerSyncui
from shinkensolutions.api.synchronizer.http_lib_external.v01_00.html.component.option_select import OptionSelectComp
from shinkensolutions.api.synchronizer.http_lib_external.v01_00.html.component.select import SelectComp
from shinkensolutions.api.synchronizer.http_lib_external.v01_00.html.distributor.option_distributor import SELECTOR_CONST
from shinkensolutions.api.synchronizer.http_lib_external.v01_00.html.distributor.selector_distributor import instance_selector_distributor
from shinkensolutions.api.synchronizer.http_lib_external.v01_00.html.html_distributor import HtmlDistributor
from shinkensolutions.api.synchronizer.http_lib_external.v01_00.html.object.html.label import LabelHtml
from shinkensolutions.api.synchronizer.source.mapping.origin_to_source.mapper import MapperOriginToSource
from shinkensolutions.api.synchronizer.source.route.route import AbstractRoute
from shinkensolutions.api.synchronizer.source.tab.tab import AbstractTab
from shinkensolutions.api.synchronizer.source.source_info import SourceInfo


class TabMappingOriginToSource(AbstractTab):
    
    def __init__(self, logger, translator, source_name, mapper_origin_to_source, type_to_import, source_info):
        # type: (PartLogger,SourceTranslatePart, str, MapperOriginToSource,List[ItemType], SourceInfo ) -> None
        super(TabMappingOriginToSource, self).__init__(logger, translator, source_name, 'mapping_origin_to_source', translator.translate('mapping_origin_to_source.tab_display_name'), source_info)
        self.mapper_origin_to_source = mapper_origin_to_source
        self.type_to_import = type_to_import
        self.html_distributor = None  # type: Optional[HtmlDistributor]
    
    
    def controller(self):
        common_translator = ComponentManagerSyncui.get_translate_component().translator()
        
        mapping_rules = []
        for mapper_name, mapping_by_type in self.mapper_origin_to_source.get_mapping().iteritems():
            for item_type, item_mapping in mapping_by_type.iteritems():
                for origin_prop_name, mapping_rule in item_mapping.iteritems():
                    mapping_rules.append((mapping_rule.get_sort_key(origin_prop_name), item_type, origin_prop_name, mapping_rule))
        mapping_rules.sort(key=lambda x: x[0])
        
        have_item_class_col = len(self.type_to_import) != 1
        have_mappers_name_col = self.mapper_origin_to_source.have_multi_mapper()
        user_mapping_files = [user_mapping_file for user_mapping_file in self.mapper_origin_to_source.user_mapping_files if 'user_mapping' in user_mapping_file['path']]
        source_exclusive_file = [exclusive_file for exclusive_file in self.mapper_origin_to_source.user_mapping_files if exclusive_file not in user_mapping_files]
        return {
            '__shinken_lang__'     : self.translator.get_lang(),
            'user_mapping_files'   : user_mapping_files,
            'source_exclusive_file': source_exclusive_file,
            'files_configuration'  : self.mapper_origin_to_source.user_mapping_files,
            'validation_state'     : self.mapper_origin_to_source.validation_state.as_html(),
            'mapping_rules'        : mapping_rules,
            'have_item_class_col'  : have_item_class_col,
            'have_mappers_name_col': have_mappers_name_col,
            'source_name'          : self.source_name,
            'translate'            : self.translator.translate,
            'common_translator'    : common_translator,
            'html_distributor'     : self.init_html_object(),
            'tab_info'             : self
        }
    
    
    def init_html_object(self):
        # type: ()-> HtmlDistributor
        if not self.html_distributor:
            _to_return = HtmlDistributor(self.translator.translate)
            # BUTTONS
            _to_return.add_button('reload', LabelHtml('''<span class="shinicon-undo"></span>'''))
            # FILTERS
            _to_return.add_list_filter('mapper-name', type_filter='select-single', content=self._build_select_mapper_name())
            _to_return.add_list_filter('item-type', type_filter='select-single', content=instance_selector_distributor.find_element(SELECTOR_CONST.FILTER_ITEM_TYPES))
            _to_return.add_list_filter('source-key', placeholder='--')
            _to_return.add_list_filter('description', placeholder='--')
            _to_return.add_list_filter('shinken-key', placeholder='--')
            _to_return.add_list_filter('shinken-name', placeholder='--')
            _to_return.add_list_filter('define_by', type_filter='select-single', content=instance_selector_distributor.find_element(SELECTOR_CONST.FILTER_DEFINE_BY))
            _to_return.add_list_filter('counters', type_filter='select-single', content=instance_selector_distributor.find_element(SELECTOR_CONST.FILTER_COUNTERS))
            self.html_distributor = _to_return
        return self.html_distributor
    
    
    def _build_select_mapper_name(self):
        select_mapper_name_options = [OptionSelectComp('select-mapper-name', mapper_name, 'user') for mapper_name in self.mapper_origin_to_source.mappers_name]
        select_mapper_name_options.insert(0, OptionSelectComp('select-mapper-name', SELECTOR_CONST.EMPTY, 'user', label=self.translator.translate('common.%s' % SELECTOR_CONST.EMPTY)))
        select_mapper_name_options.insert(0, OptionSelectComp('select-mapper-name', SELECTOR_CONST.ALL, 'user', label=self.translator.translate('common.%s' % SELECTOR_CONST.ALL)))
        select_mapper_name = SelectComp(self.translator, options=select_mapper_name_options, _id='select-mapper-name')
        return select_mapper_name


class RouteReloadMappingOriginToSource(AbstractRoute):
    
    def __init__(self, logger, source_name, mapper_origin_to_source, mapper_event, source_info):
        # type: (PartLogger, str, MapperOriginToSource, EventHandler, SourceInfo) -> None
        super(RouteReloadMappingOriginToSource, self).__init__(logger, source_name, 'mapping_origin_to_source/reload', source_info, have_http_template=False)
        self.mapper_origin_to_source = mapper_origin_to_source
        self.mapper_event = mapper_event
    
    
    def controller(self):
        if not self.share_info.import_in_progressed:
            self.mapper_origin_to_source.reload_mapper()
            self.mapper_event.send_event()
            return SOURCE_STATE.OK
        else:
            return SOURCE_STATE.RUNNING
