#!/usr/bin/python
# -*- coding: utf-8 -*-
#
# Copyright (C) 2013-2023:
# This file is part of Shinken Enterprise, all rights reserved.

from shinken.misc.type_hint import TYPE_CHECKING
from shinken.util import to_bool
from shinkensolutions.component.abstract_component import AbstractComponent
from shinkensolutions.logs.create_user_log_file import CreateUserLogFile
from shinkensolutions.logs.logger_authentication import LoggerAuthentication
from shinkensolutions.logs.logger_session import LoggerSession

if TYPE_CHECKING:
    from typing import Optional


class LoggerLogUserComponent(AbstractComponent):
    
    KEY_LOG_USERS__ENABLED = u'log_users__enabled'
    KEY_LOG_USERS__FILE_PATH = u'log_users__file_path'
    KEY_LOG_USERS__ADD_USER_NAME = u'log_users__add_user_name'
    
    
    def __init__(self, requester_prefix, conf, logger_error=None):
        self.log_enable = to_bool(getattr(conf, u'%s__%s' % (requester_prefix, self.KEY_LOG_USERS__ENABLED), False))
        self.requester_prefix = requester_prefix
        self.conf = conf
        
        _default_path = u'/var/log/shinken/%s/log_users.log' % requester_prefix
        self.log_file = getattr(conf, u'%s__%s' % (requester_prefix, self.KEY_LOG_USERS__FILE_PATH), _default_path)
        if not self.log_file or not isinstance(self.log_file, basestring):
            self.log_file = _default_path
        
        self.add_user_name = to_bool(getattr(conf, u'%s__%s' % (requester_prefix, self.KEY_LOG_USERS__ADD_USER_NAME), False))
        # This logger are use for log errors of this logger,
        # classically use the default logger of the module or daemon
        self.logger_error = logger_error
        
        self.create_user_log_file = None  # type: Optional[CreateUserLogFile]
        self.log_users_session = None  # type: Optional[LoggerSession]
        self.log_users_authentication = None  # type: Optional[LoggerAuthentication]
    
    
    def init(self):  # type: () -> None
        self.create_user_log_file = CreateUserLogFile(self.requester_prefix, self.conf, self.logger_error)
        self.log_users_session = LoggerSession(self.requester_prefix, self.conf, self.create_user_log_file, self.logger_error)
        self.log_users_authentication = LoggerAuthentication(self.requester_prefix, self.conf, self.create_user_log_file, self.logger_error)
        self.conf = None
    
    
    def login(self, user_uuid, user_name, requester_ip, run_time, result_code=200, requester_module=None, auth_module=None):
        self.log_users_authentication.login(user_uuid, user_name, requester_ip, run_time, result_code, requester_module, auth_module)
    
    
    def login_failed(self, user_uuid, user_name, requester_ip, run_time, result_code=401, requester_module=None, auth_module=None, error=None):
        self.log_users_authentication.login_failed(user_uuid, user_name, requester_ip, run_time, result_code, requester_module, auth_module, error)
    
    
    def logout(self, user_uuid, user_name, requester_ip, run_time, result_code=200, requester_module=None):
        self.log_users_authentication.logout(user_uuid, user_name, requester_ip, run_time, result_code, requester_module)
    
    
    def start_session(self, user_uuid, user_name, requester_ip, requester_module=None):
        self.log_users_session.start_session(user_uuid, user_name, requester_ip, requester_module)
