#!/usr/bin/python
# -*- coding: utf-8 -*-
#
# Copyright (C) 2013-2022:
# This file is part of Shinken Enterprise, all rights reserved.

import os
import shutil
import sys

from shinken.misc.os_utils import check_file_or_directory_access, write_file_set_owner, ShinkenFileNotFoundError, ShinkenAccessRightsError, is_run_by_root, get_lock_file_name, get_cur_user_name
from shinken.misc.type_hint import TYPE_CHECKING
from shinkensolutions.data_hub.data_hub_driver.abstract_data_hub_driver import AbstractDataHubDriver, AbstractDataHubDriverConfig
from shinkensolutions.data_hub.data_hub_exception.data_hub_exception import DataHubItemNotFound
from shinkensolutions.locking.shinken_locking.shinken_interprocess_rlock import ShinkenInterProcessRLock
from shinkensolutions.os_helper import make_dirs_and_chown_shinken, set_shinken_owner_on_file_or_directory, verify_if_shinken_own_file_or_directory

if TYPE_CHECKING:
    from shinken.log import PartLogger
    from shinken.misc.type_hint import Any, List, Final

if sys.platform.startswith('win'):
    import tempfile
    
    
    class DATA_HUB_FILE_DEFAULT_DIRECTORY(object):
        DEV_SHM = os.path.join(tempfile.gettempdir(), u'dev', u'shm', u'shinken')  # type: Final
        SHINKEN_LIB_PERSISTENT_DATA = os.path.join(tempfile.gettempdir(), u'shinken', u'persistent_data')  # type: Final
else:
    class DATA_HUB_FILE_DEFAULT_DIRECTORY(object):
        DEV_SHM = os.path.join(os.sep, u'dev', u'shm', u'shinken' if is_run_by_root() or get_cur_user_name() == u'shinken' else u'shinken_%s' % get_cur_user_name())  # type: Final
        SHINKEN_LIB_PERSISTENT_DATA = os.path.join(os.sep, u'var', u'lib', u'shinken', u'persistent_data')  # type: Final


class AbstractDataHubDriverConfigFile(AbstractDataHubDriverConfig):
    def __init__(self, name, base_directory, namespace=u'', data_location_name=u'', daemon_name=u'', module_name=u'', submodule_name=u'', file_ext=u'', file_prefix=u'shinken_'):
        # type: (unicode, unicode, unicode, unicode, unicode, unicode, unicode, unicode, unicode) -> None
        super(AbstractDataHubDriverConfigFile, self).__init__(name, namespace)
        self.base_directory = base_directory
        self.daemon_name = daemon_name
        self.data_location_name = data_location_name
        self.module_name = module_name
        self.submodule_name = submodule_name
        self.file_ext = file_ext
        self.file_prefix = file_prefix


class AbstractDataHubDriverFile(AbstractDataHubDriver):
    
    def __init__(self, logger, driver_config):
        # type: (PartLogger, AbstractDataHubDriverConfigFile) -> None
        super(AbstractDataHubDriverFile, self).__init__(logger, driver_config)
        
        self._is_root = False
        self._base_directory = driver_config.base_directory
        self._daemon_name = driver_config.daemon_name
        self._data_location_name = driver_config.data_location_name
        self._module_name = driver_config.module_name
        self._submodule_name = driver_config.submodule_name
        self._file_prefix = driver_config.file_prefix
        self._file_ext = driver_config.file_ext
        self._data_folder = self._build_folder_path()
        
        self.rights_access_correctly_granted_on_directory = True
        self._locks = {}
    
    
    def init(self):
        # type: () -> None
        self._check_if_is_root()
        self._ensure_data_folder_exists(self._logger_init)
    
    
    def _get_non_lock_files(self, _logger):
        # type: (PartLogger) -> List[unicode]
        self._ensure_data_folder_exists(_logger)
        files = next(os.walk(self._data_folder), (None, None, []))[2]
        return [file_name for file_name in files if not file_name.endswith(u'.lock')]
    
    
    def get_number_of_stored_data(self):
        # type () -> int
        return len(self._get_non_lock_files(self._logger_reader))
    
    
    def get_all_data_id(self):
        # type: () -> List[unicode]
        all_data_id = []
        for file_name in self._get_non_lock_files(self._logger_reader):
            try:
                file_name_without_prefix = file_name.split(self._file_prefix)[1]
            except Exception:
                file_name_without_prefix = file_name
            all_data_id.append(file_name_without_prefix.split(u'.')[0])
        return all_data_id
    
    
    def _check_if_is_root(self):
        self._is_root = is_run_by_root()
    
    
    def _build_folder_path(self):
        # type: () -> unicode
        _directory = os.path.join(self._base_directory, self._namespace) if self._namespace else self._base_directory
        if self._daemon_name:
            _directory = os.path.join(_directory, u'daemons', self._daemon_name)
            if self._module_name:
                _directory = os.path.join(_directory, u'modules', self._module_name)
                if self._submodule_name:
                    _directory = os.path.join(_directory, u'modules', self._submodule_name)
        
        if self._data_location_name:
            _directory = os.path.join(_directory, self._data_location_name)
        
        return _directory
    
    
    def _ensure_data_folder_exists(self, logger, check_write=True):
        # type: (PartLogger, bool) -> None
        
        if os.path.exists(self._data_folder):
            check_file_or_directory_access(self._data_folder, logger, check_write=check_write)
        else:
            try:
                logger.info(u'The folder %s does not exists. Will create it' % self._data_folder)
                make_dirs_and_chown_shinken(self._data_folder)  # This will create dir recursively and give the own to shinken:shinken
            except OSError as e:
                # errno 17 means that the folder already exists, no need to create it, it just means that someone create it at the same time
                if e.errno == 17:
                    return
                logger.error(u'No write access rights on target: %s' % self._data_folder)
                logger.error('Received exception: %s' % str(e).decode(u'utf-8', u'ignore'))
                raise IOError(u'Got access issues for target: %s' % self._data_folder)
    
    
    def _make_file_path(self, data_id):
        # type: (unicode) -> unicode
        return os.path.join(self._data_folder, u'%s%s.%s' % (self._file_prefix, data_id, self._file_ext))
    
    
    def _check_path_before_read(self, data_id, logger):
        # type: (unicode, PartLogger) -> unicode
        self._ensure_data_folder_exists(logger, check_write=False)
        file_path = self._make_file_path(data_id)
        check_file_or_directory_access(file_path, logger=logger, check_write=False)
        return file_path
    
    
    def is_data_correct(self, data_id):
        # type: (unicode) -> bool
        try:
            file_path = self._check_path_before_read(data_id, self._logger_init)
            with open(file_path, 'r') as file_handler:
                serialized_data = file_handler.read()
            self._deserialized_data(serialized_data)
        except Exception:
            return False
        return True
    
    
    def _serialized_data(self, data):
        # type: (Any) -> str
        raise NotImplementedError()
    
    
    def _deserialized_data(self, serialized_data):
        # type: (str) -> Any
        raise NotImplementedError()
    
    
    def get_lock(self, data_id):
        # type: (unicode) -> Any
        self._ensure_data_folder_exists(self._logger_writer)
        if data_id not in self._locks:
            self._locks[data_id] = ShinkenInterProcessRLock(filename=get_lock_file_name(self._make_file_path(data_id)))
        return self._locks[data_id]
    
    
    def write(self, data_id, data):
        # type: (unicode, Any) -> None
        file_path = self._make_file_path(data_id)
        with self.get_lock(data_id):
            write_file_set_owner(file_path, self._serialized_data(data), set_owner=((data_id not in self.rights_access_per_data_id) or self._is_root))
    
    
    def read(self, data_id, log_error=True):
        # type: (unicode, bool) -> Any
        try:
            file_path = self._check_path_before_read(data_id, self._logger_reader)
            with self.get_lock(data_id):
                with open(file_path, 'rb') as file_handler:
                    serialized_data = file_handler.read()
                return self._deserialized_data(serialized_data)
        except ShinkenFileNotFoundError:
            raise DataHubItemNotFound(self._data_type, data_id)
        except Exception as e:
            raise DataHubItemNotFound(message=e.message)
    
    
    def remove(self, data_id):
        # type: (unicode) -> None
        file_path = self._make_file_path(data_id)
        if os.path.exists(file_path):
            os.remove(file_path)
    
    
    def hard_remove(self, data_id):
        # type: (unicode) -> None
        file_path = self._make_file_path(data_id)
        if os.path.exists(file_path):
            os.remove(file_path)
        lock_file_path = get_lock_file_name(file_path)
        if os.path.exists(lock_file_path):
            os.remove(lock_file_path)
    
    
    def get_last_modification_date(self, data_id):
        # type: (unicode) -> int
        try:
            file_path = self._check_path_before_read(data_id, self._logger_reader)
            file_time = int(os.path.getmtime(file_path))
        except ShinkenFileNotFoundError:
            raise DataHubItemNotFound(self._data_type, data_id)
        except Exception as e:
            raise DataHubItemNotFound(message=e.message)
        return file_time
    
    
    def destroy(self):
        # type: () -> None
        shutil.rmtree(self._data_folder)
    
    
    def stop(self):
        # type: () -> None
        pass
    
    
    def get_total_size(self):
        # type: () -> int
        total_size = 0
        for filename in self._get_non_lock_files(self._logger_reader):
            fp = os.path.join(self._data_folder, filename)
            # skip if it is symbolic link
            if not os.path.islink(fp):
                total_size += os.path.getsize(fp)
        
        return total_size
    
    
    def check_right_access_issues_then_fix_them(self):
        self.verify_path_and_right_access_on_all_data()
        self.fix_right_access_issues()
    
    
    def verify_path_and_right_access_on_all_data(self):
        # type: () -> None
        try:
            self._ensure_data_folder_exists(self._logger_init)
            access = verify_if_shinken_own_file_or_directory(self._data_folder)
            self.rights_access_correctly_granted_on_directory = access
            self.rights_access_correctly_granted = self.rights_access_correctly_granted and access
        except ShinkenAccessRightsError:
            self.rights_access_correctly_granted_on_directory = False
            self.rights_access_correctly_granted = False
            access = False
        
        for data_id in self.get_all_data_id():
            file_path = self._make_file_path(data_id)
            if os.path.exists(file_path):
                access = verify_if_shinken_own_file_or_directory(file_path)
            self.rights_access_correctly_granted = self.rights_access_correctly_granted and access
            self.rights_access_per_data_id[data_id] = access
    
    
    def fix_right_access_issues(self):
        if self.rights_access_correctly_granted:
            return
        
        if not self.rights_access_correctly_granted_on_directory:
            set_shinken_owner_on_file_or_directory(self._data_folder, self._logger_init)
            self.rights_access_correctly_granted_on_directory = True
        
        for data_id in self.get_all_data_id():
            if not self.rights_access_per_data_id.get(data_id, True):
                file_path = self._make_file_path(data_id)
                lock_file_path = os.path.join(os.path.dirname(file_path), u'.%s.lock' % os.path.basename(file_path))
                
                file_exists = os.path.exists(file_path)
                lock_file_exists = os.path.exists(lock_file_path)
                
                if file_exists:
                    set_shinken_owner_on_file_or_directory(file_path, self._logger_init)
                
                if lock_file_exists:
                    set_shinken_owner_on_file_or_directory(lock_file_path, self._logger_init)
                
                if file_exists and lock_file_exists:
                    self.rights_access_per_data_id[data_id] = True
                else:
                    del self.rights_access_per_data_id[data_id]
        
        self.rights_access_correctly_granted = True
