#!/usr/bin/python
# -*- coding: utf-8 -*-
#
# Copyright (C) 2013-2023:
# This file is part of Shinken Enterprise, all rights reserved.

import logging
import os

from shinken.log import FixedTimedRotatingFileHandler
from shinken.misc.type_hint import TYPE_CHECKING
from shinken.util import to_bool
from shinkensolutions.system_tools import set_ownership

if TYPE_CHECKING:
    from typing import Optional, Any
    from shinken.log import PartLogger
    from logging import Logger


class CreateUserLogFile(object):
    KEY_LOG_USERS__ENABLED = u'log_users__enabled'
    KEY_LOG_USERS__FILE_PATH = u'log_users__file_path'
    KEY_LOG_USERS__ADD_USER_NAME = u'log_users__add_user_name'
    
    
    def __init__(self, requester_prefix, conf, logger_error=None):
        #  type: (unicode, Any, Optional[PartLogger]) -> None
        self.log_enable = to_bool(getattr(conf, u'%s__%s' % (requester_prefix, self.KEY_LOG_USERS__ENABLED), False))
        
        _default_path = u'/var/log/shinken/%s/log_users.log' % requester_prefix
        self.log_file = getattr(conf, u'%s__%s' % (requester_prefix, self.KEY_LOG_USERS__FILE_PATH), _default_path)
        if not self.log_file or not isinstance(self.log_file, basestring):
            self.log_file = _default_path
        
        # This logger are use for log errors of this logger,
        # classically use the default logger of the module or daemon
        self.logger_error = logger_error
    
    
    def create_path_and_add_right(self):
        # type: () -> None
        
        path_as_array = self.log_file.split(u'/')
        path_of_dirs = path_as_array[:-1]
        
        for i in range(1, len(path_of_dirs)):
            sub_path = u'/'.join(path_as_array[:i + 1])
            if not os.access(sub_path, os.F_OK):
                try:
                    os.makedirs(sub_path)
                except Exception as e:
                    if self.logger_error:
                        self.logger_error.error(u'Could not create directories [%s] -> Error [%s]' % (sub_path, e))
                    return
                
                try:
                    set_ownership(sub_path)
                except Exception as e:
                    if self.logger_error:
                        self.logger_error.error(u'Could not set user right [%s] -> Error [%s]' % (sub_path, e))
                    return
        
        if not os.access(self.log_file, os.F_OK):
            try:
                with open(self.log_file, "a+") as f:
                    f.write('')
            except Exception as e:
                if self.logger_error:
                    self.logger_error.error(u'Could not create file [%s] -> Error [%s]' % (self.log_file, e))
                return
        
        if not os.access(self.log_file, os.W_OK):
            try:
                set_ownership(self.log_file)
            except Exception as e:
                if self.logger_error:
                    self.logger_error.error(u'Could not set user right [%s] -> Error [%s]' % (self.log_file, e))
                return
    
    
    def init_internal_logger(self):
        # type: () -> Logger
        self.create_path_and_add_right()
        try:
            log_handler = FixedTimedRotatingFileHandler(self.log_file, u'midnight', backupCount=5)
            log_handler.setFormatter(logging.Formatter('%(message)s'))
            logger = logging.getLogger('INTERNAL_LOGGER')
            if len(logger.handlers) == 0:
                logger.addHandler(log_handler)
        except Exception as e:
            self.log_enable = False
            if self.logger_error:
                self.logger_error.error(u'Failed to open file [%s] -> Error [%s]' % (self.log_file, e))
            raise
        
        return logger
