#!/usr/bin/python
# -*- coding: utf-8 -*-
#
# Copyright (C) 2013-2023:
# This file is part of Shinken Enterprise, all rights reserved.

from shinken.misc.type_hint import TYPE_CHECKING
from shinkensolutions.logs.logger_abstract import LoggerAbstract

if TYPE_CHECKING:
    from typing import Optional


class LoggerAuthentication(LoggerAbstract):
    FORMATS = [
        (u'log_time', u'[ %(log_time)s ]'),
        (u'mode', u'[ %(mode)-5s ]'),
        (u'type_request', u'[ %(type_request)-6s ]'),
        (u'result_code', u'[ RESULT:%(result_code)-3s ]'),
        (u'run_time', u'[ TIME:   %(run_time)6.0fms ]'),
        (u'user_info', u'[ USER:%(user_info)-9s ]'),
        (u'requester_ip', u'[ CALL_BY:%(requester_ip)-15s ]'),
        (u'authentication_status', u'[ AUTHENTICATED:%(authentication_status)-6s ]'),
        (u'requester_module', u'[ BY:%(requester_module)s ]'),
        (u'auth_module', u'[ AUTHENTICATED BY THE MODULE:%(auth_module)s ]'),
        (u'error', u'[ ERROR:%(error)-9s ]'),
    ]
    
    
    def login(self, user_uuid, user_name, requester_ip, run_time, result_code=200, requester_module=None, auth_module=None):
        # type: (unicode, unicode, unicode, float, int, Optional[unicode], Optional[unicode]) -> None
        if not self.log_enable or not self._logger:
            return
        log_data = self._build_common_log_data(
            user_uuid=user_uuid,
            user_name=user_name,
            requester_ip=requester_ip,
            run_time=run_time,
            result_code=result_code,
            requester_module=requester_module,
            auth_module=auth_module,
        )
        
        log_data.update({
            u'type_request'         : u'LOGIN',
            u'authentication_status': u'OK',
        })
        
        self._log(self._build_message(log_data, self.FORMATS))
    
    
    def login_failed(self, user_uuid, user_name, requester_ip, run_time, result_code=401, requester_module=None, auth_module=None, error=None):
        # type: (unicode, unicode, unicode, float, int, Optional[unicode], Optional[unicode], Optional[unicode]) -> None
        if not self.log_enable or not self._logger:
            return
        log_data = self._build_common_log_data(
            user_uuid=user_uuid,
            user_name=user_name,
            requester_ip=requester_ip,
            run_time=run_time,
            result_code=result_code,
            requester_module=requester_module,
            auth_module=auth_module,
            error=error,
        )
        
        log_data.update({
            u'type_request'         : u'LOGIN',
            u'authentication_status': u'FAILED',
        })
        
        self._log(self._build_message(log_data, self.FORMATS))
    
    
    def logout(self, user_uuid, user_name, requester_ip, run_time, result_code=200, requester_module=None):
        # type: (unicode, unicode, unicode, float, int, Optional[unicode]) -> None
        if not self.log_enable or not self._logger:
            return
        log_data = self._build_common_log_data(
            user_uuid=user_uuid,
            user_name=user_name,
            requester_ip=requester_ip,
            run_time=run_time,
            result_code=result_code,
            requester_module=requester_module,
        )
        
        log_data.update({
            u'type_request': u'LOGOUT',
        })
        
        self._log(self._build_message(log_data, self.FORMATS))
