#!/usr/bin/python
# -*- coding: utf-8 -*-

# Copyright (C) 2013-2018:
#
# This file is part of Shinken Enterprise, all rights reserved.
import re

SELECT_PROP_FOR_OVERRIDE = (
    "contacts",
    "contact_groups",
    "escalations",
    "business_impact_modulations",
    "macromodulations",
    "resultmodulations",
    "notification_period",
    "maintenance_period"
)

OVERRIDE_PATTERN = re.compile(ur'^([^,]+),\s*([^\s]+)\s+(.*)$', re.UNICODE | re.DOTALL)


def parse_service_override_property_list_errors(service_overrides_property):
    """
    Parses a service_overrides property and returns a dict of dicts {'service_description': { 'prop1': 'value1', 'prop2': 'value2'}}
    This property is provided in the form $check,$property $value-=#=-$check2,$property2 $value-=#=- ...
    
    :param service_overrides_property:  value of the service_overrides property
    :type service_overrides_property:  unicode
    :return: Parsed service_override property
    :rtype:  dict
    """
    
    from shinken.objects import Host
    
    if not service_overrides_property:
        return {'parsed_overrides': {}, 'duplicates': [], 'syntax': []}

    separator = Host.properties['service_overrides'].separator
    service_overrides = [so.strip() for so in service_overrides_property.split(separator)]
    
    duplicate_errors = []
    syntax_errors = []
    parsed_overrides = {}
    for ovr in service_overrides:
        # Checks service override syntax
        match = OVERRIDE_PATTERN.match(ovr)
        if match is None:
            syntax_errors.append(ovr)
            continue
        service_description, prop, value = match.groups()
        if not (service_description and prop and value):
            syntax_errors.append(ovr)
            continue
        service_description = service_description.strip()
        prop = prop.strip()
        value = value.strip()
        if prop in SELECT_PROP_FOR_OVERRIDE:
            value = value.strip(',')
        if service_description not in parsed_overrides:
            parsed_overrides[service_description] = {}
        
        if prop in parsed_overrides[service_description]:
            del parsed_overrides[service_description][prop]
            duplicate_errors.append((prop, service_description))
        else:
            parsed_overrides[service_description][prop] = value
    
    return {'parsed_overrides': parsed_overrides, 'duplicates': duplicate_errors, 'syntax': syntax_errors}


def parse_service_override_property(service_overrides_property):
    overrides = parse_service_override_property_list_errors(service_overrides_property)
    if overrides['syntax']:
        raise SyntaxError(overrides['syntax'])
    
    if overrides['duplicates']:
        raise KeyError(overrides['duplicates'])
    
    return overrides['parsed_overrides']


def unparse_service_override(parsed):
    from shinken.objects import Host
    unparsed = []
    for check, so_value in parsed.iteritems():
        for prop_name, value in so_value.iteritems():
            unparsed.append('%s, %s %s' % (check, prop_name, value))
    separator = Host.properties['service_overrides'].separator
    return separator.join(unparsed)
