#!/usr/bin/python
# -*- coding: utf-8 -*-

# Copyright (C) 2013-2020:
# This file is part of Shinken Enterprise, all rights reserved.

import os
import inspect
from shinken.log import PartLogger
from shinken.misc.type_hint import Any, Dict
from shinkensolutions.api.synchronizer import SourceTranslatePart, ItemType
from shinkensolutions.api.synchronizer.http_lib_external.v01_00.html.template_html import template_format

SourceModule = object  # type of SourceModule (can import here because import cycle)


class AbstractDisplayOriginItem(object):
    def __init__(self, logger, translator, source_module, _format, raw_data, title):
        # type: (PartLogger, SourceTranslatePart, SourceModule , str, Any, str) -> None
        self.format = _format
        self.raw_data = raw_data
        self.title = title
        self.logger = logger
        self.translator = translator
        self.source_module = source_module
        self.template_name = self.get_template_name()
    
    
    def get_template_name(self):
        # type: () -> str
        return self.__class__.__module__.rsplit('.', 1)[1]
    
    
    def to_json(self):
        # type: () -> Dict
        return {
            'format'  : self.format,
            'raw_data': self.raw_data,
            'title'   : self.title,
        }
    
    
    @classmethod
    def from_json(cls, logger, translator, source_module, _json):
        # type: (PartLogger, SourceTranslatePart,SourceModule, Dict) -> AbstractDisplayOriginItem
        return cls(
            logger,
            translator,
            source_module,
            _json['format'],
            _json['raw_data'],
            _json['title']
        )
    
    
    def controller(self, item, item_type):
        # type: (Dict, ItemType) -> Dict
        raise NotImplementedError()
    
    
    def to_html(self, item, item_type):
        # type: (Dict, ItemType) -> str
        try:
            rendering_arguments = self.controller(item, item_type)
            template_path = os.path.join(os.path.dirname(inspect.getfile(self.__class__)), 'http_template', '%s.tpl' % self.template_name)
            return template_format(template_path, rendering_arguments)
        except:
            self.logger.print_stack()
            raise
