#!/usr/bin/env python
# -*- coding: utf-8 -*-
#
# Copyright (C) 2009-2012:
#    Gabes Jean, naparuba@gmail.com
#    Gerhard Lausser, Gerhard.Lausser@consol.de
#    Gregory Starck, g.starck@gmail.com
#    Hartmut Goebel, h.goebel@goebel-consult.de
#
# This file is part of Shinken.
#
# Shinken is free software: you can redistribute it and/or modify
# it under the terms of the GNU Affero General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# Shinken is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Affero General Public License for more details.
#
# You should have received a copy of the GNU Affero General Public License
# along with Shinken.  If not, see <http://www.gnu.org/licenses/>.

import cPickle
import time
import threading


class Brok(object):
    # A Brok is a piece of information exported by Shinken to the Broker.
    # Broker can do whatever he wants with it.
    
    __slots__ = ('id', 'type', 'data', 'prepared', 'instance_id', 'creation_date', 'sent_to_sched_externals', 'part_configuration_incarnation')
    counter_id = 0
    my_type = 'brok'
    brok_lock = threading.RLock()
    
    
    def __init__(self, brok_type, data):
        self.type = brok_type
        with self.brok_lock:
            # Use Brok instead self.__class__ to make sure that subclass have the same 'count'
            self.id = Brok.counter_id
            Brok.counter_id += 1
        data['creation_date'] = int(time.time())
        self.data = cPickle.dumps(data, cPickle.HIGHEST_PROTOCOL)
        self.prepared = False
        self.creation_date = int(time.time())
        self.part_configuration_incarnation = None
        self.instance_id = None
    
    
    def set_part_configuration_incarnation(self, part_configuration_incarnation):
        self.instance_id = part_configuration_incarnation.get_part_id()
        self.part_configuration_incarnation = part_configuration_incarnation
    
    
    def __str__(self):
        return 'Brok[%s]' % ','.join('%s:%s' % (slot, getattr(self, slot)) for slot in Brok.__slots__)
    
    
    def __repr__(self):
        return self.__str__()
    
    
    def __getstate__(self):
        res = {}
        for prop in Brok.__slots__:
            if hasattr(self, prop):
                res[prop] = getattr(self, prop)
        return res
    
    
    def __setstate__(self, state):
        for (prop, value) in state.iteritems():
            setattr(self, prop, value)
    
    
    # We deserialize the data, and if some prop were add after the serialize pass, we integer them in the data
    def prepare(self):
        # Maybe the brok is a old daemon one or was already prepared
        # if so, the data is already ok
        if hasattr(self, 'prepared') and not self.prepared:
            self.data = cPickle.loads(self.data)
            if getattr(self, 'instance_id', None) is not None:
                self.data['instance_id'] = self.instance_id
        self.prepared = True


# A subclass of Brok that is used to know if the brok need to be remove once sent
class PersistantBrok(Brok):
    pass
