#!/usr/bin/python
# -*- coding: utf-8 -*-
#
# Copyright (C) 2013-2019:
# This file is part of Shinken Enterprise, all rights reserved.
from shinken.misc.fast_copy import fast_deepcopy
from .massive_change_helper import LINKED_PROPERTIES_FLAT_WHEN_CHANGED, parse_form_link_to_html, parse_form_links_to_html, parse_command_link_to_html
from .massive_change_operator import MASSIVE_CHANGE_OPERATOR
from .massive_change_property import MassiveChangeProperty
from .massive_change_returns import MASSIVE_CHANGE_RETURNS
from .massive_change_status import MASSIVE_CHANGE_STATUS
from .massive_change_type import MASSIVE_CHANGE_TYPE
from ..exceptions import MassiveChangeError
from ....dao.def_items import LINKIFY_MANAGE_STATES, SERVICE_EXCLUDES_BY_ID, STOP_INHERITANCE_VALUES, ITEM_TYPE, DEF_ITEMS, VALUE_FORCE_DEFAULT
from ....dao.helpers import escape_XSS, get_default_value, update_frontend_links_into_links, update_frontend_link_into_link
from ....dao.helpers import get_default_label


class MassiveChangeAction(object):
    
    def __init__(self, prop_name, item_type, action, type_modification, datamanagerV2, extra_action=None, value=None, is_protected=False, target=None, category=None, app=None, synchronized_prop=None, level=0,
                 manage_plus=False, plus_action=None, prop_is_linked=False):
        self.prop_name = prop_name
        self.item_type = item_type
        self.action = action
        self._action = '%s_%s' % (action, extra_action) if action and extra_action else action
        self.type_modification = type_modification
        self.datamanagerV2 = datamanagerV2
        self.extra_action = extra_action
        self.value = value
        self.is_protected = is_protected
        self.target = target
        self.category = category  # To delete when MassiveChangeProperty will be create in MassiveChangeController and not in MassiveChangeAction
        self.app = app
        self.synchronized_prop = synchronized_prop
        self.level = level
        self._manage_plus = manage_plus
        self.plus_action = plus_action
        
        self._updated_value = None
        
        self.html_values = None
        self.html_target = None
        
        self.prop_is_linked = prop_is_linked
        
        self._compute_html()
    
    
    # Actions
    def _add_first(self, change_element):
        if change_element.raw_item.get(self.prop_name, None):
            value_to_add = [link for link in self._updated_value.get('links', []) if link not in change_element.raw_item[self.prop_name].get('links', [])]
            if not value_to_add:
                raise MassiveChangeError(text=MASSIVE_CHANGE_RETURNS.ALREADY_CONTAINS_VALUE)
            
            value_to_add.extend(change_element.raw_item[self.prop_name].get('links', []))
            change_element.raw_item[self.prop_name]['links'] = value_to_add
            return MASSIVE_CHANGE_STATUS.MODIFIED
        
        else:
            change_element.raw_item[self.prop_name] = fast_deepcopy(self._updated_value)
            if self._updated_value.get('has_plus', None) is None:
                change_element.raw_item[self.prop_name]['has_plus'] = False
            
            return MASSIVE_CHANGE_STATUS.MODIFIED
    
    
    def _add_last(self, change_element):
        if change_element.raw_item.get(self.prop_name, None):
            value_to_add = [link for link in self._updated_value.get('links', []) if link not in change_element.raw_item[self.prop_name].get('links', [])]
            if not value_to_add:
                raise MassiveChangeError(text=MASSIVE_CHANGE_RETURNS.ALREADY_CONTAINS_VALUE)
            
            final_value = change_element.raw_item[self.prop_name].get('links', [])
            final_value.extend(value_to_add)
            change_element.raw_item[self.prop_name]['links'] = final_value
            return MASSIVE_CHANGE_STATUS.MODIFIED
        
        else:
            change_element.raw_item[self.prop_name] = fast_deepcopy(self._updated_value)
            if self._updated_value.get('has_plus', None) is None:
                change_element.raw_item[self.prop_name]['has_plus'] = False
            
            return MASSIVE_CHANGE_STATUS.MODIFIED
    
    
    def _add_before(self, change_element):
        return self._set_item_in_list(change_element, move_if_existing=False)
    
    
    def _add_after(self, change_element):
        return self._set_item_in_list(change_element, set_after=True, move_if_existing=False)
    
    
    def _move_before(self, change_element):
        return self._set_item_in_list(change_element)
    
    
    def _move_after(self, change_element):
        return self._set_item_in_list(change_element, set_after=True)
    
    
    def _move_first(self, change_element):
        links = change_element.raw_item.get(self.prop_name, {}).get('links', [])
        if not links:
            raise MassiveChangeError(text=MASSIVE_CHANGE_RETURNS.ITEMS_WITHOUT_PROP)
        target = links[0]
        return self._set_item_in_list(change_element, target=target)
    
    
    def _move_last(self, change_element):
        links = change_element.raw_item.get(self.prop_name, {}).get('links', [])
        if not links:
            raise MassiveChangeError(text=MASSIVE_CHANGE_RETURNS.ITEMS_WITHOUT_PROP)
        target = links[-1]
        return self._set_item_in_list(change_element, target=target, set_after=True)
    
    
    def _set_item_in_list(self, change_element, target=None, set_after=False, move_if_existing=True):
        
        _target = target or self._updated_target
        
        if not change_element.raw_item.get(self.prop_name):
            raise MassiveChangeError(text=MASSIVE_CHANGE_RETURNS.MISSING_TARGET, missing_values=[self._updated_target])
        
        links_on_item = change_element.raw_item[self.prop_name]['links']
        if _target not in links_on_item:
            raise MassiveChangeError(text=MASSIVE_CHANGE_RETURNS.MISSING_TARGET, missing_values=[self._updated_target])
        
        original_value = fast_deepcopy(change_element.raw_item[self.prop_name])
        _index = None
        if move_if_existing:
            missing_values = [_link for _link in self._updated_value['links'] if _link not in links_on_item]
            if missing_values:
                raise MassiveChangeError(text=MASSIVE_CHANGE_RETURNS.NO_VALUE_ON_ITEM, missing_values=missing_values)
            value_to_set = self._updated_value['links']
            for _link in value_to_set:
                # Special case your link is your target o_O ? Yes ! In the case of move_first or move_last, the link is the first or last item in list
                if _link == _target:
                    _index = links_on_item.index(_target)
                links_on_item.remove(_link)
        else:
            value_to_set = [_link for _link in self._updated_value['links'] if _link not in links_on_item]
            # If a value is already in property, raise (rule is add if missing)
            if len(value_to_set) != len(self._updated_value['links']):
                raise MassiveChangeError(text=MASSIVE_CHANGE_RETURNS.ALREADY_CONTAINS_VALUE)
        
        # Do not remove "is not None" because it's can be 0
        try:
            index = links_on_item.index(_target)
        except ValueError:
            index = _index
        
        if set_after:
            index += 1
        
        value_to_set.reverse()
        for link in value_to_set:
            links_on_item.insert(index, link)
        
        return MASSIVE_CHANGE_STATUS.MODIFIED if original_value != change_element.raw_item[self.prop_name] else MASSIVE_CHANGE_STATUS.UNMODIFIED
    
    
    def _add(self, change_element):
        
        # SEE : SEF-2516
        # For view_contacts with default value set to everyone, Do not add contact on a empty field if t's a synchronized action
        if self.prop_name == 'view_contacts' and self.synchronized_prop and get_default_value(self.item_type, self.prop_name)[0] == 'everyone':
            if change_element.raw_item.get(self.prop_name, None) is None:
                return MASSIVE_CHANGE_STATUS.UNMODIFIED
        
        # For view_contacts with default value set to everyone if we add a value on a emtpy field, we have to add also other contact present in edition and notification
        if self.prop_name == 'view_contacts' and not self.synchronized_prop and get_default_value(self.item_type, self.prop_name)[0] == 'everyone':
            if change_element.raw_item.get(self.prop_name, None) is None:
                full_list = []
                full_list.extend(self._updated_value.get('links', []))
                full_list.extend(change_element.raw_item.get('edition_contacts', {}).get('links', []))
                full_list.extend(change_element.raw_item.get('notification_contacts', {}).get('links', []))
                change_element.raw_item[self.prop_name] = {
                    'has_plus': self._updated_value.get('has_plus', False),
                    'links'   : full_list
                }
                return MASSIVE_CHANGE_STATUS.MODIFIED
        
        if change_element.raw_item.get(self.prop_name, None):
            
            if not self._updated_value.get('links', []):
                return MASSIVE_CHANGE_STATUS.UNMODIFIED
            item_links = change_element.raw_item[self.prop_name].get('links', [])
            links_to_add = [link for link in self._updated_value['links'] if link not in item_links]
            if not links_to_add:
                raise MassiveChangeError(text=MASSIVE_CHANGE_RETURNS.ALREADY_CONTAINS_VALUE)
            if len(item_links) == 1 and not item_links[0]['exists'] and item_links[0]['name'] in STOP_INHERITANCE_VALUES:
                change_element.raw_item[self.prop_name]['links'] = links_to_add
                return MASSIVE_CHANGE_STATUS.MODIFIED
            
            change_element.raw_item[self.prop_name]['links'].extend(links_to_add)
            return MASSIVE_CHANGE_STATUS.MODIFIED
        
        else:
            change_element.raw_item[self.prop_name] = fast_deepcopy(self._updated_value)
            if self._updated_value.get('has_plus', None) is None:
                change_element.raw_item[self.prop_name]['has_plus'] = False
            
            return MASSIVE_CHANGE_STATUS.MODIFIED
    
    
    def _remove(self, change_element):
        
        # SEE : SEF-5216
        # If we remove a contact from views_contacts and prop came empty, synchronized action on edition_contacts and notification_contacts must be stop
        if self.prop_name in ('edition_contacts', 'notification_contacts') and self.synchronized_prop and get_default_value(self.item_type, 'view_contacts')[0] == 'everyone':
            if change_element.raw_item.get('view_contacts', None) is None:
                return MASSIVE_CHANGE_STATUS.UNMODIFIED
        
        if not change_element.raw_item.get(self.prop_name):
            return MASSIVE_CHANGE_STATUS.UNMODIFIED
        links_on_item = change_element.raw_item[self.prop_name].get('links', [])
        links_to_remove = [link for link in self._updated_value.get('links', []) if link in links_on_item]
        if not links_to_remove:
            return MASSIVE_CHANGE_STATUS.UNMODIFIED
        for link in links_to_remove:
            links_on_item.remove(link)
        
        # No more links on item and no plus ? drop it !
        if len(links_on_item) == 0 and not change_element.raw_item[self.prop_name].get('has_plus', None):
            change_element.raw_item.pop(self.prop_name)
        
        return MASSIVE_CHANGE_STATUS.MODIFIED
    
    
    def _replace(self, change_element):
        modified = self._set_item_in_list(change_element, move_if_existing=False)
        if modified == MASSIVE_CHANGE_STATUS.MODIFIED:
            change_element.raw_item[self.prop_name]['links'].remove(self._updated_target)
        return modified
    
    
    def _set(self, change_element):
        original_value = fast_deepcopy(change_element.raw_item.get(self.prop_name, None))
        if self.prop_name in LINKED_PROPERTIES_FLAT_WHEN_CHANGED[self.item_type] or (self.prop_name in DEF_ITEMS[self.item_type]['item_links'].get(ITEM_TYPE.COMMANDS, []) and self.item_type != ITEM_TYPE.NOTIFICATIONWAYS):
            change_element.raw_item[self.prop_name] = self._updated_value
        elif self.prop_is_linked:
            new_links = self._updated_value['links'] if 'links' in self._updated_value else None
            if not new_links:
                # This case is for manage plus (SET without value)
                return MASSIVE_CHANGE_STATUS.UNMODIFIED
            if not change_element.raw_item.get(self.prop_name, None):
                change_element.raw_item[self.prop_name] = {'has_plus': False}
            change_element.raw_item[self.prop_name]['links'] = new_links
        else:
            change_element.raw_item[self.prop_name] = self._updated_value
        return MASSIVE_CHANGE_STATUS.MODIFIED if original_value != change_element.raw_item[self.prop_name] else MASSIVE_CHANGE_STATUS.UNMODIFIED
    
    
    def _set_if_exist(self, change_element):
        if change_element.raw_item.get(self.prop_name, None) is None:
            return MASSIVE_CHANGE_STATUS.UNMODIFIED
        return self._set(change_element)
    
    
    def _empty(self, change_element):
        prop = change_element.raw_item.pop(self.prop_name, None)
        return MASSIVE_CHANGE_STATUS.MODIFIED if prop else MASSIVE_CHANGE_STATUS.UNMODIFIED
    
    
    def _exclude(self, change_element):
        backup_value = self._updated_value
        modified = MASSIVE_CHANGE_STATUS.UNMODIFIED
        checks_to_add_in_excludes = [check['check_link'] for check in self._updated_value if check['value']]
        checks_to_remove_in_excludes = [check['check_link'] for check in self._updated_value if not check['value']]
        
        if checks_to_add_in_excludes:
            try:
                self._updated_value = {'links': checks_to_add_in_excludes}
                _modified = self._add(change_element)
            except MassiveChangeError:
                _modified = MASSIVE_CHANGE_STATUS.UNMODIFIED
            
            modified = _modified
        if checks_to_remove_in_excludes:
            try:
                self._updated_value = {'links': checks_to_remove_in_excludes}
                _modified = self._remove(change_element)
            except MassiveChangeError:
                _modified = MASSIVE_CHANGE_STATUS.UNMODIFIED
            
            if _modified == MASSIVE_CHANGE_STATUS.MODIFIED:
                modified = _modified
        
        self._updated_value = backup_value
        return modified
    
    
    def _remove_all_except(self, change_element):
        if change_element.raw_item.get(self.prop_name, None):
            value_to_keep = [link for link in self._updated_value.get('links', []) if link in change_element.raw_item[self.prop_name].get('links', [])]
            if not value_to_keep:
                change_element.raw_item.pop(self.prop_name, None)
                return MASSIVE_CHANGE_STATUS.MODIFIED
            
            change_element.raw_item[self.prop_name]['links'] = value_to_keep
            return MASSIVE_CHANGE_STATUS.MODIFIED
        
        else:
            return MASSIVE_CHANGE_STATUS.UNMODIFIED
    
    
    def _set_synchro(self, change_element):
        actual_value = change_element.raw_item.get(self.prop_name, None)
        if not actual_value:
            return MASSIVE_CHANGE_STATUS.UNMODIFIED
        
        if self.prop_name == 'sla_warning_threshold' and actual_value < self._updated_value:
            change_element.raw_item[self.prop_name] = self._updated_value
            return MASSIVE_CHANGE_STATUS.MODIFIED
        if self.prop_name == 'sla_critical_threshold' and actual_value > self._updated_value:
            change_element.raw_item[self.prop_name] = self._updated_value
            return MASSIVE_CHANGE_STATUS.MODIFIED
        return MASSIVE_CHANGE_STATUS.UNMODIFIED
    
    
    # Plus
    def manage_plus(self, change_element):
        # We manage 'has_plus' ONLY if it set in value (can be True or False)
        if not self._updated_value or self._updated_value.get('has_plus', None) is None:
            return MASSIVE_CHANGE_STATUS.UNMODIFIED
        
        # If item has no value, add it only is has_plus
        if not change_element.raw_item.get(self.prop_name, None):
            if self._updated_value['has_plus']:
                change_element.raw_item[self.prop_name] = {'links': [], 'has_plus': True}
                return MASSIVE_CHANGE_STATUS.MODIFIED
            else:
                return MASSIVE_CHANGE_STATUS.UNMODIFIED
        
        item_links = change_element.raw_item[self.prop_name].get('links', [])
        
        # If no more value (no links and na has_plus), empty the field
        if not item_links and not self._updated_value['has_plus']:
            change_element.raw_item.pop(self.prop_name, None)
            return MASSIVE_CHANGE_STATUS.MODIFIED
        
        # If the link is to STOP_INHERITANCE_VALUES, we need to empty the link if we modify the plus
        if item_links and len(item_links) == 1:
            if not item_links[0]['exists'] and item_links[0]['name'] in STOP_INHERITANCE_VALUES and self._updated_value['has_plus']:
                change_element.raw_item[self.prop_name]['links'] = []
        
        # In other case, just set it !
        change_element.raw_item[self.prop_name]['has_plus'] = self._updated_value['has_plus']
        return MASSIVE_CHANGE_STATUS.MODIFIED
    
    
    # Public
    def apply(self, change_element):
        modified = False
        
        change_property = MassiveChangeProperty(self.prop_name, self.item_type, self.category, self._manage_plus, change_element.get_uuid(), self)
        change_property.set_previous_value(change_element.raw_item.get(self.prop_name, None), self.app, self.datamanagerV2)
        
        self._updated_value = fast_deepcopy(self.value)
        self._updated_target = fast_deepcopy(self.target)
        
        if self.prop_is_linked:
            if self.prop_name == SERVICE_EXCLUDES_BY_ID and self._updated_value:
                _value = [check['check_link'] for check in self._updated_value]
                if _value:
                    update_frontend_links_into_links(self.prop_name, _value, self.item_type, self.datamanagerV2, linkify=False)
            
            elif self.prop_name in DEF_ITEMS[self.item_type]['item_links'].get(ITEM_TYPE.COMMANDS, []) and self._updated_value and self.item_type != ITEM_TYPE.NOTIFICATIONWAYS:
                update_frontend_link_into_link(self.prop_name, self._updated_value['node']['link'], self.item_type, self.datamanagerV2, linkify=False)
            
            elif self._updated_value and 'links' in self._updated_value:
                update_frontend_links_into_links(self.prop_name, self._updated_value['links'], self.item_type, self.datamanagerV2, linkify=False)
            
            if self._updated_target:
                update_frontend_links_into_links(self.prop_name, [self._updated_target], self.item_type, self.datamanagerV2, linkify=False)
        
        # Manage field "exclude" on timeperiods and "parents" on Host and hosttpls and field use
        if ((self.item_type == ITEM_TYPE.TIMEPERIODS and self.prop_name == 'exclude') or \
            (self.item_type in [ITEM_TYPE.HOSTS, ITEM_TYPE.HOSTTPLS] and self.prop_name == 'parents') or \
            self.prop_name == 'use') and self.value:
            uuid_to_add = [l['_id'] for l in self.value.get('links', []) if l['exists']]
            if change_element.get_uuid() in uuid_to_add:
                change_property.set_error(MASSIVE_CHANGE_RETURNS.NOT_MYSELF)
                change_element.add_property(change_property, modified)
                return modified
        
        try:
            specific_method = getattr(self, '_%s' % self._action.lower(), None)
            prop_modified = specific_method(change_element)
            modified = bool(prop_modified)
        except MassiveChangeError as e:
            change_property.set_error(e.text)
            if e.missing_values:
                change_property.missing_values = e.missing_values
        
        plus_modified = False
        if self._manage_plus:
            try:
                plus_modified = self.manage_plus(change_element)
                modified |= bool(plus_modified)
            except MassiveChangeError as e:
                change_property.set_error(e.text)
                if e.missing_values:
                    change_property.missing_values = e.missing_values
        
        change_property.set_actual_value(change_element.raw_item.get(self.prop_name, None), modified, self.app, self.datamanagerV2, plus_modified=plus_modified)
        change_element.add_property(change_property, modified)
        
        return modified
    
    
    # HTML
    def _compute_html(self):
        if self.value in STOP_INHERITANCE_VALUES:
            default_value = get_default_value(self.item_type, self.prop_name, app=self.app, same_as_host=True)
            if default_value == 'None':  # pourquoi on a un string après get_default_value???
                default_value = get_default_label(self.item_type, self.prop_name, self.value, self.app)
            if not default_value:
                default_value = self.app._('element.none')
            if self.value == VALUE_FORCE_DEFAULT:
                self.html_values = '%s [%s]' % (self.app._('element.default'), escape_XSS(default_value))
            else:
                self.html_values = '%s [%s]' % (self.app._('element.forced_default'), escape_XSS(default_value))
        elif self.prop_name in LINKED_PROPERTIES_FLAT_WHEN_CHANGED[self.item_type]:
            flattened_value = self.datamanagerV2.flatten_value(self.value, self.item_type, self.prop_name, LINKIFY_MANAGE_STATES)
            self.html_values = escape_XSS(flattened_value)
        elif self.is_protected and self.action in (MASSIVE_CHANGE_OPERATOR.SET_IF_EXIST, MASSIVE_CHANGE_OPERATOR.SET):
            self.html_values = self.app._('element.password_protected')
        elif self.prop_name == 'business_impact' or self.prop_name == 'min_business_impact':
            self.html_values = self.app._('element.business_impact_%s' % self.value) if self.value else ""
        elif self.prop_name == 'duplicate_foreach' and self.value and self.value not in STOP_INHERITANCE_VALUES:
            self.html_values = self.value.upper().lstrip('_')
        elif self.prop_is_linked:
            if self.prop_name == SERVICE_EXCLUDES_BY_ID:
                # Must have 2 lines add and remove
                self.html_values = (
                    parse_form_links_to_html([c['check_link'] for c in self.value if c['value']], self.prop_name, self.item_type, self.app, self.datamanagerV2),
                    parse_form_links_to_html([c['check_link'] for c in self.value if not c['value']], self.prop_name, self.item_type, self.app, self.datamanagerV2)
                )
            elif self.prop_name in DEF_ITEMS[self.item_type]['item_links'].get(ITEM_TYPE.COMMANDS, []) and self.value and self.item_type != ITEM_TYPE.NOTIFICATIONWAYS:
                self.html_values = parse_command_link_to_html(self.value, self.prop_name, self.item_type, self.app, self.datamanagerV2)
            elif self.value and 'links' in self.value:
                self.html_values = parse_form_links_to_html(self.value['links'], self.prop_name, self.item_type, self.app, self.datamanagerV2)
        elif self.value:
            self.html_values = escape_XSS(self.value)
        
        if self.target:
            self.html_target = parse_form_link_to_html(self.target, self.prop_name, self.item_type, self.app, self.datamanagerV2)
    
    
    def get_action_in_html(self, app):
        _html_actions = []
        action_model = {
            'property_label': app.helper.get_item_property_display_name(self.item_type, self.prop_name),
            'action_label'  : '',
            'values'        : self.html_values or '',
            'target'        : self.html_target,
            'plus'          : "",
            'plus_filter'   : ""
        }
        
        if self.prop_name == SERVICE_EXCLUDES_BY_ID:
            # Need to set two actions : add and remove
            if self.html_values[0]:
                _add = action_model.copy()
                # Here the label we be remove because if we add an exclude, we remove the check
                _add['action_label'] = app._('mass-change.step-4.action.exclude_remove')
                _add['values'] = self.html_values[0]
                _html_actions.append(_add)
            
            if self.html_values[1]:
                _remove = action_model.copy()
                # Here the label will be add because if we remove an exclude, we add the check
                _remove['action_label'] = app._('mass-change.step-4.action.exclude_add')
                _remove['values'] = self.html_values[1]
                _html_actions.append(_remove)
        
        else:
            if self.type_modification == MASSIVE_CHANGE_TYPE.LOCAL_DATAS and self.action == MASSIVE_CHANGE_OPERATOR.SET:
                action_model['action_label'] = app._('mass-change.step-4.action.SET_OR_CREATE')
            elif self.synchronized_prop:
                _action = app._('mass-change.step-4.action.%s' % self.action)
                _text = '<span class="shinken-highlight-data-user">%s</span>' % app.helper.get_item_property_display_name(self.item_type, self.synchronized_prop)
                _synchronized = '<span class="shinken-synchronized-prop">%s</span>' % app._('mass-change.synchronized_action').format(_text)
                action_model['action_label'] = '%s<br>%s' % (_action, _synchronized)
            else:
                action_model['action_label'] = app._('mass-change.step-4.action.%s' % self.action)
            
            if self.extra_action:
                action_model['extra_action_label'] = app._('mass-change.input.extra-action.%s' % self.extra_action)
            
            if self.action == MASSIVE_CHANGE_OPERATOR.REPLACE:
                action_model['extra_action_label'] = app._('mass-change.step-4.action_label.%s' % (self.action))
            
            if self._manage_plus and self.plus_action is not None:
                if self.plus_action:
                    action_model['plus'] = app.helper.helper_object.build_icon_plus('ON')
                    action_model['plus_filter'] = 'ON'
                else:
                    action_model['plus'] = app.helper.helper_object.build_icon_plus('OFF')
                    action_model['plus_filter'] = 'OFF'
            _html_actions.append(action_model)
        return _html_actions
    
    
    def __repr__(self):
        return '<div style="background-color:red;z-index:9999;width: fit-content;position: relative;">action=%s <br>manage_plus=%s <br> self.plus_action=%s<div>' % (self._action, self._manage_plus, self.plus_action)
