#!/usr/bin/python
# -*- coding: utf-8 -*-
#
# Copyright (C) 2013-2018:
# This file is part of Shinken Enterprise, all rights reserved.


import time
from . import PreSaver, PreDeleter, PreMover
from ..def_items import HISTORY_ACTION, ITEM_STATE, ITEM_TYPE
from ..helpers import compute_diff


def add_history_info(data_manager_v2, item, item_type, item_state, user, action, old_item):
    """ Add history information to object (especially changes)
    :param data_manager_v2:
    :type data_manager_v2: DataManagerV2
    :param item: item on which to add infos
    :param item_type: Type of item
    :param item_state: State in which the Item is
    :param user: User responsible for the change
    :param action: Nature of the change
    :param old_item: Item from which to compute changes
    :return: None
    """
    from ..items.baseitem import BaseItem
    
    start_time = time.time()
    last_modif = dict()
    if user.get(u'_id', u'-1') == u'-1':
        last_modif[u'contact'] = {u'has_plus': False, u'links': [{u'name': user.get(u'contact_name'), u'exists': False}]}
    else:
        last_modif[u'contact'] = {
            u'has_plus': False,
            u'links'   : [
                {u'_id'      : user[u'_id'],
                 u'exists'   : True,
                 u'item_type': ITEM_TYPE.CONTACTS,
                 }
            ]
        }
    if isinstance(item, BaseItem):
        last_modif[u'contact'][u'links'][0][u'@link'] = {ITEM_STATE.STAGGING: user}
    last_modif[u'action'] = action
    last_modif[u'date'] = time.time()
    last_modif[u'change'] = []
    
    raw_old_item = data_manager_v2.get_raw_item(old_item, item_type=item_type, flatten_links=False)
    raw_item = data_manager_v2.get_raw_item(item, item_type=item_type, flatten_links=False)
    
    show_diffs_state = {}
    if old_item:
        show_diffs_state = old_item.get(u'last_modification', {}).get(u'show_diffs_state', {})
        
        diffs_properties = compute_diff(raw_item, raw_old_item, item_type)
        if len(diffs_properties) > 0:
            for _prop in diffs_properties:
                old_action = show_diffs_state.get(_prop, None)
                if old_action is None or old_action == HISTORY_ACTION.AUTO_MODIFICATION:
                    show_diffs_state[_prop] = action
                else:
                    show_diffs_state[_prop] = old_action
                if _prop != u"last_modification":
                    last_modif[u'change'].append({u"prop": _prop, u"old": raw_old_item.get(_prop, ''), u"new": raw_item.get(_prop, '')})
    
    show_diffs = bool(next((a for a in show_diffs_state.itervalues() if a != HISTORY_ACTION.AUTO_MODIFICATION), False))
    # If there aren't change when asking a VALIDATED_FROM_WORK_AREA or for AUTO_MODIFICATION we don't change the current last_modification
    if old_item and ((action == HISTORY_ACTION.ELEMENT_MODIFICATION and not last_modif[u'change']) or action == HISTORY_ACTION.AUTO_MODIFICATION):
        last_mod = old_item.get(u'last_modification', {})
        if not isinstance(item, BaseItem) and last_mod.get(u'contact', None):
            last_mod[u'contact'][u'links'][0].pop(u'@link', None)
        last_mod[u'show_diffs_state'] = show_diffs_state
        last_mod[u'show_diffs'] = show_diffs
    else:
        last_mod = last_modif
        last_mod[u'show_diffs_state'] = show_diffs_state
        last_mod[u'show_diffs'] = show_diffs
    
    if hasattr(item, u'set_value'):
        item.set_value(u'last_modification', last_mod)
    else:
        item[u'last_modification'] = last_mod


class CallbackHistoryInfo(PreSaver, PreDeleter, PreMover):
    def __init__(self):
        super(CallbackHistoryInfo, self).__init__()
    
    
    def called_before_save(self, item_id='', item_type='', item_state='', item=None, old_item=None, user=None, action='', datamanagerV2=None):
        if item_state not in (ITEM_STATE.CHANGES, ITEM_STATE.MERGE_SOURCES):
            if action == HISTORY_ACTION.PUT_IN_WORK_AREA:
                old_item = datamanagerV2.find_item_by_id(item_id, item_type, ITEM_STATE.STAGGING)
            self.log(item_type, item_state, item_id, action, 'add history info')
            add_history_info(datamanagerV2, item, item_type, item_state, user, action, old_item)
    
    
    def called_before_delete(self, item_id='', item_type='', item_state='', item=None, old_item=None, user=None, action='', datamanagerV2=None):
        if item_state not in (ITEM_STATE.CHANGES, ITEM_STATE.MERGE_SOURCES, ITEM_STATE.NEW):
            self.log(item_type, item_state, item_id, action, 'add history info')
            if item_state == ITEM_STATE.STAGGING and action == HISTORY_ACTION.ELEMENT_DELETE:
                item_in_production = datamanagerV2.find_item_by_id(item_id, item_type, ITEM_STATE.PRODUCTION)
                if item_in_production:
                    add_history_info(datamanagerV2, item_in_production, item_type, ITEM_STATE.PRODUCTION, user, action, old_item)
                    datamanagerV2.save_item(item_in_production, user, item_type, ITEM_STATE.PRODUCTION)
            add_history_info(datamanagerV2, item, item_type, item_state, user, action, old_item)


    called_before_move = called_before_save

callback_history_info = CallbackHistoryInfo()