#!/usr/bin/python
# -*- coding: utf-8 -*-

# Copyright (C) 2013-2019:
# This file is part of Shinken Enterprise, all rights reserved.

import re


def _split_parenthetic_contents(string):
    level = 0
    stack = []
    last_is_and = False
    between_quote = False
    skip_next = False
    for i, c in enumerate(string):
        if skip_next:
            skip_next = False
            continue
        if c == '"':
            if between_quote:
                if i + 1 != len(string) and string[i + 1] == '<':
                    stack.append("$")
                    skip_next = True
                yield level, ''.join(stack)
                stack = []
            between_quote = not between_quote
            continue
        if between_quote:
            stack.append(re.escape(c))
        else:
            if c == '>':
                stack.append("^")
            elif c == '<':
                stack.append("$")
            elif c == '*':
                stack.append(".*")
            elif c == '(':
                if last_is_and:
                    last_is_and = False
                level += 1
            elif c == ')':
                if stack:
                    yield level, ''.join(stack).strip()
                stack = []
                level -= 1
                if last_is_and:
                    level -= 1
                    last_is_and = False
            elif c in ['&', '|']:
                if not last_is_and and c == '&':
                    level += 1
                    last_is_and = True
                if stack:
                    yield level, ''.join(stack).strip()
                if last_is_and and c == '|':
                    level -= 1
                    last_is_and = False
                yield level, c
                stack = []
            elif c == '!' and i + 1 != len(string) and string[i + 1] == '(':
                yield level, c
            elif c in ['=', ' ', '!']:
                stack.append(c)
            else:
                stack.append(re.escape(c))
    if stack:
        yield level, ''.join(stack).strip()


def _match_equal(filter_to_apply, value):
    return filter_to_apply == re.escape(value)


def _try_match(filter_to_apply, value):
    negate = False
    value = value.lower()
    match = None
    filter_to_apply = filter_to_apply.lower()
    if filter_to_apply.startswith('!'):
        filter_to_apply = filter_to_apply[1:]
        negate = True
    if filter_to_apply.startswith('='):
        match = _match_equal(filter_to_apply[1:], value)
    if match is None:
        match = True if re.search(filter_to_apply, value) else False
    return match if not negate else not match


# Refer to SEF-4729
def _filter_nodes_match_with_value(nodes, value, depth=-1, deep=False):
    match = None
    operator = ''
    skip_to = -1
    node_index = 0
    negate = False
    
    if depth == -1:
        depth_nodes = [node_depth for node_depth, node_value in nodes]
        if not depth_nodes:
            return True
        depth = min(depth_nodes)
    
    for node_index, node in enumerate(nodes):
        actual_depth, filter_to_apply = node
        
        if filter_to_apply == '' or node_index <= skip_to:
            continue
        
        if filter_to_apply == '!':
            negate = True
            continue
        
        if actual_depth < depth and deep:
            return [match, node_index - 1]
        
        if filter_to_apply in ('&', '|'):
            operator = filter_to_apply
            continue
        
        if actual_depth != depth:
            depth_result = _filter_nodes_match_with_value(nodes[node_index:], value, depth + 1, True)
            actual_match = depth_result[0]
            if negate:
                actual_match = not actual_match
            skip_to = node_index + depth_result[1]
        else:
            actual_match = _try_match(filter_to_apply, value)
            
        if operator:
            if operator == '&':
                match = match and actual_match
            if operator == '|':
                match = match or actual_match
        else:
            match = actual_match
    if deep:
        return [match, node_index]
    return match


def filter_match_with_item(filter_pattern, value):
    return _filter_nodes_match_with_value(filter_pattern, value)
