#!/usr/bin/python
# -*- coding: utf-8 -*-
#
# Copyright (C) 2013-2018
# This file is part of Shinken Enterprise, all rights reserved.

import json
import uuid
from shinkensolutions.netaddr import valid_nmap_range
from ...dao.def_items import METADATA

OK = 0
BAD_PORT = 1
BAD_PORT_ORDER = 2

app = None


def set_app(_app):
    global app
    app = _app


def key_sort(s1, s2):
    if s1.startswith('_') and s2.startswith('_'):
        return cmp(s1.lower(), s2.lower())
    if s1.startswith('_') and not s2.startswith('_'):
        return 1
    if not s1.startswith('_') and s2.startswith('_'):
        return -1
    return cmp(s1, s2)


def get_ip_ranges(source_name):
    return app.discovery_confs.find({'source_name': source_name}).sort([['discovery_name', 1]])


def get_discovery_confs(source_name):
    user = app.get_user_auth()
    confs = get_ip_ranges(source_name)
    # we return values for the template (view). But beware, theses values are the
    # only one the template will have, so we must give it an app link and the
    # user we are logged with (it's a contact object in fact)
    return {'app': app, 'user': user, 'confs': confs}


def _get_source_data(source_name):
    # logger.debug("[Sources.py]::get_source - app.sources = %s" % app.sources)
    sources = [s for s in app.sources]
    source = next((s for s in sources if s.get_name() == source_name.strip()), None)
    if not source:
        return None
    source.update_from_last_synchronization(app.last_synchronization)
    return source


def get_discovery_conf(sname, id):
    user = app.get_user_auth()
    conf = {}
    if id:
        source = _get_source_data(sname)
        conf = app.discovery_confs.find_one({'_id': id})
        if source is None:
            app.abort(404, app._('source.no_source_with_this_name') % sname)
        if conf is None:
            app.abort(404, app._('discovery.no_conf_with_this_id') % id)
        METADATA.update_metadata(conf, METADATA.IN_CREATION, False)
        METADATA.update_metadata(conf, METADATA.ITEM_TYPE, 'discovery')
        
        return {
            'source'     : source,
            'app'        : app,
            'user'       : user,
            'item'       : conf,
            'source_name': sname
        }


def new_discovery_conf(sname):
    user = app.get_user_auth()
    conf = {
        '_id'           : uuid.uuid1().hex,
        'discovery_name': '',
        'enabled'       : "1"
    }
    source = _get_source_data(sname)
    if source is None:
        app.abort(404, app._('source.no_source_with_this_name') % sname)
    METADATA.update_metadata(conf, METADATA.IN_CREATION, True)
    METADATA.update_metadata(conf, METADATA.ITEM_TYPE, 'discovery')
    
    return {
        'source'     : source,
        'app'        : app,
        'user'       : user,
        'item'       : conf,
        'source_name': sname
    }


def post_discovery_conf(sname):
    forms_item = json.loads(app.request.forms['item'])
    conf_id = forms_item.get('_id', '')
    discovery_name = forms_item.get('discovery_name', '').strip()
    iprange = forms_item.get('iprange', '').strip()
    port_range = forms_item.get('port_range', '').strip()
    extra_option = forms_item.get('extra_option', '').strip()
    scan_interval = int(forms_item.get('scan_interval', '60').strip())
    notes = forms_item.get('notes', '').strip()
    enabled = (forms_item.get('enabled', '1') == '1')

    bad_fields = []

    if not discovery_name:
        bad_fields.append('\'' + app._('discovery.discovery_name') + '\'')
    if not iprange:
        bad_fields.append('\'' + app._('discovery.iprange') + '\'')
    else:
        if not __validate_iprange_field(iprange):
            bad_fields.append('\'' + app._('discovery.iprange') + '\'')

    if port_range:
        if not __validate_port_range_field(port_range):
            bad_fields.append('\'' + app._('discovery.port_range') + '\'')

    nbr_bad_fields = len(bad_fields)
    if nbr_bad_fields > 0:
        app.response.status = 400
        if nbr_bad_fields == 1:
            fields = bad_fields[nbr_bad_fields - 1]
            return app._('validator.disco_bad_field') % fields
        elif nbr_bad_fields == 2:
            fields = (' ' + app._('element.and') + ' ').join(bad_fields)
        else:
            fields = ', '.join(bad_fields[:-1])
            fields += ' ' + app._('element.and') + ' '
            fields += bad_fields[nbr_bad_fields - 1]
        return app._('validator.disco_bad_fields') % fields
    
    new_conf = {
        'discovery_name'    : discovery_name,
        'iprange'           : iprange,
        'scan_interval'     : scan_interval,
        'notes'             : notes,
        'enabled'           : enabled,
        'port_range'        : port_range,
        'extra_option'      : extra_option
    }

    _to_return = app.save_source_conf_to_backend(sname, conf_id, new_conf)
    if _to_return == "name_already_exist":
        app.response.status = 400
        return app._('validator.disco_already_exist') % discovery_name

def __validate_iprange_field(iprange):
    for _ip_range in iprange.split(' '):
        try:
            _valide_ip_range(_ip_range)
        except ValueError:
            return False
    return True


def _valide_ip_range(ip_range):
    if not valid_nmap_range(ip_range):
        raise ValueError
    
    if "/" in ip_range and int(ip_range.split('/')[1]) < 16:
        raise ValueError


def __validate_port_range_field(port_range):
    try:
        p_ranges = port_range.split(',')
        for p_range in p_ranges:
            ports = p_range.split('-')
            size = len(ports)
            if size == 1:
                if not __validate_port_value(ports[0]):
                    return False
            elif size == 2:
                status = __validate_port_range(ports[0], ports[1])
                if status == BAD_PORT:
                    app.response.status = 400
                    return False
                elif status == BAD_PORT_ORDER:
                    app.response.status = 400
                    return False
            else:
                app.response.status = 400
                return False
    except ValueError:
        app.response.status = 400
        return False
    return True


def __validate_port_value(port):
    if not 0 < int(port) < 65536:
        return False
    return True


def __validate_port_range(port1, port2):
    if not (__validate_port_value(port1) and __validate_port_value(port2)):
        return BAD_PORT
    if int(port1) > int(port2):
        return BAD_PORT_ORDER
    return OK


def delete_discovery_conf(sname):
    conf_id = app.request.forms.get('_id', '')
    if not conf_id:
        app.abort(400, "Missing property _id")
        return

    app.delete_source_conf_to_backend(sname, conf_id)
    
    return
