#!/usr/bin/python
# -*- coding: utf-8 -*-

# Copyright (C) 2013-2020:
# This file is part of Shinken Enterprise, all rights reserved.

from shinken.log import PartLogger
from shinken.misc.type_hint import Dict

from shinkensolutions.api.synchronizer import ITEM_TYPE, ItemType, SourceConfItem
from shinkensolutions.api.synchronizer.source.item.source_item import SourceItem, PRIVATE_PROP
from shinkensolutions.api.synchronizer.source.mapping.source_to_conf import *

Mapping = Dict[str, str]


class GenericMapperSourceToConf(object):
    
    def __init__(self, item_type, mapping=None):
        # type: (ItemType, Mapping) -> None
        self.item_type = item_type
        self.mapping = mapping
    
    
    @staticmethod
    def stringify(macro_property_value):
        if isinstance(macro_property_value, unicode):
            macro_property_value = macro_property_value.encode('utf8', 'ignore')
        else:
            macro_property_value = str(macro_property_value)
        return macro_property_value
    
    
    @staticmethod
    def map_macro_properties(source_conf_item, macros):
        # type: (SourceConfItem, Dict) -> None
        for macro_property_name, macro_property_value in macros.iteritems():
            if macro_property_value:
                source_conf_item[macro_property_name] = GenericMapperSourceToConf.stringify(macro_property_value)
    
    
    def map(self, source_item):
        # type: (SourceItem) -> SourceConfItem
        source_conf_item = {}
        
        for source_item_prop_name, conf_item_prop_name in self.mapping.iteritems():
            source_item_value = getattr(source_item, source_item_prop_name, '')
            if source_item_prop_name not in PRIVATE_PROP:
                source_item_value = GenericMapperSourceToConf.stringify(source_item_value)
            if source_item_value:
                source_conf_item[conf_item_prop_name] = source_item_value
        self.map_macro_properties(source_conf_item, getattr(source_item, 'macros', ''))
        
        return source_conf_item


class MapperSourceToConf(object):
    
    def __init__(self, logger):
        # type: (PartLogger) -> None
        self._mapping = {
            ITEM_TYPE.BUSINESSIMPACTMODULATIONS: CONFIGURATION_MAPPING_BUSINESS_IMPACT_MODULATION,
            ITEM_TYPE.CLUSTERS                 : CONFIGURATION_MAPPING_CLUSTER,
            ITEM_TYPE.CLUSTERTPLS              : CONFIGURATION_MAPPING_CLUSTER_TEMPLATE,
            ITEM_TYPE.COMMANDS                 : CONFIGURATION_MAPPING_COMMAND,
            ITEM_TYPE.CONTACTS                 : CONFIGURATION_MAPPING_CONTACT,
            ITEM_TYPE.CONTACTTPLS              : CONFIGURATION_MAPPING_CONTACT_TEMPLATE,
            ITEM_TYPE.CONTACTGROUPS            : CONFIGURATION_MAPPING_CONTACT_GROUP,
            ITEM_TYPE.ESCALATIONS              : CONFIGURATION_MAPPING_ESCALATION,
            ITEM_TYPE.HOSTS                    : CONFIGURATION_MAPPING_HOST,
            ITEM_TYPE.HOSTTPLS                 : CONFIGURATION_MAPPING_HOST_TEMPLATE,
            ITEM_TYPE.HOSTGROUPS               : CONFIGURATION_MAPPING_HOST_GROUP,
            ITEM_TYPE.MACROMODULATIONS         : CONFIGURATION_MAPPING_MACRO_MODULATION,
            ITEM_TYPE.NOTIFICATIONWAYS         : CONFIGURATION_MAPPING_NOTIFICATION_WAY,
            ITEM_TYPE.RESULTMODULATIONS        : CONFIGURATION_MAPPING_RESULT_MODULATION,
            ITEM_TYPE.SERVICESCLUSTERS         : CONFIGURATION_MAPPING_SERVICE_CLUSTER,
            ITEM_TYPE.SERVICESCLUSTERTPLS      : CONFIGURATION_MAPPING_SERVICE_CLUSTER_TEMPLATE,
            ITEM_TYPE.SERVICESHOSTS            : CONFIGURATION_MAPPING_SERVICE_HOST,
            ITEM_TYPE.SERVICESHOSTTPLS         : CONFIGURATION_MAPPING_SERVICE_HOST_TEMPLATE,
            ITEM_TYPE.SERVICETPLS              : CONFIGURATION_MAPPING_SERVICE_TEMPLATE,
            ITEM_TYPE.TIMEPERIODS              : CONFIGURATION_MAPPING_TIME_PERIOD,
            
        }
        self._mappers = {}  # type: Dict[ItemType, GenericMapperSourceToConf]
        self.logger = logger
        
        self._init_mapper()
    
    
    def _init_mapper(self):
        for item_type in ITEM_TYPE.ALL_TYPES:
            self._mappers[item_type] = GenericMapperSourceToConf(item_type, self._mapping.get(item_type, {}))
    
    
    def map(self, source_item):
        # type: (SourceItem) -> SourceConfItem
        source_item_factory = self._mappers[source_item.item_type]
        return source_item_factory.map(source_item)
