#!/usr/bin/python
# -*- coding: utf-8 -*-
#
# Copyright (C) 2013-2018:
# This file is part of Shinken Enterprise, all rights reserved.

import errno
import os
import subprocess
from shinken.log import logger
from shinken.util import make_unicode
from stat import S_ISREG, S_IRUSR, S_IWUSR, S_IXUSR, S_IRGRP, S_IWGRP, S_IXGRP, S_IROTH, S_IWOTH, S_IXOTH

try:
    from pwd import getpwnam
    from grp import getgrnam
except ImportError as e:
    # temporary workaround:
    logger.debug("pwd and grp modules not available on this platform: %s => using mocks instead", e)
    from collections import namedtuple
    
    PasswordEntry = namedtuple('PasswordEntry', 'pw_name pw_passwd pw_uid pw_gid pw_gecos pw_dir pw_shell')
    
    
    def getpwnam(user):
        return PasswordEntry(pw_name='shinken',
                             pw_passwd='',
                             pw_uid=0,
                             pw_gid=0,
                             pw_gecos='Shinken User',
                             pw_dir='C:/Users/shinken',
                             pw_shell='cmd.exe')
    
    
    GroupEntry = namedtuple('GroupEntry', 'gr_name gr_passwd gr_gid gr_mem')
    
    
    def getgrnam(group):
        return GroupEntry(gr_name='shinken',
                          gr_passwd='',
                          gr_gid=0,
                          gr_mem='shinken')

# This dict use the uniw mask to compute the rights
unix_rights = {
    'user' : {
        'read'   : S_IRUSR,
        'write'  : S_IWUSR,
        'execute': S_IXUSR
    },
    'group': {
        'read'   : S_IRGRP,
        'write'  : S_IWGRP,
        'execute': S_IXGRP
        
    },
    'other': {
        'read'   : S_IROTH,
        'write'  : S_IWOTH,
        'execute': S_IXOTH
    }
}


def check_right_on_path(path, mode='read', path_type='directory', user=None, group=None):
    ''' Use this function to check the rights on folder and file.
    :param path: the path to check
    :param mode: which mode you want for your path : 'read', 'write', 'execute'
    :param path_type: is your path is a directory or a file
    :param user: on which user check right ? if None, it will use the current process’s effective user id
    :param group: on which group check right ? if None, it will use the current process’s effective group id
    :return: True if rights are OK. Raise if you can access to your file
    '''
    
    path = os.path.join(path)
    # First check if the path exists
    if not os.path.exists(path):
        raise OSError(errno.ENOENT, 'The path doesn\'t exists', path)
    
    # Is the target is the good one ?
    if (path_type == 'directory' and not os.path.isdir(path)) or \
            (path_type == 'file' and not os.path.isfile(path)):
        raise OSError(errno.ENOENT, 'This is not a %s  : ' % (path_type), path)
    
    # Will check if we use the user, the group or other
    user_uid = getpwnam(user).pw_uid if user else os.geteuid()
    
    if user_uid == 0:
        # You're root ? ... ok you can go
        return True
    
    group_gid = getgrnam(group).gr_gid if group else os.getegid()
    file_stat = os.stat(path)
    
    if user_uid == file_stat.st_uid:
        key_used = 'user'
    elif group_gid == file_stat.st_gid:
        key_used = 'group'
    else:
        key_used = 'other'
    
    # Now will check the rights with the good mask
    if bool(file_stat.st_mode & unix_rights[key_used][mode]):
        return True
    else:
        raise OSError(errno.EACCES, 'User %s can not access to the %s with the %s mode ' % (user, path_type, mode), path)


def check_right_on_file(path, mode='read', user=None, group=None):
    ''' Use this function to check the rights on file.
    :param path: the path to check
    :param mode: which mode you want for your path : 'read', 'write', 'execute'
    :param user: on which user check right ? if None, it will use the current process’s effective user id
    :param group: on which group check right ? if None, it will use the current process’s effective group id
    :return: True if rights are OK. Raise if you can access to your file
    '''
    return check_right_on_path(path, mode, 'file', user, group)


def check_right_on_directory(path, mode='read', user=None, group=None):
    ''' Use this function to check the rights on folder.
    :param path: the path to check
    :param mode: which mode you want for your path : 'read', 'write', 'execute'
    :param user: on which user check right ? if None, it will use the current process’s effective user id
    :param group: on which group check right ? if None, it will use the current process’s effective group id
    :return: True if rights are OK. Raise if you can access to your folder
    '''
    return check_right_on_path(path, mode, 'directory', user, group)


def create_tree(path, user='shinken', group='shinken', mode=0755):
    # Warning! the mode will be overwrite by os.umask if set
    full_path = [p for p in path.split(os.sep) if p]
    # to manage relatif path or not
    parents = '/' if os.path.isabs(path) else ''
    for dir in full_path:
        
        current_path = os.path.join(parents, dir)
        
        if not os.path.exists(current_path):
            try:
                check_right_on_directory(parents, 'write')
            except OSError:
                raise OSError(errno.EACCES, 'Cannot create the full tree. Permission denied ', parents)
            os.mkdir(current_path, mode)
            set_ownership(current_path, user, group)
        
        try:
            check_right_on_directory(current_path, 'read', user, group)
            check_right_on_directory(current_path, 'execute', user, group)
        except OSError:
            raise OSError(errno.EACCES, 'The path already exists but cannot jump in. Permission denied ', current_path)
        
        # Now, next level
        parents = current_path


def run_command(command):
    _process = subprocess.Popen(command, stdout=subprocess.PIPE, stderr=subprocess.PIPE, shell=True)
    ret = _process.communicate()
    return '%s\n%s' % (make_unicode(ret[0][:-1]), make_unicode(ret[1][:-1]))


def run_command_with_return_code(command):
    _process = subprocess.Popen(command, stdout=subprocess.PIPE, stderr=subprocess.PIPE, shell=True)
    ret = _process.communicate()
    # return returncode, stdout, stderr
    return _process.returncode, make_unicode(ret[0][:-1]), make_unicode(ret[1][:-1])


def is_regular_file(path):
    if os.path.exists(path):
        mode = os.stat(path).st_mode
        return S_ISREG(mode)
    return False


def set_ownership(path, user='shinken', group='shinken'):
    try:
        uid = getpwnam(user).pw_uid
    except KeyError:
        raise OSError(errno.EINVAL, 'This user doesn\'t exists', user)
    try:
        gid = getgrnam(group).gr_gid
    except KeyError:
        raise OSError(errno.EINVAL, 'This group doesn\'t exists', group)
    os.chown(path, uid, gid)
