#!/usr/bin/env python

# Copyright (C) 2009-2012:
#    Gabes Jean, naparuba@gmail.com
#    Gerhard Lausser, Gerhard.Lausser@consol.de
#    Gregory Starck, g.starck@gmail.com
#    Hartmut Goebel, h.goebel@goebel-consult.de
#
# This file is part of Shinken.
#
# Shinken is free software: you can redistribute it and/or modify
# it under the terms of the GNU Affero General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# Shinken is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Affero General Public License for more details.
#
# You should have received a copy of the GNU Affero General Public License
# along with Shinken.  If not, see <http://www.gnu.org/licenses/>.


import os
import pycurl
import urllib
import sys
import json
from cStringIO import StringIO


from shinken.objects import Host
from shinken.log import logger, cprint



# Will be populated by the shinken CLI command
CONFIG = None



def __get(page, args):
    srv = CONFIG['conf']['address']
    master_key = CONFIG['conf']['master_key']
    
    # Ok we will push the file with a 10s timeout
    c = pycurl.Curl()
    c.setopt(c.POST, 0)
    c.setopt(c.CONNECTTIMEOUT, 30)
    c.setopt(c.TIMEOUT, 300)

    args['master_key'] = master_key
    c.setopt(c.URL, str( ('localhost:7765/%s?'%page)+urllib.urlencode(args)))
    response = StringIO()
    c.setopt(pycurl.WRITEFUNCTION, response.write)
    #c.setopt(c.VERBOSE, 1)
    try:
        c.perform()
    except pycurl.error, exp:
        logger.error("There was a critical error : %s" % exp)
        return

    r = c.getinfo(pycurl.HTTP_CODE)
    c.close()
    if r != 200:
        logger.error("There was a critical error : %s" % response.getvalue())
        sys.exit(2)
    else:
        ret  = json.loads(response.getvalue().replace('\\/', '/'))
        return ret

    

################" *********************** LIST *************** ##################

def _do_list(cls='host', _filter=''):
    args = {'table':cls,'filter':_filter}
    return __get('get_element_list', args)


def print_hosts(elts):
    cprint('#name;address;tags;hostgroups;_id', 'magenta')
    for e in elts:
        name = e.get('host_name', e.get('name', '(no name)'))
        is_tpl = e.get('register', '1') == '0'

        cprint('%s' % name , 'green', end='')
        cprint(';%s;%s;%s;%s' %  (e.get('address',''), e.get('use', ''), e.get('hostgroups',''), e.get('_id')))


def print_services(elts):
    cprint('#name;host_name;tags;hostgroups;check_command;_id', 'magenta')
    for e in elts:
        name = e.get('service_description', e.get('name', '(no name)'))
        is_tpl = e.get('register', '1') == '0'
        cprint('%s' % name , 'green', end='')
        cprint(';%s;%s;%s;%s;%s' %  (e.get('host_name',''), e.get('use', ''), e.get('hostgroups',''), e.get('check_command',''), e.get('_id')))
    
    
    
def do_list(table='host', templates=None):
    logger.debug("CALLING", table, templates)
    _filter = {}
    if templates == '1':
        print "ONLY TEMPLATES"
        _filter['register'] = '0'
    else:
        _filter['register'] = {'$ne': '0'}
    _filter = json.dumps(_filter)
    r = _do_list(table, _filter)
    rc = r.get('rc')
    if rc != 200:
        logger.error(r.get('error','unkown error'))
        return
    elts = r['result']
    if table == 'host':
        print_hosts(elts)
    if table == 'service':
        print_services(elts)
    if table == 'commands':
        pass
    #print r
    return


########################################      ADD      #############################

def _do_add_host(values):
    # value should be 
    # Name;display_name;address;tags;hostgroups
    try:
        elts = values.split(';', 4)
        name, display_name, address, tags, hostgroups = elts
    except ValueError: # bad number
        logger.error('Missing arguments. Value string should match hostname;display_name;address;use;hostgroups')
        return
    # Maybe there is already such a host? if so we will erase it
    r = _do_list(cls='host', _filter='{"host_name":"%s"}' % name)
    _id = ''
    _SYNC_KEYS = []
    if len(r['result']) >= 1:
        # we need to erase it... so take it's _id
        _id = r['result'][0]['_id']
        _SYNC_KEYS = r['result'][0]['_SYNC_KEYS']
        print "Erasing _id", _id, _SYNC_KEYS

    args = {}
    e = {}
    if _id:
        e['_id'] = _id
        e['_SYNC_KEYS'] = _SYNC_KEYS
    e['host_name'] = name
    e['display_name'] = display_name
    e['address'] = address
    e['use'] = tags
    e['hostgroups'] = hostgroups
    args['obj'] = json.dumps(e)
    r = __get('put_new_element', args)
    print r



def _do_add_service(values):
    # value should be 
    # hostname;service_description;use
    try:
        elts = values.split(';', 2)
        hname, sdesc, use = elts
    except ValueError: # bad number
        logger.error('Missing arguments. Value string should match hostname;service_description;use')
        return
    # Maybe there is already such a host? if so we will erase it
    r = _do_list(cls='service', _filter='{"host_name":"%s", "service_description":"%s"}' % (hname, sdesc))
    _id = ''
    _SYNC_KEYS = []
    if len(r['result']) >= 1:
        # we need to erase it... so take it's _id
        _id = r['result'][0]['_id']
        _SYNC_KEYS = r['result'][0]['_SYNC_KEYS']
        print "Erasing _id", _id, _SYNC_KEYS
    
    args = {'table':'service'}
    e = {}
    if _id:
        e['_id'] = _id
        e['_SYNC_KEYS'] = _SYNC_KEYS
    e['host_name'] = hname
    e['service_description'] = sdesc
    e['use'] = use
    args['obj'] = json.dumps(e)
    r = __get('put_new_element', args)
    print r
    
    

def do_add(table='host', value=None):
    logger.debug("CALLING add", table, value)
    if table == 'host':    
        r = _do_add_host(value)
    if table == 'service':    
        r = _do_add_service(value)

    return





########################################      update      #############################

def _do_update_host(values):
    # value should be         
    # Name;display_name;address;tags;hostgroups
    try:
        elts = values.split(';', 2)
        name, prop, value = elts
    except ValueError: # bad number
        logger.error('Missing arguments. Value should match Name;display_name;address;tags;hostgroups')
        return
    # Maybe there is already such a host? if so we will erase it
    r = _do_list(cls='host', _filter='{"host_name":"%s"}' % name)
    _id = ''
    _SYNC_KEYS = []
    
    if len(r['result']) == 0:
        logger.error('No such element')
        return
    
    # we need to erase it... so take it's _id
    _id = r['result'][0]['_id']
    
    args = {'_id':_id}
    e = {}
    e[prop] = value
    args['obj'] = json.dumps(e)
    r = __get('update_element', args)
    print r
    

def _do_update_service(values):
    # value should be 
    # hostname;service_description;prop;value
    try:
        elts = values.split(';', 3)
        hname, sdesc, prop, value = elts
    except ValueError: # bad number
        logger.error('Missing arguments. Value string should match hostname;service_description;property;value')
        return
    # Maybe there is already such a host? if so we will erase it
    r = _do_list(cls='service', _filter='{"host_name":"%s", "service_description":"%s"}' % (hname, sdesc))
    _id = ''
    _SYNC_KEYS = []
    
    if len(r['result']) == 0:
        logger.error('No such element')
        return
    
    # we need to erase it... so take it's _id
    _id = r['result'][0]['_id']
    
    args = {'_id':_id, 'table':'service'}
    e = {}
    e[prop] = value
    args['obj'] = json.dumps(e)
    r = __get('update_element', args)
    print r
    
    

def do_update(table='host', value=None):
    logger.debug("CALLING update", table, value)
    if value is None:
        logger.error('No values')
        return
    if table == 'host':    
        r = _do_update_host(value)
    if table == 'service':
        r = _do_update_service(value)
    return




exports = {
    do_list : {
        'keywords': ['cfg-list'],
        'args': [
            {'name' : '--table', 'default':'host', 'description':'Object type to describe'},
            {'name' : '--templates', 'default':'0', 'description':'Show only templates'},
            ],
        'description': 'List the elements'
        },
    do_add : {
        'keywords': ['cfg-add'],
        'args': [
            {'name' : '--table', 'default':'host', 'description':'Object type to insert'},
            {'name' : '--value', 'default':'', 'description':'For host: Name;display_name;address;tags;hostgroups'},
            ],
        'description': 'Add a new element',
        },
    do_update : {
        'keywords': ['cfg-update'],
        'args': [
            {'name' : '--table', 'default':'host', 'description':'Object type to update'},
            {'name' : '--value', 'default':'', 'description':'For host: hotname;property;value'},
            ],
        'description': 'Update an element',
        },

    }


configurations = {
    'conf':{
        'comment': '''# Set your option for the synchronizer communication
# master_key: key access to talk to your synchronizer tool
# address: address of your synchronizer daemon, by default http://localhost:7765/
''',
            'values'  : [('address','http://localhost:7765/'), ('master_key','')]}    
}
