#!/usr/bin/python
# -*- coding: utf-8 -*-

# Copyright (C) 2009-2012:
#    Gabes Jean, naparuba@gmail.com
#    Gerhard Lausser, Gerhard.Lausser@consol.de
#    Gregory Starck, g.starck@gmail.com
#    Hartmut Goebel, h.goebel@goebel-consult.de
#
# This file is part of Shinken.
#
# Shinken is free software: you can redistribute it and/or modify
# it under the terms of the GNU Affero General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# Shinken is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Affero General Public License for more details.
#
# You should have received a copy of the GNU Affero General Public License
# along with Shinken.  If not, see <http://www.gnu.org/licenses/>.

import httplib
import json
import time
import urllib
import time

from shinken.log import logger
from shinken.misc.filter import is_authorized
from shinken.util import get_sec_from_morning, get_wday
from shinken.webui.bottlewebui import HTTPError

### Will be populated by the UI with it's own value
app = None
widget_display_name = 'Graphs'
widget_desc = '''Show the perfdata graph'''
default_options = [
    {'name': 'search', 'value': '', 'type': 'hst_AND_srv_AND_met', 'label': 'Metrics', 'required': True},
    {'name': 'show_warning', 'value': False, 'type': 'bool', 'label': 'Show warning thresholds (if available)', 'required': False},
    {'name': 'show_critical', 'value': False, 'type': 'bool', 'label': 'Show critical thresholds (if available)', 'required': False},
    {'name': 'y_min', 'value': '', 'type': 'int', 'label': 'Graph minimum', 'required': False},
    {'name': 'y_max', 'value': '', 'type': 'int', 'label': 'Graph maximum', 'required': False},
]

HTTP_ERROR_GRAPHITE_SERVER_NOT_FOUND = 512
HTTP_ERROR_GRAPHITE = 513

last_graphite_errors = {}

GRAPHITE_PING_TIMEOUT = 10
GRAPHITE_DATA_TIMEOUT = 20
MIN_TIME_BETWEEN_GRAPHITE_RETRY = 120


# clean error count oldest than 24h
def _clean_http_graphite_error(app):
    try:
        hours_since_epoch = int(time.time()) / 3600
        for key in app.graphs_errors.keys():
            if (int(key) < (hours_since_epoch - 24)) or int(key) > hours_since_epoch:
                del app.http_errors_count[key]
    except Exception:
        # if an error occurec here, do nothing
        pass


# record errors in app.graphs_errors and count them by hour
def _add_http_graphite_error(app, error):
    hours_since_epoch = int(time.time()) / 3600
    if hours_since_epoch in app.graphs_errors:
        app.graphs_errors[int(hours_since_epoch)] += 1
    else:
        app.graphs_errors[int(hours_since_epoch)] = 1


# Internal
def _parse_common_graphite_return(item, item_name, check_name, data):
    res = []
    try:
        for line in data.splitlines():
            metric_res = {'trending': False}
            elts = line.split('|')
            meta = elts[0]
            raw_values = elts[1]
            elts = meta.split(',')
            name = elts[0]
            start = int(elts[1])
            end = int(elts[2])
            step = int(elts[3])
            
            metric_res['name'] = name
            metric_res['host_name'] = item_name
            metric_res['check_name'] = check_name
            metric_res['mname'] = name.split('.')[-1]
            metric_res['metric_name'] = metric_res['mname'].replace('SLASH', '/')
            metric_res['start'] = start
            metric_res['end'] = end
            metric_res['has_value'] = False
            metric_res['uuid'] = item.get_instance_uuid()
            metric_res['interval'] = step
            
            values = []
            elts = raw_values.split(',')
            elts.reverse()
            
            pos = end
            last_val = end
            for e in elts:
                if not e.startswith('None'):
                    v = float(e)
                    t = [pos * 1000, v]
                    values.append(t)
                    metric_res['has_value'] = True
                    last_val = pos
                elif last_val - pos > (item.check_interval * 2 * 60):
                    values.append([pos * 1000, None])
                pos -= step
            
            values.reverse()
            metric_res['values'] = values
            res.append(metric_res)
    except:
        logger.error('data from graphite : [%s]' % data)
        logger.print_stack()
        return app.abort(HTTP_ERROR_GRAPHITE_SERVER_NOT_FOUND, "broker error see log")
        pass
    
    return res


def _parse_host_graphite_return(data, host_uuid):
    _clean_http_graphite_error(app)
    item_host = app.datamgr.get_host_by_uuid(host_uuid)
    if not item_host:
        return app.abort(404, "[_parse_graphite_return ] object %s was not found" % host_uuid)
    logger.debug("[_parse_graphite_return] check time:[%s]" % item_host.check_interval)
    return _parse_common_graphite_return(item_host, item_host.get_name(), app._('graphics.check_host_alive'), data)


def _parse_service_graphite_return(data, check_uuid):
    _clean_http_graphite_error(app)
    item_check = app.datamgr.get_service_by_uuid(check_uuid)
    if not item_check:
        return app.abort(404, "[_parse_graphite_return ] object %s was not found" % item_check.get_instance_uuid())
    logger.debug("[_parse_graphite_return] check time:[%s]" % item_check.check_interval)
    return _parse_common_graphite_return(item_check, item_check.host.get_name(), item_check.get_name(), data)


def _get_graphite_uri(host):
    _clean_http_graphite_error(app)
    graphite_backends = app.myconf.graphite_backends.split(',')
    backends = []
    for gb in graphite_backends:
        elts = gb.split(':')
        if len(elts) != 2:
            continue
        backends.append((elts[0].strip(), elts[1].strip()))
    
    if not backends:
        logger.error('[_get_graphite_uri] no backends found')
        return None
    
    realm = host.realm
    for b in backends:
        (r, u) = b
        if r == realm:
            return u
    # Ok not found? take the default one, the '*' entry
    realm = '*'
    for b in backends:
        (r, u) = b
        if r == realm:
            return u
    return None


def _sort_services(s1, s2):
    return cmp(s1.service_description.lower(), s2.service_description.lower())


def _sort_have_metrics(e1, e2):
    return cmp(e2['have_metrics'], e1['have_metrics'])


def _get_timezone_offset():
    is_dst = time.daylight and time.localtime().tm_isdst > 0
    return int(time.altzone if is_dst else time.timezone) / 60


# We will try to query the front page of graphite in a small time 5s, and if fail, kill this server
# for a while (60s)
def _get_graphite_connection(uri):
    now = int(time.time())
    
    min_time_between_graphite_retry = int(getattr(app.myconf, 'metrology__after_error_wait_before_retry', MIN_TIME_BETWEEN_GRAPHITE_RETRY))
    graphite_ping_timeout = int(getattr(app.myconf, 'metrology__ping_timeout', GRAPHITE_PING_TIMEOUT))
    graphite_data_timeout = int(getattr(app.myconf, 'metrology__query_timeout', GRAPHITE_DATA_TIMEOUT))
    
    # If we are too close of a graphite error
    if now < last_graphite_errors.get(uri, 0.0) + min_time_between_graphite_retry:
        return None
    # Try a ping
    ping_conn = httplib.HTTPConnection(uri, timeout=graphite_ping_timeout)
    try:
        ping_conn.request("GET", "/")  # ping, small static page
        response = ping_conn.getresponse()
        buf = response.read()  # be sure to read, to be sure we did test the whole connection way
        ping_conn.close()
    except Exception as err:
        last_graphite_errors[uri] = int(time.time())
        logger.warning('[METROLOGY] [graphite:%s] Cannot join graphite server in %s timeout (%s). The graphite server is not available until %ss before a retry.' % (uri, graphite_ping_timeout, err, min_time_between_graphite_retry))
        _add_http_graphite_error(app, HTTP_ERROR_GRAPHITE)
        return None
    
    # We did succeed, but the conn is too low in timeout, need a bigger one to allow
    data_conn = httplib.HTTPConnection(uri, timeout=graphite_data_timeout)
    return data_conn


def _get_metrics_from_graphite(uri, graphite_query):
    conn = _get_graphite_connection(uri)
    if conn is None:
        return app.abort(HTTP_ERROR_GRAPHITE, "[_get_metric_info] Cannot request the graphite server")
    
    logger.debug("[_get_metric_info] target : %s" % graphite_query)
    
    t0 = time.time()
    try:
        # search_by_name: let graphite know we are using uuids directly
        conn.request("GET", "/render/?target=%s&rawData=true&from=-1minutes&search_by_name=0" % graphite_query)
        response = conn.getresponse()
        graphite_return = response.read()
        conn.close()
    except Exception as err:
        logger.info('GRAPHITE ERROR: %s %s %s' % (err, type(err), time.time() - t0))
        return app.abort(HTTP_ERROR_GRAPHITE_SERVER_NOT_FOUND, err.message)
    
    # graphite return an error
    if response.status == 500:
        # if the data1 contain some string, return some special errors
        if "[Errno 13] Permission denied" in graphite_return:
            _add_http_graphite_error(app, HTTP_ERROR_GRAPHITE)
            return app.abort(HTTP_ERROR_GRAPHITE, "[_get_metric_info] cannot access metrology data, check the /opt/graphite/storage/whisper/ read access for the host %s." % item_check.host.get_name())
        else:
            return app.abort(HTTP_ERROR_GRAPHITE, "[_get_metric_info] %s" % graphite_return)
    
    return graphite_return


def _compute_metric_infos(metrics):
    metric_infos = []
    metric_names = map(lambda metric: metric['name'], metrics)
    for metric in filter(lambda metric: not (metric['name'].endswith('_warn') or metric['name'].endswith('_crit')), metrics):
        metric_dict = {}
        name = metric['name']
        t = name.split('.')
        if len(t) == 0:
            continue
        mname = t[-1]
        mname = mname.replace('SLASH', '/')
        metric_dict['name'] = mname
        metric_dict['warn'] = ("%s_warn" % name) in metric_names
        metric_dict['crit'] = ("%s_crit" % name) in metric_names
        
        metric_infos.append(metric_dict)
    return metric_infos


def _get_service_metric_info(item_check):
    _clean_http_graphite_error(app)
    uri = _get_graphite_uri(item_check.host)
    if not uri:
        return app.abort(HTTP_ERROR_GRAPHITE_SERVER_NOT_FOUND, '[_get_metric_info] %s not found in what graphite server datas are.' % item_check.host.get_name())
    graphite_query = '%s.*' % (item_check.get_instance_uuid().replace('-', '.'))
    graphite_return = _get_metrics_from_graphite(uri, graphite_query)
    
    metrics = _parse_service_graphite_return(graphite_return, item_check.get_instance_uuid())
    
    return _compute_metric_infos(metrics)


def _get_host_metric_info(item_host):
    _clean_http_graphite_error(app)
    uri = _get_graphite_uri(item_host)
    if not uri:
        return app.abort(HTTP_ERROR_GRAPHITE_SERVER_NOT_FOUND, '[_get_metric_info] %s not found in what graphite server datas are.' % item_host.get_name())
    
    graphite_query = '%s.__HOST__.*' % (item_host.get_instance_uuid().replace('-', '.'))
    graphite_return = _get_metrics_from_graphite(uri, graphite_query)
    
    metrics = _parse_host_graphite_return(graphite_return, item_host.get_instance_uuid())
    
    return _compute_metric_infos(metrics)


# iframe
def get_graph_json(uuid):
    _clean_http_graphite_error(app)
    user = app.get_user_auth()
    item = app.datamgr.get_host_by_uuid(uuid)
    _item_is_a_service = False
    
    if not item:
        item = app.datamgr.get_service_by_uuid(uuid)
        _item_is_a_service = True
    
    if item is None:
        item = {
            'uuid'      : uuid,
            'is_unknown': True
        }
        return app.abort(404, item, True)
    
    if not is_authorized(item, user, app):
        logger.info('[get_graph_json] User is not authorized to view this element')
        item = {
            'uuid'      : uuid,
            'is_unknown': False
        }
        return app.abort(403, item, True)
    
    request = app.request
    metric = request.GET.get('metric', '*')
    _from = request.GET.get('from', '-1d')
    do_trending = (request.GET.get('trending', '0') == '1')
    
    _from_orig = _from
    # If it's not a -, it's a +, so in the future, so get from few past
    if not _from.startswith('-'):
        _from = '-1h'
    
    # Hook metric / into SLASH
    metric = metric.replace('/', 'SLASH')
    if _item_is_a_service:
        uri = _get_graphite_uri(item.host)
    else:
        uri = _get_graphite_uri(item)
    
    if not uri:
        return app.abort(HTTP_ERROR_GRAPHITE_SERVER_NOT_FOUND, "get_graph_json can not contact graphite", True)
    
    conn = _get_graphite_connection(uri)
    if conn is None:
        return app.abort(HTTP_ERROR_GRAPHITE, "[_get_metric_info] Cannot request the graphite server")
    
    targets = []
    if _item_is_a_service:
        if metric == '*':
            targets.append('target=%s.%s' % (uuid.replace('-', '.'), metric))
        else:
            for suf in ['', '_crit', '_warn']:
                targets.append('target=%s.%s%s' % (uuid.replace('-', '.'), metric, suf))
    else:
        if metric == '*':
            targets.append('target=%s.__HOST__.%s' % (uuid, metric))
        else:
            for suf in ['', '_crit', '_warn']:
                targets.append('target=%s.__HOST__.%s%s' % (uuid, metric, suf))
    target = "&".join(targets)
    logger.debug("[get_graph_json] target %s" % target)
    
    try:
        # search_by_name: let graphite know we are using uuids directly
        conn.request("GET", "/render/?%s&rawData=true&from=%s&search_by_name=0" % (target, _from))
        response = conn.getresponse()
        graphite_return = response.read()
        conn.close()
    except:
        return app.abort(HTTP_ERROR_GRAPHITE_SERVER_NOT_FOUND, '[get_graph_json] can not contact graphite', True)
    
    # graphite return an error
    if response.status == 500:
        # if the data1 contain some string, return some special errors
        if "[Errno 13] Permission denied" in graphite_return:
            logger.error("[get_graph_json] Error code 513, cannot access metrology data, check the /opt/graphite/storage/whisper/ read access for the host %s." % uuid)
            _add_http_graphite_error(app, HTTP_ERROR_GRAPHITE)
            return app.abort(HTTP_ERROR_GRAPHITE, "Graphite don't have access to your whisper storage", True)
    
    if _item_is_a_service:
        datas = _parse_service_graphite_return(graphite_return, uuid)
    else:
        datas = _parse_host_graphite_return(graphite_return, uuid)
    
    metrics_to_trend = []
    for data in datas:
        elts = data['name'].split(',', 1)
        name = elts[0]
        elts = name.split('.')
        if len(elts) < 3:  # Bad name?
            continue
        host_uuid = elts[0]
        
        metric = elts[-1]
        if metric.endswith('_crit') or metric.endswith('_warn'):  # not a real metric
            continue
        check_uuid = '.'.join(elts[1:-1])  # take between
        logger.debug("[get_graph_json] LOOKING FOR TRENDING FOR %s %s %s" % (host_uuid, check_uuid, metric))
        metrics_to_trend.append((host_uuid, check_uuid, metric, data['start'], data['end']))
    
    if do_trending:
        logger.debug("[get_graph_json] do_trending")
        # first look at the module and so the db
        db = None
        col = None
        for mod in app.modules_manager.get_all_instances():
            if mod.properties['type'] != 'trending':
                continue
            # the module should already got the db
            db = getattr(mod, 'db', None)
            if db:
                col = getattr(db, 'trending', None)
        if col is None:
            logger.error('[get_graph_json]  No trending module found')
        else:  # do the job :)
            logger.debug("[get_graph_json] metrics_to_trend %s " % metrics_to_trend)
            for (host_uuid, check_uuid, metric, start, end) in metrics_to_trend:
                # maybe we are asking for forecast, if so hack the future :)
                if not _from_orig.startswith('-'):
                    now = int(time.time())
                    start = now - 3600  # minus 1 hour
                    if _from_orig == '1d':
                        end = now + 86400
                    if _from_orig == '1w':
                        end = now + 86400 * 7
                    if _from_orig == '1month':
                        end = now + 86400 * 31
                    if _from_orig == '1y':
                        end = now + 86400 * 366
                metric_res = {
                    'name'    : '.'.join([host_uuid, check_uuid, metric + '_trend']),
                    'mname'   : metric + '_trend',
                    'start'   : start,
                    'end'     : end,
                    'trending': True,
                }
                logger.debug("[get_graph_json] metric_res %s " % metric_res)
                values = []
                metric_res['values'] = values
                CHUNK_INTERVAL = 900
                t = (start / 900) * 900
                # We will got all db entries to do just one query instead of numerous ones
                all_trend = {}
                for e in col.find({"hname": host_uuid.replace(' ', '_').replace('DOT', '_'), "sdesc": check_uuid.replace(' ', '_'), "metric": metric, "cycle": 'week'}):
                    all_trend[e['_id']] = e
                while t <= end:
                    # if we got a forecast, look in the future from now
                    if not _from_orig.startswith('-'):
                        nb_week_offset = int((t - start) / (86400 * 7))
                    else:
                        nb_week_offset = int((end - t) / (86400 * 7))
                    # but if past, look from the end, so end
                    sec_from_morning = get_sec_from_morning(t)
                    wday = get_wday(t)
                    chunk_nb = sec_from_morning / CHUNK_INTERVAL
                    t += CHUNK_INTERVAL
                    key = '%s.%s.%s.week.%d.Vtrend.%d' % (host_uuid.replace('.', 'DOT').replace(' ', '_'), check_uuid.replace(' ', '_'), metric, wday, chunk_nb)
                    doc = all_trend.get(key, None)
                    v = None
                    if doc:
                        v = doc['VtrendSmooth']
                    # If we got a value, apply the number of week offset
                    if v:
                        if not _from_orig.startswith('-'):
                            v += nb_week_offset * doc['VevolutionSmooth']
                        else:
                            v -= nb_week_offset * doc['VevolutionSmooth']
                    
                    values.append([t * 1000, v])
                datas.append(metric_res)
    
    return json.dumps(datas)


# Widget
def get_host_metric_list(uuid):
    _clean_http_graphite_error(app)
    item = app.datamgr.get_host_by_uuid(uuid)
    if not item:
        logger.debug('[get_host_metric_list] host not found')
        return app.abort(404, [], True)
    
    list_checks = []
    for service in item.services:
        
        try:
            metric_info = _get_service_metric_info(service)
        except HTTPError:
            metric_info = []
        
        data = {
            'uuid'               : service.get_instance_uuid(),
            'service_description': service.service_description,
            'metrics'            : metric_info,
            'sort_key'           : service.service_description
        }
        list_checks.append(data)
    
    _host_metric_infos = _get_host_metric_info(item)
    if not item.got_business_rule:
        data_host = {
            'uuid'               : item.get_instance_uuid(),
            'service_description': app._('graphics.check_host_alive'),
            'metrics'            : _host_metric_infos,
            # We want this metric to appear at the top, so we use the first printable character (other than space) at the start of the sork key
            'sort_key'           : "!%s" % app._('graphics.check_host_alive')
        }
        list_checks.insert(0, data_host)
    
    return json.dumps(list_checks)


def loading_graphs_widget(uuid_dashboard, uuid_widget):
    width = app.request.GET.get('width', '')
    height = app.request.GET.get('height', '')
    metrics = app.request.GET.get('metrics', '')
    graph_min = app.request.GET.get('graph-min', '')
    graph_max = app.request.GET.get('graph-max', '')
    
    param_url = {
        "uuid_dashboard": uuid_dashboard,
        "uuid_widget"   : uuid_widget
    }
    
    param_request = {
        'width'    : width,
        'height'   : height,
        "type"     : "widget",
        "metrics"  : metrics,
        "graph_min": graph_min,
        "graph_max": graph_max
    }
    return {
        'app'                     : app,
        'load'                    : 'api/widget/graphs/page',
        'param_url'               : param_url,
        'offset_time_zone_backend': _get_timezone_offset(),
        'param_request'           : param_request
    }


def get_graphs_widget(uuid_dashboard, uuid_widget):
    width = app.request.GET.get('width', '250')
    height = app.request.GET.get('height', '275')
    
    return {
        'code_http'      : 200,
        'uuid_dashboard' : uuid_dashboard,
        'uuid_widget'    : uuid_widget,
        'widget_width'   : width,
        'widget_height'  : height,
        'colors_graphics': app.colors_graphics,
        'app'            : app,
        'name_elt_info'  : json.dumps({'code_http': 200})
    }


# Panel
def loading_graphs_panel(uuid):
    width = app.request.GET.get('width', '')
    height = app.request.GET.get('height', '')
    metric = app.request.GET.get('metric', '')
    
    if metric:
        metric = urllib.quote_plus(metric)
    
    param_request = {
        'width'     : width,
        'type'      : "detail",
        'height'    : height,
        'metric'    : metric,
        'width_host': app.request.GET.get('width_host', '200')
    }
    
    return {
        'app'                     : app,
        'load'                    : 'api/graphs/panel/detail/page',
        'offset_time_zone_backend': _get_timezone_offset(),
        'param_url'               : uuid,
        'param_request'           : param_request
    }


def get_graphs_panel(uuid):
    _clean_http_graphite_error(app)
    user = app.get_user_auth()
    
    # Look for an host or a service?
    host = None
    sdesc = ""
    if app.is_check_uuid(uuid):
        check = app.datamgr.get_service_by_uuid(uuid)
        if check:
            host = check.host
            sdesc = check.service_description
    else:
        host = app.datamgr.get_host_by_uuid(uuid)
    
    if not host:
        logger.info('[get_graphs_panel] host not found')
        return {'app': app, 'code_http': 404, 'uuid': uuid}
    
    if not is_authorized(host, user, app):
        logger.info('[get_graphs_panel] User is not authorized to view this element')
        return {'app': app, 'code_http': 403, 'uuid': uuid}
    
    width = int(app.request.GET.get('width', '500'))
    height = int(app.request.GET.get('height', '571'))
    metric = app.request.GET.get('metric', '')
    
    services = host.services[:]
    services.sort(_sort_services)
    
    data = {
        'uuid'    : host.uuid,
        'hname'   : host.get_name(),
        'services': []
    }
    for service in services:
        try:
            tmp_metric_info = _get_service_metric_info(service)
        except HTTPError as err:
            logger.info(err.message)
            return {'app': app, 'code_http': err.status, 'uuid': uuid}
        
        data_service = {
            'sdesc'       : service.get_name(),
            'state'       : service.state.lower(),
            '_id'         : app.helper.get_html_id(service),
            'uuid'        : service.get_instance_uuid(),
            'metrics'     : tmp_metric_info,
            'have_metrics': len(tmp_metric_info) != 0
        }
        
        data['services'].append(data_service)
    data['services'].sort(_sort_have_metrics)
    
    _host_metric_infos = _get_host_metric_info(host)
    if not host.got_business_rule:
        data_host = {
            'sdesc'       : app._('graphics.check_host_alive'),
            'state'       : host.state.lower(),
            '_id'         : app.helper.get_html_id(host),
            'uuid'        : host.get_instance_uuid(),
            'metrics'     : _host_metric_infos,
            'have_metrics': len(_host_metric_infos) != 0
        }
        data['services'].insert(0, data_host)
    
    return {
        'app'            : app,
        'h'              : host,
        '_sort_services' : _sort_services,
        'width'          : width,
        'height'         : height,
        'uuid'           : uuid,
        'sdesc'          : sdesc,
        'colors_graphics': app.colors_graphics,
        'metric_selected': metric,
        'metric'         : metric,
        'data'           : data,
        'code_http'      : 200,
        'width_host'     : app.request.GET.get('width_host', '200')
    }


def get_graphs_errors():
    return json.dumps(sum(app.graphs_errors.itervalues()))


pages = {
    get_graph_json       : {'routes': ['/api/graphs/data/:uuid'], 'view': None, 'static': False, 'wrappers': ['auth', 'json']},
    
    get_host_metric_list : {'routes': ['/api/widget/graphs/checks-with-metrics/:uuid'], 'view': None, 'static': False, 'wrappers': ['auth', 'json']},
    loading_graphs_widget: {
        'routes'             : ['/api/widget/graphs/loading/:uuid_dashboard/:uuid_widget'],
        'view'               : 'loading_graphs',
        'static'             : True,
        'widget'             : ['dashboard'],
        'widget_display_name': widget_display_name,
        'widget_desc'        : widget_desc,
        'widget_name'        : 'graphs',
        'widget_size'        : {'width': 6, 'height': 4},
        'widget_options'     : default_options,
        'widget_favoritable' : False,
        'resizable'          : False,
        'wrappers'           : []
    },
    get_graphs_widget    : {'routes': ['/api/widget/graphs/page/:uuid_dashboard/:uuid_widget'], 'view': 'widget_graphs', 'static': True},
    
    loading_graphs_panel : {'routes': ['/api/graphs/panel/detail/loading/:uuid'], 'view': 'loading_graphs', 'static': True},
    get_graphs_panel     : {'routes': ['/api/graphs/panel/detail/page/:uuid'], 'view': 'detail_graphs', 'static': True},
    get_graphs_errors    : {'routes': ['/api/graphs/errors'], 'static': True, 'wrappers': []},
    
}
