#!/bin/bash

# Copyright (C) 2013-2018:
# This file is part of Shinken Enterprise, all rights reserved.


# load common variables (versions and co), and common functions
. lib/common.sh

function display_help {
  printf "./update.sh\n"
  printf "This updater is designed to easily update Shinken Enterprise on a RedHat/Centos 6 or 7 64bits edition\n"
  printf "\n"
  printf "  --disable-important-notices-user-input:  removes the need to hit a key when required during installation\n"
  printf "  --disable-daemons-restart-after-update:  the shinken daemons will be not restart after the installation is done\n"
  printf "  --activate-encryption [<keyname>]:       creates a new encryption key with the name you give, and activates encryption.\n"
  printf "                                               If no name is given, you will be prompted for it.\n"
  printf "                                               Note that this can be done at a later stage.\n"
  printf "  --package-update-only-on-conflict:       Will not try to update package by default if already installed, and will only update\n"
  printf "                                           them if need by a conflict on a new package installation.\n"
  exit 0
}  

# By default we will try to update packages only on conflict for new packages
export PACKAGE_UPDATE_ONLY_ON_CONFLICT="false"


SANATIZE_LOGFILE="${OUTPUT_FOLDER}/sanitize.log"
activate_database_encryption=0

while :
do
    case "$1" in
      -h | --help)
        display_help  # Call your function
	  # no shifting needed here, we're done.
	  exit 0
	    ;;
      --disable-important-notices-user-input)
        disable_user_input="disableuserinput"
        shift
        ;;
      --activate-encryption)
        activate_database_encryption=1
        if [ $# -ge 2 ]; then
            encryption_key_name="$2"
            shift
        else
            encryption_key_name=
        fi
        check_encryption_status
        shift
        ;;
      --disable-daemons-restart-after-update)
        disable_daemons_restart_after_update="disabledaemonsrestartafterupdate"
        shift
        ;;
      --package-update-only-on-conflict)
        export PACKAGE_UPDATE_ONLY_ON_CONFLICT="true"
        shift
        ;;
      --) # End of all options
        shift
	break
	;;
      -*)
        echo "Error: Unknown option: $1" >&2
	  exit 1
	    ;;
      *)  # No more options
	      break
	        ;;
    esac
done

# Must be run as root
assert_is_root


# Be sure the tarball rights are valid
restore_all_tarball_valid_rights


# Check if we are with a redhat or a centos OS. If not, exit
assert_redhat_or_centos


# Check if already installed. if so exit and use install.sh instead
assert_is_installed

# Check if the Synchronizer configuration file is corrupted regarding protected fields
assert_protected_fields_conf_not_corrupted

# Check if we do not upgrade from an older version
assert_current_version_not_too_old

# Check that if there is a mongo installed, the version is known and valid
assert_only_valid_mongo

print_header "Welcome to Shinken Enterprise installer"

print_std_line "This installer is designed to update Shinken Enterprise to the $DISPLAY_VERSION version"

# Check if the database contains several objects with same name and type in one state
IS_SYNCHRONIZER=$(shinken-daemons-has synchronizer 2>/dev/null)
if [[ "$IS_SYNCHRONIZER" != "NO" ]];
then
    assert_no_duplicates
fi


printf "\033[0m\n"
# Check if mongod is running
mongod_check
# Check global system state (dnsname, not running shinken, etc etc)
system_check

set_maintenance_mode

# As we already have all dependencies, we can check for selinux rules now (need selinux package)
assert_selinux

# Before update, backup the old configuration
# Only backup configuration if we are the synchronizer
if [[ "$IS_SYNCHRONIZER" != "NO" ]];
then
    printf '\n\n'
    print_header "Saving current configuration before update"
    backup_result=$(shinken-backup --configuration --user --output-name "backup-preupdate-version-$DISPLAY_VERSION" --output-directory "$OUTPUT_FOLDER" 2>&1)
    rc=$?
    echo "$backup_result"
    if [[ $rc -eq 2 ]]; then
        doexit "An error has occurred during the pre-update backup. The update is aborted, please contact your Support."
    fi
fi

printf '\n\n'
print_header "Updating Shinken Enterprise to version $DISPLAY_VERSION"

# Fix missing /etc/shinken directories is missing at the update (like if someone did move the packs directory)
assume_etc_shinken_directories

# Fix if need writes in the /etc/shinken
chown -R shinken:shinken /etc/shinken/ 2>/dev/null

# And we do NOT want to update/erase some previous files because the user can have edit them (but
# for a reason, the system&rpm can miss the modification and still erase them...
for ii in `find /etc/shinken -type f | grep -v _default | grep -v packs | grep -v discovery_rules | grep -v local-import | grep -E '\.cfg$'`; do
   touch "$ii"
done

printf "\n\n"



# MAIN DEPS="python-cherrypy python-simplejson nagios-plugins-all python-pymongo nmap python-pymongo-gridfs perl-Config-IniFiles perl-DateTime fping perl-Crypt-DES perl-Net-SNMP perl-Socket6 perl-Digest-HMAC perl-Digest-SHA1 perl-IO-stringy perl-DBD-MySQL python-gevent perl-Class-MethodMaker perl-Archive-Zip perl-Data-Dump perl-SOAP-Lite perl-XML-SAX perl-XML-NamespaceSupport perl-XML-LibXML perl-libxml-perl libuuid-devel perl-Nagios-Plugin redhat-lsb-core perl-Time-HiRes python-paramiko python-ldap pycairo"


print_std_line_no_end "Updating Shinken Enterprise dependencies... "
install_enterprise_deps
print_ok_result_line

# Remove comment on the pidfile line in mongo configuration files (bug in mongo init scripts on CentOS 6)
# Do it before updating mongo or else mongod daemon cannot be stopped and the update fails
fix_02_07_03_remove_pidfile_comment_in_mongodb_config_files

# Mongodb have it's own update way
print_std_line "Updating mongodb"
install_or_update_mongodb_rpm

# These fix must be launched before rpm to work on files before they are replaced by new ones
# We merge sources and modules configuration.
fix_02_05_00_merge_sources_modules

# TODO : This issue should be done using rpm --replacepkgs
CURRENT_VERSION=$(get_current_version)
if [[ "$CURRENT_VERSION" != "$VERSION" ]]; then
    # Clean old angular app leave after patch
    rm -fr /var/lib/shinken/modules/webui/htdocs/ui/scripts/* >/dev/null 2>/dev/null
fi

print_std_line "Updating Shinken Enterprise package ... "
dorpminstall "packages/shinken-enterprise-$VERSION.el$CENTOSRELEASE.x86_64.rpm" "Shinken Enterprise package"  "forceupdate"
# copy the 27 edition init.d script (shinken+shinken-gatherer). Theses line MUST BE EXECUTED just after the rpminstall
cp -rp dependencies/enterprise/init.d/* /etc/init.d/
chmod a+x /etc/init.d/shinken*
print_ok_result_line

# Be sure it start on the boot
chkconfig --add shinken

# We overwrite shinken-packs file
overwrite_shinken_packs

# we need to change the nmap-mac-prefixes file
assume_nmap_mac_prefixes_file

# we should create user directories
assume_user_directories

# we should add discovery_rules default user file
assume_discovery_rules_file

# we should add nmap_mac_prefixes default user file
assume_nmap_mac_prefixes_user_file

setup_shinken_ssh_key

update_security_parameters

###################### Graphite
print_std_line "Updating graphite ... "
# be sure to force the whisper lib update, for reads boost
install_only_graphite_whisper
set_whisper_rights

# update graphite webapp for data migration ( names to IDs)
update_graphite_webapp

# update init script for carbon-cache
/bin/cp -rp dependencies/graphite/init.d/carbon-cache  /etc/init.d/

# in all case, be sure to setup the clean of graphite logs once a day, and force a clean now
assume_graphite_clean
print_ok_result_line


###################### NagVis
arbiternode=$(shinken-daemons-has arbiter)
brokernode=$(shinken-daemons-has broker)
if [ "$arbiternode" == "NO" ]; then
    arbiternode=""
fi
if [ "$brokernode" == "NO" ]; then
    brokernode=""
fi

nagvis_shinken_setup_needed=0
nagvis_user_setup_needed=0
if [ ! -d $NAGVIS_SHINKEN_PATH ]; then
    nagvis_shinken_setup_needed=1
fi
if [ ! -d $NAGVIS_USER_PATH ]; then
    nagvis_user_setup_needed=1
fi

assume_nagvis "" "$arbiternode" "$brokernode"
NAGVIS_INSTALL_FAILED=$?

if [ $nagvis_shinken_setup_needed -eq 1 ]; then
    # If Nagvis is not installed, perform full install instead of update
    set_nagvis_configuration $NAGVIS_SHINKEN_PATH
    set_nagvis_architecture_export_configuration $NAGVIS_SHINKEN_PATH

    # Make sure addon is disabled
    shinken-addons-disable --force nagvis-shinken-architecture > /dev/null 2>&1
fi
if [ $nagvis_user_setup_needed -eq 1 ]; then
    # If Nagvis is not installed, perform full install instead of update
    set_nagvis_configuration $NAGVIS_USER_PATH

    # Make sure addon is disabled
    shinken-addons-disable --force nagvis > /dev/null 2>&1
fi

#### Also force the update (will erase previous version)
install_server_analyzer


# The libXrender install for shinken can break http/graphite, so force apache to restart
# and so load the new lib version
/etc/init.d/httpd stop >/dev/null 2>/dev/null
/etc/init.d/httpd start >/dev/null 2>/dev/null

shinken-context-manage --update-version "$VERSION"


if [ $activate_database_encryption -eq 1 ]; then
    printf "\n\033[33mThis Shinken installation will enable protected fields encryption in the Synchronizer\n"
    if [ -n "$encryption_key_name" ]; then
        printf "The encryption key will be named \033[35m${encryption_key_name}.\033[33m\n"
    else
        printf "You will be asked for the encryption key name during the installation process.\033[33m\n"
    fi
    printf "\n"
fi


print_header "Launching data and configuration checks and fixes"
####Launch various fix on the local files, but before restart the daemons.
# Some various fix specific for the update
# NOTE: must be launch in the good order, and must be idempotent (you can launch it as many time as you want)
fix_02_03_02_r43_sources_enabled_string
# fix missing lang parameter in webui (TODO: set other thank just fr)
fix_02_03_03_forcewebui_lang
# Fix: missing http_use_ssl (and such parameters) in the synchronizer configuration file (was ass in 2.03.03)
fix_02_03_03_forcehttp_use_ssl_in_synchronizer
# same for webui
fix_02_03_03_forcehttp_use_ssl_in_webui
# set enabled string in the cfg daemon files, so we can enabled/disable them if need
fix_02_03_03_r44_satellite_cfg_enabled_enabled_string
# Remove packs from synchronizer start
fix_02_03_03_r51_remove_packs_from_synchronizer
# Fix cherrypy SSL bug
fix_02_03_03_r76_cherrypyssl
# We move all discovery.cfg from pack to discovery_rules folder.
fix_02_03_03_u01_rc116_move_discovery_rules_in_discovery_rules_dir
# We empty contactgroup file in /etc/shinken/contactgroups.cfg
fix_02_03_03_u01_rc116_empty_contactgroup
# We change the check SLA so we update graphite metric
fix_02_03_03_u01_rc128_rename_metric_sla
# Add remove_user auth options in cfg files if missing
fix_02_03_03_u01_add_remote_auth_options
# Fix the default value for reactionner and sub realms (was not enabled by default before 03-U01)
fix_02_03_03_u01_fix_default_reactionner_sub_realms
# We destroy "__for_git" in "/source-data-cfg-sample" folder.
fix_02_03_03_u01_destroy_file_for_git
# Replace "service_check_timeout" by "check_running_timeout"
fix_02_05_00_service_check_timeout_renaming
# Fix the missing daemon_thread_poll_size of the various ini files
fix_02_03_03_u01_fix_missing_daemon_thread_pool_size
# Add contact mail mapping in ldap import configuration if missing
fix_02_03_03_u01_fix_ldap_mail_mapping
# cleanup changes and news persistent data
fix_02_05_00_remove_persistent_changes_and_news_data
# update sync_keys in stagging see SEF-1199
fix_02_03_03_u01_fix_double_sync_keys
# Convert all existing SE_UUIDs in synchronizer database items to use types rather than classes (core-hosttpls-0012139A14343 instead of core-host-29385629375623523)
fix_02_07_00_normalize_se_uuids
# update item id see SEF-1131
fix_02_03_03_u01_fix_item_uuid
# add new configuration key for auth-active-dir module
fix_02_03_03_u01_auth_active_dir
# remove shinken-core user
fix_02_03_03_u01_fix_remove_shinken_core
# Will clean deprecated check after poller update see SEF-1053 / SEF-1143
fix_02_05_00_fix_remove_deprecated_check
# Will add lines about language export in arbiter
fix_02_05_00_fix_language_arbiter
# Remap active directory configuration to the new format, split in 3 files
fix_02_05_00_remap_active_directory
# Make sure the synchronizer config have only existing sources defined in its list
fix_02_05_00_fix_synchronizer_sources_list
# Remove undefined templates from aix checks
fix_02_05_00_remove_undefined_aix_templates
# Force custom data and duplicate_foreach to be uppercase
fix_02_05_00_force_custom_data_uppercase
# Removes host_name from check templates
fix_02_05_00_host_name_inheritance
# Force high flapping thresholds to be higher than low ones
fix_02_05_00_fix_flapping_thresholds
# Move check_command bp_rule to bp_rule key to allow inheritance of check_command of cluster templates
fix_02_05_00_fix_bp_rule_args
# Fix wrong uuid in sla info
fix_02_05_00_fix_uuid_in_sla_info
# Will add default element properties lines to arbiter and synchronizer conf is missing (pre-2.04 install)
fix_02_04_assume_default_element_properties_present_in_configuration
# Add the overload file parameter in the shinken.cfg (arbiter) file (prior to 2.04)
fix_02_04_assume_shinken_cfg_overload
# Add the overload file parameter in the synchronizer.cfg
fix_02_04_assume_synchronizer_cfg_overload
# in 2.04 we clean the shinken.cfg files from the old useless files
fix_02_04_clean_shinken_cfg_from_useless_files
# in 2.04 we add the lang parameter if missing in the synchronizer
fix_02_04_assume_synchronizer_cfg_lang
# in 2.04 we added a configuration options for the webui
fix_02_04_add_overload_paths_to_webui
# Remove incoherent business impact
fix_02_04_fix_business_impact
# The default reactionner file did have passive parameter miss, and a typo in reactionner_tags
fix_02_03_03_u01_default_reactionner_file_arguments
# Remove all empty data in database except user data starts with '_'
fix_02_03_03_fix_empty_value_from_synchronizer_data
# Add the human_timestamp_log
fix_02_04_01_missing_human_timestamp_log
# add default acl for share in UI visualisation SEF-1813
fix_02_04_add_acl_share_ui_visualisation
# add default acl_try_check_contact
fix_02_04_acl_try_check_contact
# Set if missing the synchronizer.cfg the bind_addr=127.0.0.1
fix_02_04_assume_synchronizer_cfg_bind_addr
# Remove old metric of check_scheduler_poller
fix_02_04_00_remove_metric_check_scheduler_poller
# Add Protected fields encryption configuration (not activated by default)
fix_02_05_00_add_crypted_passwords_update
# SEF-2092 :  SLA calculation can be configured based on status : no_data and unknow can now be count as OK
fix_02_04_00_migrate_sla_option
# SEF-2073 :  User right - Add configuration option of history tab : show history and/or SLA, with or without theirs ranges
fix_02_04_acl_in_tab_history
# Add the remote_user_case_sensitive in pre-2.04.00 versions (allow user filter to be case insensitive)
fix_02_04_00_missing_remote_user_case_sensitive_in_livestatus_cfg
# The metric archive_sla_load isn't give anymore so we clean file in graphite
fix_02_05_00_RC001_remove_metric_archive_sla_load
# For update before 2.5.0, we need to add the listener_rest source
fix_02_05_00_missing_listener_rest_in_synchronizer_master_cfg
# Set if missing the synchronizer.cfg entries for listeners & analyzers
fix_02_05_00_missing_listeners_analyzers_in_synchronizer_cfg
# Add nb_stored_days parameter to sla existing module.
fix_02_05_00_add_nb_stored_days_parameter
# Fix paths for overload files in /etc/shinken-user
fix_02_05_00_move_overload_configuration_files
# Add warning_threshold_cpu_usage option in shinken.cfg
fix_02_04_04_fix_add_warning_threshold_cpu_usage_option
# remove unused synchronizer plugins  (aka Bad magic number import on __init__.pyc)
fix_02_05_00_remove_unused_synchronizer_plugins
# Add cfg_dir=synchronizers in shinken.cfg when absent
fix_02_05_00_add_cfg_dir_synchronizers_in_shinken_cfg
# Add map_base_url parameter to /etc/shinken/modules/architecture-export.cfg if absent
fix_02_05_00_add_map_base_url_param_to_architecture_export_cfg
# Remove item hostgroups_in_working_area if there are any
fix_02_05_00_remove_hostgroups_in_working_area
# Migrate links name into id
fix_02_06_00_migrate_links_name_into_id
# update double link in prod and stagging see SEF-1184 (update for 02.06.00)
fix_02_03_03_u01_fix_double_link_for_all_items
# Migrate links name into id
fix_02_06_00_encrypt_service_overrides_protected_fields
# Make index for sla module
fix_02_06_00_make_index_for_sla_module
# Enable Synchronizer_database_module
fix_02_06_00_enable_synchronizer_database_module
# Fix xss encode in prop
fix_02_06_00_fix_escape_properties_for_xss
# Fix old command shinken-protected-fields-properties-manage to shinken-protected-fields-data-manage
fix_02_06_00_rename_shinken_protected_fields_properties_manage
# Fix location of overload files, which changed in 02.05.00
fix_02_06_00_all_move_overload_files
# Add news elements to the listener-shinken (use_ssl, ssl_cert, ssl_key)
fix_02_07_00_add_news_elements_to_listener_shinken
# Add news elements to the architecture-export module (use_ssl, ssl_cert, ssl_key)
fix_02_07_00_add_news_elements_to_architecture_export
# Add discovery rules path in the discovery cfg
fix_02_07_00_add_rules_path_discovery
# Add discovery nmap_mac_prefixes path in the discovery cfg
fix_02_07_00_add_nmap_mac_prefixes_path_discovery
# Update notes_multi_url syntax with new elements : icon and display type
fix_02_07_00_update_notes_multi_url
# Add internal options to the mongodb-retention module
fix_02_07_04_add_internal_option_to_retention_mongo_cfg
# Update default user file with sla threshold
fix_02_08_add_default_sla_threshold
# Add SLA storing output options to the module's cfg
fix_02_08_00_add_sla_output_store_and_worker_options
# Add retry options to the synchronizer import module cfg
fix_02_08_01_add_retry_options_to_synchronizer_import_module
# Add map_realm_layout to the architecture-export module cfg
fix_02_08_01_add_map_realm_layout_to_architecture_export_module
# in 02.05 carbon cache was set as root & perf limited, so we remove theses limitations in carbon.conf
fix_02_07_04_fix_carbon_cache_parameters
# in 02.07.05, remove presence protection from elements
fix_02_07_05_fix_presence_protection_on_items
# in 02.07.06 we want to be sure the samba packages are updated for centos7 #SEF-5801
fix_02_07_06_fix_force_samba_update_for_centos7
# Delete tmp old tmp files in nagvis-shinekn folder
fix_02_08_01_clean_old_nagvis_shinken_tmp_files



if [ "$SANATIZE_ERRORS" = "1" ] ; then
    printf "\n\033[31mSome errors occurred while running the fixes.\n"
    printf "Please check the log file for more information (\033[32m${SANATIZE_LOGFILE}\033[31m).\n"
    printf "Send this file to your Shinken support if needed.\033[0m\n"
fi

fix_02_06_00_create_default_synchronizer_backup_directory

# Remove the maintenance file to let shinken start
rm -fr ${MAINTENANCE_FILE}


if [ -z "$disable_daemons_restart_after_update" ];then
    print_std_line "Starting Shinken ... "
    # Since user groups need to be refreshed before starting Shinken (NagVis installation adds root and shinken users to apache group), we need to start Shinken using 'su -'
    # This fix is only needed for CentOS 6, the problem was not observed on CentOS 7
    su -c "bash /etc/init.d/shinken start 2>/dev/null" -
    print_ok_result_line
fi

# Also look if the gatherer collector is re-started (can be updated since last version)
# NOTE: AFTER we exit maintenance
restart_gatherer

printf "\n\n"
print_header "Checking update"

print_note_line "Note that you can check a Shinken Enterprise installation with the command shinken-healthcheck"


# warn that the vmware sdk is NOT installed by default
check_vmware_sdk


# Save this update/install path so we know after the install where to ask more install things if need
save_intall_path

display_protected_fields_warning

current_version_major=$(echo $CURRENT_VERSION | cut -f 1 -d.)
current_version_minor=$(echo $CURRENT_VERSION | cut -f 2 -d.)
if [[ $current_version_major == 02  &&  $current_version_minor == 07 ]]; then
    display_event_container_info
fi


printf '\n\n'
print_header "End"
print_std_line_no_end "Your update to the version $DISPLAY_VERSION is "
print_OK

if [ $NAGVIS_INSTALL_FAILED -eq 0 ]; then
    if [ $nagvis_shinken_setup_needed -eq 1 ] || [ $nagvis_user_setup_needed -eq 1 ]; then
        printf "\nShinken update status: \n\n"

        if [ $nagvis_shinken_setup_needed -eq 1 ]; then
            printf "  - The addon 'nagvis-shinken-architecture' has been installed during the update.\n"

            printf "    \033[33mThis addon is currently disabled.\n"
            printf "    With an enabled Arbiter on this server, it can be enabled with the 'shinken-addons-enable nagvis-shinken-architecture' command.\033[0m\n"
            printf "    Once enabled, it will be accessible with the following address.\n"

            printf "    NagVis Shinken architecture:  \033[92mhttp://$EXTERNAL_IP$NAGVIS_SHINKEN_WEB_PATH\033[0m\n\n\n"
        fi
        if [ $nagvis_user_setup_needed -eq 1 ]; then
            printf "  - The addon 'nagvis' has been installed during the update.\n"

            printf "    \033[33mThis addon is currently disabled.\n"
            printf "    It can be enabled manually with the 'shinken-addons-enable nagvis' command.\033[0m\n"
            printf "    Once enabled, it will be accessible with the following address.\n"

            printf "    NagVis:                       \033[92mhttp://$EXTERNAL_IP$NAGVIS_USER_WEB_PATH\033[0m\n\n\n"
        fi
     fi
fi

# If on CentOS 6, restart terminal to refresh groups of user (needed for architecture-export)
if [ $nagvis_shinken_setup_needed -eq 1 ]; then
    if [[ "X$LINUX_VERSION" == "XCENTOS-6" || "X$LINUX_VERSION" == "XREDHAT-6" ]]; then
        printf "\033[33mDue to technical limitations on CentOS 6, the shell session needs to be restarted to account for user rights changed by Shinken update before continuing.\n"
        printf "It can be done with the command \"su -\", or by opening a new terminal window.\033[0m\n\n"
    fi
fi