#!/usr/bin/python

import json
import shutil
import optparse
import os
import sys
import re

FILES_PATH_AND_HELP = """
    # In order to configure this source, you must copy or edit the sample active-directory source-data available in the directory
    # /etc/shinken-user/source-data/source-data-active-directory-sample/ with theses files:
    #  * _configuration/active-directory-connection.json   ==> how to connect to your active directory server
    #  * _configuration/active-directory-rules.json        ==> what are the import rule to setup (like OU=>template rules)
    #  * _configuration/active-directory-mapping.json      ==> if you have custom Ldap fields, you can defined them in this file (ex: for openldap users)
    #
    # Note: If a file is commented, the configuration will be load from the default directory /etc/shinken/_default/sources/active-directory-source

    # Configuration file for your Active Directory connection (server, user, password, ...)
    connection_configuration_file  %s

    # Configuration file for your import rules (like OU=>template rules)
    rules_configuration_file       %s

    # Configuration file for your ldap fields mapping (like for openldap users)
    mapping_configuration_file     %s
"""

DEFAULT_CONN = {
  "url": "ldap://YOUR-DC-FQDN/",
  "ldap_protocol":3,
  "base": "dc=YOUR,dc=DOMAIN,dc=com",
  "hosts_base": "OU=DataCenter Servers,dc=YOUR,dc=DOMAIN,dc=com",
  "contacts_base": "dc=YOUR,dc=DOMAIN,dc=com",
  "hostgroups_base": "OU=DataCenter Servers,dc=YOUR,dc=DOMAIN,dc=com",
  "username": "SHINKEN@YOURDOMAIN.com",
  "password": "PASSWORD"
}

# Determine to which file belongs a key, and rename it if needed.
def dispatch_key(conf_key):
    # Keys that are copied as-is in the connection file.
    conn_keys = [
        "url",
        "base",
        "hosts_base",
        "contacts_base",
        "username",
        "password"
    ]
    # Keys that are copied as-is in the rules file.
    rules_keys = [
        "hosts_tag",
        "contacts_tag",
        "hosts_tag_\S+",
        "hosts_match_\S+"
    ]

    if conf_key in conn_keys:
        return 'conn', conf_key
    for rule in rules_keys:
        if re.match(rule, conf_key):
            return 'rules', conf_key
    if conf_key == 'contacts_filters':
        return 'rules', 'contact_group_filter'
    m = re.match("contacts_match_([^_]+)_(\S+)", conf_key)
    if m:
        return 'rules', 'AddLast_template_(%s)_to_contact_matching_[%s]' % (m.group(2), m.group(1))
    return 'source', conf_key

if __name__ == '__main__':
    parser = optparse.OptionParser("%prog --sources-dir SOURCES_DIR --modules-dir MODULES_DIR", version="%prog: ", description='This tool is used for merge a source with its associated module\nWARNING: Only supports a single module and source per file')
    parser.add_option('--sources-dir', dest='sources_dir', help="Sources directory")
    parser.add_option('--sources-data', dest='sources_data', help="Sources data directory")

    opts, args = parser.parse_args()

    # Look if the user ask for local or global, and if not, guess

    if not opts.sources_dir or not opts.sources_data:
        parser.print_help()
        sys.exit(1)

    s_dir = os.path.abspath(opts.sources_dir)
    s_data_dir = os.path.abspath(opts.sources_data)

    if not os.path.exists(s_dir):
        print "Error: Cannot find sources directory: %s" % s_dir
        sys.exit(2)
    if not os.path.exists(s_data_dir):
        print "Error: Cannot find sources data directory: %s" % s_data_dir
        sys.exit(2)
    connection_files = {}
    rules_files = {}
    mapping_files = []
    for root, _, files in os.walk(s_dir):
        for name in files:
            if not name.endswith(".cfg"):
                continue
            path = os.path.join(root, name)
            name = name[:-4]
            conf_dir = os.path.join(s_data_dir, name, '_configuration')
            conn_path = os.path.join(conf_dir, '%s-connection.json' % name)
            rules_path = os.path.join(conf_dir, '%s-rules.json' % name)
            mapping_path = os.path.join(conf_dir, '%s-mapping.json' % name)
            conf = {'rules': {}, 'conn': {}}

            must_split = False
            new_file = []
            with open(path) as f:
                for line in f.readlines():
                    line_striped = line.strip()
                    if re.search("^module_type", line_striped):
                        if line_striped[11:].strip() != "active-dir-import":
                            break
                        else:
                            must_split = True
                            new_file.append("    module_type    ldap-import\n")
                            new_file.append(FILES_PATH_AND_HELP % (conn_path, rules_path, mapping_path))
                    else:
                        m = re.match("(\S+)\s+(\S.*)", line_striped)
                        if m:
                            where, key = dispatch_key(m.group(1))
                            if where == 'source':
                                new_file.append(line)
                            else:
                                conf[where][key] = m.group(2)
                        else:
                            new_file.append(line)

            if must_split:
                connection_files[conn_path] = conf['conn']
                rules_files[rules_path] = conf['rules']
                mapping_files.append(mapping_path)
                os.makedirs(conf_dir)
                with open(path, 'w') as f:
                    f.write(''.join(new_file))
                print "* Migration of your active directory source \033[35m%s\033[0m configuration into the new format: \033[92mOK\033[0m" % name

    for path in mapping_files:
        # Simple copy from active directory mapping
        shutil.copyfile('/etc/shinken/_default/sources/active-directory-source/active-directory-mapping.json', path)

    # Rules and connection will be default with the additional keys retrived from old conf
    for path, conf in connection_files.iteritems():
        json_conf = DEFAULT_CONN.copy()
        json_conf.update(conf)
        with open(path, 'w') as f:
            f.write(json.dumps(json_conf, indent=4, sort_keys=True))

    for path, conf in rules_files.iteritems():
        with open(path, 'w') as f:
            f.write(json.dumps(conf, indent=4, sort_keys=True))