#!/bin/bash

# the script can be launch outside the common.sh file so load it first
if [ "X$LINUX_VERSION" == "X" ];then
   . lib/common.sh
fi


if [ "X$LINUX_VERSION" == "X" ];then
   doexit "ERROR: the script must be launch from the tarball directory"
fi

# Ce script va avoir pour role de faire passer mongodb de la version 2.6
# a la version 3.0, et surtout bien faire attention aux cas des clusters

# On a trois cas en matière de cluster/pas cluster:
# * le pas cluster du tout
# * le cas d'un cluster mais à la sauce de certains partenaires (juste ha mais sans mongos)
# * le cas qu'on recommande avec mongod+mongo-config+mongos

TMP_DIR="/tmp/mongo_3_update"
IS_MONGO_CLUSTER="false"
MONGO_CLUSTER_SERVERS=""
IS_MONGO_INSTALLED="true"
IS_MONGO3_INSTALLED="false"
HAVE_MONGOS="false"
CONCURRENT_UPDATE_LOCK="/tmp/mongo-update-`date +%F`.lock"
# Time when an update is ALREADY running
MAX_CONCURRENT_UPDATE_WAIT=300
# Time when we cannot update and we are waiting ANOTHER node to start it
MAX_LOCK_WAITING_TIME=60
RANDOM_WAIT_TIME=60  # to avoid have 2 servers run at the same time, we randomize the lock set a bit


function print_step {
   print_std_line "$1"
}


# We only support some mongodb versions. So if at INSTALL we already have a mongo, it MUST
# be the valid one or no one at all
function check_mongo_version {
    # 2.6.9 => 2.6.0 and before
    # 3.0.15 => 2.6.1
    VALID_VERSIONS="2.6.9 3.0.15"
    MONGODB_VERSION=$(mongod --version 2>/dev/null | grep 'db version' | grep "$MONGODB_VERSION_ASSERTED" )

    # Maybe there is no mongo, so we are OK
    if [ $? != 0 ]; then
        IS_MONGO_INSTALLED="false"
        return
    fi

    for valid_version in $VALID_VERSIONS; do
        echo "$MONGODB_VERSION"  | grep "$valid_version" > /dev/null
        # Valid version founded
        if [ $? == 0 ]; then
            if [ $valid_version == "3.0.15" ];then
                IS_MONGO3_INSTALLED="true"
            fi
            return
        fi
    done
    doexit "ERROR: Mongodb is already installed but your Mongodb version ($MONGODB_VERSION) is not supported for install/update"
}


function get_local_rpms {
   if [[ "X$LINUX_VERSION" == "XCENTOS-6" || "X$LINUX_VERSION" == "XREDHAT-6" ]]; then
        echo "dependencies/mongodb/packages/redhat-centos/6.X/*.rpm"
   fi
   if [[ "X$LINUX_VERSION" == "XCENTOS-7" || "X$LINUX_VERSION" == "XREDHAT-7" ]]; then
        echo "dependencies/mongodb/packages/redhat-centos/7.X/*.rpm"
   fi
}


function install_mongodb_rpm {
    LOCAL_RPMS=$(get_local_rpms)
    dorpminstall "$LOCAL_RPMS" "mongodb" "forceupdate"

    # Copy our mongod init.d that have the --smallfiles in it and latest fixes
    cp -rp dependencies/mongodb/mongod /etc/init.d/mongod
    chmod a+x /etc/init.d/mongod

    if [[ "X$LINUX_VERSION" == "XCENTOS-7" || "X$LINUX_VERSION" == "XREDHAT-7" ]]; then
        systemctl daemon-reload
    fi
}


function assert_ssh_connection {
   # If fail: do not exit but just return an error
   OUT=$(launch_ssh_command "$1" "echo 'OK'" "true")
   res=$?
   # NOTE: ssh give us \n on the output
   #OUT=$(echo "$OUT" | tr -d '\n')
   # If there was an error, maybe we are a secondary node and we must wait for the primary node to connect
   if [ $res == 2 ];then
      echo ""
      wait_for_concurrent_lock
      print_step "Cannot connect to SSH: maybe we are a secondary node and we must wait for the primary node to start the update."
      echo "    └→ first update server : please check that this server have the SSH keys to execute commands to all mongo server (itself included) as root"
      echo "                             $ ssh-keygen (if do not already have ssh-keys)"
      echo "                             $ ssh-copy-id -i /root/.ssh/id_rsa.pub root@$1"
      echo "    └→ other servers       : please re-launch the update after the primary server did finish"
      doexit "Exiting"
   fi
}


function launch_ssh_command {
   NODE="$1"
   COMMAND="$2"
   IF_FAIL_DO_RETURN="$3"
   LOCAL_EXEC="$4"
   RETURN_STDOUT="$5"

   if [ "$LOCAL_EXEC" == "true" ]; then
        sh -c "${COMMAND}"
        return $?
   fi

   # NOTE: do not look at distant host key as we are in a script
   ssh -o BatchMode=yes -o LogLevel=ERROR -o UserKnownHostsFile=/dev/null -o StrictHostKeyChecking=no root@$NODE "$COMMAND" > /tmp/ssh.try
   if [ $? != 0 ];then
       if [ "$IF_FAIL_DO_RETURN" == "true" ];then
          cat /tmp/ssh.try
          return 2
       fi
       cat /tmp/ssh.try
       doexit "ERROR: the command $COMMAND did fail on the node $NODE. Cannot continue the update."
   fi

   if [ "$RETURN_STDOUT" == "true" ];then
      cat /tmp/ssh.try
   fi
}


function launch_ssh_command_all_nodes {
   COMMAND="$1"
   for node in ${MONGO_NODES[@]}; do
       launch_ssh_command "$node" "$COMMAND"
   done
}


function launch_scp {
   NODE="$1"
   FROM="$2"
   DEST="$3"
   LOCAL_EXEC="$4"

   if [ "$LOCAL_EXEC" == "true" ]; then
        cp "$FROM" "$DEST"
        return
   fi

   # NOTE: do not look at distant host key as we are in a script
   SCP=$(scp -o BatchMode=yes -o LogLevel=ERROR -o UserKnownHostsFile=/dev/null -o StrictHostKeyChecking=no $FROM "root@$NODE:$DEST")
   if [ $? != 0 ];then
       doexit "ERROR: cannot copy the files $FROM to the server $NODE: $SCP. Cannot continue the update."
   fi
}


function launch_scp_all_nodes {
    FROM="$1"
    DEST="$2"
    for node in ${MONGO_NODES[@]}; do
        launch_scp "$node" "$FROM" $DEST
    done
}


function clean_lock_everywhere {
    launch_ssh_command_all_nodes "rm -f $CONCURRENT_UPDATE_LOCK"
}


function set_lock_everywhere {
    HOST_NAME=`hostname`
    launch_ssh_command_all_nodes "echo $HOST_NAME > $CONCURRENT_UPDATE_LOCK"
}


function check_concurrent_update {
    # if there is a lock, cannot go more, but warn the user about the lock
    if [ -f $CONCURRENT_UPDATE_LOCK ];then
       LOCK_HOLDER=$(cat $CONCURRENT_UPDATE_LOCK)
       echo "IMPORTANT: there is already a lock file $CONCURRENT_UPDATE_LOCK (from the server $LOCK_HOLDER) meaning a mongo cluster update is already running. Waiting for the other process to finish."
       for ii in `seq 1 $MAX_CONCURRENT_UPDATE_WAIT`; do
          printf "."
          if [ ! -f $CONCURRENT_UPDATE_LOCK ];then
             printf "\nOK: The other process did finish and did update our mongo daemon to the new version.\n"
             exit 0
          fi
          sleep 1
       done

       doexit "ERROR: another update process (on the server $LOCK_HOLDER) did create the lock file $CONCURRENT_UPDATE_LOCK and did not finish in $MAX_CONCURRENT_UPDATE_WAIT seconds. When you are sure the other server is done, please delete this file and start this updater again"
    fi
}


# Maybe we cannot start the update because we do not have SSH keys, but then we can just wait
# for the lock to be created by the first update node
function wait_for_concurrent_lock {
    print_step "Waiting if another node is currently trying to launch the update via SSH (waiting time=$MAX_LOCK_WAITING_TIME seconds)"
    for ii in `seq 1 $MAX_LOCK_WAITING_TIME`; do
       printf "."
       if [ -f $CONCURRENT_UPDATE_LOCK ];then
           check_concurrent_update
           # note: check_concurrent_update should exit already
           exit 0
       fi
       sleep 1
    done
    doexit "ERROR: we cannot start the mongo update on other node, and no other node did start the update too. Please check at your SSH keys."
}

# Get running mongo daemon pid on choosen node
function get_mongo_daemon_pid {
    local NODE_ADDR=$1
    local DAEMON_NAME=$2
    local LOCAL_EXEC=$3
    local daemon_running_pid=$(launch_ssh_command "${NODE_ADDR}" "ps -u mongod -o pid,cmd --no-headers | grep \"${DAEMON_NAME}.conf\" | awk '{\$1=\$1};1' | cut -d' ' -f 1" "false" "${LOCAL_EXEC}" "true")

    echo $daemon_running_pid
}

# Start specified mongo daemon and check if it has started correctly
function start_and_check_mongo_daemon {
    local NODE_ADDR=$1
    local DAEMON_NAME=$2
    local LOCAL_EXEC=$3

    launch_ssh_command "${NODE_ADDR}" "/etc/init.d/${DAEMON_NAME} start" "false" "${LOCAL_EXEC}"

    sleep 2

    running_pid=$(get_mongo_daemon_pid ${NODE_ADDR} ${DAEMON_NAME} ${LOCAL_EXEC})
    if [ "X$running_pid" == "X" ]; then
        doexit "ERROR: cannot start the mongo '${DAEMON_NAME}' daemon on '${NODE_ADDR}'. Cannot update in a safe way. Exiting."
    fi
}

# Check if specified mongo daemon is running and its pid file
# When daemon cannot be started correctly, the script exits
# When the function ends without exiting, the specified mongo daemon on the choosen node is running and its pidfile is consistent with the daemon pid
function check_mongo_running {
    local NODE_ADDR=$1
    local DAEMON_NAME=$2
    local LOCAL_EXEC=$3

    CONFIGFILE="/etc/${DAEMON_NAME}.conf"
    PIDINFILE=0
    PIDFILEPATH=$(launch_ssh_command "${NODE_ADDR}" "echo \"`awk -F'[:=]' -v IGNORECASE=1 '/^[[:blank:]]*(processManagement\.)?pidfilepath[[:blank:]]*[:=][[:blank:]]*/{print \$2}' "$CONFIGFILE" | tr -d '[:blank:]' | awk -F'#' '{print $1}'`\"" "false" "${LOCAL_EXEC}" "true")

    daemon_running_pid=$(get_mongo_daemon_pid ${NODE_ADDR} ${DAEMON_NAME} ${LOCAL_EXEC})
    if [ "X$daemon_running_pid" == "X" ]; then
        # Daemon is not running
        if [ "X${PIDFILEPATH}" != "X" ]; then
            PID_FILE_EXISTS_ON_REMOTE_MONGO_SERVER=$(launch_ssh_command "${NODE_ADDR}" "stat \"${PIDFILEPATH}\" 2>&1 > /dev/null" "true" "${LOCAL_EXEC}")
            if [ $? == 0 ]; then
                launch_ssh_command "${NODE_ADDR}" "rm \"${PIDFILEPATH}\"" "false" "${LOCAL_EXEC}"
            fi
        fi
        start_and_check_mongo_daemon "${NODE_ADDR}" "${DAEMON_NAME}" "${LOCAL_EXEC}"
        return
    fi

    if [ -f "$PIDFILEPATH" ]; then
        PIDINFILE=$(launch_ssh_command "${NODE_ADDR}" "cat ${PIDFILEPATH}" "false" "${LOCAL_EXEC}" "true")
    fi

    if [ "X$PIDINFILE" != "X$daemon_running_pid" ]; then
        # PIDs do not match, writing correct pid in file
        launch_ssh_command "${NODE_ADDR}" "echo \"${daemon_running_pid}\" > ${PIDFILEPATH}" "false" "${LOCAL_EXEC}"
    fi
}

function check_mongo_running_all_nodes {
    local DAEMON_NAME=$1
    local LOCAL_EXEC=$2

    for node in ${MONGO_NODES[@]}; do
        check_mongo_running "${node}" "${DAEMON_NAME}" "${LOCAL_EXEC}"
    done
}

function disable_mongodb_unix_sockets {
    UNIXDOMAINSOCKET=$(grep -i 'unixDomainSocket\|nounixsocket' /etc/mongod.conf | grep -v ^#)
    MONGO_CFG_INI_FORMAT=$(grep -E "logpath|pidfilepath|bind_ip" /etc/mongod.conf | grep -v ^# | wc -l)
    MONGO_CFG_YAML_FORMAT=$(grep -E "systemLog|pidFilePath|bindIp" /etc/mongod.conf | grep -v ^# | wc -l)
    if [ -z $UNIXDOMAINSOCKET ];then
        if [ $MONGO_CFG_YAML_FORMAT != '0' ]; then
            print_note_line "Disabling unix socket for mongo:"
            printf "\nnet:\n unixDomainSocket:\n  enabled: false\n\n" >> /etc/mongod.conf
        elif [ $MONGO_CFG_INI_FORMAT != '0'  ]; then
            print_note_line "Disabling unix socket for mongo:"
            printf "\nnounixsocket=yes" >> /etc/mongod.conf
        fi
        print_ok_result_line
    fi
}

######################################################
       #     #     #     ###  #     #
       ##   ##    # #     #   ##    #
       # # # #   #   #    #   # #   #
       #  #  #  #     #   #   #  #  #
       #     #  #######   #   #   # #
       #     #  #     #   #   #    ##
       #     #  #     #  ###  #     #
######################################################


# If we have a bad starting version of mongo, we can exit now
check_mongo_version

# If there is no mongo installed, we are ok with cluster, and just install it)
if [ $IS_MONGO_INSTALLED == "false" ];then
    print_std_line_no_end "Installing mongodb server:"
    install_mongodb_rpm
    print_OK

    print_std_line_no_end "Enabling fast mongodb engine:"
    # At install, we do enable wiredtiger (cannot do at update because we need to
    # migrate data first)
    printf "\nstorage:\n    engine: wiredTiger\n\n" >> /etc/mongod.conf
    print_OK
    print_std_line_no_end "Disabling unix socket for mongo:"
    printf "\nnet:\n unixDomainSocket:\n  enabled: false\n\n" >> /etc/mongod.conf
    print_OK

    check_mongo_running "localhost" "mongod" "true"

    exit 0
fi

# if mongo is already mongo3, update mongo locally only and exit
if [ $IS_MONGO3_INSTALLED == "true" ];then
    print_step "Mongo 3 is already installed. No need to perform cluster update, only the local mongodb installation will be updated"
    print_std_line "Updating mongodb server"
    disable_mongodb_unix_sockets
    /etc/init.d/mongod restart >/dev/null 2>/dev/null
    install_mongodb_rpm
    check_mongo_running "localhost" "mongod" "true"
    exit 0
fi

# if there is a lock, cannot go more, but warn the user about the lock
check_concurrent_update


# Maybe the mongo is stopped:
print_note_line "Checking if mongodb is in cluster mode"
mongo --quiet --eval "db.version();" >/dev/null 2>/dev/null
if [ $? != 0 ];then
   echo "Starting the mongo daemon to check for a cluster mode."
   /etc/init.d/mongod start >/dev/null 2>/dev/null
   /etc/init.d/mongo-configsrv start >/dev/null 2>/dev/null
   /etc/init.d/mongos start >/dev/null 2>/dev/null
   OUT=$(mongo --quiet --eval "db.version();")
   if [ $? != 0 ];then
      doexit "ERROR: cannot start the mongo server and so cannot check if the mongo is in a cluster or not: $OUT. Cannot update in a safe way. Exiting."
   fi
fi


# Here mongo is installed, so beware of cluster/not cluster setup
# but beware: for cluster, as we cannot rely on config files, ask mongo directly
MONGO_REPLICASET=$(mongo --quiet --eval "JSON.stringify(rs.config().members.map(function(m) { return m.host }).join());")
MONGODB_HAS_REPLICASET=$?
MONGODB_SHARD=$(mongo --quiet --eval "JSON.stringify(db.adminCommand( { listShards: 1 } ).shards[0].host);")
MONGODB_HAS_SHARDING=$?
if [ $MONGODB_HAS_SHARDING != 0 ] && [ $MONGODB_HAS_REPLICASET != 0 ]; then
    print_step "The local mongodb is not a cluster. We can update in a safe way."
    print_std_line_no_end "Updating mongodb server"
    install_mongodb_rpm
    print_OK

    # Check that mongo is running and that pidfile contents are correct (needed to have a clean stop)
    check_mongo_running "localhost" "mongod" "true"

    # When updating mongo 3, restart mongo after update to ensure all updated files/daemons are taken into account
    /etc/init.d/mongod stop > /dev/null 2> /dev/null
    disable_mongodb_unix_sockets
    /etc/init.d/mongod start > /dev/null 2> /dev/null
    check_mongo_running "localhost" "mongod" "true"

    exit 0
fi

print_header "Updating a mongo cluster (on all nodes)"
printf "\nIMPORTANT: We detected a Mongodb installation in a cluster mode. In such an environnement, all Mongo updates should be done in the correct order\n"
printf "           The first node of the Mongodb cluster that launches this update will automatically launch the update of the mongo daemons on the two other nodes\n"
printf "             * The first updated Mongodb node will need to be able to connect to the root account via SSH to the other Mongodb nodes in order to execute the mongo update (with SSH keys set up)\n"
printf "             * Please wait until the first node is updated to launch the update on the other Mongodb nodes.\n"
printf "               -> The other nodes will automatically detect that mongo is already updated and will continue the Shinken Enterprise update\n"


if [ $MONGODB_HAS_SHARDING == 0 ]; then
    ONLY_NODES=$(echo $MONGODB_SHARD | sed 's#\".*/\(.*\)\"#\1#')
else
    ONLY_NODES=$(echo $MONGO_REPLICASET | sed 's#\".*/\(.*\)\"#\1#' | sed 's/^"//' | sed 's/"$//')
fi

mongo_nodes_with_ports=$(echo $ONLY_NODES | awk 'BEGIN{RS=","; ORS=" "} {print}')
declare -a MONGO_NODES=()
for node in $mongo_nodes_with_ports; do
    MONGO_NODES+=( $(echo $node | cut -d':' -f 1) )
done

print_note_line "${#MONGO_NODES[@]} nodes cluster detected: ${MONGO_NODES[*]}"


# Now we need to test ssh for theses nodes, as root
print_note_line "Checking if all mongo nodes can exchange with SSH"
for node in ${MONGO_NODES[@]}; do
    check_concurrent_update
    assert_ssh_connection $node
done


print_step "Setting a lock file ($CONCURRENT_UPDATE_LOCK) on the server to avoid update corruption"
WAIT_RANDOM_TIME=$(($RANDOM % $RANDOM_WAIT_TIME))
# Before setting our own lock, do another protection against concurrent runs
check_concurrent_update
sleep $WAIT_RANDOM_TIME
# recheck concurrent after the wait to let the other node set the lock
check_concurrent_update
set_lock_everywhere

if [ $IS_MONGO3_INSTALLED != "true" ];then
    print_note_line "Copying update files to all nodes"
    # Dispatch the files to the other nodes with SSH (even local one)
    # => scp mongodb-org-*rpm   NODE 1->3  (pas grave si on recolle sur le 1)
    LOCAL_RPMS=$(get_local_rpms)

    launch_ssh_command_all_nodes "rm -fr $TMP_DIR; mkdir $TMP_DIR"
    launch_scp_all_nodes "$LOCAL_RPMS" "$TMP_DIR"
fi

if [ -f /etc/mongos.conf ];then
    print_step "Mongos router is detected"
    HAVE_MONGOS="true"
fi


if [ "$HAVE_MONGOS" == "true" ];then
    print_note_line "Preparing the mongo cluster before the update"
    # On the local server, we set the mongo cluster as "unbalanced" as it block the mongos update
    mongo --quiet --eval 'sh.setBalancerState(false);'
    if [ $? != 0 ];then
       doexit "The mongo cluster was not able to be set in a un-balanced mode. Cannot continue the update."
    fi
fi


# Copy new init scripts before shutting down daemons to avoid pid bug in older init scripts
launch_scp_all_nodes "dependencies/mongodb/mongod" /etc/init.d/mongod
launch_ssh_command_all_nodes "chmod a+x /etc/init.d/mongod"
if [ "$HAVE_MONGOS" == "true" ];then
    launch_scp_all_nodes "tools/mongo-cluster/mongos" /etc/init.d/mongos
    launch_ssh_command_all_nodes "chmod a+x /etc/init.d/mongos"
    launch_scp_all_nodes "tools/mongo-cluster/mongo-configsrv" /etc/init.d/mongo-configsrv
    launch_ssh_command_all_nodes "chmod a+x /etc/init.d/mongo-configsrv"
fi

if [[ "X$LINUX_VERSION" == "XCENTOS-7" || "X$LINUX_VERSION" == "XREDHAT-7" ]]; then
    launch_ssh_command_all_nodes "systemctl daemon-reload"
fi

# Check that everything is running before shutting down. The check running routines also ensure that pidfiles and running daemon pids are in sync, which prevents errors while shutting down
check_mongo_running_all_nodes "mongod"
if [ "$HAVE_MONGOS" == "true" ];then
    check_mongo_running_all_nodes "mongos"
    check_mongo_running_all_nodes "mongo-configsrv"
fi

print_step "Stopping mongo daemons before the update"
# We can now stop ALL the mongo daemons, across the servers
if [ "$HAVE_MONGOS" == "true" ];then
    launch_ssh_command_all_nodes "/etc/init.d/mongos stop 2>&1 && /etc/init.d/mongo-configsrv stop 2>&1 && /etc/init.d/mongod stop 2>&1 "
else
    launch_ssh_command_all_nodes "/etc/init.d/mongod stop 2>&1 "
fi

# In theory, Mongo should remove pid files when shutting down, but it is not always the case. Remove then manually to avoid errors when restarting
launch_ssh_command_all_nodes "rm -f /var/run/mongodb/mongo*.pid"

# For mongo3, we cannot have a keyFile parameter as it force the user auth
# =>> sed -i 's/^keyFile/#keyFile/' /etc/mongos.conf
# ==> sed -i 's/^keyFile/#keyFile/' /etc/mongod.conf
# ==> sed -i 's/^keyFile/#keyFile/' /etc/mongo-configsrv.conf
if [ "$HAVE_MONGOS" == "true" ];then
    launch_ssh_command_all_nodes "sed -i 's/^keyFile/#keyFile/' /etc/mongos.conf && sed -i 's/^keyFile/#keyFile/' /etc/mongod.conf && sed -i 's/^keyFile/#keyFile/' /etc/mongo-configsrv.conf"
else
    launch_ssh_command_all_nodes "sed -i 's/^keyFile/#keyFile/' /etc/mongod.conf"
fi


# Launching the update on the servers
if [ $IS_MONGO3_INSTALLED == "true" ];then
    print_step "Mongo3 is already installed, skipping package update"
else
    print_step "Updating mongo to the new version"
    launch_ssh_command_all_nodes "rpm -Uvh /tmp/mongo_3_update/*rpm 2>&1"
fi

if [ "$HAVE_MONGOS" == "true" ];then
    print_step "Restarting data & config server daemons"
    # We can start daemons, but only the mongod & mongo-configsrv as mongos will need a special case
    launch_ssh_command_all_nodes "/etc/init.d/mongod start 2>&1 && /etc/init.d/mongo-configsrv start 2>&1 "

    check_mongo_running_all_nodes "mongod"
    check_mongo_running_all_nodes "mongo-configsrv"
else
    print_step "Restarting mongo data server"
    launch_ssh_command_all_nodes "/etc/init.d/mongod start 2>&1"

    check_mongo_running_all_nodes "mongod"
fi


if [ "$HAVE_MONGOS" == "true" ];then
    print_step "Updating mongos relay data"
    # Mongos: need a special command to update it's data, on the local node:
    MONGOS=$(mongos --config /etc/mongos.conf --upgrade)
    if [ $? != 0 ];then
       doexit "The mongos command was not able to update it's data: $MONGOS. Cannot continue the update."
    fi
    # The pid file is now with root user, need to fix this
    chown mongod:mongod /var/run/mongodb/*pid

    # Mongos configuration upgrade can leave a pid file that prevents mongos service from starting.
    if [ -f /var/run/mongodb/mongos.pid ]; then
        rm /var/run/mongodb/mongos.pid
    fi

    # restart the mongos as daemon on all servers
    launch_ssh_command_all_nodes "/etc/init.d/mongos start"

    check_mongo_running_all_nodes "mongos"

    print_step "Finish to reenable mongo cluster"
    # We can now reactivate the cluster balacing, launch on the local node
    # => sh.setBalancerState(true);
    mongo --quiet --eval 'sh.setBalancerState(true);'
    if [ $? != 0 ];then
       doexit "The mongo cluster was not able to be set in a balanced mode."
    fi
fi



print_step "Cleaning the lock file from all servers"
clean_lock_everywhere

print_ok_result_line "The cluster is updated"


exit 0