#!/usr/bin/python
# -*- coding: utf-8 -*-

# Copyright (C) 2009-2012:
#    Gabes Jean, naparuba@gmail.com
#    Gerhard Lausser, Gerhard.Lausser@consol.de
#    Gregory Starck, g.starck@gmail.com
#    Hartmut Goebel, h.goebel@goebel-consult.de
#
# This file is part of Shinken.
#
# Shinken is free software: you can redistribute it and/or modify
# it under the terms of the GNU Affero General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# Shinken is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Affero General Public License for more details.
#
# You should have received a copy of the GNU Affero General Public License
# along with Shinken.  If not, see <http://www.gnu.org/licenses/>.

import time
import re
import threading
import traceback

# Import all objects we will need
from shinken.brok import Brok
from shinken.objects.host import Host, Hosts
from shinken.objects.hostgroup import Hostgroup, Hostgroups
from shinken.objects.service import Service, Services
from shinken.objects.servicegroup import Servicegroup, Servicegroups
from shinken.objects.contact import Contact, Contacts
from shinken.objects.contactgroup import Contactgroup, Contactgroups
from shinken.objects.notificationway import NotificationWay, NotificationWays
from shinken.objects.timeperiod import Timeperiod, Timeperiods
from shinken.objects.command import Command, Commands
from shinken.objects.config import Config
from shinken.schedulerlink import SchedulerLink, SchedulerLinks
from shinken.reactionnerlink import ReactionnerLink, ReactionnerLinks
from shinken.pollerlink import PollerLink, PollerLinks
from shinken.brokerlink import BrokerLink, BrokerLinks
from shinken.receiverlink import ReceiverLink, ReceiverLinks
from shinken.util import safe_print
from shinken.message import Message
from shinken.log import logger
from shinken.objects.proxyitem import proxyitemsgraph
from shinken.configuration_incarnation import PartConfigurationIncarnationContainer, PartConfigurationIncarnation
from shinken.misc.filter import init_user_item_access_cache

# Only for scheduler mode
# Ok you are wondering why we don't add initial_broks_done? It's because the Livestatus modules need this part to do internal things.
# But don't worry, the vanilla regenerator will just skip it in all_done_linking :D
WANT_BROK_TYPE = (
    'program_status',
    'initial_host_status',
    'initial_hostgroup_status',
    'initial_service_status',
    'initial_servicegroup_status',
    'initial_contact_status',
    'initial_contactgroup_status',
    'initial_timeperiod_status',
    'initial_command_status'
)

INITIAL_BROKS_FOR_CONFIGURATION = (
    'initial_host_status',
    'initial_hostgroup_status',
    'initial_service_status',
    'initial_servicegroup_status',
    'initial_contact_status',
    'initial_contactgroup_status',
    'initial_timeperiod_status',
    'initial_command_status',
    'initial_scheduler_status',
    'initial_poller_status',
    'initial_reactionner_status',
    'initial_broker_status',
    'initial_receiver_status',
    'initial_broks_done',
    'proxy_items_graph',
    'program_status',
)


def _natural_key(string_):
    natural_key = []
    for s in re.split(r'(\d+)', string_):
        if s.isdigit():
            natural_key.append(int(s))
        else:
            natural_key.append(s.lower())
    return natural_key


def _stringify(_input):
    return _input.encode('utf-8', 'ignore') if callable(getattr(_input, 'encode', None)) else str(_input)


# Class for a Regenerator. It will get broks, and 'regenerate' real objects from them :)
class Regenerator(object):
    def __init__(self, module_conf, import_in='(unknown)'):
        
        # Our Real datas
        self.configs = {}
        self.currently_asking_full_broks = {}
        self.hosts = Hosts([])
        self.services = Services([])
        self.notificationways = NotificationWays([])
        self.contacts = Contacts([])
        self.hostgroups = Hostgroups([])
        self.servicegroups = Servicegroups([])
        self.contactgroups = Contactgroups([])
        self.timeperiods = Timeperiods([])
        self.commands = Commands([])
        self.schedulers = SchedulerLinks([])
        self.pollers = PollerLinks([])
        self.reactionners = ReactionnerLinks([])
        self.brokers = BrokerLinks([])
        self.receivers = ReceiverLinks([])
        # From now we only look for realms names
        self.realms = set()
        self.tags = {}
        self.services_tags = {}
        
        # And in progress one
        self.inp_hosts = {}
        self.inp_services = {}
        self.inp_hostgroups = {}
        self.inp_servicegroups = {}
        self.inp_contactgroups = {}
        
        self.contact_names = set()
        
        # Flag to say if our data came from the scheduler or not
        # (so if we skip *initial* broks)
        self.in_scheduler_mode = False
        
        # The Queue where to launch message, will be fill from the broker
        self.from_module_to_main_daemon_queue = None
        
        # We will have default value overridden by the module configurations
        self.default_properties_values = getattr(module_conf, 'configuration_default_value', {})
        self.minimal_time_before_an_element_become_missing_data = getattr(module_conf, 'minimal_time_before_an_element_become_missing_data', 0)
        
        # Sorted cache of all host and services
        self.all_hosts_and_services = []
        
        # Regenerator can have a specific thread that look at objects freshness
        self.freshness_thread = None
        self.freshness_lock = threading.RLock()
        
        # Keep track of the last time an Item went into UNKNOWN because the scheduler
        # didn't send any brok for a long time
        self.item_last_unknown_date = {}
        
        self.part_configuration_incarnation_container = PartConfigurationIncarnationContainer()
        
        # Know in which module we are loaded
        self.import_in = import_in
    
    
    def get_configuration_incarnation_uuid(self):
        # type: () -> basestring
        return self.part_configuration_incarnation_container.get_configuration_handle_uuid()
    
    
    def get_last_part_configuration_incarnation(self):
        # type: () -> PartConfigurationIncarnation
        return self.part_configuration_incarnation_container.get_last_part_configuration_incarnation()
    
    
    def get_default_property_value(self, cls, property_name, missing=''):
        if cls not in self.default_properties_values:
            return missing
        if property_name not in self.default_properties_values[cls]:
            return missing
        return self.default_properties_values[cls][property_name]
    
    
    # Load an external queue for sending messages
    def load_external_queue(self, from_module_to_main_daemon_queue):
        self.from_module_to_main_daemon_queue = from_module_to_main_daemon_queue
    
    
    # If we are called from a scheduler it self, we load the data from it
    def load_from_scheduler(self, sched):
        # Ok, we are in a scheduler, so we will skip some useless steps
        self.in_scheduler_mode = True
        
        # Go with the data creation/load
        c = sched.conf
        # Simulate a drop conf
        b = sched.get_program_status_brok()
        b.prepare()
        self.manage_program_status_brok(b)
        
        # Now we will lie and directly map our objects :)
        self.hosts = c.hosts
        self.services = c.services
        self.notificationways = c.notificationways
        self.contacts = c.contacts
        self.hostgroups = c.hostgroups
        self.servicegroups = c.servicegroups
        self.contactgroups = c.contactgroups
        self.timeperiods = c.timeperiods
        self.commands = c.commands
        # We also load the realm
        for h in self.hosts:
            self.realms.add(h.realm)
            break
    
    
    def want_brok(self, brok):
        if self.in_scheduler_mode:
            # If we are in a scheduler mode, some broks are dangerous, so we will skip them
            return brok.type not in WANT_BROK_TYPE
        return True
    
    
    def _is_a_new_configuration(self, brok):
        # type: (Brok) -> bool
        brok_type = brok.type
        if brok_type in INITIAL_BROKS_FOR_CONFIGURATION and brok.part_configuration_incarnation:
            can_accept_brok, cause = self.part_configuration_incarnation_container.check(brok.part_configuration_incarnation)
            if not can_accept_brok:
                # logger.debug('[regenerator] brok [%s] refused because : %s' % (brok_type, cause))
                
                # We log only for brok_type 'initial_broks_done' for not spam log
                if brok_type == 'initial_broks_done':
                    logger.info('[ CONFIGURATION ] [ NEW ] [ REGENERATOR ] No need to reload the configuration part because %s' % cause)
                return False
        
        return True
    
    
    def manage_brok(self, brok):
        # Look for a manager function for a brok, and call it
        manage = getattr(self, 'manage_' + brok.type + '_brok', False)
        
        managed = manage and self.want_brok(brok) and self._is_a_new_configuration(brok)
        # If we can and want it, got for it :)
        if managed:
            return manage(brok)
    
    
    @staticmethod
    def update_element(element, data):
        # Update all, but not the id as we can't follow it between restarts
        if 'id' in data:
            del data['id']
        
        # We let the freshness thread know we did update this element
        element.last_broker_data_update = int(time.time())
        for prop in data:
            setattr(element, prop, data[prop])
        
        # Invalidation of list cache (see webui-enterprise/plugins/tabular/tabular.py test_remote_tabular)
        if hasattr(element, 'list_cache_var'):
            delattr(element, 'list_cache_var')
    
    
    # We have a thread that is doing freshness checks and modification if need
    def launch_freshness_thread(self, lang='en'):
        self.freshness_thread = threading.Thread(target=self._do_freshness_thread, args=(lang,), name='regenerator-freshness')
        self.freshness_thread.daemon = True
        self.freshness_thread.start()
    
    
    def _do_freshness_thread(self, lang):
        while True:
            with self.freshness_lock:
                try:
                    now = int(time.time())
                    logger.debug('[regenerator-freshness] freshness thread')
                    _types = (self.hosts, self.services)
                    for _type in _types:
                        for element in _type:
                            element.look_for_fake_unknown_state_due_to_missing_data_in_broker(self, lang=lang, now=now)
                except Exception:
                    logger.error('[regenerator-freshness] thread did error: %s' % traceback.format_exc())
            # ok let's wait a bit, we are in a minute interval because we don't care about 1s delay here
            time.sleep(60)
    
    
    def create_reversed_list(self):
        self.hosts.create_reversed_list()
        self.hostgroups.create_reversed_list()
        self.contacts.create_reversed_list()
        self.contactgroups.create_reversed_list()
        self.notificationways.create_reversed_list()
        self.services.create_reversed_list()
        self.servicegroups.create_reversed_list()
        self.timeperiods.create_reversed_list()
        self.commands.create_reversed_list()
    
    
    # Now we get all data about an instance, link all this stuff :)
    def all_done_linking(self, part_configuration_incarnation):
        # type: (PartConfigurationIncarnation) -> None
        inst_id = part_configuration_incarnation.get_part_id()
        # In a scheduler we are already 'linked' so we can skip this
        if self.in_scheduler_mode:
            safe_print('Regenerator: We skip the all_done_linking phase because we are in a scheduler')
            return
        
        start = time.time()
        logger.debug('[regenerator] In ALL Done linking phase for instance: %d' % inst_id)
        # check if the instance is really defined, so got ALL the init phase
        if inst_id not in self.configs.keys():
            logger.error('[regenerator] The instance %s is not fully given, bailout' % part_configuration_incarnation)
            return
        
        self.part_configuration_incarnation_container.add(part_configuration_incarnation)
        logger.info('[ CONFIGURATION ] [ NEW ] [ REGENERATOR ] configuration part retrieved  : %s' % part_configuration_incarnation)
        
        # Try to load the in progress list and make them available for finding
        try:
            inp_hosts = self.inp_hosts[inst_id]
            inp_hosts.create_reversed_list()
            inp_hostgroups = self.inp_hostgroups[inst_id]
            inp_contactgroups = self.inp_contactgroups[inst_id]
            inp_services = self.inp_services[inst_id]
            inp_services.create_reversed_list()
            inp_servicegroups = self.inp_servicegroups[inst_id]
        except Exception, exp:
            logger.error('[regenerator] Warning in all done:  %s ' % exp)
            return
        
        # We want to know if all local checks have really valid hosts, so first index them
        local_conf_host_name_idx = set([h.get_name() for h in inp_hosts])
        for s in inp_services:
            hname = s.host_name
            if hname not in local_conf_host_name_idx:
                logger.error('[regenerator] IMPORTANT ERROR: the shard %d does have an error: the check %s do not have a known host (%s).\nIn order to solve this, we need to ask again to the scheduler the element definitions.\n%s\n' %
                             (inst_id, s.service_description, s.host_name, '*' * 30)
                             )
                del self.inp_hosts[inst_id]
                del self.inp_hostgroups[inst_id]
                del self.inp_contactgroups[inst_id]
                del self.inp_services[inst_id]
                del self.inp_servicegroups[inst_id]
                
                del self.configs[inst_id]
                self._ask_new_fill_brok(inst_id)
                return
        
        t0 = time.time()
        # Link HOSTGROUPS with hosts
        for hg in inp_hostgroups:
            new_members = []
            for (i, hname) in hg.members:
                h = inp_hosts.find_by_name(hname)
                if h:
                    new_members.append(h)
            hg.members = new_members
        logger.debug('[regenerator] Link hostgroups=>hosts: %.2f' % (time.time() - t0))
        
        # We need to delete the reversed list here
        # Because we may override id so that the reversed_list
        # is inconsistent (used in find by name)
        # Inconsistency comes from a mix of host group declaration :
        # within a host or with the define hostgroups statement
        # No worry, we recreate it just after!
        delattr(self.hostgroups, 'reversed_list')
        
        t1 = time.time()
        # Merge HOSTGROUPS with real ones
        for inphg in inp_hostgroups:
            hostgroup_name = inphg.hostgroup_name
            hg = self.hostgroups.find_by_name(hostgroup_name)
            # If the hostgroup already exist, just add the new
            # hosts into it
            if not hg:
                self.hostgroups[inphg.id] = inphg
        # We can declare hostgroups done
        self.hostgroups.create_reversed_list()
        logger.debug('[regenerator] Merging hostgroups: %.2f' % (time.time() - t1))
        
        t2 = time.time()
        # Now link HOSTS with hostgroups, and commands
        for h in inp_hosts:
            new_hostgroups = []
            for hostgroup_name in h.hostgroups.split(','):
                hg = self.hostgroups.find_by_name(hostgroup_name)
                if hg:
                    new_hostgroups.append(hg)
            h.hostgroups = new_hostgroups
            
            # Now link Command() objects
            self.linkify_a_command(h, 'check_command')
            self.linkify_a_command(h, 'event_handler')
            
            # Now link timeperiods
            self.linkify_a_timeperiod_by_name(h, 'notification_period')
            self.linkify_a_timeperiod_by_name(h, 'check_period')
            self.linkify_a_timeperiod_by_name(h, 'maintenance_period')
            
            # And link contacts too
            self.linkify_contacts(h, 'contacts')
            self.linkify_contacts(h, 'view_contacts')
            self.linkify_contacts(h, 'notification_contacts')
            self.linkify_contacts(h, 'edition_contacts')
            
            # Linkify tags
            for t in h.tags:
                if t not in self.tags:
                    self.tags[t] = 0
                self.tags[t] += 1
            
            # find the old host and remove associated service
            old_host = self.hosts.find_by_name(h.get_name())
            if old_host:
                for service in old_host.services:
                    hname = service.host_name
                    sdesc = service.service_description
                    # First remove the old service if present
                    self.services.remove_by_name_and_hostname(hname, sdesc)
            
            # Remove the old host is present
            self.hosts.remove_by_name(h.get_name())
            
            self.hosts[h.id] = h
        
        self.hosts.create_reversed_list()
        logger.debug('[regenerator] Link hosts with others: %.2f' % (time.time() - t2))
        self.services.remove_orphan_services()
        
        t3 = time.time()
        # Link SERVICEGROUPS with services
        for sg in inp_servicegroups:
            new_members = []
            for (i, sname) in sg.members:
                if i not in inp_services:
                    continue
                s = inp_services[i]
                new_members.append(s)
            sg.members = new_members
        logger.debug('[regenerator] Link servicegroups=>service: %.2f' % (time.time() - t3))
        
        # We need to delete the reversed list here
        # Because we may override id so that the reversed_list
        # is inconsistent (used in find by name)
        # Inconsistency comes from a mix of service group declaration :
        # within a service or with the define servicegroup statement
        # No worry, we recreate it just after!
        delattr(self.servicegroups, 'reversed_list')
        
        t4 = time.time()
        # Merge SERVICEGROUPS with real ones
        for inpsg in inp_servicegroups:
            sgname = inpsg.servicegroup_name
            sg = self.servicegroups.find_by_name(sgname)
            # If the servicegroup already exist, just add the new
            # services into it
            if not sg:
                self.servicegroups[inpsg.id] = inpsg
        # We can declare servicegroups done
        self.servicegroups.create_reversed_list()
        logger.debug('[regenerator] Merge sericegroups: %.2f' % (time.time() - t4))
        
        t5 = time.time()
        
        t50 = time.time()
        # Optimize service lookup before clean them
        self.services.optimize_service_search(self.hosts)
        t51 = time.time()
        
        sum1 = 0.0
        sum2 = 0.0
        sum3 = 0.0
        sum4 = 0.0
        
        # Now link SERVICES with hosts, servicesgroups, and commands
        for s in inp_services:
            t52 = time.time()
            new_servicegroups = []
            for sgname in s.servicegroups.split(','):
                sg = self.servicegroups.find_by_name(sgname)
                if sg:
                    new_servicegroups.append(sg)
            s.servicegroups = new_servicegroups
            
            # Now link with host
            hname = s.host_name
            s.host = self.hosts.find_by_name(hname)
            if s.host:
                s.host.services.append(s)
            
            sum1 += (time.time() - t52)
            
            t53 = time.time()
            # Now link Command() objects
            self.linkify_a_command(s, 'check_command')
            self.linkify_a_command(s, 'event_handler')
            
            # Now link timeperiods
            self.linkify_a_timeperiod_by_name(s, 'notification_period')
            self.linkify_a_timeperiod_by_name(s, 'check_period')
            self.linkify_a_timeperiod_by_name(s, 'maintenance_period')
            
            sum2 += (time.time() - t53)
            
            t54 = time.time()
            # And link contacts too
            self.linkify_contacts(s, 'contacts')
            
            # Linkify services tags
            for t in s.tags:
                if t not in self.services_tags:
                    self.services_tags[t] = 0
                self.services_tags[t] += 1
            
            sum3 += (time.time() - t54)
            
            t55 = time.time()
            
            # We can really declare this host OK now
            hname = s.host_name
            sdesc = s.service_description
            # First remove the old service if present
            self.services.remove_by_name_and_hostname(hname, sdesc)
            self.services[s.id] = s
            sum4 += (time.time() - t55)
            # logger.debug('Creation the service %s/%s with id: %s' % (hname, sdesc, s.id))
        
        logger.debug('[regenerator] Linking services with others: %.2f' % (time.time() - t5))
        logger.debug('[regenerator]    Sub times: 1:%.2f  2:%.2f  3:%.2f  4:%.2f' % (sum1, sum2, sum3, sum4))
        
        t6 = time.time()
        # Link services into service groups
        # First clean members
        for sg in self.servicegroups:
            sg.members = []
        # Then set the services into the service groups
        for s in self.services:
            for sg in s.servicegroups:
                sg.members.append(s)
        logger.debug('[regenerator] Linking services into servicegroups %.2f' % (time.time() - t6))
        
        t7 = time.time()
        # Same for hosts
        # Link hosts into hostgroups
        # First clean members
        for hg in self.hostgroups:
            hg.members = []
        # Then set the services into the service groups
        for h in self.hosts:
            for hg in h.hostgroups:
                hg.members.append(h)
        
        logger.debug('[regenerator] Linking hosts into hostgroups: %.2f' % (time.time() - t7))
        
        # Add realm of theses hosts. Only the first is useful
        for h in inp_hosts:
            self.realms.add(h.realm)
            break
        
        t8 = time.time()
        # Now we can link all impacts/source problem list
        # but only for the new ones here of course
        for h in inp_hosts:
            self.linkify_host_and_hosts(h, 'parents')
            self.linkify_host_and_hosts(h, 'childs')
            self.linkify_dict_srv_and_hosts(h, 'parent_dependencies')
            self.linkify_dict_srv_and_hosts(h, 'child_dependencies')
        logger.debug('[regenerator] Linking host impacts: %.2f' % (time.time() - t8))
        
        t9 = time.time()
        # Now services too
        for s in inp_services:
            self.linkify_dict_srv_and_hosts(s, 'parent_dependencies')
            self.linkify_dict_srv_and_hosts(s, 'child_dependencies')
        logger.debug('[regenerator] Linking services impacts: %.2f' % (time.time() - t9))
        
        # Linking TIMEPERIOD exclude with real ones now
        for tp in self.timeperiods:
            new_exclude = []
            for ex in tp.exclude:
                exname = ex.timeperiod_name
                t = self.timeperiods.find_by_name(exname)
                if t:
                    new_exclude.append(t)
            tp.exclude = new_exclude
        
        t10 = time.time()
        # Merge contactgroups with real ones
        for inpcg in inp_contactgroups:
            cgname = inpcg.contactgroup_name
            cg = self.contactgroups.find_by_name(cgname)
            # If the contactgroup already exist, just add the new
            # contacts into it
            if not cg:
                self.contactgroups[inpcg.id] = inpcg
        
        # We can declare contactgroups done
        self.contactgroups.create_reversed_list()
        logger.debug('[regenerator] Merging contactgroups: %.2f' % (time.time() - t10))
        
        t11 = time.time()
        # Same for contacts
        # Link contacts into contactgroups
        # First clean members
        for cg in self.contactgroups:
            cg.members = []
        # Then set the contacts into the contact groups
        for c in self.contacts:
            # We must set our contact into the real contactgroups. but there are 2 cases:
            # * it's our first start/shard, our inp_cg are already the self.contactgroups
            # * it's not our first start/shard, then our inp_cg will be droped and we just get it's name to get the self.contactgroups one
            for inp_cg in c.contactgroups:
                self.contactgroups.find_by_name(inp_cg.get_name()).members.append(c)
        logger.debug('[regenerator] Linking contacts into cgroups: %.2f' % (time.time() - t11))
        
        for cg in self.contactgroups:
            logger.debug('[regenerator] Contactgroup: %s, size:%d' % (cg.get_name(), len(cg.members)))
        
        self.all_hosts_and_services = []
        self.all_hosts_and_services.extend(self.hosts)
        self.all_hosts_and_services.extend(self.services)
        self.all_hosts_and_services.sort(key=lambda e: _natural_key(_stringify(e.get_full_name())))

        init_user_item_access_cache(self)
        
        logger.debug('[regenerator] load : ')
        logger.debug('[regenerator] hosts : %d ' % len(self.hosts))
        logger.debug('[regenerator] services : %d ' % len(self.services))
        logger.debug('[regenerator] contacts : %d ' % len(self.contacts))
        
        logger.debug('[regenerator] Linking configuration %d time: %.2f' % (inst_id, (time.time() - start)))
        
        # clean old objects
        to_del = []
        for c in self.contacts:
            if c.get_name() not in self.contact_names:
                to_del.append(c.id)
        for c_id in to_del:
            del self.contacts[c_id]
        self.contacts.create_reversed_list()
        del self.inp_hosts[inst_id]
        del self.inp_hostgroups[inst_id]
        del self.inp_contactgroups[inst_id]
        del self.inp_services[inst_id]
        del self.inp_servicegroups[inst_id]
    
    
    # We look for o.prop (CommandCall) and we link the inner Command() object with our real ones
    def linkify_a_command(self, o, prop):
        cc = getattr(o, prop, None)
        # if the command call is void, bypass it
        if not cc:
            setattr(o, prop, None)
            return
        cmdname = cc.command
        c = self.commands.find_by_name(cmdname)
        cc.command = c
    
    
    # We look at o.prop and for each command we relink it
    def linkify_commands(self, o, prop):
        v = getattr(o, prop, None)
        if not v:
            # If do not have a command list, put a void list instead
            setattr(o, prop, [])
            return
        
        for cc in v:
            cmdname = cc.command
            c = self.commands.find_by_name(cmdname)
            cc.command = c
    
    
    # We look at the timeperiod() object of o.prop and we replace it with our true one
    def linkify_a_timeperiod(self, o, prop):
        t = getattr(o, prop, None)
        if not t:
            setattr(o, prop, None)
            return
        tpname = t.timeperiod_name
        tp = self.timeperiods.find_by_name(tpname)
        setattr(o, prop, tp)
    
    
    # same than before, but the value is a string here
    def linkify_a_timeperiod_by_name(self, o, prop):
        tpname = getattr(o, prop, None)
        if not tpname:
            setattr(o, prop, None)
            return
        tp = self.timeperiods.find_by_name(tpname)
        setattr(o, prop, tp)
    
    
    # We look at o.prop and for each contacts in it,  we replace it with true object in self.contacts
    def linkify_contacts(self, o, prop):
        v = getattr(o, prop)
        
        if not v:
            return
        
        new_v = []
        for cname in v:
            c = self.contacts.find_by_name(cname)
            if c:
                new_v.append(c)
        setattr(o, prop, new_v)
    
    
    # We got a service/host dict, we want to get back to a flat list
    def linkify_dict_srv_and_hosts(self, o, prop):
        v = getattr(o, prop)
        
        if not v:
            setattr(o, prop, [])
        
        new_v = []
        
        for name in v['services']:
            elts = name.split('/')
            hname = elts[0]
            sdesc = '/'.join(elts[1:])
            s = self.services.find_srv_by_name_and_hostname(hname, sdesc)
            if s:
                new_v.append(s)
        for hname in v['hosts']:
            h = self.hosts.find_by_name(hname)
            if h:
                new_v.append(h)
        setattr(o, prop, new_v)
    
    
    def linkify_host_and_hosts(self, o, prop):
        v = getattr(o, prop)
        
        if not v:
            setattr(o, prop, [])
        
        new_v = []
        for hname in v:
            h = self.hosts.find_by_name(hname)
            if h:
                new_v.append(h)
        setattr(o, prop, new_v)
    
    
    # ------- Brok management part
    
    def before_after_hook(self, brok, obj):
        # This can be used by derived classes to compare the data in the brok with the object which will be updated by these data.
        # For example, it is possible to find out in this method whether the state of a host or service has changed.
        
        if 'next_schedule' in brok.type:
            return
        
        if 'host_' not in brok.type and 'service_' not in brok.type:
            return
        
        # IMPORTANT: we want to manage case where the scheduler WAS OUTDATED (no broks since long)
        # and we put all its elements into UNKNOWN, but now this scheduler does give us back
        # old broks and we are unhappy about this because we did say to the user that the state was UNKNOWN
        # and now we have a OK/CRITICAL data. So we must FAKE broks data to set this in UNKNOWN
        # during the period when we said that the scheduler was off, and now.
        with self.freshness_lock:
            instance_uuid = brok.data['instance_uuid']
            if instance_uuid in self.item_last_unknown_date:
                last_change_from_broker = self.item_last_unknown_date[instance_uuid]
                last_change_from_scheduler = brok.data.get('last_state_change', 0)
                # We were in UNKNOWN state (the scheduler had not sent any brok for a long time
                # The scheduler just sent us a new brok
                # - If the change from the scheduler is more recent than the change to UNKNOWN :
                #    => The data from the scheduler is correct
                #    => We do not need the date when we got into unknown anymore
                # - If the change from scheduler is before the change to UNKNOWN, then we juste got back communication :
                #    => The data about the previous state from the scheduler is obsolete
                #    => The new state is the state given by the scheduler
                if last_change_from_broker < last_change_from_scheduler:
                    del self.item_last_unknown_date[instance_uuid]
                else:
                    if obj.last_state_as_string != 'UNKNOWN':
                        brok.data['last_state_change'] = int(time.time())
                    else:
                        # If the state did not change, the change date should stay the same as well
                        brok.data['last_state_change'] = obj.last_state_change
                    brok.data['last_state_id'] = 3
                    brok.data['last_state_as_string'] = 'UNKNOWN'
    
    
    # ------ INITIAL PART
    def manage_program_status_brok(self, b):
        # We do not want to modify elements while switching elements
        # NOTE: we do not lock all access, because we will only change atomic elements on the elements
        with self.freshness_lock:
            self.do_manage_program_status_brok(b)
    
    
    def do_manage_program_status_brok(self, b):
        data = b.data
        instance_id = data['instance_id']
        logger.info('[regenerator] Creating instance %d ' % instance_id)
        
        config = Config()
        self.update_element(config, data)
        
        # If we were asking for this config, remove it from protection list
        if instance_id in self.currently_asking_full_broks:
            del self.currently_asking_full_broks[instance_id]
        
        # Clean all in_progress things.
        self.inp_hosts[instance_id] = Hosts([])
        self.inp_services[instance_id] = Services([])
        self.inp_hostgroups[instance_id] = Hostgroups([])
        self.inp_servicegroups[instance_id] = Servicegroups([])
        self.inp_contactgroups[instance_id] = Contactgroups([])
        self.contact_names = set()
        
        # And we save it
        self.configs[instance_id] = config
        
        # Clean the old 'hard' objects
        
        # We should clean all previously added hosts and services
        to_del_h = [h for h in self.hosts if h.instance_id == instance_id]
        to_del_srv = [s for s in self.services if s.instance_id == instance_id]
        
        logger.debug('[regenerator] Cleaning %d hosts  -- %d services from previous instance_id %d' % (len(to_del_h), len(to_del_srv), instance_id))
        
        # Clean hosts from hosts and hostgroups
        for h in to_del_h:
            del self.hosts[h.id]
        
        # Now clean all hostgroups too
        for hg in self.hostgroups:
            # Exclude from members the hosts with this inst_id
            hg.members = [h for h in hg.members if h.instance_id != instance_id]
        
        for s in to_del_srv:
            del self.services[s.id]
        
        # Now clean service groups
        for sg in self.servicegroups:
            sg.members = [s for s in sg.members if s.instance_id != instance_id]
        
        # We now regenerate reversed list so the client will find only real objects
        self.create_reversed_list()
    
    
    def manage_proxy_items_graph_brok(self, b):
        son_to_fathers = b.data['proxy_items_graph_son_to_fathers']
        proxyitemsgraph.update_from_son_to_fathers(son_to_fathers)
    
    
    # Get a new host. Add in in in progress tab
    def manage_initial_host_status_brok(self, b):
        data = b.data
        inst_id = data['instance_id']
        
        # Try to get the inp progress Hosts
        try:
            inp_hosts = self.inp_hosts[inst_id]
        except Exception as exp:
            logger.error('Error in manage_initial_host_status_brok : [%s]' % exp)
            logger.print_stack()
            return
        
        # Whatever the data is giving us as id, create our own increasing one
        h = Host({}, skip_useless_in_configuration=True)  # we do not care about running properties, they will be take from data
        self.before_after_hook(b, h)
        self.update_element(h, data)
        
        # We need to rebuild Downtime and Comment relationship
        for dtc in h.downtimes:
            dtc.ref = h
        
        # Ok, put in in the in progress hosts
        inp_hosts[h.id] = h
        # logger.debug('Creating the host %s as id %s' % (h.get_name(), h.id))
    
    
    # From now we only create an hostgroup in the in prepare part. We will link at the end.
    def manage_initial_hostgroup_status_brok(self, b):
        data = b.data
        hgname = data['hostgroup_name']
        inst_id = data['instance_id']
        
        # Try to get the inp progress Hostgroups
        try:
            inp_hostgroups = self.inp_hostgroups[inst_id]
        except Exception as exp:
            logger.error('Error in manage_initial_hostgroup_status_brok : [%s]' % exp)
            logger.print_stack()
            return
        
        logger.debug('[regenerator] Creating an hostgroup: %s in instance %d' % (hgname, inst_id))
        
        # With void members
        hg = Hostgroup([])
        
        # populate data
        self.update_element(hg, data)
        
        # We will link hosts into hostgroups later so now only save it
        inp_hostgroups[hg.id] = hg
    
    
    def manage_initial_service_status_brok(self, b):
        data = b.data
        inst_id = data['instance_id']
        
        # Try to get the inp progress Hosts
        try:
            inp_services = self.inp_services[inst_id]
        except Exception as exp:
            logger.error('Error in manage_initial_service_status_brok : [%s]' % exp)
            logger.print_stack()
            return
        
        s = Service({}, skip_useless_in_configuration=True)  # we do not care about running properties, they will be take from data
        self.before_after_hook(b, s)
        self.update_element(s, data)
        
        # We need to rebuild Downtime and Comment relationship
        for dtc in s.downtimes:
            dtc.ref = s
        
        # Ok, put in in the in progress hosts
        inp_services[s.id] = s
    
    
    # We create a servicegroup in our in progress part we will link it after
    def manage_initial_servicegroup_status_brok(self, b):
        data = b.data
        sgname = data['servicegroup_name']
        inst_id = data['instance_id']
        
        # Try to get the inp progress
        try:
            inp_servicegroups = self.inp_servicegroups[inst_id]
        except Exception, exp:
            logger.error('Error in manage_initial_servicegroup_status_brok : [%s]' % exp)
            logger.print_stack()
            return
        
        logger.debug('[regenerator] Creating a servicegroup: %s in instance %d' % (sgname, inst_id))
        
        # With void members
        sg = Servicegroup([])
        
        # populate data
        self.update_element(sg, data)
        
        # We will link hosts into hostgroups later
        # so now only save it
        inp_servicegroups[sg.id] = sg
    
    
    # For Contacts, it's a global value, so 2 cases:
    # We got it -> we update it
    # We don't -> we create it
    # In both cases we need to relink it
    def manage_initial_contact_status_brok(self, brok):
        data = brok.data
        cname = data['contact_name']
        inst_id = data['instance_id']
        self.contact_names.add(cname)
        
        contact = self.contacts.find_by_name(cname)
        if contact:
            self.update_element(contact, data)
        else:
            contact = Contact({}, skip_useless_in_configuration=True)  # we do not care about running properties, they will be take from data
            self.update_element(contact, data)
            self.contacts[contact.id] = contact
        
        # Delete some useless contact values
        contact.host_notification_commands = []
        contact.service_notification_commands = []
        contact.host_notification_period = []
        contact.service_notification_period = []
        
        # Now manage notification ways too
        # Same than for contacts. We create or
        # update
        nws = contact.notificationways
        new_notifways = []
        for cnw in nws:
            nwname = cnw.notificationway_name
            nw = self.notificationways.find_by_name(nwname)
            if not nw:
                nw = NotificationWay([])
                self.notificationways[nw.id] = nw
            # Now update it
            for prop in NotificationWay.properties:
                if hasattr(cnw, prop):
                    setattr(nw, prop, getattr(cnw, prop))
            new_notifways.append(nw)
            
            # Linking the notification way
            # With commands
            self.linkify_commands(nw, 'host_notification_commands')
            self.linkify_commands(nw, 'service_notification_commands')
            
            # Now link timeperiods
            self.linkify_a_timeperiod(nw, 'host_notification_period')
            self.linkify_a_timeperiod(nw, 'service_notification_period')
        
        contact.notificationways = new_notifways
        
        # linkify contacts with newly receive contactsgroups and
        # at the end (in all_done_linking) they will be re-linkify with the linkified groups
        ncgs = []
        
        # contact.contactgroups = contact.contactgroups.split(',')
        for cg_ in contact.contactgroups:
            cgf_ = self.inp_contactgroups[inst_id].find_by_name(cg_)
            if cgf_:
                ncgs.append(cgf_)
        contact.contactgroups = ncgs
        
        # Ok, declare this contact now :)
        # And notif ways too
        self.contacts.create_reversed_list()
        self.notificationways.create_reversed_list()
    
    
    # From now we only create an hostgroup with unlink data in the in prepare list. We will link all of them at the end.
    def manage_initial_contactgroup_status_brok(self, b):
        data = b.data
        cgname = data['contactgroup_name']
        inst_id = data['instance_id']
        
        # Try to get the inp progress Contactgroups
        try:
            inp_contactgroups = self.inp_contactgroups[inst_id]
        except Exception as exp:
            logger.error('Error in manage_initial_contactgroup_status_brok : [%s]' % exp)
            logger.print_stack()
            return
        
        logger.debug('[regenerator] Creating an contactgroup: %s in instance %d' % (cgname, inst_id))
        
        # With void members
        cg = Contactgroup([])
        
        # populate data
        self.update_element(cg, data)
        
        # We will link contacts into contactgroups later
        # so now only save it
        inp_contactgroups[cg.id] = cg
    
    
    # For Timeperiods we got 2 cases: do we already got the command or not.
    # if got: just update it
    # if not: create it and declare it in our main commands
    def manage_initial_timeperiod_status_brok(self, b):
        data = b.data
        tpname = data['timeperiod_name']
        
        tp = self.timeperiods.find_by_name(tpname)
        if tp:
            self.update_element(tp, data)
        else:
            tp = Timeperiod({})
            self.update_element(tp, data)
            self.timeperiods[tp.id] = tp
            # We add a timeperiod, we update the reversed list
            self.timeperiods.create_reversed_list()
    
    
    # For command we got 2 cases: do we already got the command or not.
    # if got: just update it
    # if not: create it and declare it in our main commands
    def manage_initial_command_status_brok(self, b):
        data = b.data
        cname = data['command_name']
        
        c = self.commands.find_by_name(cname)
        if c:
            # print 'Already existing command', cname, 'updating it'
            self.update_element(c, data)
        else:
            # print 'Creating a new command', cname
            c = Command({})
            self.update_element(c, data)
            self.commands[c.id] = c
            # Ok, we can regenerate the reversed list so
            self.commands.create_reversed_list()
    
    
    def manage_initial_scheduler_status_brok(self, b):
        data = b.data
        scheduler_name = data['scheduler_name']
        realm = data['realm']
        self.realms.add(realm)
        sched = SchedulerLink({})
        self.update_element(sched, data)
        self.schedulers[scheduler_name] = sched
    
    
    def manage_initial_poller_status_brok(self, b):
        data = b.data
        poller_name = data['poller_name']
        poller = PollerLink({})
        self.update_element(poller, data)
        self.pollers[poller_name] = poller
    
    
    def manage_initial_reactionner_status_brok(self, b):
        data = b.data
        reactionner_name = data['reactionner_name']
        reac = ReactionnerLink({})
        self.update_element(reac, data)
        self.reactionners[reactionner_name] = reac
    
    
    def manage_initial_broker_status_brok(self, b):
        data = b.data
        broker_name = data['broker_name']
        broker = BrokerLink({})
        self.update_element(broker, data)
        self.brokers[broker_name] = broker
    
    
    def manage_initial_receiver_status_brok(self, b):
        data = b.data
        receiver_name = data['receiver_name']
        receiver = ReceiverLink({})
        self.update_element(receiver, data)
        self.receivers[receiver_name] = receiver
    
    
    # This brok is here when the WHOLE initial phase is done. So we got all data, we can link all together :)
    def manage_initial_broks_done_brok(self, b):
        part_configuration_incarnation = b.part_configuration_incarnation
        # We do not want to modify elements while switching elements
        # NOTE: we do not lock all access, because we will only change atomic elements on the  elements
        with self.freshness_lock:
            self.all_done_linking(part_configuration_incarnation)
    
    
    # ------- Status Update part
    def _ask_new_fill_brok(self, instance_id, scheduler_name='unknown'):
        # Maybe we did already ask for a regeneration, so don't hammer it
        if instance_id in self.currently_asking_full_broks:
            logger.debug('[regenerator] skipping asking for instance id %d full brok information, because we already did it' % instance_id)
            return
        
        if self.from_module_to_main_daemon_queue is None:
            return
        
        logger.info("[ELEMENTS ] [ NEED DATA ] Requesting initial broks from the broker about the scheduler '%s' (managing shard %s)." % (scheduler_name, instance_id))
        msg = Message(id=0, type='NeedData', data={'full_instance_id': instance_id}, source=self.import_in)
        self.from_module_to_main_daemon_queue.put(msg)
        self.currently_asking_full_broks[instance_id] = True
    
    
    # A scheduler send us a 'I'm alive' brok.
    # If we never heard about this one, we got some problem and we ask him some initial data :)
    def manage_update_program_status_brok(self, b):
        data = b.data
        instance_id = data['instance_id']
        
        # If we got an update about an unknown instance, cry and ask for a full version!
        if instance_id not in self.configs.keys():
            scheduler_name = data['instance_name']
            self._ask_new_fill_brok(instance_id, scheduler_name=scheduler_name)
            return
        
        # Ok, good conf, we can update it
        config = self.configs[instance_id]
        self.update_element(config, data)
    
    
    def manage_update_host_status_brok(self, b):
        # There are some properties that should not change and are already linked so just remove them
        clean_prop = [
            'check_command',
            'hostgroups',
            'contacts',
            'view_contacts',
            'notification_contacts',
            'edition_contacts',
            'notification_period',
            'contact_groups',
            'check_period',
            'event_handler',
            'maintenance_period',
            'realm',
            'escalations'
        ]
        
        # some are only use when a topology change happened
        toplogy_change = b.data['topology_change']
        if not toplogy_change:
            other_to_clean = ['childs', 'parents', 'child_dependencies', 'parent_dependencies']
            clean_prop.extend(other_to_clean)
        
        data = b.data
        for prop in clean_prop:
            del data[prop]
        
        hname = data['host_name']
        h = self.hosts.find_by_name(hname)
        
        if h:
            # If we are updating the host with old instance_id, we will have several issues, so just skip this
            # Note: this must be a old brok, because initial broks are always before this brok
            if h.instance_id != data['instance_id']:
                return
            
            self.before_after_hook(b, h)
            self.update_element(h, data)
            
            # If the topology change, update it
            if toplogy_change:
                logger.debug('[regenerator] Topology change for host %s and hostparent: %s' % (h.get_name(), h.parent_dependencies))
                self.linkify_host_and_hosts(h, 'parents')
                self.linkify_host_and_hosts(h, 'childs')
                self.linkify_dict_srv_and_hosts(h, 'parent_dependencies')
                self.linkify_dict_srv_and_hosts(h, 'child_dependencies')
            
            # Relink downtimes
            for dtc in h.downtimes:
                dtc.ref = h
    
    
    def manage_update_service_status_brok(self, b):
        # There are some properties that should not change and are already linked
        # so just remove them
        clean_prop = ['check_command', 'servicegroups',
                      'contacts', 'notification_period', 'contact_groups',
                      'check_period', 'event_handler',
                      'maintenance_period', 'escalations']
        
        # some are only use when a topology change happened
        toplogy_change = b.data['topology_change']
        if not toplogy_change:
            other_to_clean = ['child_dependencies', 'parent_dependencies']
            clean_prop.extend(other_to_clean)
        
        data = b.data
        for prop in clean_prop:
            del data[prop]
        
        hname = data['host_name']
        sdesc = data['service_description']
        s = self.services.find_srv_by_name_and_hostname(hname, sdesc)
        if s:
            # If we are updating the check with old instance_id, we will have several issues, so just skip this
            # Note: this must be a old brok, because initial broks are always before this brok
            if s.instance_id != data['instance_id']:
                return
            
            self.before_after_hook(b, s)
            self.update_element(s, data)
            
            # If the topology change, update it
            if toplogy_change:
                self.linkify_dict_srv_and_hosts(s, 'parent_dependencies')
                self.linkify_dict_srv_and_hosts(s, 'child_dependencies')
            
            # Relink downtimes with the service
            for dtc in s.downtimes:
                dtc.ref = s
    
    
    def manage_update_broker_status_brok(self, b):
        data = b.data
        broker_name = data['broker_name']
        try:
            s = self.brokers[broker_name]
            self.update_element(s, data)
        except Exception:
            pass
    
    
    def manage_update_receiver_status_brok(self, b):
        data = b.data
        receiver_name = data['receiver_name']
        try:
            s = self.receivers[receiver_name]
            self.update_element(s, data)
        except Exception:
            pass
    
    
    def manage_update_reactionner_status_brok(self, b):
        data = b.data
        reactionner_name = data['reactionner_name']
        try:
            s = self.reactionners[reactionner_name]
            self.update_element(s, data)
        except Exception:
            pass
    
    
    def manage_update_poller_status_brok(self, b):
        data = b.data
        poller_name = data['poller_name']
        try:
            s = self.pollers[poller_name]
            self.update_element(s, data)
        except Exception:
            pass
    
    
    def manage_update_scheduler_status_brok(self, b):
        data = b.data
        scheduler_name = data['scheduler_name']
        
        realm = data['realm']
        self.realms.add(realm)
        try:
            s = self.schedulers[scheduler_name]
            self.update_element(s, data)
        except Exception:
            pass
    
    
    # ------- Check result and schedule part
    def manage_host_check_result_brok(self, b):
        data = b.data
        hname = data['host_name']
        
        h = self.hosts.find_by_name(hname)
        if h:
            # If we are updating the host with old instance_id, we will have several issues, so just skip this
            # Note: this must be a old brok, because initial broks are always before this brok
            if h.instance_id != data['instance_id']:
                return
            self.before_after_hook(b, h)
            self.update_element(h, data)
    
    
    # this brok should arrive within a second after the host_check_result_brok
    def manage_host_next_schedule_brok(self, b):
        self.manage_host_check_result_brok(b)
    
    
    # A service check have just arrived, we UPDATE data info with this
    def manage_service_check_result_brok(self, b):
        data = b.data
        hname = data['host_name']
        sdesc = data['service_description']
        s = self.services.find_srv_by_name_and_hostname(hname, sdesc)
        if s:
            # If we are updating the check with old instance_id, we will have several issues, so just skip this
            # Note: this must be a old brok, because initial broks are always before this brok
            if s.instance_id != data['instance_id']:
                return
            self.before_after_hook(b, s)
            self.update_element(s, data)
    
    
    # A service check update have just arrived, we UPDATE data info with this
    def manage_service_next_schedule_brok(self, b):
        self.manage_service_check_result_brok(b)
