#!/usr/bin/python
# -*- coding: utf-8 -*-

# Copyright (C) 2013-2018:
# This file is part of Shinken Enterprise, all rights reserved.

import time
from threading import Event
from shinken.log import logger
from shinken.misc.fast_copy import fast_deepcopy
from ..dao.helpers import split_and_strip_list, compute_diff, get_default_value
from ..dao.def_items import DEF_ITEMS, ITEM_STATE, ITEM_TYPE, WORKING_AREA_STATUS, METADATA, SOURCE_STATE, WORKING_AREA_LAST_ACTION, PROP_DEFAULT_VALUE, VALUE_FORCE_DEFAULT, STOP_INHERITANCE_VALUES, SERVICE_OVERRIDE
from ..dao.validators.validator import Validator


class SyncUICommon(object):
    def __init__(self):
        self.app = None
        self._configuration_stats = {}
        self._diff_staging_production = None
        self._diff_staging_production_event = Event()
        self.validator = None
        self.call_stats = {}
    
    
    def set_app(self, app):
        self.app = app
        self.validator = Validator(app=app, datamanagerV2=app.datamanagerV2)
    
    
    #     _    ____ _
    #    / \  / ___| |
    #   / _ \| |   | |
    #  / ___ \ |___| |___
    # /_/   \_\____|_____|
    #
    
    def check_acl(self, item, user, current_page):
        can_edit = True
        can_view = True
        cause = ''
        
        item_type = item.get_type()
        item_state = item.get_state()
        
        if item_type == ITEM_TYPE.HOSTS:
            can_edit = self.look_for_host_acl(user, item, what_for='edition')
            if not can_edit:  # maybe they even not have the view rights?
                can_view = self.look_for_host_acl(user, item, what_for='view')
        elif item_type == ITEM_TYPE.CONTACTS:
            can_view = can_edit = self.look_for_contact_acl(user, item)
        if not can_edit:
            cause = "ACL"
        if item and item_state != ITEM_STATE.NEW:
            work_area_status = WORKING_AREA_STATUS.get_work_area_status(item, item_state) if item_type == ITEM_TYPE.HOSTS else WORKING_AREA_STATUS.VALIDATED
            work_area_last_action = WORKING_AREA_LAST_ACTION.get_work_area_action(item, item_state)
            if current_page == 'working_area':
                can_edit = can_edit and (work_area_status == WORKING_AREA_STATUS.VALIDATED or work_area_status == WORKING_AREA_STATUS.WORKING or work_area_status == WORKING_AREA_STATUS.REJECTED)
            else:
                can_edit = can_edit and work_area_status == WORKING_AREA_STATUS.VALIDATED
            if not can_edit and not cause:
                cause = "WORK"
            if work_area_status == WORKING_AREA_STATUS.PROPOSED:
                can_edit = False
                cause = "PROPOSE"
            if work_area_last_action in (WORKING_AREA_LAST_ACTION.WORKING_SUPPRESSED, WORKING_AREA_LAST_ACTION.REJECTED_SUPPRESSED):
                can_edit = False
                cause = "DELETED"
        return {'can_edit': can_edit, 'can_view': can_view, 'cause': cause}
    
    
    # Look for an host ACL:
    # return is can do the thing we want like:
    # * view: can view
    # * edit: can edit or not
    def look_for_host_acl(self, user, item, what_for='view'):
        user_name = user.get_name().strip()
        contacts_property = {'view': 'view_contacts', 'edition': 'edition_contacts'}.get(what_for)
        contactgroups_property = {'view': 'view_contact_groups', 'edition': 'edition_contact_groups'}.get(what_for)
        # 1: is an admin, can see it   :)
        if user.is_admin():
            # logger.debug("[look_for_host_acl] Look ALC for [%s] with user [%s] access OK" % (item.get('host_name'), user_name))
            return True
        # 2: is the contact a direct contact?
        if not user_name:  # is the user playing?
            # logger.debug("[look_for_host_acl] Look ALC for [%s] with user [%s] access refuse" % (item.get('host_name'), user_name))
            return False
        if not item:
            logger.error('[look_for_host_acl] Item not found')
        item_contact = set((i.get_name() for i in item.get_link_items(contacts_property, states=[ITEM_STATE.STAGGING])))
        item_contact_groups = set((i.get_name() for i in item.get_link_items(contactgroups_property, states=[ITEM_STATE.STAGGING])))
        user_contactgroups = set((i.get_name() for i in user.get_link_items('contactgroups', states=[ITEM_STATE.STAGGING])))
        if user_name in set(item_contact):
            # logger.debug("[look_for_host_acl] Look ALC for [%s] with user [%s] access OK" % (item.get('host_name'), user_name))
            return True
        # Check if user contactgroups contains one or more item groups
        if item_contact_groups.intersection(user_contactgroups):
            # logger.debug("[look_for_host_acl] Look ALC for [%s] with user [%s] access OK" % (item.get('host_name'), user_name))
            return True
        if what_for == 'view':
            # everyone have access?
            _default_view_contacts = get_default_value(ITEM_TYPE.HOSTS, 'view_contacts') or 'nobody'
            
            # access can be granted only if everyone set for default value (not the default)
            # must have nothing in contacts and contact_group, for both object and templates
            if 'everyone' in _default_view_contacts:
                if (not item_contact or item_contact.intersection(set(("everyone", PROP_DEFAULT_VALUE, VALUE_FORCE_DEFAULT)))) and (not item_contact_groups or item_contact_groups.intersection(STOP_INHERITANCE_VALUES)):
                    # logger.debug("[look_for_host_acl] Look ALC for [%s] with user [%s] access OK" % (item.get('host_name'), user_name))
                    return True
        return False
    
    
    def look_for_contact_acl(self, user, item):
        # We will have several ways to allow a user to see the element
        # *1: it can be an admin
        # *2: is the contact
        
        # 1: is an admin, can see it :)
        if user.is_admin():
            return True
        # 2: is the contact the same?
        user_name = user.get_name().strip()
        if not user_name:  # is the user playing?
            return False
        item_contact_name = item.get('contact_name', '')
        return item_contact_name == user_name
    
    
    #                   __ _                       _   _                     _        _
    #   ___ ___  _ __  / _(_) __ _ _   _ _ __ __ _| |_(_) ___  _ __      ___| |_ __ _| |_ ___
    #  / __/ _ \| '_ \| |_| |/ _` | | | | '__/ _` | __| |/ _ \| '_ \    / __| __/ _` | __/ __|
    # | (_| (_) | | | |  _| | (_| | |_| | | | (_| | |_| | (_) | | | |   \__ \ || (_| | |_\__ \
    # \___\___/|_| |_|_| |_|\__, |\__,_|_|  \__,_|\__|_|\___/|_| |_|___|___/\__\__,_|\__|___/
    #                       |___/                                 |_____|
    def apply_staging_in_production(self):
        self.app.datamanagerV2.update_all_item(ITEM_STATE.STAGGING, {'$set': {'last_modification.show_diffs_state': {}, 'last_modification.show_diffs': False}})
        self.app.datamanagerV2.copy_state(ITEM_STATE.STAGGING, ITEM_STATE.PRODUCTION)
    
    
    def get_configuration_stats(self, user):
        try:
            return fast_deepcopy(self._get_configuration_stats(user))
        except Exception as err:
            logger.error('Error in get_configuration_stats [%s]' % err)
            logger.print_stack()
            return {'error': True}
    
    
    def reset_configuration_stats_cache(self):
        try:
            self._configuration_stats.clear()
            return True
        except Exception as err:
            logger.error('Error in get_configuration_stats [%s]' % err)
            logger.print_stack()
            return False
    
    
    def _get_configuration_stats(self, user):
        user_name = user['contact_name']
        TAG_PERF_LOG = 'get_configuration_stats[%s]' % user_name
        # fast exit if user already have configuration stat in cache
        cached_value = self._configuration_stats.get(user_name, None)
        if cached_value:
            # logger.debug("[%s] from cached " % TAG_PERF_LOG)
            return cached_value
        logger.debug("[%s] not cached " % TAG_PERF_LOG)
        
        ZERO_VALUES = [0, 0]
        t0 = time.time()
        
        r = {
            'overall':
                {
                    'working_area': [0, 0],
                    'rejected'    : [0, 0],
                    'proposed'    : [0, 0],
                    'new'         : 0,
                    'change'      : 0,
                    'stagging'    : 0,
                    'to_apply'    : 0
                },
            'detail' : {},
            'error'  : False
        }
        
        is_admin = user.is_admin()
        item_types = DEF_ITEMS.keys() if is_admin else [ITEM_TYPE.HOSTS, ITEM_TYPE.CONTACTS]
        
        for item_type in item_types:
            r['detail'][item_type] = {}
            
            # find items in working area, only for hosts (for now)
            if item_type == ITEM_TYPE.HOSTS:
                # Count elements that are being edited in working area (the ones i can edit and the one i can see)
                items = self.app.datamanagerV2.find_merge_state_items(item_type, (ITEM_STATE.WORKING_AREA, ITEM_STATE.STAGGING), where={'work_area_info.status': WORKING_AREA_STATUS.WORKING})
                visible, mine = self.count_visible(item_type, items, user)
                r['detail'][item_type]['working_area'] = [mine, visible]
                r['overall']['working_area'][0] += mine
                r['overall']['working_area'][1] += visible
                
                # Counts elements that are being rejected in working area
                items = self.app.datamanagerV2.find_merge_state_items(item_type, (ITEM_STATE.WORKING_AREA, ITEM_STATE.STAGGING), where={'work_area_info.status': WORKING_AREA_STATUS.REJECTED})
                visible, mine = self.count_visible(item_type, items, user)
                r['detail'][item_type]['rejected'] = [mine, visible]
                r['overall']['rejected'][0] += mine
                r['overall']['rejected'][1] += visible
                
                # Count elements that are being submitted to stagging
                items = self.app.datamanagerV2.find_merge_state_items(item_type, (ITEM_STATE.WORKING_AREA, ITEM_STATE.STAGGING), where={'work_area_info.status': WORKING_AREA_STATUS.PROPOSED})
                visible, mine = self.count_visible(item_type, items, user)
                r['detail'][item_type]['proposed'] = [mine, visible]
                r['overall']['proposed'][0] += mine
                r['overall']['proposed'][1] += visible
            else:
                r['detail'][item_type]['working_area'] = ZERO_VALUES
                r['detail'][item_type]['rejected'] = ZERO_VALUES
                r['detail'][item_type]['proposed'] = ZERO_VALUES
            
            nb_new = self.app.datamanagerV2.count_items(item_type, ITEM_STATE.NEW) if is_admin else 0
            r['detail'][item_type]['new'] = nb_new
            r['overall']['new'] += nb_new
            
            if is_admin:
                nb_stagging = self.app.datamanagerV2.count_items(item_type, ITEM_STATE.STAGGING)
            else:
                items = self.app.datamanagerV2.find_merge_state_items(item_type, (ITEM_STATE.STAGGING, ITEM_STATE.WORKING_AREA))
                visible, _ = self.count_visible(item_type, items, user)
                nb_stagging = visible
            
            r['detail'][item_type]['stagging'] = nb_stagging
            r['overall']['stagging'] += nb_stagging
            
            diff = self.get_diff_staging_production()[item_type]
            nb_to_apply = 0
            if is_admin:
                for (diff_type, diff_list) in diff.iteritems():
                    for diff in diff_list:
                        if diff_type == 'changed':
                            try:
                                if diff.get('modification_info', {}).get('show_diffs', True) is True:
                                    nb_to_apply += 1
                            except AttributeError:
                                nb_to_apply += 1
                        else:
                            nb_to_apply += 1
            r['detail'][item_type]['to_apply'] = nb_to_apply
            r['overall']['to_apply'] += nb_to_apply
            
            r['detail'][item_type]['change'] = 0
            
            if is_admin:
                # can't count_items because stagging and workarea can be mixed and changes ca be count as double
                count = 0
                count += len(self.app.datamanagerV2.find_merge_state_items(item_type, (ITEM_STATE.WORKING_AREA, ITEM_STATE.STAGGING), where={'@metadata.%s' % METADATA.SOURCE_STATE: SOURCE_STATE.CHANGE}))
                r['detail'][item_type]['change'] = count
                r['overall']['change'] += count
        
        r['detail'][ITEM_TYPE.ELEMENTS] = r['overall']
        t1 = time.time()
        r['time'] = t1 - t0
        r['since'] = t1
        self._configuration_stats[user_name] = r
        # logger.debug('[get_configuration_stats] give back for [%s] : [%s]' % (user_name, r))
        logger.log_perf(t0, TAG_PERF_LOG, 'end')
        return r
    
    
    def count_visible(self, item_type, items, user):
        if user.is_admin():
            my_item = len([i for i in items if i.get('work_area_info', {}).get('get_by_user', '') == user['_id']])
            return len(items), my_item
        
        if item_type == ITEM_TYPE.CONTACTS:
            return min(len(items), 1), min(len(items), 1)
        
        if item_type == ITEM_TYPE.HOSTS:
            visible_count = 0
            my_item = 0
            for item in items:
                work_area_user = item.get('work_area_info', {}).get('get_by_user', '')
                if self.look_for_host_acl(user, item, what_for='view'):
                    visible_count += 1
                    if work_area_user == user['_id']:
                        my_item += 1
            return visible_count, my_item
        
        return len(items), min(len(items), 1)
    
    
    #
    #      _ _  __  __       _        _        _                                      _            _   _
    #   __| (_)/ _|/ _|  ___| |_ __ _| |_ __ _(_)_ __   __ _      _ __  _ __ ___   __| |_   _  ___| |_(_) ___  _ __
    #  / _` | | |_| |_  / __| __/ _` | __/ _` | | '_ \ / _` |    | '_ \| '__/ _ \ / _` | | | |/ __| __| |/ _ \| '_ \
    # | (_| | |  _|  _| \__ \ || (_| | || (_| | | | | | (_| |    | |_) | | | (_) | (_| | |_| | (__| |_| | (_) | | | |
    #  \__,_|_|_| |_|___|___/\__\__,_|\__\__, |_|_| |_|\__, |____| .__/|_|  \___/ \__,_|\__,_|\___|\__|_|\___/|_| |_|
    #              |_____|               |___/         |___/_____|_|
    #
    def compute_all_diff_staging_production(self):
        from ..dao.transactions.transactions import DBTransaction
        with DBTransaction():
            self._diff_staging_production_event.clear()
            self._diff_staging_production = {}
            for item_type in DEF_ITEMS.keys():
                self._diff_staging_production[item_type] = {'new': [], 'removed': [], 'changed': []}
                _ids = {}
                items_from_staging = self.app.datamanagerV2.find_items(item_type, ITEM_STATE.STAGGING)
                items_from_production = self.app.datamanagerV2.find_items(item_type, ITEM_STATE.PRODUCTION)
                
                for item in items_from_staging:
                    _ids[item['_id']] = {'staging': item}
                for item in items_from_production:
                    if item['_id'] in _ids:
                        _ids[item['_id']]['production'] = item
                    else:
                        _ids[item['_id']] = {'production': item}
                
                for item_id in _ids:
                    item_staging = _ids[item_id].get('staging', None)
                    item_production = _ids[item_id].get('production', None)
                    self._compute_one_diff_staging_production(item_id, item_type, item_staging, item_production, all_change_must_be_show=True)
            
            self._diff_staging_production_event.set()
        # logger.debug("[compute_all_diff_staging_production] (staging / production) diff = %s" % pformat(self.diffs))
    
    
    def compute_one_diff_staging_production(self, item_id, item_type, item_staging, item_production, all_change_must_be_show=False):
        self._diff_staging_production_event.wait()
        self._compute_one_diff_staging_production(item_id, item_type, item_staging, item_production, all_change_must_be_show)
    
    
    def _compute_one_diff_staging_production(self, item_id, item_type, item_staging, item_production, all_change_must_be_show=False):
        # logger.debug("[compute_one_diff_staging_production] items staging / production =\n%s" % logger.diff(item_staging, item_production))
        modification_info = {}
        if item_staging or item_production:
            modification_info = item_staging.get('last_modification', {}) if item_staging else item_production.get('last_modification', {})
        
        diff_entry = {
            'stagging'         : item_staging,
            'production'       : item_production,
            '_id'              : item_id,
            'changed'          : [],
            'modification_info': modification_info
        }
        
        diffs_type = self._diff_staging_production[item_type]
        # Filter item in previous diff info
        for t in ('new', 'changed', 'removed'):
            diffs_type[t] = [d for d in diffs_type[t] if d['_id'] != item_id]
        
        # If present in stagging but not in production it's a new element
        if item_staging and not item_production:
            diffs_type['new'].append(diff_entry)
        # The inverse: it's a removed element :)
        elif not item_staging and item_production:
            diffs_type['removed'].append(diff_entry)
        # Ok both are present, and now? look at  values :)
        elif item_staging and item_production:
            staging_raw_item = item_staging.get_raw_item(flatten_links=(ITEM_STATE.STAGGING,))
            production_raw_item = item_production.get_raw_item(flatten_links=(ITEM_STATE.PRODUCTION,))
            
            diff_entry['stagging'] = staging_raw_item
            diff_entry['production'] = production_raw_item
            
            if staging_raw_item.get(SERVICE_OVERRIDE, None):
                diff_entry['stagging'][SERVICE_OVERRIDE] = item_staging.get_raw_value(SERVICE_OVERRIDE, flatten_links=None)
            if production_raw_item.get(SERVICE_OVERRIDE, None):
                diff_entry['production'][SERVICE_OVERRIDE] = item_production.get_raw_value(SERVICE_OVERRIDE, flatten_links=None)
            
            changed = compute_diff(staging_raw_item, production_raw_item, item_type)
            
            # We remove diff for list with same value but in different order.
            item_class = DEF_ITEMS[item_type]['class']
            to_remove = []
            for attr_with_change in changed:
                prop_entry = item_class.properties.get(attr_with_change, None)
                if prop_entry is not None and prop_entry.merging == 'join':
                    value_stagging = staging_raw_item.get(attr_with_change, '')
                    value_production = production_raw_item.get(attr_with_change, '')
                    s_vo = set(split_and_strip_list(value_production))
                    s_ve = set(split_and_strip_list(value_stagging))
                    if s_vo == s_ve:
                        to_remove.append(attr_with_change)
            changed = filter(lambda x: x not in to_remove, changed)
            
            diff_entry['changed'] = changed
            
            if all_change_must_be_show:
                if len(changed) != 0:
                    diffs_type['changed'].append(diff_entry)
            else:
                if len(changed) == 0:
                    item_staging['last_modification']['show_diffs'] = False
                    item_staging['last_modification']['show_diffs_state'] = {}
                    # save with only one callback
                    self.app.database_cipher.cipher(item=item_staging, item_type=item_type)
                    self.app.datamanagerV2.data_provider.save_item(item_staging, item_type, ITEM_STATE.STAGGING)
                else:
                    must_add_diff_entry = False
                    # If there are one difference which isn't automatic we add the change in the apply page.
                    show_diffs_state = item_staging.get('last_modification', {}).get('show_diffs_state', {})
                    for attr_with_change in changed:
                        if show_diffs_state.get(attr_with_change, 'AUTO_MODIFICATION') != 'AUTO_MODIFICATION':
                            must_add_diff_entry = True
                    
                    if must_add_diff_entry:
                        diffs_type['changed'].append(diff_entry)
    
    
    # "Short" version than diffs, with far less verbose changed entry, with only the keys that did changed instead of the whole objects
    def get_diff_staging_production_summary(self):
        diffs = self.get_diff_staging_production()
        summary = {}
        for item_type, full_diff in diffs.iteritems():
            changed_summary = []
            summary[item_type] = {
                "new"    : full_diff['new'],
                "removed": full_diff['removed'],
                "changed": changed_summary
            }
            # in changed, we have a list of ('stagging':FULL OBJ, 'production':FULL OBJECT, 'changed':['key that did changed']
            for full_change_entry in full_diff['changed']:
                stag_obj = full_change_entry['stagging']
                prod_obj = full_change_entry['production']
                changed_key = full_change_entry['changed']
                n_stag_obj = {}
                n_prod_obj = {}
                n_e = {
                    'stagging'         : n_stag_obj,
                    'production'       : n_prod_obj,
                    'changed'          : changed_key,
                    '_id'              : full_change_entry['_id'],
                    'modification_info': full_change_entry['modification_info']
                }
                # we will only keep useful keys in the stag and prod obj
                for old_o, new_o in [(prod_obj, n_prod_obj), (stag_obj, n_stag_obj)]:
                    for k2, v2 in old_o.iteritems():
                        if k2 in changed_key or k2.startswith('_') or k2.endswith('name') or k2.endswith('description') or k2 == 'register' or k2 == 'last_modification' or k2 == 'work_area_info':
                            new_o[k2] = v2
                changed_summary.append(n_e)
        
        return summary
    
    
    def get_diff_staging_production(self):
        self._diff_staging_production_event.wait()
        return self._diff_staging_production
    
    
    def get_call_stats(self):
        call_stats_dump = {}
        for (path, avg) in syncuicommon.call_stats.iteritems():
            call_stats_dump[path] = avg.get_avg(avg_on_time=False)
        return call_stats_dump


syncuicommon = SyncUICommon()
