#!/usr/bin/python
# -*- coding: utf-8 -*-
#
# Copyright (C) 2013-2018
# This file is part of Shinken Enterprise, all rights reserved.

import itertools
import re
from sys import _getframe
from urlparse import urlparse

from shinken.log import logger
from shinken.macroresolver import MacroResolver
from shinken.objects import Timeperiod
from shinken.property import _boolean_states, AclShareProp, BoolProp
from shinken.util import get_key_value_sequence, GET_KEY_VALUE_SEQUENCE_ERROR_NOERROR, make_unicode, WeakMethod
from shinkensolutions.service_override_parser import unparse_service_override, parse_service_override_property_list_errors
from shinkensolutions.time_period_parser import TimePeriodParser
from ...dao.checks_inheritance import inherited_checks_from_hosts_templates
from ...dao.def_items import ADVANCED_PERIODS, DEF_ITEMS, FORBIDDEN_DATA, HISTORY_ACTION, ITEM_STATE, ITEM_TYPE, NOT_TO_LOOK, PROP_DEFAULT_VALUE, SERVICE_EXCLUDES_BY_NAME, SERVICE_OVERRIDE, SERVICE_OVERRIDE_UUID_SEP, STOP_INHERITANCE_VALUES, \
    TIMEPERIOD_BASIC_PROPS, VALUE_FORCE_DEFAULT, METADATA
from ...dao.helpers import escape_XSS, get_item_property, get_name_from_type, get_property_def, split_and_strip_list, unescape_XSS, read_frontend_links
from ...dao.items import BaseItem
from ...dao.parsers.bp_rule_parser import BpRuleParser
from ...dao.parsers.complex_exp_parser import parse_complex_exp
from ...front_end.helper import PopUpUrlSelect

ILLEGAL_CHARS = re.compile(r'''[`~!$%^&*"|'<>?,()=/+]''')
ILLEGAL_CHARS_SERVICE = re.compile(r'''[`~!$%^&*"|'<>?,()=+]''')
ILLEGAL_CHARS_SERVICE_EXCLUDES = re.compile(r'''[`~!%^&"'|<>?()=+]''')
ILLEGAL_CHARS_SERVICE_OVERRIDES = re.compile(r'''[`~$%^&*"|'<>?,()=+]''')
ILLEGAL_CHARS_HOST_AND_HOSTGROUP_NAME = re.compile(r'''[~%^"'<>?=]''')

RESTRICT_ALLOWED_CHARS_URL = re.compile(r'''["' ]+''')
#  Regexp explanation :               Scheme | Credentials | Netloc (with macros)      | Port (with macros)            |     Path + Query String
RESTRICT_EXTERNAL_URL = re.compile('''^\w+://(\w+(:\w+)?@)?[^$]*(\$[^"':/ ]+\$[^$]*)*(:[^$]*(\$[0-9A-Za-z_]+\$[^$]*)*)?(/|/([\w#!:.?+=&%@\-/$]+))?$''')
RESTRICT_POSITIVE_DECIMAL = re.compile("""^\d+(\.\d\d?\d?)?$""")

# some properties need to be validated as number
INTEGER_MANDATORY_PROPERTY = set(('timeout', 'max_check_attempts', 'check_running_timeout', 'check_interval', 'retry_interval', 'warning_threshold_cpu_usage', 'notification_interval', 'first_notification_time', 'last_notification_time'))
VALIDE_BOOL_VALUE = _boolean_states.keys()
VALIDE_BOOL_VALUE.append(None)
VALIDE_BOOL_VALUE.append(PROP_DEFAULT_VALUE)
VALIDE_BOOL_VALUE.append(VALUE_FORCE_DEFAULT)

MARKER_FOR_KEY_FROM_ADVANCED_PERIODS = "adv_#_#_#"


def is_integer(s):
    try:
        int(s)
        return True
    except ValueError:
        pass
    return False


class Validator(object):
    _LEVEL_VALIDATOR_SUCCESS = 10
    _LEVEL_VALIDATOR_WARNING = 20
    _LEVEL_VALIDATOR_ERROR = 30
    
    
    def __init__(self, app, datamanagerV2):
        self._TYPE_VALIDATOR = {
            ITEM_TYPE.CONTACTS                 : WeakMethod(self._validate_contacts),
            ITEM_TYPE.CONTACTTPLS              : WeakMethod(self._validate_contacttpls),
            ITEM_TYPE.HOSTS                    : WeakMethod(self._validate_hosts),
            ITEM_TYPE.HOSTTPLS                 : WeakMethod(self._validate_hosttpls),
            ITEM_TYPE.CLUSTERS                 : WeakMethod(self._validate_clusters),
            ITEM_TYPE.CLUSTERTPLS              : WeakMethod(self._validate_clustertpls),
            ITEM_TYPE.SERVICESHOSTS            : WeakMethod(self._validate_services),
            ITEM_TYPE.SERVICESHOSTTPLS         : WeakMethod(self._validate_services),
            ITEM_TYPE.SERVICESCLUSTERS         : WeakMethod(self._validate_services),
            ITEM_TYPE.SERVICESCLUSTERTPLS      : WeakMethod(self._validate_services),
            ITEM_TYPE.SERVICETPLS              : WeakMethod(self._validate_servicetpls),
            ITEM_TYPE.MACROMODULATIONS         : WeakMethod(self._validate_macromodulations),
            ITEM_TYPE.TIMEPERIODS              : WeakMethod(self._validate_timeperiods),
            ITEM_TYPE.RESULTMODULATIONS        : WeakMethod(self._validate_resultmodulations),
            ITEM_TYPE.BUSINESSIMPACTMODULATIONS: WeakMethod(self._validate_businessimpactmodulations),
            ITEM_TYPE.COMMANDS                 : WeakMethod(self._validate_commands),
            ITEM_TYPE.NOTIFICATIONWAYS         : WeakMethod(self._validate_notificationways),
            ITEM_TYPE.ESCALATIONS              : WeakMethod(self._validate_escalations),
            ITEM_TYPE.CONTACTGROUPS            : WeakMethod(self._validate_contactgroups),
            ITEM_TYPE.HOSTGROUPS               : WeakMethod(self._validate_hostgroups),
        }
        self._COMMON_RULES = [
            WeakMethod(self._rule_valide_item_name),
            WeakMethod(self._rule_is_XSS_protected),
            WeakMethod(self._rule_validate_number_properties),
            WeakMethod(self._rule_valide_bool_format),
            WeakMethod(self._rule_validate_no_rogue_plus),
            WeakMethod(self._rule_valide_index_size),
            WeakMethod(self._rule_remove_invisible_characters_in_data),
        ]
        self.app = app
        self.datamanagerV2 = datamanagerV2
        
        self.XSS_PROTECTION_WHITELIST = {
            ITEM_TYPE.CLUSTERS                 : {
                'XSS_BLOCK'    : set(['pack']),
                'XSS_BLOCK_URL': set(['notes_url']),
            },
            ITEM_TYPE.CLUSTERTPLS              : {
                'XSS_BLOCK'    : set(['pack']),
                'XSS_BLOCK_URL': set(['notes_url']),
            },
            ITEM_TYPE.HOSTS                    : {
                'XSS_REPLACE'  : set(['display_name']),
                'XSS_BLOCK'    : set(['address', 'pack']),
                'XSS_BLOCK_URL': set(['notes_url']),
            },
            ITEM_TYPE.HOSTTPLS                 : {
                'XSS_BLOCK'    : set(['pack']),
                'XSS_BLOCK_URL': set(['notes_url']),
            },
            ITEM_TYPE.HOSTGROUPS               : {
                'XSS_BLOCK': set(['pack']),
            },
            ITEM_TYPE.SERVICESHOSTS            : {
                'XSS_BLOCK'    : set(['pack']),
                'XSS_BLOCK_URL': set(['notes_url']),
            },
            ITEM_TYPE.SERVICESHOSTTPLS         : {
                'XSS_BLOCK'    : set(['pack']),
                'XSS_BLOCK_URL': set(['notes_url']),
            },
            ITEM_TYPE.SERVICESCLUSTERS         : {
                'XSS_BLOCK'    : set(['pack']),
                'XSS_BLOCK_URL': set(['notes_url']),
            },
            ITEM_TYPE.SERVICESCLUSTERTPLS      : {
                'XSS_BLOCK'    : set(['pack']),
                'XSS_BLOCK_URL': set(['notes_url']),
            },
            ITEM_TYPE.SERVICETPLS              : {
                'XSS_BLOCK'    : set(['pack']),
                'XSS_BLOCK_URL': set(['notes_url']),
            },
            ITEM_TYPE.CONTACTS                 : {
                'XSS_REPLACE': set(['display_name']),
                'XSS_BLOCK'  : set(['pack']),
            },
            ITEM_TYPE.CONTACTTPLS              : {
                'XSS_BLOCK': set(['pack']),
            },
            ITEM_TYPE.CONTACTGROUPS            : {
                'XSS_BLOCK': set(['pack']),
            },
            ITEM_TYPE.ESCALATIONS              : {
                'XSS_BLOCK': set(['pack']),
            },
            ITEM_TYPE.NOTIFICATIONWAYS         : {
                'XSS_BLOCK': set(['pack']),
            },
            ITEM_TYPE.COMMANDS                 : {
                'XSS_BLOCK': set(['pack']),
            },
            ITEM_TYPE.BUSINESSIMPACTMODULATIONS: {
                'XSS_BLOCK': set(['pack']),
            },
            ITEM_TYPE.MACROMODULATIONS         : {
                'XSS_BLOCK': set(['pack']),
            },
            ITEM_TYPE.RESULTMODULATIONS        : {
                'XSS_BLOCK': set(['pack']),
            },
            ITEM_TYPE.TIMEPERIODS              : {
                'XSS_BLOCK': set(['pack']),
            },
        }
        
        self.OVERRIDE_FORBIDDEN_PROPERTIES = set([
            'hostgroup_name',
            'host_name',
            'service_description',
            'name',
            'register',
            'enabled',
            'pack',
            'duplicate_foreach',
            'default_value',
            'poller_tag',
            'reactionner_tag',
            'definition_order',
            'event_handler',
            'use',
        
        ])
        
        self.OVERRIDE_REPLACED_KEYWORDS = set([
            'check_command',
        ])
        
        self.OVERRIDE_SPECIFIC_KEYWORDS = set([
            'check_command_args',
        ])
        
        PROPERTIES_NOT_IN_TYPE = {
            ITEM_TYPE.SERVICETPLS        : set(['hostgroup_name', 'host_name']),
            ITEM_TYPE.SERVICESCLUSTERS   : set(['hostgroup_name']),
            ITEM_TYPE.SERVICESCLUSTERTPLS: set(['hostgroup_name']),
            ITEM_TYPE.HOSTTPLS           : set(['address']),
            ITEM_TYPE.CLUSTERS           : set([
                'parents',
                'for_all_users',
                'max_check_attempts',
                'check_interval',
                'retry_interval',
                'active_checks_enabled',
                'passive_checks_enabled',
                'poller_tag',
                'process_perf_data',
                'check_freshness',
                'freshness_threshold',
                'service_excludes',
            ]),
            ITEM_TYPE.CLUSTERTPLS        : set([
                'parents',
                'for_all_users',
                'max_check_attempts',
                'check_interval',
                'retry_interval',
                'active_checks_enabled',
                'passive_checks_enabled',
                'poller_tag',
                'process_perf_data',
                'check_freshness',
                'freshness_threshold',
                'service_excludes',
            ]),
            ITEM_TYPE.CONTACTS           : set(['for_all_users']),
            ITEM_TYPE.CONTACTTPLS        : set(['password']),
        }
        
        self.KNOWN_PROPERTIES = {}
        for item_type in DEF_ITEMS:
            cls = DEF_ITEMS[item_type]['class']
            self.KNOWN_PROPERTIES[item_type] = {
                'properties'               : set(getattr(cls, 'properties', {}).keys()) - PROPERTIES_NOT_IN_TYPE.get(item_type, set()),
                'passthrough'              : set(getattr(cls, 'passthrough', {}).keys()) - PROPERTIES_NOT_IN_TYPE.get(item_type, set()),
                'blacklist'                : set(getattr(cls, 'blacklist', {}).keys()) - PROPERTIES_NOT_IN_TYPE.get(item_type, set()),
                'not_inherited_passthrough': set(getattr(cls, 'not_inherited_passthrough', {}).keys()) - PROPERTIES_NOT_IN_TYPE.get(item_type, set()),
            }
            
            known_props = self.KNOWN_PROPERTIES[item_type]
            known_props['all'] = NOT_TO_LOOK | known_props['properties'] | known_props['passthrough'] | known_props['not_inherited_passthrough']
    
    
    def validate(self, item_type, item):
        # before validate, we add a metadata to validate the number of dollar on item.
        # After the validator passed, we can removed the metadata
        METADATA.update_metadata(item, METADATA.VALIDATE_DOLLAR_NUMBER, set())
        
        validation = []
        type_validator = self._TYPE_VALIDATOR.get(item_type, None)
        
        validation.extend(self._validate_common(item, item_type))
        
        if type_validator:
            validation.extend(type_validator(item, item_type))
        
        METADATA.remove_metadata(item, METADATA.VALIDATE_DOLLAR_NUMBER)
        
        result = {
            'has_error'        : False,
            'error_messages'   : [],
            'has_warning'      : False,
            'warning_messages' : [],
            'has_messages'     : False,
            'messages'         : [],
            'qualified_message': []
        }
        
        if len(validation) != 0:
            for _validation in validation:
                level = _validation['level']
                rule_name = _validation['rule_name']
                message = _validation['message']
                
                if level == Validator._LEVEL_VALIDATOR_ERROR:
                    result['has_error'] = True
                    result['error_messages'].append(message)
                
                if level == Validator._LEVEL_VALIDATOR_WARNING:
                    result['has_warning'] = True
                    result['warning_messages'].append(message)
                result['has_messages'] = True
                result['messages'].append(message)
                result['qualified_message'].append((level, rule_name, message))
        return result
    
    
    @staticmethod
    def _create_message(message, level=None):
        if level is None:
            level = Validator._LEVEL_VALIDATOR_ERROR
        
        return {
            'level'    : level,
            'rule_name': _getframe(1).f_code.co_name,
            'message'  : message
        }
    
    
    def _validate_common(self, item, item_type):
        validation = []
        
        for rule in self._COMMON_RULES:
            validation.extend(rule(item, item_type))
        
        return validation
    
    
    def _validate_contacts(self, item, item_type):
        validation = []
        validation.extend(self._rule_valide_no_disable_self(item))
        validation.extend(self._rule_ACL_visualisation_UI(item))
        validation.extend(self._rule_valide_data_name(item))
        validation.extend(self._rule_valide_business_impact(item, item_type))
        validation.extend(self._rule_valide_dollar_in_data(item))
        validation.extend(self._rule_valide_duplicate_name(item, item_type))
        validation.extend(self._rule_valide_acl_in_tab_history(item))
        validation.extend(self._rule_no_loop_in_template(item, item_type))
        return validation
    
    
    def _validate_contacttpls(self, item, item_type):
        validation = []
        validation.extend(self._rule_ACL_visualisation_UI(item))
        validation.extend(self._rule_valide_data_name(item))
        validation.extend(self._rule_valide_business_impact(item, item_type))
        validation.extend(self._rule_valide_dollar_in_data(item))
        validation.extend(self._rule_valide_duplicate_name(item, item_type))
        validation.extend(self._rule_valide_acl_in_tab_history(item))
        validation.extend(self._rule_no_loop_in_template(item, item_type))
        return validation
    
    
    def _validate_hosts(self, item, item_type):
        validation = []
        validation.extend(self._rule_valide_data_name(item))
        validation.extend(self._rule_validate_host_dfe_data(item, item_type))
        validation.extend(self._rule_valide_dollar_in_data(item))
        validation.extend(self._rule_valide_macros_in_check_command(item))
        validation.extend(self._rule_valide_business_impact(item, item_type))
        validation.extend(self._rule_valide_duplicate_name_for_types(item, (ITEM_TYPE.HOSTS, ITEM_TYPE.CLUSTERS)))
        validation.extend(self._rule_valide_check_interval_gte_1(item))
        validation.extend(self._rule_valide_flap_threshold(item))
        validation.extend(self._rule_valide_sla_threshold(item, item_type))
        validation.extend(self._rule_valide_service_overrides(item, item_type))
        validation.extend(self._rule_valide_check_running_timeout_property(item))
        validation.extend(self._rule_valide_warning_threshold_cpu_usage_property(item))
        validation.extend(self._rule_valide_notification_interval_property(item))
        validation.extend(self._validate_service_excludes(item))
        validation.extend(self._rule_no_loop_in_template(item, item_type))
        validation.extend(self._rule_valide_external_url(item))
        return validation
    
    
    def _validate_hosttpls(self, item, item_type):
        validation = []
        validation.extend(self._rule_valide_data_name(item))
        validation.extend(self._rule_validate_host_dfe_data(item, item_type))
        validation.extend(self._rule_valide_dollar_in_data(item))
        validation.extend(self._rule_valide_macros_in_check_command(item))
        validation.extend(self._rule_valide_business_impact(item, item_type))
        validation.extend(self._rule_valide_duplicate_name_for_types(item, (ITEM_TYPE.HOSTTPLS, ITEM_TYPE.CLUSTERTPLS)))
        validation.extend(self._rule_valide_check_interval_gte_1(item))
        validation.extend(self._rule_valide_flap_threshold(item))
        validation.extend(self._rule_valide_sla_threshold(item, item_type))
        validation.extend(self._rule_valide_service_overrides(item, item_type))
        validation.extend(self._rule_valide_check_running_timeout_property(item))
        validation.extend(self._rule_valide_warning_threshold_cpu_usage_property(item))
        validation.extend(self._rule_valide_notification_interval_property(item))
        validation.extend(self._validate_service_excludes(item))
        validation.extend(self._rule_valide_external_url(item))
        validation.extend(self._rule_no_loop_in_template(item, item_type))
        return validation
    
    
    def _validate_clusters(self, item, item_type):
        validation = []
        validation.extend(self._rule_valide_data_name(item))
        validation.extend(self._rule_valide_business_impact(item, item_type))
        validation.extend(self._rule_validate_host_dfe_data(item, item_type))
        validation.extend(self._rule_valide_dollar_in_data(item))
        validation.extend(self._rule_valide_duplicate_name_for_types(item, (ITEM_TYPE.HOSTS, ITEM_TYPE.CLUSTERS)))
        validation.extend(self._rule_valide_check_interval_gte_1(item))
        validation.extend(self._rule_valide_flap_threshold(item))
        validation.extend(self._rule_valide_sla_threshold(item, item_type))
        validation.extend(self._rule_valide_service_overrides(item, item_type))
        validation.extend(self._validate_service_excludes(item))
        validation.extend(self._rule_valide_external_url(item))
        validation.extend(self._validate_bp_rule(item))
        validation.extend(self._rule_no_loop_in_template(item, item_type))
        return validation
    
    
    def _validate_clustertpls(self, item, item_type):
        validation = []
        validation.extend(self._rule_valide_data_name(item))
        validation.extend(self._rule_valide_business_impact(item, item_type))
        validation.extend(self._rule_validate_host_dfe_data(item, item_type))
        validation.extend(self._rule_valide_dollar_in_data(item))
        validation.extend(self._rule_valide_duplicate_name_for_types(item, (ITEM_TYPE.HOSTTPLS, ITEM_TYPE.CLUSTERTPLS)))
        validation.extend(self._rule_valide_check_interval_gte_1(item))
        validation.extend(self._rule_valide_flap_threshold(item))
        validation.extend(self._rule_valide_sla_threshold(item, item_type))
        validation.extend(self._rule_valide_service_overrides(item, item_type))
        validation.extend(self._validate_service_excludes(item))
        validation.extend(self._rule_valide_external_url(item))
        validation.extend(self._validate_bp_rule(item))
        validation.extend(self._rule_no_loop_in_template(item, item_type))
        return validation
    
    
    def _validate_services(self, item, item_type):
        on_check_override = False
        if item_type.startswith('override_'):
            on_check_override = True
            item_type = item_type.replace('override_', '')
        validation = []
        validation.extend(self._rule_valide_data_name(item))
        validation.extend(self._rule_valide_data_duplicate_foreach(item))
        validation.extend(self._rule_validate_host_dfe_default_value(item))
        validation.extend(self._rule_valide_dollar_in_data(item))
        validation.extend(self._rule_valide_macros_in_check_command(item))
        validation.extend(self._rule_valide_host_name(item, item_type))
        validation.extend(self._rule_valide_hostgroup_names(item, item_type))
        validation.extend(self._rule_valide_business_impact(item, item_type))
        validation.extend(self._rule_valide_external_url(item))
        if not on_check_override:
            validation.extend(self._rule_valide_duplicate_name_for_service(item, item_type))
            validation.extend(self._rule_no_loop_in_template(item, item_type))
        validation.extend(self._rule_valide_flap_threshold(item))
        validation.extend(self._rule_valide_sla_threshold(item, item_type))
        validation.extend(self._rule_valide_check_interval_gte_1(item))
        validation.extend(self._rule_no_duplicate_service_overrides(item, item_type))
        validation.extend(self._rule_valide_check_running_timeout_property(item))
        validation.extend(self._rule_valide_warning_threshold_cpu_usage_property(item))
        validation.extend(self._rule_valide_notification_interval_property(item))
        if item_type in (ITEM_TYPE.SERVICESHOSTTPLS, ITEM_TYPE.SERVICESCLUSTERTPLS):
            validation.extend(self._rule_valide_complex_expression(item, item_type))
        return validation
    
    
    def _validate_servicetpls(self, item, item_type):
        validation = []
        validation.extend(self._rule_valide_data_name(item))
        validation.extend(self._rule_valide_data_duplicate_foreach(item))
        validation.extend(self._rule_validate_host_dfe_default_value(item))
        validation.extend(self._rule_valide_dollar_in_data(item))
        validation.extend(self._rule_valide_macros_in_check_command(item))
        validation.extend(self._rule_valide_host_name(item, item_type))
        validation.extend(self._rule_valide_hostgroup_names(item, item_type))
        validation.extend(self._rule_valide_business_impact(item, item_type))
        validation.extend(self._rule_valide_duplicate_name(item, item_type))
        validation.extend(self._rule_valide_flap_threshold(item))
        validation.extend(self._rule_valide_sla_threshold(item, item_type))
        validation.extend(self._rule_valide_check_interval_gte_1(item))
        validation.extend(self._rule_valide_check_running_timeout_property(item))
        validation.extend(self._rule_valide_warning_threshold_cpu_usage_property(item))
        validation.extend(self._rule_valide_notification_interval_property(item))
        validation.extend(self._rule_valide_external_url(item))
        validation.extend(self._rule_no_loop_in_template(item, item_type))
        return validation
    
    
    def _validate_macromodulations(self, item, item_type):
        validation = []
        validation.extend(self._rule_valide_data_name(item))
        validation.extend(self._rule_valide_dollar_in_data(item))
        validation.extend(self._rule_valide_data_duplicate_foreach(item))
        validation.extend(self._rule_valide_duplicate_name(item, item_type))
        return validation
    
    
    def _validate_timeperiods(self, item, item_type):
        validation = []
        validation.extend(self._rule_valide_duplicate_name(item, item_type))
        validation.extend(self._rule_valide_periods(item))
        return validation
    
    
    def _validate_resultmodulations(self, item, item_type):
        validation = []
        validation.extend(self._rule_valide_duplicate_name(item, item_type))
        return validation
    
    
    def _validate_businessimpactmodulations(self, item, item_type):
        validation = []
        validation.extend(self._rule_valide_business_impact(item, item_type))
        validation.extend(self._rule_valide_duplicate_name(item, item_type))
        return validation
    
    
    def _validate_commands(self, item, item_type):
        validation = []
        validation.extend(self._rule_valide_duplicate_name(item, item_type))
        validation.extend(self._rule_valide_macro_in_command_line(item))
        validation.extend(self._rule_valide_timeout_property(item))
        validation.extend(self._rule_valide_warning_threshold_cpu_usage_property(item))
        return validation
    
    
    def _validate_notificationways(self, item, item_type):
        validation = []
        validation.extend(self._rule_valide_business_impact(item, item_type))
        validation.extend(self._rule_valide_duplicate_name(item, item_type))
        return validation
    
    
    def _validate_escalations(self, item, item_type):
        validation = []
        validation.extend(self._rule_valide_duplicate_name(item, item_type))
        validation.extend(self._rule_valide_notification_interval_property_on_escalations(item))
        return validation
    
    
    def _validate_contactgroups(self, item, item_type):
        validation = []
        validation.extend(self._rule_valide_duplicate_name(item, item_type))
        return validation
    
    
    def _validate_hostgroups(self, item, item_type):
        validation = []
        validation.extend(self._rule_valide_duplicate_name(item, item_type))
        return validation
    
    
    def _rule_valide_SE_UUID(self, item, item_type):
        validation = []
        
        se_uuid = item.get('_SE_UUID', '')
        if se_uuid and (se_uuid.count('-') != 2 or se_uuid.split('-')[1] != item_type):
            validation.append(self._create_message(self.app._('validator.malformed_se_uuid'), Validator._LEVEL_VALIDATOR_WARNING))
            del item['_SE_UUID']
        
        return validation
    
    
    def _rule_valide_no_disable_self(self, item):
        validation = []
        user = self._get_auth_user()
        
        if user['_id'] == item['_id'] and item.get('enabled', '1') == '0':
            validation.append(self._create_message(self.app._('element.exception_disable_not_authorized')))
        
        return validation
    
    
    # see SEF-1813 : Share - Adding user rights for share viewing, creating, managing, deleting, moving
    def _rule_ACL_visualisation_UI(self, item):
        validation = []
        
        acl_share_private = item.get('acl_share_private')
        acl_share_group = item.get('acl_share_group')
        acl_share_everybody = item.get('acl_share_everybody')
        
        if acl_share_private:
            validation.extend(self._rule_ACL_visualisation_UI_generic(acl_share_private, 'acl_share_private'))
        if acl_share_group:
            validation.extend(self._rule_ACL_visualisation_UI_generic(acl_share_group, 'acl_share_group'))
        if acl_share_everybody:
            validation.extend(self._rule_ACL_visualisation_UI_generic(acl_share_everybody, 'acl_share_everybody'))
        
        return validation
    
    
    def _rule_is_XSS_protected(self, item, item_type):
        validation = []
        
        _XSS_BLOCK = self.XSS_PROTECTION_WHITELIST[item_type].get('XSS_BLOCK', set())
        _XSS_BLOCK_URL = self.XSS_PROTECTION_WHITELIST[item_type].get('XSS_BLOCK_URL', set())
        _XSS_REPLACE = self.XSS_PROTECTION_WHITELIST[item_type].get('XSS_REPLACE', set())
        
        for property_name in item:
            property_value = item.get(property_name, '')
            if not property_value or not isinstance(property_value, basestring):
                continue
            if _XSS_BLOCK and property_name in _XSS_BLOCK and re.search(r'[<>&"\'/]', property_value):
                validation.append(self._create_message(self.app._('validator.invalid_characters') % ('<>&"\'/', make_unicode(property_name))))
            elif _XSS_BLOCK_URL and property_name in _XSS_BLOCK_URL and re.search(r'["\']', property_value):
                validation.append(self._create_message(self.app._('validator.invalid_characters') % ('<>"\'', make_unicode(property_name))))
            elif ((_XSS_REPLACE and property_name in _XSS_REPLACE) or property_name.startswith('_')) and re.search(r'[<>"\'/]', property_value):
                property_value = escape_XSS(property_value)
                item[property_name] = property_value
        
        return validation
    
    
    def _rule_validate_host_dfe_default_value(self, item):
        validation = []
        dollar_already_checked_for_props = METADATA.get_metadata(item, METADATA.VALIDATE_DOLLAR_NUMBER)
        
        dfe_default_value = item.get('default_value')
        if dfe_default_value and dfe_default_value not in STOP_INHERITANCE_VALUES:
            if dfe_default_value.endswith(","):  # This case is allowed by the Arbiter but not by the Front-End
                validation.append(self._create_message("%s : %s " % ('default_value', self.app._('element.tooltip_dfe_syntax_issue'))))
            
            parser_regex = re.compile(r'\$\((?P<val>.*?)\)\$')
            
            found_matches_iterator = parser_regex.finditer(dfe_default_value)
            data_found = []
            for index, match in enumerate(found_matches_iterator):
                value = match.group('val')
                if value is not None:
                    data_found.append("$(%s)$" % value)
                    updated_value = self.__validate_dollar_count_in_data_values('default_value', validation, value, dollar_already_checked_for_props, key='', arg=index + 1)
                    if updated_value != value:
                        value_to_replace = '$(%s)$' % value
                        updated_value = '$(%s)$' % updated_value
                        item['default_value'] = item['default_value'].replace(value_to_replace, updated_value, 1)
            if not data_found:
                validation.append(self._create_message("<span class='shinken-field'>%s</span> : %s " % ('default_value', self.app._('element.tooltip_dfe_syntax_issue'))))
            elif "".join(data_found) != dfe_default_value:
                validation.append(self._create_message("<span class='shinken-field'>%s</span> : %s " % ('default_value', self.app._('element.tooltip_dfe_syntax_issue'))))
        return validation
    
    
    def _rule_validate_host_dfe_data(self, item, item_type):
        validation = []
        dollar_already_checked_for_props = METADATA.get_metadata(item, METADATA.VALIDATE_DOLLAR_NUMBER)
        
        item_templates = item.get('use', '')
        if not isinstance(item_templates, basestring):
            item_templates = self.datamanagerV2.flatten_prop(item, item_type, 'use')
        templates_list = split_and_strip_list(item_templates)
        
        item_name = get_name_from_type(item_type, item)
        checks = inherited_checks_from_hosts_templates(templates_list, get_name_from_type(item_type, item), item['_id'], item_type, self.datamanagerV2)
        templates_list.insert(0, item_name)
        checked_properties = set()
        key_template = re.compile(r'VALUE[0-9]+')
        for tpl in templates_list:
            for check in checks.get(tpl, []):
                dfe_property = check.get('duplicate_foreach')
                if dfe_property and dfe_property not in checked_properties:
                    checked_properties.add(dfe_property)
                    dfe_property_value = item.get(dfe_property, '')
                    if not dfe_property_value:  # The property may be defined in a template, in which case the check is not needed here
                        continue
                    
                    if dfe_property_value.endswith(","):  # This case is allowed by the Arbiter but not by the Front-End
                        validation.append(self._create_message("%s : %s " % (make_unicode(dfe_property), self.app._('element.tooltip_dfe_syntax_issue'))))
                        continue
                    
                    parsed_dfe_property_value, error = get_key_value_sequence(dfe_property_value)
                    if error == GET_KEY_VALUE_SEQUENCE_ERROR_NOERROR:
                        known_keys = set()
                        duplicate_keys_errors = set()
                        for key_to_check in parsed_dfe_property_value:
                            # Hum ... this is dirty : The value is XSS safe for save and show it but here, we need to know the real value to know if it's valid or not
                            if ILLEGAL_CHARS_SERVICE.search(unescape_XSS(key_to_check['KEY'])):
                                validation.append(self._create_message("%s : %s " % (make_unicode(dfe_property), self.app._('element.tooltip_dfe_syntax_issue'))))
                            
                            if key_to_check['KEY'] in known_keys and key_to_check['KEY'] not in duplicate_keys_errors:
                                validation.append(self._create_message(self.app._('validator.dfe_key_duplicate') % (make_unicode(dfe_property), key_to_check['KEY'])))
                                duplicate_keys_errors.add(key_to_check['KEY'])
                            else:
                                known_keys.add(key_to_check['KEY'])
                            for key, value in key_to_check.iteritems():
                                if value is not None and key_template.match(key):
                                    updated_value = self.__validate_dollar_count_in_data_values(dfe_property, validation, value, dollar_already_checked_for_props, key=key_to_check['KEY'], arg=key)
                                    if updated_value != value:
                                        value_to_replace = '$(%s)$' % value
                                        updated_value = '$(%s)$' % updated_value
                                        item[dfe_property] = item[dfe_property].replace(value_to_replace, updated_value, 1)
                    else:
                        validation.append(self._create_message("%s : %s " % (make_unicode(dfe_property), self.app._('element.tooltip_dfe_syntax_issue'))))
        
        return validation
    
    
    def _rule_valide_data_name(self, item):
        validation = []
        for property_name in item:
            if not property_name.startswith('_'):
                continue
            
            if property_name in FORBIDDEN_DATA:
                validation.append(self._create_message(self.app._('validator.data_name_id_not_allowed').format(property_name)))
            
            if property_name == '_':
                validation.append(self._create_message(self.app._('element.empty_data_name')))
            
            if re.search(r'[^0-9a-zA-Z_-]', property_name):
                validation.append(self._create_message(self.app._('element.invalide_data_name') % make_unicode(property_name)))
        
        return validation
    
    
    def _rule_valide_index_size(self, item, item_type):
        validation = []
        
        # Due to a mongo limitation, index must be lower than 1024 bytes. The index len is index_key + value.
        # For each item, index is set on key_name.
        _item_property = DEF_ITEMS[item_type]['key_name']
        value = item.get(_item_property, None)
        if not value:
            # No value to check ? No problem for me
            return validation
        
        # To be large, we limit the len to 900 bytes
        if len(bytearray(value.encode('utf8'))) > 900:
            validation.append(self._create_message(self.app._('element.validator_index_too_long') % (self.app._("%s.%s" % (item_type, _item_property)))))
        
        return validation
    
    
    def _rule_valide_data_duplicate_foreach(self, item):
        validation = []
        
        dfe_property = item.get('duplicate_foreach')
        if dfe_property:
            if not dfe_property.startswith('_'):
                validation.append(self._create_message(self.app._('element.invalide_data_duplicate_foreach')))
            
            if dfe_property.startswith('_') and re.search(r'[^0-9a-zA-Z_-]', dfe_property):
                validation.append(self._create_message(self.app._('element.invalide_data_name') % make_unicode(dfe_property)))
        
        return validation
    
    
    def _rule_validate_no_rogue_plus(self, item, item_type):
        allowed_fields_with_plus = DEF_ITEMS[item_type].get('fields_with_plus', set())
        validation = []
        for prop in allowed_fields_with_plus.intersection(set(item.keys())):
            value = item[prop]
            if isinstance(value, list):
                value_to_test = ','.join(value)
            elif isinstance(value, basestring):
                value_to_test = value
            else:
                continue  # Values which are neither strings nor lists can't have a "+" anyway
            if value_to_test.find('+', 1) > -1:
                rogue_plus_message = self.app._('validator.rogue_plus') % prop
                del item[prop]
                validation.append(self._create_message(rogue_plus_message, level=Validator._LEVEL_VALIDATOR_WARNING))
        
        return validation
    
    
    def _rule_valide_item_name(self, item, item_type):
        validation = []
        if ITEM_TYPE.has_template(item_type):
            template_type = DEF_ITEMS[item_type]['template']
            keys_name_son = set([DEF_ITEMS[son_type]['key_name'] for son_type in ITEM_TYPE.SONS[template_type]])
            if isinstance(item, BaseItem):
                # We don't want all possible keys but only keys actually defined in the item
                item_keys = item.get_raw_item().keys()
            else:
                item_keys = item.keys()
            invalide_name_key = keys_name_son.intersection(set(item_keys))
            if len(invalide_name_key) > 1:
                invalide_name_key = list(invalide_name_key)
                validation.append(self._create_message(self.app._('element.invalide_name_extra_key') % (invalide_name_key[0], invalide_name_key[1])))
        
        # Must have a name (or service description if service)
        _name = get_name_from_type(item_type, item)
        # Refuse objects with no or invalid name
        if not _name:
            key_name = DEF_ITEMS[item_type]['key_name']
            validation.append(self._create_message(self.app._('element.missing_object_name') % key_name))
        _name = _name.strip()
        if _name in ('null', 'shinken-service', 'shinken-host', 'shinken-cluster'):
            validation.append(self._create_message(self.app._('validator.invalid_name')))
        if not Validator._rule_valide_no_special_char_in_item_name(item_type, _name):
            illegal_chars = ILLEGAL_CHARS.pattern
            if item_type in ITEM_TYPE.ALL_SERVICES:
                illegal_chars = ILLEGAL_CHARS_SERVICE.pattern
            validation.append(self._create_message(self.app._('element.invalid_object_name') % illegal_chars[1:-1]))
        return validation
    
    
    def _rule_valide_host_name(self, item, item_type):
        validation = []
        if item_type not in ITEM_TYPE.ALL_SERVICES:
            return validation
        
        hosts_value = item.get('host_name', '')
        if not isinstance(hosts_value, basestring):
            return validation
        
        for hosts_name in split_and_strip_list(hosts_value):
            if ILLEGAL_CHARS_HOST_AND_HOSTGROUP_NAME.search(hosts_name):
                validation.append(self._create_message(self.app._('validator.invalid_characters') % (ILLEGAL_CHARS_HOST_AND_HOSTGROUP_NAME.pattern[1:-1], "host_name")))
        
        return validation
    
    
    def _rule_valide_hostgroup_names(self, item, item_type):
        validation = []
        if item_type not in ITEM_TYPE.ALL_SERVICES:
            return validation
        
        hostgroup_value = item.get('hostgroup_name', '')
        if not isinstance(hostgroup_value, basestring):
            return validation
        
        for hostgroup_name in split_and_strip_list(hostgroup_value):
            if ILLEGAL_CHARS_HOST_AND_HOSTGROUP_NAME.search(hostgroup_name):
                validation.append(self._create_message(self.app._('validator.invalid_characters') % (ILLEGAL_CHARS_HOST_AND_HOSTGROUP_NAME.pattern[1:-1], 'hostgroup_name')))
        
        return validation
    
    
    def _rule_validate_number_properties(self, item, item_type):
        validation = []
        for property_name in INTEGER_MANDATORY_PROPERTY:
            if property_name in item and not is_integer(item[property_name]) and not item[property_name] in STOP_INHERITANCE_VALUES:
                del item[property_name]
                validation.append(self._create_message(self.app._('element.property_should_be_a_number') % property_name, level=Validator._LEVEL_VALIDATOR_WARNING))
        return validation
    
    
    def _rule_valide_duplicate_name_for_service(self, item, item_type):
        if not item.get('service_description', None):
            return []
        
        key_name = 'service_description'
        item_name = item[key_name]
        where = {
            'service_description_lower': item_name.lower(),
            '_id'                      : {'$ne': item['_id']}
        }
        
        item_host_name = item.get('host_name', '')
        if not isinstance(item_host_name, basestring):
            item_host_name = self.datamanagerV2.flatten_prop(item, item_type, 'host_name')
        
        item_hostgroup_name = item.get('hostgroup_name', '')
        if not isinstance(item_hostgroup_name, basestring):
            item_hostgroup_name = self.datamanagerV2.flatten_prop(item, item_type, 'hostgroup_name')
        
        has_sources_datamanager = hasattr(self, 'datamanagerV2_for_merged_items')
        item_states = [ITEM_STATE.STAGGING, ITEM_STATE.NEW]
        if has_sources_datamanager:
            item_states.append(ITEM_STATE.MERGE_SOURCES)
        
        for item_state in item_states:
            same_name_items = self.datamanagerV2.find_items(item_type, item_state, where=where, lookup={key_name: 1, 'host_name': 1, 'hostgroup_name': 1})
            if has_sources_datamanager:
                same_name_items.extend(self.datamanagerV2_for_merged_items.find_items(item_type, item_state, where=where, lookup={key_name: 1, 'host_name': 1, 'hostgroup_name': 1}))
            for same_name_item in same_name_items:
                if get_item_property(same_name_item, 'host_name'):
                    same_name_host = self.datamanagerV2.flatten_prop(same_name_item, item_type, 'host_name')
                else:
                    same_name_host = ''
                
                if get_item_property(same_name_item, 'hostgroup_name'):
                    same_name_hostgroup = self.datamanagerV2.flatten_prop(same_name_item, item_type, 'hostgroup_name')
                else:
                    same_name_hostgroup = ''
                
                # We get all sources from all same_name_items + all sources from item we add it all and put in a set for remove same source name so
                # We get all sources of all items in item + same_name_items
                from_sources = itertools.chain.from_iterable((i.get('sources', '').split(',') for i in same_name_items))
                from_sources = set(itertools.chain(from_sources, item.get('sources', '').split(',')))
                from_sources = ','.join(('<span class="shinken-data-user">%s</span>' % i for i in from_sources))
                if item_host_name and (item_host_name == same_name_host):
                    if item_state == ITEM_STATE.MERGE_SOURCES:
                        error_msg = self.app._("validator.same_check_with_same_host_in_%s" % item_state) % (self.app._('type.%s' % item_type), item_name, item_host_name, from_sources)
                    else:
                        error_msg = self.app._("validator.same_check_with_same_host_in_%s" % item_state) % (self.app._('type.%s' % item_type), item_name, item_host_name)
                    return [self._create_message(error_msg)]
                if item_hostgroup_name and (item_hostgroup_name == same_name_hostgroup):
                    if item_state == ITEM_STATE.MERGE_SOURCES:
                        error_msg = self.app._("validator.same_check_with_same_hostgroup_in_%s" % item_state) % (self.app._('type.%s' % item_type), item_name, item_hostgroup_name, from_sources)
                    else:
                        error_msg = self.app._("validator.same_check_with_same_hostgroup_in_%s" % item_state) % (self.app._('type.%s' % item_type), item_name, item_hostgroup_name)
                    return [self._create_message(error_msg)]
        return []
    
    
    def _rule_valide_duplicate_name_for_types(self, item, item_types):
        validation = []
        for item_type in item_types:
            validation.extend(self._rule_valide_duplicate_name(item, item_type))
        return validation
    
    
    def _rule_valide_duplicate_name(self, item, item_type):
        validation = []
        
        key_name = DEF_ITEMS[item_type]['key_name']
        item_name = item.get(key_name, '')
        where = {"%s_lower" % key_name: item_name.lower(), "_id": {"$ne": item['_id']}}
        
        item_states = [ITEM_STATE.STAGGING, ITEM_STATE.NEW]
        if ITEM_TYPE.has_work_area(item_type):
            item_states.append(ITEM_STATE.WORKING_AREA)
        
        for item_state in item_states:
            same_name_items = self.datamanagerV2.find_items(item_type, item_state, where=where)
            if same_name_items:
                validation.append(self._create_message(self.app._("element.already_in_%s" % item_state) % item_name))
                break
        return validation
    
    
    def _rule_ACL_visualisation_UI_generic(self, acl_share, acl_name):
        validation = []
        
        if acl_share == PROP_DEFAULT_VALUE:
            return validation
        
        if not (re.match(r'[0-1]{5}', acl_share) or acl_share in ('none', 'all', 'null')):
            _acls = acl_share.split(',')
            for _acl in _acls:
                if _acl not in AclShareProp.KEY_WORD:
                    validation.append(self._create_message(self.app._('element.acl_error_unknow_right') % (acl_name, _acl)))
        
        acl_share = AclShareProp().pythonize(acl_share)
        # If user set create or organize or modify or delete it need read
        if acl_share != 'null' and acl_share[4] == '0' and int(acl_share[0:4]) > 0:
            validation.append(self._create_message(self.app._('element.acl_error_missing_read') % (acl_name, AclShareProp().unpythonize(acl_share))))
        
        return validation
    
    
    def _rule_valide_macro_in_command_line(self, item):
        validation = []
        command_line = item.get('command_line', '')
        if command_line:
            item['command_line'] = self._validate_macro_in_command_line(command_line, validation)
        return validation
    
    
    def _rule_valide_macros_in_check_command(self, item):
        validation = []
        
        check_command = item.get('check_command', '')
        
        if isinstance(check_command, basestring):
            check_command = check_command.split('!', 1)
            if len(check_command) > 1 and check_command[1]:
                cmd = check_command[0]
                params = self._validate_macro_in_command_line(check_command[1], validation)
                item['check_command'] = '%s!%s' % (cmd, params)
        else:
            check_command['node']['args'] = self._validate_macro_in_command_line(check_command['node']['args'], validation)
        return validation
    
    
    def _rule_valide_dollar_in_data(self, item):
        validation = []
        dollar_already_checked_for_props = METADATA.get_metadata(item, METADATA.VALIDATE_DOLLAR_NUMBER)
        for prop, value in item.iteritems():
            if prop.startswith('_') and prop != '__SYNC_IDX__':
                item[prop] = self.__validate_dollar_count_in_data_values(prop, validation, value, dollar_already_checked_for_props)
        
        return validation
    
    
    def __validate_dollar_count_in_data_values(self, prop, validation, value, dollar_already_checked_for_props, key=None, arg=None):
        updated_value = value
        nb_dollar = value.count('$')
        dollar_already_checked_for_props.add(prop)
        if nb_dollar == 1:
            updated_value = value.replace('$', '$$')
            if arg and key:
                validation.append(self._create_message(self.app._('validator.escape_dollar_in_dfe') % (prop, key, arg.replace('VALUE', '')), level=Validator._LEVEL_VALIDATOR_WARNING))
            else:
                validation.append(self._create_message(self.app._('validator.escape_dollar_in_data') % prop, level=Validator._LEVEL_VALIDATOR_WARNING))
        
        elif nb_dollar % 2 == 1:
            # There is an odd number of $ in the macro
            
            if arg and key:
                validation.append(self._create_message(self.app._('validator.invalid_number_dollar_in_dfe_value') % (prop, key, arg.replace('VALUE', ''))))
            else:
                if prop not in dollar_already_checked_for_props:
                    validation.append(self._create_message(self.app._('validator.invalide_number_dollar_in_data') % prop))
        return updated_value
    
    
    def _rule_valide_flap_threshold(self, item):
        validation = []
        try:
            low_flap_threshold = int(item.get('low_flap_threshold', ''))
            high_flap_threshold = int(item.get('high_flap_threshold', ''))
        except ValueError:
            return validation
        
        if low_flap_threshold > high_flap_threshold:
            validation.append(self._create_message(self.app._('element.incoherent_flapping_values'), level=Validator._LEVEL_VALIDATOR_ERROR))
        
        return validation
    
    
    def _rule_valide_sla_threshold(self, item, item_type):
        validation = []
        skip_warning_threshold = False
        skip_critical_threshold = False
        sla_warning_threshold_str = item.get('sla_warning_threshold', '').strip()
        sla_critical_threshold_str = item.get('sla_critical_threshold', '').strip()
        
        if sla_warning_threshold_str in STOP_INHERITANCE_VALUES:
            skip_warning_threshold = True
        if sla_critical_threshold_str in STOP_INHERITANCE_VALUES:
            skip_critical_threshold = True
        
        if skip_warning_threshold and skip_critical_threshold:
            return validation
        
        if sla_warning_threshold_str and not skip_warning_threshold and not re.search(RESTRICT_POSITIVE_DECIMAL, sla_warning_threshold_str):
            validation.append(self._create_message(self.app._('element.bad_sla_threshold') % self.app._(item_type + '.sla_warning_threshold'), level=Validator._LEVEL_VALIDATOR_ERROR))
        
        if sla_critical_threshold_str and not skip_critical_threshold and not re.search(RESTRICT_POSITIVE_DECIMAL, sla_critical_threshold_str):
            validation.append(self._create_message(self.app._('element.bad_sla_threshold') % self.app._(item_type + '.sla_critical_threshold'), level=Validator._LEVEL_VALIDATOR_ERROR))
        
        try:
            if sla_warning_threshold_str and not skip_warning_threshold:
                sla_warning_threshold = float(sla_warning_threshold_str)
                if not (0.0 <= sla_warning_threshold <= 100.0):
                    validation.append(self._create_message(self.app._('element.out_of_range_sla_value') % self.app._(item_type + '.sla_warning_threshold'), level=Validator._LEVEL_VALIDATOR_ERROR))
                item['sla_warning_threshold'] = '%0.3f' % sla_warning_threshold
            if sla_critical_threshold_str and not skip_critical_threshold:
                sla_critical_threshold = float(sla_critical_threshold_str)
                if not (0.0 <= sla_critical_threshold <= 100.0):
                    validation.append(self._create_message(self.app._('element.out_of_range_sla_value') % self.app._(item_type + '.sla_critical_threshold'), level=Validator._LEVEL_VALIDATOR_ERROR))
                item['sla_critical_threshold'] = '%0.3f' % sla_critical_threshold
            
            if sla_warning_threshold_str and sla_critical_threshold_str and not (skip_critical_threshold or skip_warning_threshold):
                if sla_critical_threshold > sla_warning_threshold:
                    validation.append(self._create_message(self.app._('element.incoherent_sla_values'), level=Validator._LEVEL_VALIDATOR_ERROR))
        except ValueError:
            return validation
        
        return validation
    
    
    def _rule_valide_acl_in_tab_history(self, item):
        validation = []
        
        item_value = item.get('acl_in_tab_history', None)
        if item_value not in (None, 'history_sla', 'history', 'sla', PROP_DEFAULT_VALUE):
            validation.append(self._create_message(self.app._('validator.invalid_value') % ('acl_in_tab_history', item_value, ','.join(('history_sla', 'history', 'sla', VALUE_FORCE_DEFAULT)))))
        
        return validation
    
    
    def _rule_valide_bool_format(self, item, item_type):
        validation = []
        
        for property_name, item_value in item.iteritems():
            item_property = get_property_def(item_type, property_name)
            if item_property and isinstance(item_property, BoolProp) and item_value not in VALIDE_BOOL_VALUE:
                validation.append(self._create_message(self.app._('validator.invalid_value') % ('property_name', item_value, ','.join(_boolean_states.iterkeys()))))
        
        return validation
    
    
    def _rule_valide_check_interval_gte_1(self, item):
        validation = []
        check_interval_value = item.get('check_interval', None)
        if check_interval_value and check_interval_value in STOP_INHERITANCE_VALUES:
            return validation
        if check_interval_value and int(check_interval_value) < 1:
            item['check_interval'] = '1'
            validation.append(self._create_message(self.app._('element.property_check_interval_should_be_gte_1'), level=Validator._LEVEL_VALIDATOR_WARNING))
        return validation
    
    
    def _validate_macro_in_command_line(self, command_line, validation):
        macros = MacroResolver.get_macros(command_line)
        for macro in macros.iterkeys():
            original_macro = macro
            macro = macro.strip()
            if original_macro != macro:
                validation.append(self._create_message(self.app._('element.macro_name_space_between_dollars') % macro, level=Validator._LEVEL_VALIDATOR_WARNING))
            prev = macro
            macro = macro.upper()
            if prev != macro:
                validation.append(self._create_message(self.app._('element.macro_name_not_in_upper') % macro, level=Validator._LEVEL_VALIDATOR_WARNING))
            if re.search(r'[^a-z0-9\-_]', macro, flags=re.IGNORECASE):
                validation.append(self._create_message(self.app._('element.macro_name_modified') % macro, level=Validator._LEVEL_VALIDATOR_ERROR))
            command_line_with_fixed_macros = command_line.replace(original_macro, macro)
            if command_line != command_line_with_fixed_macros:
                command_line = command_line_with_fixed_macros
        return command_line
    
    
    @staticmethod
    def _rule_valide_no_special_char_in_item_name(item_type, name):
        illegal_chars = ILLEGAL_CHARS
        protected_strings = []
        if item_type in ITEM_TYPE.ALL_SERVICES:
            illegal_chars = ILLEGAL_CHARS_SERVICE
            protected_strings = ['$KEY$']  # ee allow one occurrence of $KEY$ in services
        for token in protected_strings:
            name = name.replace(token, '', 1)
        return not re.search(illegal_chars, name)
    
    
    def _rule_valide_business_impact(self, item, item_type):
        validation = []
        for properties_to_validate in ('business_impact', 'min_business_impact'):
            if properties_to_validate in item and not (item[properties_to_validate] in STOP_INHERITANCE_VALUES or (item[properties_to_validate].isdigit() and 0 <= int(item[properties_to_validate]) and int(item[properties_to_validate]) <= 5)):
                _msg = self._create_message(self.app._('validator.removed') % (item_type, get_name_from_type(item_type, item), item.get('imported_from', self.app._('validator.unknown_source'))), level=Validator._LEVEL_VALIDATOR_WARNING)
                validation.append(_msg)
                del item[properties_to_validate]
        return validation
    
    
    def _rule_valide_external_url(self, item):
        validation = []
        if 'notes_url' in item:
            validation.extend(self._rule_valide_url_syntax('notes_url', item['notes_url']))
        
        if 'notes_multi_url' in item:
            (validations, new_value) = self._rule_multi_url(item['notes_multi_url'])
            item['notes_multi_url'] = new_value
            validation.extend(validations)
        
        return validation
    
    
    def _rule_valide_service_overrides(self, item, item_type):
        validation = []
        
        raw_value = item.get(SERVICE_OVERRIDE)
        if not raw_value:
            return validation
        
        check_type = DEF_ITEMS[item_type]['check_type']
        if isinstance(raw_value, basestring):
            # Check syntax
            parsed_overrides = parse_service_override_property_list_errors(raw_value)
            for error in parsed_overrides['syntax']:
                _msg = self._create_message(self.app._('validator.service_overrides_syntax') % error, level=Validator._LEVEL_VALIDATOR_WARNING)
                validation.append(_msg)
            for error in parsed_overrides['duplicates']:
                # Note for trads : It must have 2 args but error is a tuple so no problems
                _msg = self._create_message(self.app._('validator.service_overrides_duplicate') % error, level=Validator._LEVEL_VALIDATOR_WARNING)
                validation.append(_msg)
            
            parsed_overrides = parsed_overrides['parsed_overrides']
            
            # Check semantics
            for check_name, override in parsed_overrides.items():
                if '$KEY$' in check_name or '$key$' in check_name:
                    _msg = self._create_message(self.app._('validator.service_overrides_invalide_name') % check_name, level=Validator._LEVEL_VALIDATOR_WARNING)
                    validation.append(_msg)
                    del parsed_overrides[check_name]
                    continue
                if SERVICE_OVERRIDE_UUID_SEP in check_name:
                    check_name = check_name.replace(SERVICE_OVERRIDE_UUID_SEP, '')
                
                mock_check = {
                    'service_description': check_name,
                    '_id'                : '0'
                }
                METADATA.update_metadata(mock_check, METADATA.VALIDATE_DOLLAR_NUMBER, set())
                
                invalide_prop = []
                for property_name, property_value in override.iteritems():
                    validation_invalide_prop = self._service_override_validate_properties(check_type, property_name)
                    validation.extend(validation_invalide_prop)
                    if validation_invalide_prop:
                        invalide_prop.append(property_name)
                    else:
                        mock_check[property_name] = property_value
                
                map(override.pop, invalide_prop)
                
                check_command_args = override.get('check_command_args', '')
                if check_command_args:
                    del mock_check['check_command_args']
                    check_command = mock_check.get('check_command', '')
                    if check_command:
                        mock_check['check_command'] = '%s!%s' % (check_command.split('!', 1)[0], check_command_args)
                
                validation.extend(self._service_override_validate_check_with_override(check_type, mock_check))
                
                # If the validators updated a property, we need to report the updates in the real object
                del mock_check['service_description']
                del mock_check['_id']
                del mock_check['@metadata']
                override.clear()
                override.update(mock_check)
                if check_command_args:
                    override['check_command_args'] = check_command_args
            # Make sure the string value is standardized to prevent unwanted differences
            item[SERVICE_OVERRIDE] = unparse_service_override(parsed_overrides)
        elif isinstance(raw_value, dict):
            if raw_value.get('has_error'):
                _msg = self._create_message(self.app._('validator.service_overrides_syntax') % raw_value.get('errors', ''), level=Validator._LEVEL_VALIDATOR_WARNING)
                validation.append(_msg)
            
            mock_checks = {}
            for link in raw_value['links']:
                key = link['key']
                value = link['value']
                check_link = link['check_link']
                if check_link['exists']:
                    check_id = check_link.get('_id', None)
                    original_check = self.datamanagerV2.find_item_by_id(check_id, item_state=ITEM_STATE.STAGGING, item_type=check_link['item_type']).get_raw_item()
                    check_name = get_name_from_type(check_link['item_type'], original_check)
                else:
                    check_name = check_id = check_link.get('name')
                    original_check = None
                
                check_id += '!key_dfe!' + link.get('dfe_key', '')
                if check_id not in mock_checks:
                    if original_check:
                        mock_checks[check_id] = original_check
                    else:
                        if SERVICE_OVERRIDE_UUID_SEP in check_name:
                            check_name = check_name.replace(SERVICE_OVERRIDE_UUID_SEP, '')
                        mock_checks[check_id] = {
                            '_id'                : check_id,
                            'service_description': check_name
                        }
                    mock_checks[check_id]['check_link_'] = check_link
                    mock_checks[check_id]['key_dfe'] = link.get('dfe_key', '')
                mock_checks[check_id][key] = value
            
            for mock_check in mock_checks.itervalues():
                METADATA.update_metadata(mock_check, METADATA.VALIDATE_DOLLAR_NUMBER, set())
                validation.extend(self._service_override_validate_check_with_override(check_type, mock_check))
                
                # If the validators updated a property, we need to report the updates in the real object
                for link in raw_value['links']:
                    if link['check_link'] == mock_check['check_link_'] and mock_check['key_dfe'] == link.get('dfe_key', ''):
                        for key, value in mock_check.iteritems():
                            if link['key'] == key:
                                link['value'] = value
        
        if not item[SERVICE_OVERRIDE]:
            del item[SERVICE_OVERRIDE]
        
        return validation
    
    
    def _validate_service_override(self, item, item_type):
        validation = []
        
        for rule in self._COMMON_RULES:
            if rule == self._rule_valide_SE_UUID:
                continue
            if rule == self._rule_valide_item_name:
                continue
            validation.extend(rule(item, item_type))
        
        return validation
    
    
    def _service_override_validate_check_with_override(self, check_type, mock_check):
        validation = []
        type_validator = self._TYPE_VALIDATOR.get(check_type, None)
        check_validations = self._validate_service_override(mock_check, check_type)
        check_validations.extend(type_validator(mock_check, 'override_%s' % check_type))
        for check_validation in check_validations:
            validation.append(self._create_message(self.app._("validator.service_overrides_check_validation") % (mock_check['service_description'], check_validation['message']), check_validation['level']))
        
        return validation
    
    
    def _service_override_validate_properties(self, check_type, property_name):
        validation = []
        if property_name in self.OVERRIDE_FORBIDDEN_PROPERTIES:
            _msg = self._create_message(self.app._('validator.service_overrides_forbidden') % property_name, level=Validator._LEVEL_VALIDATOR_WARNING)
            validation.append(_msg)
        
        return validation
    
    
    def _rule_no_duplicate_service_overrides(self, item, item_type):
        validation = []
        
        if not hasattr(self.datamanagerV2.data_provider, 'items_linked_with_name'):
            logger.warning('The rule _rule_no_duplicate_service_overrides should not call here.')
            return validation
        
        check_name = get_name_from_type(item_type, item)
        no_exist_links = self.datamanagerV2.data_provider.items_linked_with_name.find_one({'name': check_name, 'type': item_type})
        
        if not no_exist_links:
            return validation
        
        for reverse_link in no_exist_links['reverse_links']:
            if reverse_link.key == SERVICE_OVERRIDE:
                item_with_link_to_update = self.datamanagerV2.find_item_by_id(reverse_link.item_id, reverse_link.item_type, reverse_link.item_state)
                if not item_with_link_to_update:
                    continue
                host_name = item_with_link_to_update.get_name()
                host_type = item_with_link_to_update.get_type()
                host_state = item_with_link_to_update.get_state()
                
                item_with_link_to_update = item_with_link_to_update.get_raw_item(flatten_links=False)
                count_next_so_on_me = {}
                for so_link in item_with_link_to_update.get(SERVICE_OVERRIDE, {}).get('links', []):
                    if so_link['check_link'].get('_id', None) == item['_id'] or check_name == so_link['check_link'].get('name', None):
                        count_next_so_on_me[so_link['key']] = count_next_so_on_me.get(so_link['key'], 0) + 1
                
                for _key, _count in count_next_so_on_me.iteritems():
                    if _count > 1:
                        new_check_name = ''
                        for so_link in item_with_link_to_update.get(SERVICE_OVERRIDE, {}).get('links', []):
                            if so_link['key'] == _key and check_name == so_link['check_link'].get('name', None):
                                new_check_name = 'override-conflict-%s' % so_link['check_link'].get('name', None)
                                so_link['check_link']['name'] = new_check_name
                        
                        user = self._get_auth_user()
                        self.datamanagerV2.save_item(item_with_link_to_update, user, host_type, host_state, HISTORY_ACTION.AUTO_MODIFICATION)
                        
                        _msg = self._create_message(self.app._('validator.no_duplicate_service_overrides') % {'key': _key, 'host_name': host_name, 'new_check_name': new_check_name}, level=Validator._LEVEL_VALIDATOR_WARNING)
                        validation.append(_msg)
        
        return validation
    
    
    def _rule_valide_periods(self, item):
        validation = []
        original_item = item
        item = item.copy()  # Because we need to regenerate the item with keys that the user can understand in validation messages
        
        if ADVANCED_PERIODS in item:
            for period in item[ADVANCED_PERIODS].split(TimePeriodParser.SEPARATOR_ADVANCED_DEF):
                if not period:
                    continue
                if TimePeriodParser.SEPARATOR_ADVANCED_KEY_VALUE not in period or len(period.split(TimePeriodParser.SEPARATOR_ADVANCED_KEY_VALUE)) != 2:
                    validation.append(self._create_message(self.app._('validator.timeperiod_empty_range') % period.replace(MARKER_FOR_KEY_FROM_ADVANCED_PERIODS, '')))
                    continue
                key, intervals = period.split(TimePeriodParser.SEPARATOR_ADVANCED_KEY_VALUE)
                item["%s%s" % (MARKER_FOR_KEY_FROM_ADVANCED_PERIODS, key)] = intervals
            item.pop(ADVANCED_PERIODS)
        
        for period_name, period_value in item.iteritems():
            if period_name.startswith('_') or period_name in TIMEPERIOD_BASIC_PROPS:
                continue
            
            raw_period_name = period_name.replace(MARKER_FOR_KEY_FROM_ADVANCED_PERIODS, '')
            periods_list = period_value.split(TimePeriodParser.SEPARATOR_DAY_DEF)
            intervals_list = []
            for period in periods_list:
                if not period:
                    continue
                
                actual_periods = period.split(TimePeriodParser.SEPARATOR_DAY_OPT)[0]
                intervals_list.append(actual_periods)
            
            for period in intervals_list:
                if ' ' in period:
                    validation.append(self._create_message(self.app._('validator.timeperiod_removed_spaces_in_ranges') % (raw_period_name, period), Validator._LEVEL_VALIDATOR_WARNING))
                    replaced_period = period.replace(' ', '')
                    if period_name.startswith(MARKER_FOR_KEY_FROM_ADVANCED_PERIODS):
                        original_item[ADVANCED_PERIODS] = original_item[ADVANCED_PERIODS].replace(period, replaced_period)
                    else:
                        original_item[period_name] = item[period_name].replace(period, replaced_period)
                
                if not TimePeriodParser.TIME_RANGES_REGEX_EXACT_MATCH.match(period):
                    validation.append(self._create_message(self.app._('validator.timeperiod_wrong_range') % (raw_period_name, period)))
                    continue
                
                intervals = period.split(',')
                self._validate_time_ranges(intervals, raw_period_name, validation)
            
            # Instantiate a Timeperiod to check if it is valid for the Arbiter ; enables Advanced timeperiods validation
            tp = Timeperiod({raw_period_name: period_value})
            tp.explode(None)
            if not tp.is_correct():
                validation.append(self._create_message(self.app._('validator.timeperiod_invalid_entry') % ("%s %s" % (raw_period_name, period_value))))
            if tp.invalid_entries:
                validation.append(self._create_message(self.app._('validator.timeperiod_ignored_invalid_entry') % ("%s %s" % (raw_period_name, period_value)), Validator._LEVEL_VALIDATOR_WARNING))
        
        return validation
    
    
    def _validate_time_ranges(self, intervals, period_name, validation):
        start_time = 0
        for interval in intervals:
            interval = interval.strip()
            
            if interval == "":
                validation.append(self._create_message(self.app._('validator.timeperiod_empty_range') % period_name))
                continue
            
            hours = interval.split('-')
            if len(hours) != 2:
                validation.append(self._create_message(self.app._('validator.timeperiod_wrong_range') % (period_name, interval)))
                continue
            
            has_wrong_range = False
            for hour in hours:
                if ":" not in hour:
                    
                    if not has_wrong_range:
                        validation.append(self._create_message(self.app._('validator.timeperiod_wrong_range') % (period_name, interval)))
                    has_wrong_range = True
                    continue
                
                hh, mm = hour.split(':')
                hh = int(hh)
                mm = int(mm)
                time = hh * 60 + mm
                if not (0 <= hh <= 24 and 0 <= mm <= 59) or time > 1440:
                    validation.append(self._create_message(self.app._('validator.timeperiod_time_out_of_range') % (period_name, interval, hour)))
                    continue
                
                if time < start_time:
                    validation.append(self._create_message(self.app._('validator.timeperiod_interval_wrong_order') % period_name))
                    break
                else:
                    start_time = time
    
    
    def _rule_valide_timeout_property(self, item):
        validation = []
        timeout = item.get('timeout', -1)
        if timeout and timeout in STOP_INHERITANCE_VALUES:
            return validation
        timeout = int(timeout)
        if timeout != -1 and timeout < 1:
            _msg = self._create_message(self.app._('validator.timeout_not_in_range'), level=Validator._LEVEL_VALIDATOR_ERROR)
            validation.append(_msg)
        
        return validation
    
    
    def _rule_valide_check_running_timeout_property(self, item):
        validation = []
        check_running_timeout = item.get('check_running_timeout', -1)
        if check_running_timeout in STOP_INHERITANCE_VALUES:
            return validation
        check_running_timeout = int(check_running_timeout)
        if check_running_timeout != -1 and check_running_timeout < 1:
            _msg = self._create_message(self.app._('validator.check_running_timeout_not_in_range'), level=Validator._LEVEL_VALIDATOR_ERROR)
            validation.append(_msg)
        
        return validation
    
    
    def _rule_multi_url(self, multi_url_value):
        validation = []
        if multi_url_value == PROP_DEFAULT_VALUE or multi_url_value == VALUE_FORCE_DEFAULT:
            return validation, multi_url_value
        
        list_of_urls = multi_url_value.split('~#~')
        updated_urls = []
        for url_spec in list_of_urls:
            url_components = url_spec.split('~=')
            if len(url_components) == 4:
                (url_name, url_value, url_tag, url_display) = url_components
            elif len(url_components) == 1:  # Simple URL
                url_name = ''
                url_value = url_spec
                url_tag = "tag"
                url_display = "OPEN_IN_NEW_TAB"
                _msg = self._create_message(self.app._('validator.notes_multi_url_upgrade') % url_spec, level=Validator._LEVEL_VALIDATOR_WARNING)
                validation.append(_msg)
            elif len(url_components) == 2:  # name~=URL
                (url_name, url_value) = url_components
                url_tag = "tag"
                url_display = "OPEN_IN_NEW_TAB"
                _msg = self._create_message(self.app._('validator.notes_multi_url_upgrade') % url_spec, level=Validator._LEVEL_VALIDATOR_WARNING)
                validation.append(_msg)
            else:
                _msg = self._create_message(self.app._('validator.notes_multi_url_syntaxe_error') % url_spec, level=Validator._LEVEL_VALIDATOR_ERROR)
                validation.append(_msg)
                continue
            
            if re.search(r'[<>&"\'/]', url_name):
                _msg = self._create_message(self.app._('validator.notes_multi_url_name_forbidden_chars'), level=Validator._LEVEL_VALIDATOR_ERROR)
                validation.append(_msg)
            
            url_display = url_display.upper()
            if url_display not in PopUpUrlSelect.LIST:
                _msg = self._create_message(self.app._('validator.notes_multi_url_non_existing_popup') % url_display, level=Validator._LEVEL_VALIDATOR_ERROR)
                validation.append(_msg)
            
            validation.extend(self._rule_valide_url_syntax('notes_multi_url', url_value))
            updated_urls.append("%s~=%s~=%s~=%s" % (url_name, url_value, url_tag, url_display))
        
        return validation, "~#~".join(updated_urls)
    
    
    def _rule_valide_url_syntax(self, prop_name, url):
        validation = []
        if url in STOP_INHERITANCE_VALUES:
            return validation
        parts = urlparse(url)
        if RESTRICT_ALLOWED_CHARS_URL.search(url):
            _msg = self._create_message(self.app._('validator.url_forbidden_chars') % (prop_name, url), level=Validator._LEVEL_VALIDATOR_ERROR)
            validation.append(_msg)
        if not RESTRICT_EXTERNAL_URL.match(url) or not parts.netloc:
            _msg = self._create_message(self.app._('validator.url_no_netloc') % (prop_name, url), level=Validator._LEVEL_VALIDATOR_WARNING)
            validation.append(_msg)
        if parts.scheme not in ('http', 'https'):
            _msg = self._create_message(self.app._('validator.url_wrong_protocol') % (prop_name, url), level=Validator._LEVEL_VALIDATOR_WARNING)
            validation.append(_msg)
        return validation
    
    
    def _rule_valide_warning_threshold_cpu_usage_property(self, item):
        validation = []
        warning_threshold_cpu_usage = item.get('warning_threshold_cpu_usage', 1)
        if warning_threshold_cpu_usage and warning_threshold_cpu_usage in STOP_INHERITANCE_VALUES:
            return validation
        if int(warning_threshold_cpu_usage) < 1:
            _msg = self._create_message(self.app._('validator.warning_threshold_cpu_usage_not_in_range'), level=Validator._LEVEL_VALIDATOR_ERROR)
            validation.append(_msg)
        
        return validation
    
    
    def _rule_valide_notification_interval_property(self, item):
        validation = []
        notification_interval = item.get('notification_interval', 0)
        if notification_interval and notification_interval in STOP_INHERITANCE_VALUES:
            return validation
        if int(notification_interval) < 0:
            _msg = self._create_message(self.app._('validator.notification_interval_not_in_range'), level=Validator._LEVEL_VALIDATOR_ERROR)
            validation.append(_msg)
        
        return validation
    
    
    def _rule_valide_notification_interval_property_on_escalations(self, item):
        validation = []
        notification_interval = item.get('notification_interval', 0)
        if notification_interval and notification_interval in STOP_INHERITANCE_VALUES:
            return validation
        if int(notification_interval) < -1:
            _msg = self._create_message(self.app._('validator.notification_interval_escalations_not_in_range'), level=Validator._LEVEL_VALIDATOR_ERROR)
            validation.append(_msg)
        
        return validation
    
    
    def _rule_valide_complex_expression(self, item, item_type):
        validation = []
        string_to_validate = item.get('host_name')
        if not string_to_validate:
            return validation
        
        if isinstance(string_to_validate, dict):
            # Already parsed, so no syntax error
            return validation
        
        try:
            parse_complex_exp(string_to_validate)
        except SyntaxError as errors:
            # property_trad = self.app._("%s.host_name" % item_type)
            for error_key in str(errors).split(','):
                _msg = self._create_message("%s, %s" % (
                    self.app._("validator.in_property") % "host_name",
                    self.app._(error_key)), Validator._LEVEL_VALIDATOR_ERROR)
                validation.append(_msg)
        
        return validation
    
    
    def _validate_service_excludes(self, item):
        validation = []
        string_to_validate = item.get('service_excludes')
        if string_to_validate and ILLEGAL_CHARS_SERVICE_EXCLUDES.search(string_to_validate):
            _msg = self._create_message(self.app._('validator.invalid_characters') % (ILLEGAL_CHARS_SERVICE_EXCLUDES.pattern, SERVICE_EXCLUDES_BY_NAME), level=Validator._LEVEL_VALIDATOR_ERROR)
            validation.append(_msg)
        
        return validation
    
    
    def _get_auth_user(self):
        user = self.app.get_user_auth(warn_no_bottle_thread=False)
        if not user:
            from ...dao.transactions import transactions
            user = transactions.get_transaction_object().user
        
        return user
    
    
    def _validate_bp_rule(self, item):
        validation = []
        if 'bp_rule' not in item:
            return validation
        
        bp_rule = item['bp_rule']
        
        if isinstance(bp_rule, basestring):
            parsed_bp_rule = BpRuleParser.compute_internal_bp_rule(bp_rule)
        else:
            parsed_bp_rule = bp_rule['node']
            bp_rule = BpRuleParser.get_unparse_bp_rule(parsed_bp_rule)
        
        for node in parsed_bp_rule:
            if node.get('has_error', False):
                token = node['token']
                substring_with_error = escape_XSS(bp_rule[token.start:token.end])
                annotated_bp_rule = bp_rule[:token.end] + '</span>' + escape_XSS(bp_rule[token.end:])
                annotated_bp_rule = escape_XSS(annotated_bp_rule[:token.start]) + '<span class="shinken-format-error">' + annotated_bp_rule[token.start:]
                _msg = self._create_message(self.app._('validator.invalid_bprule') % (substring_with_error, annotated_bp_rule))
                validation.append(_msg)
        return validation
    
    
    def _rule_remove_invisible_characters_in_data(self, item, item_type):
        # The invisible character \u200b is forbidden in all data
        # SEE : SEF-6015
        validation = []
        
        impacted_data = []
        impacted_properties = []
        
        for prop_name, value in ((key, value) for key, value in item.iteritems() if isinstance(value, basestring) and key not in NOT_TO_LOOK):
            
            new_value = value.replace(u'\u200b', '')
            if new_value != value:
                item[prop_name] = new_value
                if prop_name.startswith('_'):
                    impacted_data.append(prop_name)
                else:
                    impacted_properties.append(prop_name)
        
        if impacted_data:
            _msg = self._create_message(self.app._('validator.invisible_characters_in_data') % ', '.join(impacted_data), level=Validator._LEVEL_VALIDATOR_WARNING)
            validation.append(_msg)
        
        if impacted_properties:
            _msg = self._create_message(self.app._('validator.invisible_characters_in_property') % ', '.join(impacted_properties), level=Validator._LEVEL_VALIDATOR_WARNING)
            validation.append(_msg)
        
        return validation
    
    
    def _rule_no_loop_in_template(self, item, item_type):
        validation = []
        
        raw_use = item.get('use', '')
        enabled = item.get('enabled', '1')
        if not raw_use or enabled != '1':
            return []
        if isinstance(raw_use, basestring):
            raw_use = split_and_strip_list(raw_use)
            _templates_to_checks = []
            for template_name in raw_use:
                t = self.datamanagerV2.find_item_by_name(template_name, ITEM_TYPE.get_template_type(item_type), ITEM_STATE.STAGGING)
                if not t:
                    t = BaseItem({'exists': False})
                    METADATA.update_metadata(t, METADATA.NAME, template_name)
                _templates_to_checks.append(t)
        
        else:
            _templates_to_checks = read_frontend_links('use', raw_use, item_type, self.datamanagerV2)
        
        
        _templates = []
        if ITEM_TYPE.is_template(item_type):
            _name = get_name_from_type(item_type, item)
            _templates.append(_name)
        
        while _templates_to_checks:
            _template_to_check = _templates_to_checks.pop()
            if not _template_to_check:
                continue
            validation = self._check_loop_in_use(_template_to_check, _templates, validation, item, item_type)
        
        return validation
    
    
    def _check_loop_in_use(self, item, templates_use, validation, original_item, item_type, max_level=32):
        if max_level <= 0:
            return validation
        
        if not item.is_enabled():
            return validation
        
        for _template in item.get_all_state_link_items('use'):
            _template_name = get_name_from_type(item_type, original_item) if _template.get('_id') == original_item['_id'] else _template.get_name()
            my_direct_fathers = templates_use[:]
            my_direct_fathers.append(item.get_name())
            if _template_name in templates_use:
                my_direct_fathers.append(_template_name)
                validation.append(self._create_message(self.app._('validator.loop_in_template').format(' >>> '.join(my_direct_fathers)), level=Validator._LEVEL_VALIDATOR_ERROR))
                return validation
            validation = self._check_loop_in_use(_template, my_direct_fathers, validation, original_item, item_type, max_level=max_level - 1)
        return validation
