#!/usr/bin/python
# -*- coding: utf-8 -*-

# Copyright (C) 2013-2020:
# This file is part of Shinken Enterprise, all rights reserved.

from shinken.log import PartLogger
from shinken.misc.type_hint import Dict
from shinkensolutions.api.synchronizer import SourceTranslatePart, ITEM_TYPE
from shinkensolutions.api.synchronizer.source.source_exception import SourceException
from shinkensolutions.api.synchronizer.source.file_loader import FileLoader, LANGUAGES
from shinkensolutions.api.synchronizer.source.validation_state import ValidationState
from shinkensolutions.api.synchronizer.source.source_configuration_value import ServiceMode, SERVICE_MODE


# see SEF-6034
class OriginItemDescription(object):
    def __init__(self, logger, translator, file_loader, lang, configuration_state):
        # type: (PartLogger, SourceTranslatePart, FileLoader, str, ServiceMode) ->None
        self.logger = logger
        self._translator = translator
        self.file_loader = file_loader
        self.lang = lang
        # In case of missing translation in our lang we load 'en' translation. But if we are in 'en' we load 'fr'
        self.other_lang = 'fr' if self.lang == 'en' else 'en'
        self.file_exist = False
        self.configuration_state = configuration_state
        self.user_file = file_loader.properties_description[self.lang].user_file
        self.user_other_lang_file = file_loader.properties_description[self.other_lang].user_file
        self.origin_item_properties_description = {}
        self.validation_state = ValidationState(self.logger, self._translator)  # type: ValidationState
        self.reload()
    
    
    def add_error(self, message=''):
        # type: (str) -> None
        self.validation_state.add_error(message=message)
    
    
    def add_warning(self, message=''):
        # type: (str) -> None
        self.validation_state.add_warning(message=message)
    
    
    def reload(self):
        # type: () -> None
        self.logger.info('reload origin item description')
        self.origin_item_properties_description, self.validation_state = self._read_origin_item_description()
        self.validate()


    # This method can be override to validate item properties description after load
    def validate(self):
        # type: () -> None
        pass
    
    
    def _read_origin_item_description(self):
        # type: () -> (Dict, ValidationState)
        validation_state = ValidationState(self.logger, self._translator)
        origin_item_properties_description = {}
        if self.configuration_state == SERVICE_MODE.OFF:
            return origin_item_properties_description, validation_state
        
        if self.lang not in LANGUAGES:
            raise SourceException('lang %s is unknown. known lang are : %s' % (self.lang, LANGUAGES))
        
        default_file = self.file_loader.properties_description[self.lang].default_file
        if not default_file.exist():
            validation_state.add_warning(translate_key='warning.missing_description_file', params=(default_file.path,))
            return origin_item_properties_description, validation_state
        
        try:
            origin_item_properties_description = default_file.load()
        except Exception as e:
            validation_state.add_error(translate_key='error.fail_to_load_file', params=(default_file.path, str(e)))
            return origin_item_properties_description, validation_state
        
        if not isinstance(origin_item_properties_description, dict):
            validation_state.add_error(translate_key='error.fail_to_load_file', params=(default_file.path, ''))
            return origin_item_properties_description, validation_state
        
        have_one_existing_user_file = next((True for lang_file in self.file_loader.properties_description.itervalues() if lang_file.user_file.exist()), False)
        if self.configuration_state == SERVICE_MODE.ON and not self.user_file.exist():
            validation_state.add_warning(translate_key='warning.missing_user_file', params=(self.user_file.path,))
        if self.configuration_state == SERVICE_MODE.ON and validation_state.is_valid() and have_one_existing_user_file:
            user_overload = {}
            other_lang_user_overload = {}
            try:
                user_overload = self.user_file.load()
            except Exception as e:
                validation_state.add_error(translate_key='error.fail_to_load_file', params=(self.user_file.path, str(e)))
            
            try:
                other_lang_user_overload = self.user_other_lang_file.load()
            except Exception as e:
                validation_state.add_error(translate_key='error.fail_to_load_file', params=(self.user_other_lang_file.path, str(e)))
            
            new_origin_item_properties_description = origin_item_properties_description
            for item_type in ITEM_TYPE.ALL_TYPES:
                if item_type not in new_origin_item_properties_description:
                    new_origin_item_properties_description[item_type] = {}
                
                _properties_description = new_origin_item_properties_description[item_type]
                _user_properties_description = user_overload.get(item_type, {})
                _user_other_lang_properties_description = other_lang_user_overload.get(item_type, {})
                
                _user_properties_overload = set(_user_properties_description.keys() + _user_other_lang_properties_description.keys())
                for _api_property_name in _user_properties_overload:
                    _user_property_description = _user_properties_description.get(_api_property_name, '')
                    _user_other_lang_property_description = _user_other_lang_properties_description.get(_api_property_name, '')
                    
                    if _user_other_lang_property_description and not _user_property_description:
                        validation_state.add_warning(translate_key='warning.missing_translation', params=(_api_property_name, self.user_file.path))
                        _properties_description[_api_property_name] = _user_other_lang_property_description
                    elif _user_property_description:
                        _properties_description[_api_property_name] = _user_property_description
            
            origin_item_properties_description = new_origin_item_properties_description
        self.file_exist = True
        
        return origin_item_properties_description, validation_state
