#!/usr/bin/python
# -*- coding: utf-8 -*-

# Copyright (C) 2013-2018:
# This file is part of Shinken Enterprise, all rights reserved.


import datetime
import imp
import json
import os
import sys
import threading
import time
import traceback

from shinken.basemodule import BaseModule
from shinken.daemon import Daemon
from shinken.log import logger
from shinken.message import Message
from shinken.misc.regenerator import Regenerator
from shinken.webui.bottlewebui import static_file, view, route, request, response, template, abort, HTTPError
from shinken.webui.cherrypybackend import CherryPyServerHTTP
from shinken.load import AvgForFixSizeCall
from shinken.misc.datamanager import datamgr
import shinken.webui.bottlewebui as bottle

ONE_HOUR = 3600

bottle.debug(True)

# Import bottle lib to make bottle happy
bottle_dir = os.path.abspath(os.path.dirname(bottle.__file__))
sys.path.insert(0, bottle_dir)

# Look at the webui module root dir too
livedatamod_dir = os.path.abspath(os.path.dirname(__file__))
htdocs_dir = os.path.join(livedatamod_dir, 'htdocs')

properties = {
    'daemons' : ['broker'],
    'type'    : 'broker_module_livedata',
    'phases'  : ['running'],
    'external': True,
}


# called by the plugin manager to get an instance
def get_instance(plugin):
    bottle.TEMPLATE_PATH.append(os.path.join(livedatamod_dir, 'views'))
    bottle.TEMPLATE_PATH.append(livedatamod_dir)
    logger.debug("Get a BrokerModuleLiveData instance for plugin %s" % plugin.get_name())
    
    instance = broker_module_livedata(plugin)
    return instance


class EnableCors(object):
    name = 'enable_cors'
    api = 2
    
    
    @staticmethod
    def apply(fn, context):
        def _enable_cors(*args, **kwargs):
            # set CORS headers
            response.headers['Access-Control-Allow-Origin'] = 'http://127.0.0.1:9000'
            response.headers['Access-Control-Allow-Methods'] = 'GET, POST, PUT, OPTIONS, DELETE, PATCH'
            response.headers['Access-Control-Allow-Headers'] = 'Origin, Accept, Content-Type, X-Requested-With, X-CSRF-Token, X-Shinken-Token'
            response.headers['Access-Control-Allow-Credentials'] = 'true'
            if bottle.request.method != 'OPTIONS':
                # actual request; reply with the actual response
                return fn(*args, **kwargs)
        
        
        return _enable_cors


class broker_module_livedata(BaseModule, Daemon):
    def __init__(self, modconf):
        BaseModule.__init__(self, modconf)
        
        self.http_start_time = int(time.time())
        
        self.plugins = []
        
        self.serveropts = {}
        umask = getattr(modconf, 'umask', None)
        if umask != None:
            self.serveropts['umask'] = int(umask)
        bind_address = getattr(modconf, 'bindAddress', None)
        if bind_address:
            self.serveropts['bindAddress'] = str(bind_address)
        self.serveropts['numthreads'] = 64
        
        self.lang = getattr(modconf, 'lang', 'en')
        self.port = int(getattr(modconf, 'port', '51000'))
        self.token = str(getattr(modconf, 'token', ''))
        self.host = getattr(modconf, 'host', '0.0.0.0')
        
        self.http_backend = getattr(modconf, 'http_backend', 'auto')
        
        self.use_ssl = getattr(modconf, 'use_ssl', '0') == '1'
        self.ssl_key = getattr(modconf, 'ssl_key', '')
        self.ssl_cert = getattr(modconf, 'ssl_cert', '')
        
        self.average_time_response = AvgForFixSizeCall(time_limit=ONE_HOUR)
        self.nb_errors_by_hour = []
        self.errors = []
        self.errors_per_ip = {}
        self.worst_response_time = {'time': 0, 'response_time': 0.0}
        self.rg = Regenerator(modconf)
        
        self.bottle = bottle
        
        self.hook_error()
    
    
    def get_module_info(self):
        module_info = {}
        tmp = BaseModule.get_submodule_states(self)
        if tmp:
            module_info.update(tmp)
        module_info['https'] = self.use_ssl
        if self.token == "change_me":
            module_info['token'] = 1
        elif self.token == "":
            module_info['token'] = 2
        else:
            module_info['token'] = 0
        
        module_info['port'] = self.port
        time_now = time.time()
        module_info['average_response_time'], request_per_hour = self.average_time_response.get_avg(avg_on_time=False, with_range_size=True)
        if time_now - self.worst_response_time['time'] <= ONE_HOUR and self.worst_response_time['time'] != 0:
            module_info['worst_response_time'] = self.worst_response_time
        module_info['request_per_hour'] = 0

        tmp = self.average_time_response.max_call_in_period()
        
        if tmp['nb'] == 0:
            module_info['best_in_one_second'] = tmp
        else:
            module_info['best_in_one_second'] = {'time': datetime.datetime.fromtimestamp(tmp['time']).strftime('%H:%M:%S'), 'nb': tmp['nb']}
        
        module_info['error_per_hour'] = 0
        if len(self.nb_errors_by_hour) >= 1:
            while self.nb_errors_by_hour and time_now - self.nb_errors_by_hour[0] >= ONE_HOUR:
                self.nb_errors_by_hour.pop(0)
            if len(self.nb_errors_by_hour) >= 1:
                module_info['error_per_hour'] = len(self.nb_errors_by_hour)
        
        for error in self.errors:
            if time_now - error['time'] >= ONE_HOUR:
                del error
            else:
                break
        
        module_info['errors'] = self.errors
        module_info['request_per_hour'] = int(request_per_hour) + len(self.errors)
        return module_info
    
    
    def error_handler(self, error):
        return_value = error.output
        if error.traceback:
            first_line = True
            for line_stack in error.traceback.splitlines():
                if first_line:
                    logger.error("ERROR stack : %s" % line_stack)
                    first_line = False
                else:
                    logger.error("%s" % line_stack)
        if response and response.content_type == 'application/json':
            return_value = json.dumps(error.output)
        return return_value
    
    
    def hook_error(self):
        
        @bottle.error(500)
        def custom500(error):
            return self.error_handler(error)
        
        
        @bottle.error(400)
        def custom500(error):
            return self.error_handler(error)
        
        
        @bottle.error(404)
        def custom404(error):
            return self.error_handler(error)
        
        
        @bottle.error(401)
        def custom401(error):
            return self.error_handler(error)
        
        
        @bottle.error(403)
        def custom403(error):
            return self.error_handler(error)
        
        
        @bottle.error(409)
        def custom409(error):
            return self.error_handler(error)
    
    
    def init(self):
        self.rg.load_external_queue(self.from_module_to_main_daemon_queue)
    
    
    def want_brok(self, b):
        return self.rg.want_brok(b)
    
    
    def main(self):
        logger.set_name(self.name)
        
        self.datamgr = datamgr
        datamgr.load(self.rg)
        
        self.request = request
        self.abort = self._abort
        self.response = response
        self.template_call = template
        
        # the regenerator is ready in the valid process, we can start it's freshness thread
        self.rg.launch_freshness_thread(lang=self.lang)
        try:
            self.do_main()
        except Exception, exp:
            msg = Message(id=0, type='ICrash', data={'name': self.get_name(), 'exception': exp, 'trace': traceback.format_exc()})
            self.from_module_to_main_daemon_queue.put(msg)
            # wait 2 sec so we know that the broker got our message, and die
            time.sleep(2)
            raise
    
    
    @staticmethod
    def _abort(code=500, text='Unknown Error: Application stopped.', warning=False, to_logger=True):
        if to_logger is True:
            logger.error('A %s HTTP error occured : %s' % (code, text))
        http_error = HTTPError(code, text)
        http_error.warning = warning
        raise http_error
    
    
    def wrap_json(self, f):
        def __wrap(*args, **kwargs):
            self.response.content_type = 'application/json'
            return f(*args, **kwargs)
        
        
        return __wrap
    
    
    # Real main function
    def do_main(self):
        
        # I register my exit function
        self.set_exit_handler()
        
        # We will protect the operations on the non read+write with a lock and 2 int
        self.global_lock = threading.RLock()
        self.nb_readers = 0
        self.nb_writers = 0
        
        self.data_thread = None
        
        core_plugin_dir = os.path.join(os.path.abspath(os.path.dirname(__file__)), 'plugins')
        
        # Step - Then look at the plugins in toe core and load all we can there
        self.load_plugins(core_plugin_dir)
        
        # Step - Launch the data thread
        self.data_thread = threading.Thread(None, self.manage_brok_thread, 'datathread')
        self.data_thread.start()
        
        # TODO: look for alive and killing
        logger.debug("starting web server")
        self.srv = bottle.run(host=self.host, port=self.port, server=CherryPyServerHTTP, use_ssl=self.use_ssl, ssl_key=self.ssl_key, ssl_cert=self.ssl_cert, **self.serveropts)
        self.srv.start()
    
    
    # It's the thread function that will get broks and update data.
    # Will lock the whole thing while updating
    def manage_brok_thread(self):
        
        while True:
            broks = self.to_q.get()
            t0 = time.time()
            for brok in broks:
                brok.prepare()
                
                # For updating, we cannot do it while answer queries, so wait for no readers
                self.wait_for_no_readers()
                try:
                    self.rg.manage_brok(brok)
                except Exception, exp:
                    msg = Message(id=0, type='ICrash', data={'name': self.get_name(), 'exception': exp, 'trace': traceback.format_exc()})
                    self.from_module_to_main_daemon_queue.put(msg)
                    # wait 2 sec so we know that the broker got our message, and die
                    time.sleep(2)
                    # No need to raise here, we are in a thread, exit!
                    os._exit(2)
                finally:
                    # We can remove us as a writer from now. It's NOT an atomic operation
                    # so we REALLY not need a lock here (yes, I try without and I got
                    # a not so accurate value there....)
                    self.global_lock.acquire()
                    self.nb_writers -= 1
                    self.global_lock.release()
            
            logger.log_perf(t0, '[Broks]', 'Managed %d broks' % len(broks))
    
    
    def load_plugin(self, fdir, plugin_dir):
        logger.debug('Loading %s from %s' % (fdir, plugin_dir))
        try:
            # Put the full qualified path of the module we want to load
            # for example we will give  webui/plugins/eltdetail/
            mod_path = os.path.join(plugin_dir, fdir)
            # Then we load the eltdetail.py inside this directory
            m = imp.load_module('%s' % (fdir), *imp.find_module(fdir, [mod_path]))
            m_dir = os.path.abspath(os.path.dirname(m.__file__))
            sys.path.append(m_dir)
            
            logger.debug('module %s is loaded from %s' % (m_dir, str(m)))
            pages = m.pages
            do_static = False
            for (f, entry) in pages.items():
                routes = entry.get('routes', None)
                v = entry.get('view', None)
                wrappers = entry.get('wrappers', ['auth'])
                
                f_name = f.__name__
                f.display_name = f_name
                for wrapper in wrappers:
                    wrap = {'json': self.wrap_json}.get(wrapper, None)
                    if wrap:
                        f_name = f.display_name
                        f = wrap(f)
                        f.display_name = f_name + '-w(' + wrapper + ')'
                
                # IMPORTANT: apply VIEW BEFORE route!
                if v:
                    f_name = f.display_name
                    f = view(v)(f)
                    f.display_name = f_name + '-v(' + v + ')'
                
                # Maybe there is no route to link, so pass
                if routes:
                    for r in routes:
                        method = entry.get('method', 'GET')
                        logger.debug('linking function [%s] and route [%s] for method [%s]' % (getattr(f, 'display_name', str(f)), r, method))
                        
                        # Ok, we will just use the lock for all
                        # plugin page, but not for static objects
                        # so we set the lock at the function level.
                        f_name = f.display_name
                        lock_version = self.lockable_function(f)
                        lock_version.display_name = f_name
                        f = route(r, callback=lock_version, method=[method, 'OPTIONS'])
            
            # And finally register me so the pages can get data and other
            # useful stuff
            m.app = self
        
        except Exception, exp:
            logger.warning("Loading plugins: %s" % exp)
            logger.print_stack()
    
    
    # Here we will load all plugins (pages) under the webui/plugins directory.
    # Each one can have a page, views and htdocs dir that we must route correctly
    def load_plugins(self, plugin_dir):
        # Load plugin directories
        if not os.path.exists(plugin_dir):
            return
        
        plugin_dirs = [fname for fname in os.listdir(plugin_dir) if os.path.isdir(os.path.join(plugin_dir, fname))]
        
        sys.path.append(plugin_dir)
        # We try to import them, but we keep only the one of our type
        for fdir in plugin_dirs:
            self.load_plugin(fdir, plugin_dir)
    
    
    # It will say if we can launch a page rendering or not.
    # We can only if there is no writer running from now
    def wait_for_no_writers(self):
        wait_start = time.time()
        start = time.time()
        wait_time = 0.001
        while True:
            self.global_lock.acquire()
            # We will be able to run
            if self.nb_writers == 0:
                # Ok, we can run, register us as readers
                self.nb_readers += 1
                self.global_lock.release()
                break
            # Oups, a writer is in progress. We must wait a bit
            self.global_lock.release()
            
            # Before checking again, we should wait a bit to not hammer the cpu.
            # Wait will be longer each call limit to 100ms
            time.sleep(wait_time)
            wait_time = max(wait_time * 2, 0.1)
            
            if time.time() - start > 30:
                logger.error("livedata: we wait for read lock for more than 30s! %0.3f" % (time.time() - wait_start))
                start = time.time()
    
    
    # It will say if we can launch a brok management or not
    # We can only if there is no readers running from now
    def wait_for_no_readers(self):
        wait_start = time.time()
        start = time.time()
        wait_time = 0.001
        while True:
            self.global_lock.acquire()
            # We will be able to run
            if self.nb_readers == 0:
                # Ok, we can run, register us as writers
                self.nb_writers += 1
                self.global_lock.release()
                break
            # Ok, we cannot run now, wait a bit
            self.global_lock.release()
            
            # Before checking again, we should wait a bit to not hammer the cpu.
            # Wait will be longer each call limit to 100ms
            time.sleep(wait_time)
            wait_time = max(wait_time * 2, 0.1)
            # We should warn if we cannot update broks
            # for more than 30s because it can be not good
            if time.time() - start > 30:
                logger.error("livedata: we wait for write lock for more than 30s! %0.3f" % (time.time() - wait_start))
                start = time.time()
    
    
    # We want a lock manager version of the plugin functions
    def lockable_function(self, f):
        def lock_version(**args):
            f = lock_version.f
            self.wait_for_no_writers()
            
            t = time.time()
            try:
                # set CORS headers
                response.headers['Access-Control-Allow-Origin'] = 'http://127.0.0.1:9000'
                response.headers['Access-Control-Allow-Methods'] = 'GET, POST, PUT, OPTIONS, DELETE, PATCH'
                response.headers['Access-Control-Allow-Headers'] = 'Origin, Accept, Content-Type, X-Requested-With, X-CSRF-Token, X-Shinken-Token'
                response.headers['Access-Control-Allow-Credentials'] = 'true'
                
                if bottle.request.method != 'OPTIONS':
                    # And disable cache on all backend calls
                    self._set_no_cache_headers()
                    return f(**args)
                return
            finally:
                logger.log_perf(t, 'perf', 'function %s executed' % getattr(f, 'display_name', str(f)))
                # We can remove us as a reader from now. It's NOT an atomic operation
                # so we REALLY not need a lock here (yes, I try without and I got
                # a not so accurate value there....)
                self.global_lock.acquire()
                self.nb_readers -= 1
                self.global_lock.release()
        
        
        lock_version.f = f
        return lock_version
    
    
    # For this request, ask for no cache at all. for example static that have always the
    # same address across versions
    @staticmethod
    def _set_no_cache_headers():
        # The Cache-Control is per the HTTP 1.1 spec for clients and proxies
        # (and implicitly required by some clients next to Expires).
        # The Pragma is per the HTTP 1.0 spec for prehistoric clients.
        # The Expires is per the HTTP 1.0 and 1.1 spec for clients and proxies.
        # In HTTP 1.1, the Cache-Control takes precedence over Expires, so it's after all for HTTP 1.0 proxies only.
        response.headers['Cache-Control'] = 'no-cache,no-store,must-revalidate'
        response.headers['Pragma'] = 'no-cache'
        response.headers['Expires'] = '0'
    
    
    @staticmethod
    def is_check_uuid(uuid):
        return '-' in uuid
    
    
    def get_name_from_uuid(self, uuid):
        r = None
        if '-' in uuid:
            check = datamgr.get_service_by_uuid(uuid)
            if check:
                r = [check.host.host_name, check.service_description]
        else:
            host = datamgr.get_host_by_uuid(uuid)
            if host:
                r = [host.host_name]
        
        if not r:
            return self.abort(404, 'unknow uuid [%s]' % uuid, True)
        return r
