#!/usr/bin/python
# -*- coding: utf-8 -*-
#
# Copyright (C) 2013-2019
# This file is part of Shinken Enterprise, all rights reserved.
import time
from datetime import timedelta

from shinken.objects.module import Module as ShinkenModuleDefinition
from shinken.withworkersandinventorymodule import WithWorkersAndInventoryModule
from shinkensolutions.date_helper import timestamp_from_datetime
from shinkensolutions.lib_modules.configuration_reader import read_int_in_configuration
from sla_abstract_component import ThreadComponent
from sla_common import shared_data
from sla_component_manager import ComponentManager
from sla_database import SLADatabase
from sla_error_handler import error_handler, ERROR_LEVEL
from sla_info import SLAInfo, DEFAULT_FIRST_MONITORING_DATE

DEFAULT_RAW_SLA_DAYS_TO_KEEP = 7
INFINITE_NB_STORED_DAYS = -1
SAMPLES_TO_STORE = 360


class SLAWriterData:
    def __init__(self, nb_stored_days):
        self.database_status = True
        
        self.workers = {}
        
        self.archive_in_progress = False
        self.total_sla_archived = 0
        self.total_sla_current_archive = 0
        self.sla_archived_during_current_archive = 0
        self.archive_progression_date = 0
        self.latest_archive_start_time = 0
        self.latest_archive_execution_time = 0
        self.previous_archive_start_time = 0
        self.previous_archive_execution_time = 0
        self.previous_archive_sla_archived = 0
        self.oldest_sla_date = 0
        
        self.all_database_migrated = False
        self.migration_in_progress = False
        self.total_sla_to_migrate = 0
        self.nb_sla_left_to_migrate = 0
        self.execution_time_last_migration = 0
        
        self.daily_clean_in_progress = False
        self.total_sla_to_remove = 0
        self.nb_sla_left_to_remove = 0
        self.execution_time_last_daily_clean = 0
        self.days_to_keep_sla = nb_stored_days
        self.sla_current_storage_size = 0
        self.total_unique_elements_stored = 0


class SLAWriterStats(ThreadComponent):
    def __init__(self, configuration, component_manager, sla_database, sla_info, module_with_worker):
        # type: (ShinkenModuleDefinition, ComponentManager,  SLADatabase, SLAInfo, WithWorkersAndInventoryModule) -> None
        super(SLAWriterStats, self).__init__(configuration, component_manager, only_one_thread_by_class=True, stop_thread_on_error=False, loop_speed=10)
        
        self.sla_database = sla_database
        self.sla_info = sla_info
        self._module_with_worker = module_with_worker
        
        self._keep_raw_sla_days = read_int_in_configuration(configuration, 'keep_raw_sla_day', DEFAULT_RAW_SLA_DAYS_TO_KEEP)
        if self._keep_raw_sla_days < 1:
            self._keep_raw_sla_days = DEFAULT_RAW_SLA_DAYS_TO_KEEP
        
        self._nb_stored_days = read_int_in_configuration(configuration, 'nb_stored_days', INFINITE_NB_STORED_DAYS)
        if self._nb_stored_days != INFINITE_NB_STORED_DAYS and self._nb_stored_days < 7:
            self._nb_stored_days = 7
        
        self._data = SLAWriterData(self._nb_stored_days)
        self._sla_stats = {}
        
        self._old_date_cache_time = 0
    
    
    def init(self):
        self.start_thread()
    
    
    def get_thread_name(self):
        return 'sla-writer-stats'
    
    
    def loop_turn(self):
        self._update_stats_from_shared_data()
        self._update_total_unique_elements_stored()
        self._update_sla_current_storage_size()
        
        sla_counts = self._module_with_worker.send_command('ask_sla_count')['workers']
        update_time = time.time()
        for worker in sla_counts:
            if sla_counts[worker] is None:
                continue
            if worker not in self._sla_stats:
                self._sla_stats[worker] = []
            if len(sla_counts[worker]) > 0:
                self._sla_stats[worker].insert(0, (update_time, sla_counts[worker][0], sla_counts[worker][1]))
            if len(self._sla_stats[worker]) > SAMPLES_TO_STORE:
                self._sla_stats[worker].pop(len(self._sla_stats[worker]) - 1)
        
        self._update_oldest_sla_date(update_time)
    
    
    def process_raw_worker_stats(self, raw_stats):
        raw_worker_data = raw_stats.get('workers', None)
        if not raw_worker_data:
            return
        
        worker_stats = {}
        for worker_id, worker_data in sorted(raw_worker_data.iteritems()):
            worker_data['sla_stats'] = self._sla_stats[worker_id] if worker_id in self._sla_stats else []
            worker_stats[worker_id + 1] = worker_data
        self._data.workers = worker_stats
    
    
    def get_raw_stats(self):
        return self._data.__dict__
    
    
    def _update_stats_from_shared_data(self):
        # Archive
        self._data.archive_in_progress = shared_data.get_archive_in_progress()
        self._data.total_sla_archived = shared_data.get_total_sla_archived()
        self._data.total_sla_current_archive = shared_data.get_total_sla_current_archive()
        self._data.sla_archived_during_current_archive = shared_data.get_sla_archived_during_current_archive()
        self._data.archive_progression_date = shared_data.get_archive_progression_date()
        self._data.latest_archive_start_time = shared_data.get_latest_archive_start_time()
        self._data.latest_archive_execution_time = shared_data.get_latest_archive_execution_time()
        self._data.previous_archive_start_time = shared_data.get_previous_archive_start_time()
        self._data.previous_archive_execution_time = shared_data.get_previous_archive_execution_time()
        self._data.previous_archive_sla_archived = shared_data.get_previous_archive_sla_archived()
        
        # Migration
        self._data.all_database_migrated = shared_data.get_migration_archive_done()
        self._data.migration_in_progress = shared_data.get_migration_in_progress()
        self._data.total_sla_to_migrate = shared_data.get_total_sla_to_migrate()
        self._data.nb_sla_left_to_migrate = shared_data.get_nb_sla_left_to_migrate()
        self._data.execution_time_last_migration = shared_data.get_execution_time_last_migration()
        
        # Daily clean
        self._data.daily_clean_in_progress = shared_data.get_daily_clean_in_progress()
        self._data.total_sla_to_remove = shared_data.get_total_sla_to_remove()
        self._data.nb_sla_left_to_remove = shared_data.get_nb_sla_left_to_remove()
        self._data.execution_time_last_daily_clean = shared_data.get_execution_time_last_daily_clean()
    
    
    def _update_oldest_sla_date(self, update_time):
        if update_time - self._old_date_cache_time > 3600:
            self._old_date_cache_time = update_time
            oldest_sla_date = self.sla_info.get_first_monitoring_start_time()
            if oldest_sla_date != DEFAULT_FIRST_MONITORING_DATE:
                self._data.oldest_sla_date = timestamp_from_datetime(oldest_sla_date + timedelta(days=1))
    
    
    def _update_sla_current_storage_size(self):
        try:
            self._data.sla_current_storage_size = self.sla_database.compute_current_storage_size(self._keep_raw_sla_days)
        except Exception as e:
            error_handler.handle_exception('Fail to get the size of database with error : %s. This will only impact your check shinken on your sup of sup.' % e, e, self.logger, ERROR_LEVEL.WARNING)
    
    
    def _update_total_unique_elements_stored(self):
        if shared_data.get_total_unique_elements_valid():
            return
        
        try:
            self._data.total_unique_elements_stored = self.sla_database.compute_total_unique_elements_stored()
            shared_data.set_total_unique_elements_valid(True)
        except Exception as e:
            error_handler.handle_exception('Fail to get the size of database with error : %s. This will only impact your check shinken on your sup of sup.' % e, e, self.logger, ERROR_LEVEL.WARNING)
            self._data.total_unique_elements_stored = -1
