#!/usr/bin/python

# -*- coding: utf-8 -*-

# Copyright (C) 2014:
#    Gabes Jean, j.gabes@shinken-solutions.com
#
# This file is part of Shinken Enterprise

import re

from shinken.basemodule import BaseModule
from shinken.log import logger

from shinkensolutions.api.synchronizer import SourceInfo, SourceInfoProperty, ITEM_TYPE, METADATA

properties = {
    'daemons': ['synchronizer'],
    'type'   : 'sync-regexp-tag',
}


# called by the plugin manager to get a module
def get_instance(plugin):
    logger.info("[Sync RegEx] Get a SyncIPTag module for plugin %s" % plugin.get_name())
    
    # Catch errors
    matched_prop = getattr(plugin, 'matched_prop', 'host_name')
    matched_regexp = plugin.matched_regexp
    prop = plugin.property
    value = plugin.value
    method = getattr(plugin, 'method', 'replace')
    ignore_hosts = getattr(plugin, 'ignore_hosts', None)
    
    instance = Sync_Name_Tag_Arbiter(plugin, matched_prop, matched_regexp, prop, value, method, ignore_hosts)
    return instance


# Just print some stuff
class Sync_Name_Tag_Arbiter(BaseModule):
    def __init__(self, mod_conf, matched_prop, matched_regexp, prop, value, method, ignore_hosts=None):
        BaseModule.__init__(self, mod_conf)
        logger.debug("[Sync RegExTag] __init__ [%s]" % matched_regexp)
        self.matched_prop = matched_prop
        self.matched_regexp = re.compile(matched_regexp)
        self.property = prop
        self.value = value
        self.method = method
        if ignore_hosts:
            self.ignore_hosts = ignore_hosts.split(', ')
            logger.info("[Sync RegExTag] Ignoring hosts : %s" % self.ignore_hosts)
        else:
            self.ignore_hosts = []
    
    
    # Called by Arbiter to say 'let's prepare yourself guy'
    def init(self):
        logger.info("[Sync RegExTag] Initialization of the regexp based tagger module")
    
    
    def hook_post_merge(self, all_hosts):
        logger.info("[Sync RegExTag] in hook post merge")
        
        for host in all_hosts:
            
            hvalue = host.get(self.matched_prop, None)
            
            # Maybe this host got nothing, bail out
            if hvalue is None or not isinstance(hvalue, basestring):
                continue
            
            source_info = METADATA.get_metadata(host, METADATA.SOURCE_INFO)
            if isinstance(source_info, dict):
                source_info = SourceInfo.from_dict(source_info, ITEM_TYPE.HOSTS)
            # We are here for this...
            if self.matched_regexp.search(hvalue.lower()):
                logger.debug("[Sync RegExTag] Is matched for %s" % hvalue)
                # 4 cases: append , replace and set
                # append will join with the value if exist (on the END)
                # prepend will join with the value if exist (on the BEGINING)
                # replace will replace it if NOT existing
                # set put the value even if the property exists
                if self.method == 'append':
                    orig_v = [e.strip() for e in host.get(self.property, '').split(',') if e.strip()]
                    suffixlst = [e.strip() for e in self.value.split(',') if e.strip()]
                    for e in suffixlst:
                        if e not in orig_v:
                            orig_v.append(e)
                    new_v = ','.join(orig_v)
                    host[self.property] = new_v
                    # Let the source info know that this value came from us
                    # Here : we did not overrite
                    source_info.update_field(self.property, self.value, self.get_name(), field_type=SourceInfoProperty.ORDERED_TYPE)
                
                # Same but we put before
                if self.method == 'prepend':
                    orig_v = [e.strip() for e in host.get(self.property, '').split(',') if e.strip()]
                    suffixlst = [e.strip() for e in self.value.split(',') if e.strip()]
                    for e in suffixlst:
                        if e not in orig_v:
                            orig_v.insert(0, e)
                    new_v = ','.join(orig_v)
                    host[self.property] = new_v
                    # Let the source info know that this value came from us
                    # Here : we did not overrite
                    source_info.update_field(self.property, self.value, self.get_name(), field_type=SourceInfoProperty.ORDERED_TYPE)
                
                if self.method == 'replace':
                    if not self.property in host:
                        # Ok, set the value!
                        host[self.property] = self.value
                        # Let the source info know that this value came from us
                        # Here : we did overwrite it
                        source_info.update_field(self.property, self.value, self.get_name(), field_type=SourceInfoProperty.ORDERED_TYPE, overwrite=True)
                
                if self.method == 'set':
                    host[self.property] = self.value
                    # Let the source info know that this value came from us
                    # Here : we did overwrite it
                    source_info.update_field(self.property, self.value, self.get_name(), field_type=SourceInfoProperty.ORDERED_TYPE, overwrite=True)
                
                logger.debug('[Sync RegExTag] Generated property:  host:%s    matched:%s   %s => %s' % (host.get('host_name', ''), hvalue, self.property, host[self.property]))
