#!/usr/bin/python
# -*- coding: utf-8 -*-

# Copyright (C) 2009-2017:
#    Gabes Jean, naparuba@gmail.com
#    Gerhard Lausser, Gerhard.Lausser@consol.de
#    Gregory Starck, g.starck@gmail.com
#    Hartmut Goebel, h.goebel@goebel-consult.de
#
# This file is part of Shinken.
#
# Shinken is free software: you can redistribute it and/or modify
# it under the terms of the GNU Affero General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# Shinken is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Affero General Public License for more details.
#
# You should have received a copy of the GNU Affero General Public License
# along with Shinken.  If not, see <http://www.gnu.org/licenses/>.


import base64
import sqlite3
import cPickle
import sys
import threading
import time
import traceback
import zlib
import os
import thread
import logging
from datetime import datetime

from sqlite3 import IntegrityError
from shinken.load import AvgInRange
from shinken.daemon import Interface, Daemon
from shinken.satellite import IForArbiter
from shinken.daemons.brokerdaemon import IStats as BrokerIStats
from shinken.daemons.brokerdaemon import Broker
from shinken.http_client import HTTPExceptions
from shinken.log import logger
from shinken.property import PathProp, IntegerProp
from shinken.satellite import BaseSatellite
from shinken.misc.perfdata import PerfDatas
from shinken.rawdata import TYPE_RAW_DATA_CHECK_RESULT, TYPE_RAW_DATA_FULL

_BEACON_LOG_LEVEL = logging.DEBUG

_SCHEMA_VERSION = 1
_CREATION_QUERY = """
CREATE TABLE CLUSTER (
    instance_id INTEGER,
    cluster_uuid TEXT PRIMARY KEY NOT NULL
);
CREATE TABLE HOST (
    instance_id INTEGER,
    host_uuid TEXT PRIMARY KEY NOT NULL
);
CREATE TABLE SERVICE (
    instance_id INTEGER,
    service_uuid TEXT ,
    service_link_host_uuid,
    service_id TEXT PRIMARY KEY
);
CREATE TABLE LINK_CLUSTER_HOST (
    instance_id INTEGER,
    link_cluster_host_host_uuid,
    link_cluster_host_cluster_uuid,
    PRIMARY KEY (link_cluster_host_host_uuid, link_cluster_host_cluster_uuid)
);
CREATE TABLE LINK_CLUSTER_SERVICE (
    instance_id INTEGER,
    link_cluster_service_service_uuid,
    link_cluster_service_cluster_uuid,
    PRIMARY KEY (link_cluster_service_service_uuid, link_cluster_service_cluster_uuid)
);
CREATE TABLE RAW_DATA (
    raw_data_id INTEGER PRIMARY KEY AUTOINCREMENT NOT NULL,
    raw_data_item_id TEXT,
    raw_data_item_type TEXT,
    raw_data_last_chk INTEGER,
    raw_data_state_type_id INTEGER,
    raw_data_state_id INTEGER,
    raw_data_context_id INTEGER
);
CREATE TABLE METRIC (
    metric_id INTEGER PRIMARY KEY AUTOINCREMENT NOT NULL,
    metric_raw_data_id,
    metric_last_chk INTEGER,
    metric_name TEXT,
    metric_value REAL,
    metric_warning REAL,
    metric_critical REAL,
    metric_uom TEXT,
    metric_min REAL,
    metric_max REAL
);
CREATE INDEX raw_data_item_id_index ON RAW_DATA (raw_data_item_id);
CREATE INDEX raw_data_last_chk_index ON RAW_DATA (raw_data_last_chk);
CREATE INDEX metric_last_chk_index ON METRIC (metric_last_chk);
PRAGMA main.user_version = %s;
""" % _SCHEMA_VERSION
# Use this line when you want debug the db model.
# _DEFAULT_PATH_DB = "/var/lib/shinken/provider-%s.db" % hashlib.md5(_CREATION_QUERY).hexdigest()
_DEFAULT_PATH_DB = "/var/lib/shinken/provider.db"
ITEM_TYPE_HOST = 'HOST'
ITEM_TYPE_CLUSTER = 'CLUSTER'
ITEM_TYPE_SERVICE = 'SERVICE'


class ProviderNotReady(BaseException):
    pass


class IInfo(Interface):
    def ask_members(self, item_id):
        return self.app.ask_members(item_id)
    
    
    ask_members.doc = 'Get the members of the cluster'
    ask_members.need_lock = False
    
    
    def ask_checks(self, item_id):
        return self.app.ask_checks(item_id)
    
    
    ask_checks.doc = 'Get the checks of the host'
    ask_checks.need_lock = False
    
    
    def ask_metrics(self, item_id):
        return self.app.ask_metrics(item_id)
    
    
    ask_metrics.doc = 'Get the metrics of the element'
    ask_metrics.need_lock = False
    
    
    def ask_item_states(self, item_id, item_type, since=-1):
        return self.app.ask_item_states(item_id, item_type, int(since))
    
    
    ask_item_states.doc = 'Get item states of the element.'
    ask_item_states.need_lock = False
    
    
    def ask_item_metrics(self, item_id, item_type, since=-1):
        return self.app.ask_item_metrics(item_id, item_type, int(since))
    
    
    ask_item_metrics.doc = 'Get metrics info of the element.'
    ask_item_metrics.need_lock = False
    
    
    def ask_item_metric(self, item_id, item_type, metric, since=-1):
        return self.app.ask_item_metric(item_id, item_type, metric, int(since))
    
    
    ask_item_metric.doc = 'Get the metric info of the element.'
    ask_item_metric.need_lock = False


class IStats(BrokerIStats):
    def get_raw_stats(self, param=''):
        app = self.app
        res = {'modules': []}
        insts = [inst for inst in app.modules_manager.instances if inst.is_external]
        for inst in insts:
            try:
                res['modules'].append({'module_name': inst.get_name(), 'queue_size': inst.to_q.qsize()})
            except Exception:
                res['modules'].append({'module_name': inst.get_name(), 'queue_size': -1})
        
        res['module_stats'] = self._get_module_stats(getattr(self.app, 'modules_manager', None), param)
        
        res['raw_data_done_by_sec'] = app.raw_data_done_by_sec.get_avg()
        return res
    
    
    get_raw_stats.doc = ''
    get_raw_stats.need_lock = False


# Our main APP class
class Provider(Broker):
    properties = BaseSatellite.properties.copy()
    properties.update({
        'pidfile'  : PathProp(default='providerd.pid'),
        'port'     : IntegerProp(default='7774'),
        'local_log': PathProp(default='providerd.log'),
    })
    
    daemon_type = 'provider'
    
    
    def __init__(self, config_file, is_daemon, do_replace, debug, debug_file, profile='', daemon_id=0):
        super(Provider, self).__init__(config_file, is_daemon, do_replace, debug, debug_file, daemon_id)
        
        # Daemon config
        self.schedulers = {}
        self.modules = None
        self.timeout = 1.0
        self.loop_turn = 0
        
        self.raw_datas = []
        
        self.raw_datas_lock = threading.RLock()
        self.satellite_lock = threading.RLock()
        
        # Now we create the interfaces
        self.interface = IForArbiter(self)
        self.istats = IStats(self)
        self.iinfo = IInfo(self)
        
        # DB
        self.initial_info = 0
        self.db_path = _DEFAULT_PATH_DB
        self.sqlite_con = {}
        self.memdb = None
        self.current_cursor = None
        self.retention_time = 24 * 60
        
        # stats
        self.nb_raw_data_done_by_sec = 0
        self.nb_insert_done_by_sec = 0
        self.raw_data_done_by_sec = AvgInRange(60)
    
    
    def create_local_db(self):
        # we delete self.db_path-journal to avoid lock of the db
        if os.path.exists(self.db_path + '-journal'):
            os.remove(self.db_path + '-journal')
        
        if os.path.exists(self.db_path):
            memdb = sqlite3.connect(self.db_path)
            current_cursor = memdb.cursor()
            schema_version = current_cursor.execute('PRAGMA main.user_version;').fetchone()[0]
            current_cursor.execute('PRAGMA synchronous = OFF;')
            current_cursor.execute('PRAGMA journal_mode = WAL;')
            memdb.commit()
            self.sqlite_con[thread.get_ident()] = memdb
        else:
            memdb = sqlite3.connect(self.db_path)
            current_cursor = memdb.cursor()
            current_cursor.executescript(_CREATION_QUERY)
            schema_version = current_cursor.execute('PRAGMA main.user_version;').fetchone()[0]
            current_cursor.execute('PRAGMA synchronous = OFF;')
            current_cursor.execute('PRAGMA journal_mode = WAL;')
            memdb.commit()
            self.sqlite_con[thread.get_ident()] = memdb
        
        logger.debug("[base] Connect to base [%s] schema_version [%s] from thread [%s]" % (self.db_path, schema_version, threading.current_thread().getName()))
        if schema_version != _SCHEMA_VERSION:
            logger.info("[base] Connect to base [%s] in version [%s] expect version [%s]." % (self.db_path, schema_version, _SCHEMA_VERSION))
            logger.info("[base] We must migrate your database.")
    
    
    def get_con_db(self):
        thread_id = thread.get_ident()
        
        if self.cur_conf is not None and thread_id not in self.sqlite_con and os.path.exists(self.db_path):
            memdb = sqlite3.connect(self.db_path)
            current_cursor = memdb.cursor()
            schema_version = current_cursor.execute('PRAGMA main.user_version;').fetchone()[0]
            current_cursor.execute('PRAGMA synchronous = OFF;')
            current_cursor.execute('PRAGMA journal_mode = WAL;')
            memdb.commit()
            self.sqlite_con[thread_id] = memdb
            logger.info("[base] Connect to base [%s] schema_version [%s] from thread [%s]" % (self.db_path, schema_version, threading.current_thread().getName()))
        
        if not thread_id in self.sqlite_con:
            raise ProviderNotReady('The provider was not contact by the arbiter. Please retry when the provider will be ready')
        return self.sqlite_con[thread_id]
    
    
    def process_raw_data(self, raw_data):
        raw_data_type_full = raw_data.type
        raw_data_data = raw_data.data
        
        raw_data_type = ''
        raw_data_item_type = ''
        
        # logger.debug("[process_raw_data] get new raw data : [%s]" % raw_data_type_full)
        if raw_data_type_full == 'clean_all_my_instance_id':
            raw_data_instance_id = raw_data.data['instance_id']
            self.clean_items(raw_data_instance_id)
        if raw_data_type_full == 'host_full':
            raw_data_type = TYPE_RAW_DATA_FULL
            raw_data_item_type = ITEM_TYPE_CLUSTER if raw_data_data.get('got_business_rule', False) else ITEM_TYPE_HOST
        elif raw_data_type_full == 'host_check_result':
            raw_data_type = TYPE_RAW_DATA_CHECK_RESULT
            raw_data_item_type = ITEM_TYPE_CLUSTER if raw_data_data.get('got_business_rule', False) else ITEM_TYPE_HOST
        elif raw_data_type_full == 'service_full':
            raw_data_type = TYPE_RAW_DATA_FULL
            raw_data_item_type = ITEM_TYPE_SERVICE
        elif raw_data_type_full == 'service_check_result':
            raw_data_type = TYPE_RAW_DATA_CHECK_RESULT
            raw_data_item_type = ITEM_TYPE_SERVICE
        elif raw_data_type_full == 'initial_raw_datas_done':
            logger.debug("[raw_data] end of get new configuration")
        
        if raw_data_type == TYPE_RAW_DATA_FULL:
            self.process_raw_data_full(raw_data_item_type, raw_data)
        elif raw_data_type == TYPE_RAW_DATA_CHECK_RESULT:
            self.process_raw_data_check_result(raw_data_item_type, raw_data)
    
    
    def clean_items(self, raw_data_instance_id):
        clean_items_query = """
        DELETE FROM LINK_CLUSTER_HOST WHERE instance_id=%s;
        DELETE FROM LINK_CLUSTER_SERVICE WHERE instance_id=%s;
        DELETE FROM SERVICE WHERE instance_id=%s;
        DELETE FROM CLUSTER WHERE instance_id=%s;
        DELETE FROM HOST WHERE instance_id=%s;
        """ % (raw_data_instance_id, raw_data_instance_id, raw_data_instance_id, raw_data_instance_id, raw_data_instance_id)
        self.current_cursor.executescript(clean_items_query)
        self.initial_info += 1
        logger.debug("[raw_data] clean items of the instance [%s], disable foreign_keys" % raw_data_instance_id)
    
    
    def process_raw_data_full(self, raw_data_item_type, raw_data, watchdog=0):
        # logger.debug("[raw_data] process_raw_data_full [%s]-[%s]" % (raw_data_item_type, raw_data))
        raw_data_data = raw_data.data
        raw_data_instance_id = raw_data_data['instance_id']
        raw_data_item_uuid = raw_data_data['uuid']
        
        try:
            if raw_data_item_type == ITEM_TYPE_CLUSTER:
                element_to_insert = [raw_data_item_uuid, raw_data_instance_id]
                self.current_cursor.execute('INSERT INTO CLUSTER (cluster_uuid, instance_id) VALUES (?,?)', element_to_insert)
                self.nb_insert_done_by_sec += 1
                for link_cluster in raw_data_data['cluster_members']:
                    # logger.debug("[raw_data] [%s] cluster_members [%s]" % (raw_data_item_uuid, link_cluster))
                    link_cluster_uuid = link_cluster[0]
                    link_cluster_type = link_cluster[1]
                    if link_cluster_type == 'host':
                        element_to_insert = [link_cluster_uuid, raw_data_item_uuid, raw_data_instance_id]
                        self.current_cursor.execute('INSERT INTO LINK_CLUSTER_HOST (link_cluster_host_host_uuid, link_cluster_host_cluster_uuid, instance_id) VALUES (?,?,?)', element_to_insert)
                        self.nb_insert_done_by_sec += 1
                    if link_cluster_type == 'service':
                        element_to_insert = [link_cluster_uuid, raw_data_item_uuid, raw_data_instance_id]
                        self.current_cursor.execute('INSERT INTO LINK_CLUSTER_SERVICE (link_cluster_service_service_uuid, link_cluster_service_cluster_uuid, instance_id) VALUES (?,?,?)', element_to_insert)
                        self.nb_insert_done_by_sec += 1
            elif raw_data_item_type == ITEM_TYPE_HOST:
                element_to_insert = [raw_data_item_uuid, raw_data_instance_id]
                self.current_cursor.execute('INSERT INTO HOST (host_uuid, instance_id) VALUES (?,?)', element_to_insert)
                self.nb_insert_done_by_sec += 1
            elif raw_data_item_type == ITEM_TYPE_SERVICE:
                uuid = '%s-%s' % (raw_data_data['host_uuid'], raw_data_item_uuid)
                element_to_insert = [uuid, raw_data_item_uuid, raw_data_data['host_uuid'], raw_data_instance_id]
                self.current_cursor.execute('INSERT INTO SERVICE (service_id, service_uuid, service_link_host_uuid, instance_id) VALUES (?,?,?,?)', element_to_insert)
                self.nb_insert_done_by_sec += 1
        except IntegrityError as ie:
            if watchdog > 1:
                raise ie
            logger.info('[data] item [%s-%s] must have change this scheduler. [%s]' % (raw_data_item_type, raw_data_item_uuid, ie))
            if raw_data_item_type == ITEM_TYPE_CLUSTER:
                element_to_insert = [raw_data_item_uuid]
                self.current_cursor.execute('DELETE FROM CLUSTER WHERE cluster_uuid=?', element_to_insert)
                self.current_cursor.execute('DELETE FROM LINK_CLUSTER_HOST WHERE link_cluster_host_cluster_uuid=?', element_to_insert)
                self.current_cursor.execute('DELETE FROM LINK_CLUSTER_SERVICE WHERE link_cluster_service_cluster_uuid=?', element_to_insert)
            elif raw_data_item_type == ITEM_TYPE_HOST:
                element_to_insert = [raw_data_item_uuid]
                self.current_cursor.execute('DELETE FROM HOST WHERE host_uuid=?', element_to_insert)
            elif raw_data_item_type == ITEM_TYPE_SERVICE:
                uuid = '%s-%s' % (raw_data_data['host_uuid'], raw_data_item_uuid)
                element_to_insert = [uuid]
                self.current_cursor.execute('DELETE FROM SERVICE WHERE service_id=?', element_to_insert)
            self.process_raw_data_full(raw_data_item_type, raw_data, watchdog)
            watchdog += 1
    
    
    def process_raw_data_check_result(self, raw_data_item_type, raw_data):
        # logger.debug("[raw_data] save_raw_data [%s]-[%s]" % (raw_data_item_type, raw_data))
        raw_data_data = raw_data.data
        
        raw_data_item_id = raw_data_data['uuid']
        if raw_data_item_type == ITEM_TYPE_SERVICE:
            raw_data_item_id = '%s-%s' % (raw_data_data['host_uuid'], raw_data_data['uuid'])
        
        raw_data_to_insert = [
            raw_data_item_id,
            raw_data_item_type,
            raw_data_data['last_chk'],
            raw_data_data['state_type_id'],
            raw_data_data['state_id'],
            raw_data_data['context_id']
        ]
        raw_data_insert = self.current_cursor.execute('INSERT INTO RAW_DATA ('
                                                      'raw_data_item_id,'
                                                      'raw_data_item_type,'
                                                      'raw_data_last_chk,'
                                                      'raw_data_state_type_id,'
                                                      'raw_data_state_id,'
                                                      'raw_data_context_id) '
                                                      'VALUES (?,?,?,?,?,?)', raw_data_to_insert)
        
        last_raw_data_id = raw_data_insert.lastrowid
        self.nb_insert_done_by_sec += 1
        # logger.debug("[raw_data] raw_data_insert : [%s] - [%s]" % (last_raw_data_id, raw_data_to_insert))
        
        perf_data = raw_data_data['perf_data']
        
        if perf_data:
            metrics = PerfDatas(perf_data)
            for metric in metrics:
                metric_name = metric.name
                metric_last_chk = raw_data_data['last_chk']
                metric_value = metric.value
                metric_warning = metric.warning
                metric_critical = metric.critical
                metric_uom = metric.uom
                metric_min = metric.min
                metric_max = metric.max
                
                metric_info_to_insert = [
                    last_raw_data_id,
                    metric_name,
                    metric_last_chk,
                    metric_value,
                    metric_warning,
                    metric_critical,
                    metric_uom,
                    metric_min,
                    metric_max,
                ]
                
                # logger.debug("[raw_data] metric insert : [%s]" % metric_info_to_insert)
                self.nb_insert_done_by_sec += 1
                self.current_cursor.execute('INSERT INTO METRIC ('
                                            'metric_raw_data_id,'
                                            'metric_name ,'
                                            'metric_last_chk ,'
                                            'metric_value ,'
                                            'metric_warning ,'
                                            'metric_critical ,'
                                            'metric_uom ,'
                                            'metric_min , '
                                            'metric_max '
                                            ') VALUES (?,?,?,?,?,?,?,?,?)', metric_info_to_insert)
                
                # logger.debug("[raw_data] metric insert : [%s]" % metric_info_to_insert)
    
    
    def clean_raw_data(self):
        start_time = time.time()
        db_con = self.get_con_db()
        c = db_con.cursor()
        from_time = time.time() - self.retention_time * 60
        
        max_raw_data = c.execute('SELECT max(raw_data_last_chk) FROM RAW_DATA;')
        max_metric = c.execute('SELECT max(metric_last_chk) FROM METRIC;')
        
        if max_metric < from_time:
            c.execute('DROP TABLE METRIC;', [from_time])
        else:
            c.execute('DELETE FROM METRIC WHERE metric_last_chk<?', [from_time])
        
        if max_raw_data < from_time:
            c.execute('DROP TABLE RAW_DATA;', [from_time])
            from_time_date = datetime.fromtimestamp(max_raw_data)
            logger.debug("[raw_data] clean_raw_data your last entry is at [%s] we drop all datas." % from_time_date.strftime("%Y-%m-%d %H:%M:%S.%f"))
        else:
            nb_raw_data_clean = c.execute('DELETE FROM RAW_DATA WHERE raw_data_last_chk<?', [from_time]).rowcount
            from_time_date = datetime.fromtimestamp(from_time)
            logger.debug("[raw_data] clean_raw_data [%s] from [%s]" % (nb_raw_data_clean, from_time_date.strftime("%Y-%m-%d %H:%M:%S.%f")))
        
        db_con.commit()
    
    
    def manage_raw_data(self, raw_data):
        raw_data.prepare()
        self.process_raw_data(raw_data)
        for mod in self.modules_manager.get_internal_instances():
            try:
                mod.manage_raw_data(raw_data)
            except Exception as exp:
                logger.debug(str(exp.__dict__))
                logger.warning("The mod %s raise an exception: %s, I'm tagging it to restart later" % (mod.get_name(), str(exp)))
                logger.warning("Exception type: %s" % type(exp))
                logger.warning("Back trace of this kill: %s" % (traceback.format_exc()))
                self.modules_manager.set_to_restart(mod)
    
    
    # Add raw_datas (a tab) to different queues for
    # internal and external modules
    def add_raw_datas_to_queue(self, raw_datas):
        # Ok now put in queue raw_datas to be managed by
        # internal modules
        with self.raw_datas_lock:
            self.raw_datas.append(raw_datas)
    
    
    # We get new raw_datas from schedulers
    def get_new_raw_datas(self, sat_entry):
        # We check for new check in each schedulers and put
        # the result in new_checks
        sat_type = sat_entry['type']
        try:
            con = sat_entry['con']
            if con is not None:  # None = not initialized
                t0 = time.time()
                # Before ask a call that can be long, do a simple ping to be sure it is alive
                con.get('ping')
                tmp_raw_datas = con.get('get_raw_datas', {'provider_name': self.name}, wait='long')
                try:
                    _t = base64.b64decode(tmp_raw_datas)
                    _t = zlib.decompress(_t)
                    tmp_raw_datas = cPickle.loads(_t)
                except (TypeError, zlib.error, cPickle.PickleError), exp:
                    logger.error('Cannot load raw_datas from %s : %s' % (sat_entry['name'], exp))
                    sat_entry['con'] = None
                    return
                
                if len(tmp_raw_datas) != 0:
                    logger.debug("[raw_data] [%s] raw_data get in [%s] from [%s-%s]" % (len(tmp_raw_datas), time.time() - t0, sat_type, sat_entry['name']))
                for b in tmp_raw_datas.values():
                    b.instance_id = sat_entry['instance_id']
                
                # Ok, we can add theses raw_datas to our queues
                self.add_raw_datas_to_queue(tmp_raw_datas.values())
            
            else:  # no con? make the connection
                self.pynag_con_init(sat_entry)
        # Ok, con is not known, so we create it
        except KeyError, exp:
            logger.debug("Key error for get_raw_datas : %s" % str(exp))
            self.pynag_con_init(sat_entry)
        except HTTPExceptions, exp:
            logger.warning("Connection problem to the %s %s: %s" % (sat_type, sat_entry['name'], str(exp)))
            sat_entry['con'] = None
        # scheduler must not #be initialized
        except AttributeError, exp:
            logger.warning("The %s %s should not be initialized: %s" % (sat_type, sat_entry['name'], str(exp)))
        # scheduler must not have checks
        #  What the F**k? We do not know what happened,
        # so.. bye bye :)
        except Exception, x:
            logger.error(str(x))
            logger.error(traceback.format_exc())
            sys.exit(1)
    
    
    # Drop others object currently, like Broks
    def add(self, o):
        pass
    
    
    def really_setup_new_conf(self):
        conf = self.new_conf
        self.new_conf = None
        self.cur_conf = conf
        # Got our name from the globals
        g_conf = conf['global']
        if 'provider_name' in g_conf:
            name = g_conf['provider_name']
        else:
            name = 'Unnamed provider'
        self.name = name
        logger.load_obj(self, name)
        
        self.save_daemon_name_into_configuration_file(name)
        
        logger.debug("[%s] Sending us configuration %s" % (self.name, conf))
        
        db_path = g_conf['db_path']
        if db_path:
            self.db_path = db_path
        self.create_local_db()
        
        # Should we enable/disable human log format
        logger.set_human_format(on=g_conf.get('human_timestamp_log', True))
        
        self.retention_time = g_conf['retention_time']
        
        # If we've got something in the schedulers, we do not
        # want it anymore
        # self.schedulers.clear()
        for sched_id in conf['schedulers']:
            # Must look if we already have it to do not overdie our raw_datas
            already_got = False
            
            # Thread that manage this connection
            thread = None
            # We can already got this conf id, but with another address
            if sched_id in self.schedulers:
                new_addr = conf['schedulers'][sched_id]['address']
                old_addr = self.schedulers[sched_id]['address']
                new_port = conf['schedulers'][sched_id]['port']
                old_port = self.schedulers[sched_id]['port']
                thread = self.schedulers[sched_id]['thread']  # always keep it, if not valid, will drop it self and will be restarted
                # Should got all the same to be ok :)
                if new_addr == old_addr and new_port == old_port:
                    already_got = True
            
            if already_got:
                raw_datas = self.schedulers[sched_id]['raw_datas']
                daemon_incarnation = self.schedulers[sched_id]['daemon_incarnation']
            else:
                raw_datas = {}
                daemon_incarnation = 0
            s = conf['schedulers'][sched_id]
            self.schedulers[sched_id] = s
            
            # replacing scheduler address and port by those defined in satellitemap
            if s['name'] in g_conf['satellitemap']:
                s = dict(s)  # make a copy
                s.update(g_conf['satellitemap'][s['name']])
            proto = 'http'
            if s['use_ssl']:
                proto = 'https'
            uri = '%s://%s:%s/' % (proto, s['address'], s['port'])
            self.schedulers[sched_id]['uri'] = uri
            
            self.schedulers[sched_id]['raw_datas'] = raw_datas
            self.schedulers[sched_id]['instance_id'] = s['instance_id']
            self.schedulers[sched_id]['daemon_incarnation'] = daemon_incarnation
            self.schedulers[sched_id]['active'] = s['active']
            self.schedulers[sched_id]['last_connection'] = 0
            self.schedulers[sched_id]['thread'] = thread
            self.schedulers[sched_id]['con'] = None
            self.schedulers[sched_id]['id'] = sched_id
            self.schedulers[sched_id]['broks'] = {}
            self.schedulers[sched_id]['type'] = 'scheduler'
            
            logger.debug("[%s] schedulers sched_id[%s] name[%s] instance_id[%s] push_flavor[%s]" % (self.name, sched_id, s['name'], s['instance_id'], s['push_flavor']))
        
        if not self.have_modules:
            self.modules = mods = conf['global']['modules']
            self.have_modules = True
            logger.info("We received modules %s " % mods)
            
            # Ok now start, or restart them!
            # Set modules, init them and start external ones
            self.modules_manager.set_modules(self.modules)
            self.do_load_modules()
            self.modules_manager.start_external_instances()
        
        # Set our giving timezone from arbiter
        self.set_tz(conf['global']['use_timezone'])
        
        # Start threads if need, not a problem as starting thread is cheap and not timeout prone
        self.assert_valid_satellite_threads()
    
    
    # An arbiter ask us to wait for a new conf, so we must clean
    # all our mess we did, and close modules too
    def clean_previous_run(self):
        with self.satellite_lock:
            # Clean all lists
            self.schedulers.clear()
        with self.raw_datas_lock:
            self.raw_datas = self.raw_datas[:]
        
        # And now modules
        self.have_modules = False
        self.modules_manager.clear_instances()
    
    
    # Provider ask for raw_data
    def get_jobs_from_distant(self, e):
        self.get_new_raw_datas(e)
    
    
    # For a new distant daemon, if it is a scheduler, ask for a new full broks generation
    def _manage_new_distant_daemon_incarnation(self, entry, old_incar, new_incar):
        con = entry['con']
        daemon_type = entry['type']
        entry['broks'].clear()
        # we must ask for a new full broks if  it's a scheduler
        if daemon_type == 'scheduler':
            logger.debug("[raw_data] Asking initial raw data generation to the scheduler [%s] with daemon incarnation [%s] (old was [%s])" % (entry['name'], new_incar, old_incar))
            con.get('fill_initial_raw_datas', {'provider_name': self.name}, wait='long')
    
    
    def do_loop_turn(self):
        loop_time = time.time()
        before_ = time.time()
        
        # Begin to clean modules
        self.check_and_del_zombie_modules()
        
        # Look if we have all need threads for our satellites
        self.assert_valid_satellite_threads()
        
        # Maybe the arbiter ask us to wait for a new conf
        # If true, we must restart all...
        if self.cur_conf is None:
            # Clean previous run from useless objects and close modules
            self.clean_previous_run()
            
            self.wait_for_initial_conf()
            # we may have been interrupted or so; then just return from this loop turn
            if not self.new_conf:
                return
            self.setup_new_conf()
        
        # Now we check if arbiter speak to us.
        # If so, we listen for it
        # When it pushes conf to us, we reinit connections
        self.watch_for_new_conf(0.0)
        if self.new_conf:
            self.setup_new_conf()
        
        after_ = time.time()
        logger.log(_BEACON_LOG_LEVEL, '[beacon] checks [%.3f]' % (after_ - before_))
        before_ = time.time()
        
        if self.loop_turn % 60 == 0:
            self.clean_raw_data()
        
        after_ = time.time()
        logger.log(_BEACON_LOG_LEVEL, '[beacon] cleaning  [%.3f]' % (after_ - before_))
        before_ = time.time()
        
        # We will works this turn with a copy of the raw_datas, so we won't be impacted by the others threads
        with self.raw_datas_lock:
            raw_data_lists = self.raw_datas[:]
            self.raw_datas = []
        
        after_ = time.time()
        logger.log(_BEACON_LOG_LEVEL, '[beacon] copy raw_datas  [%.3f]' % (after_ - before_))
        before_ = time.time()
        
        raw_datas = []
        for raw_data_list in raw_data_lists:
            raw_datas.extend(raw_data_list)
        
        after_ = time.time()
        logger.log(_BEACON_LOG_LEVEL, '[beacon] extend raw_datas  [%.3f]' % (after_ - before_))
        before_ = time.time()
        
        logger.debug("[raw_data] Loop - managing [%s] raw datas." % len(raw_datas))
        
        # We put to external queues raw_datas that was not already send
        t0 = time.time()
        
        # We are sending raw_datas as a big list, more efficient than one by one
        queues = self.modules_manager.get_external_to_queues()
        to_send = [b for b in raw_datas if getattr(b, 'need_send_to_ext', True)]
        
        for q in queues:
            try:
                q.put(to_send)
            except Exception:  # we catch but the kill detector on the next loop will detect the fail module and will manage it
                logger.error('FAIL TO PUSH DATA TO EXTERNAL MODULE     the provider did fail to push data to an external module, this module will be detected and restart.')
        
        # No more need to send them
        for b in to_send:
            b.need_send_to_ext = False
        if time.time() - t0 > 0.1:
            logger.debug("[raw_data] Time to send [%s] raw_datas to module ([%.3f] secs)" % (len(to_send), time.time() - t0))
        
        after_ = time.time()
        logger.log(_BEACON_LOG_LEVEL, '[beacon] send to external module  [%.3f]' % (after_ - before_))
        before_ = time.time()
        
        self.memdb = self.get_con_db()
        self.current_cursor = self.memdb.cursor()
        start_time = time.time()
        while len(raw_datas) != 0:
            now = time.time()
            
            # Do not 'manage' more than 1s, we must get new raw_datas every 1s
            if now - start_time > 1:
                # so we must remerge our last raw_datas with the main raw_datas to do not lost them
                with self.raw_datas_lock:
                    logger.info('Cannot manage all remaining raw_datas [%d] in a loop turn, push back this raw_datas in the queue.' % len(raw_datas))
                    self.raw_datas.insert(0, raw_datas)
                break
            
            try:
                raw_data = raw_datas.pop()
            except IndexError:  # no more raw_datas, maybe a daemon stop, not a problem, catch it
                break
            
            self.manage_raw_data(raw_data)
            self.nb_raw_data_done_by_sec += 1
        insert_time = time.time() - start_time
        
        after_ = time.time()
        logger.log(_BEACON_LOG_LEVEL, '[beacon] inserting  [%.3f]' % (after_ - before_))
        before_ = time.time()
        
        commit_time = -1
        try:
            start_time = time.time()
            self.memdb.commit()
            commit_time = time.time() - start_time
        except Exception as e:
            logger.error('Commit fail ! [%s]' % e)
            logger.print_stack()
        
        after_ = time.time()
        logger.log(_BEACON_LOG_LEVEL, '[beacon] commit  [%.3f]' % (after_ - before_))
        before_ = time.time()
        
        # Stats
        self.raw_data_done_by_sec.update_avg(self.nb_raw_data_done_by_sec)
        timer = time.time() - start_time
        speed = 0 if self.nb_raw_data_done_by_sec == 0 or timer == 0 else self.nb_raw_data_done_by_sec / timer
        logger.debug("[process_raw_data] stats raw done[%4d]-[%4d]/[%d](by/s) keep todo [%d] insert[%5d] insert[%0.4f]s commit[%0.4f]s" % (
            self.nb_raw_data_done_by_sec,
            self.raw_data_done_by_sec.get_avg(),
            speed,
            len(raw_datas),
            self.nb_insert_done_by_sec,
            insert_time,
            commit_time
        ))
        self.nb_raw_data_done_by_sec = 0
        self.nb_insert_done_by_sec = 0
        
        after_ = time.time()
        logger.log(_BEACON_LOG_LEVEL, '[beacon] stats  [%.3f]' % (after_ - before_))
        
        # Maybe external modules raised 'objects' we should get them
        self.get_objects_from_from_queues()
        
        # Say to modules it's a new tick :)
        self.hook_point('tick')
        
        daemon_load = (time.time() - loop_time)
        logger.debug("[raw_data] daemon_load [%.3f]s." % daemon_load)
        self.sleep(1.0 - daemon_load)
    
    
    #  Main function, will loop forever
    def main(self):
        try:
            self.load_config_file()
            
            for line in self.get_header():
                logger.info(line)
            
            logger.info("[%s] Using working directory [%s]", self.name, os.path.abspath(self.workdir))
            
            # Look if we are enabled or not. If ok, start the daemon mode
            self.look_for_early_exit()
            self.do_daemon_init_and_start()
            self.load_modules_manager()
            
            self.http_daemon.register(self.interface)
            self.http_daemon.register(self.istats)
            self.http_daemon.register(self.iinfo)
            
            #  We wait for initial conf
            self.wait_for_initial_conf()
            if not self.new_conf:
                return
            
            self.setup_new_conf()
            
            # Do the modules part, we have our modules in self.modules
            # REF: doc/broker-modules.png (1)
            self.hook_point('load_retention')
            
            # Now the main loop
            self.do_mainloop()
        
        except Exception:
            logger.critical("The daemon did have an unrecoverable error. It must exit.")
            logger.critical("You can log a bug to your Shinken integrator with the error message:")
            logger.critical("%s" % (traceback.format_exc()))
            raise
    
    
    def ask_members(self, item_id):
        logger.debug("asking members list of [%s]", item_id)
        try:
            memdb = self.get_con_db()
        except ProviderNotReady as e:
            return e.message
        
        return_val = {'hosts': [], 'services': []}
        request_param = [item_id]
        request = """
            SELECT link_cluster_host_host_uuid
            FROM LINK_CLUSTER_HOST
            WHERE LINK_CLUSTER_HOST.link_cluster_host_cluster_uuid=?
        """
        c = memdb.cursor()
        return_val['hosts'] = [e[0] for e in c.execute(request, request_param)]
        
        request = """
            SELECT link_cluster_service_service_uuid
            FROM LINK_CLUSTER_SERVICE
            WHERE LINK_CLUSTER_SERVICE.link_cluster_service_cluster_uuid=?
        """
        c = memdb.cursor()
        return_val['services'] = [e[0] for e in c.execute(request, request_param)]
        
        return return_val
    
    
    def ask_checks(self, item_id):
        logger.debug("asking checks list of [%s]", item_id)
        try:
            memdb = self.get_con_db()
        except ProviderNotReady as e:
            return e.message
        
        request = """
            SELECT service_id
            FROM SERVICE
            WHERE SERVICE.service_link_host_uuid=?
        """
        request_param = [item_id]
        
        c = memdb.cursor()
        return {'checks': [e[0] for e in c.execute(request, request_param)]}
    
    
    def ask_metrics(self, item_id):
        logger.debug("asking metrics list of [%s]", item_id)
        try:
            memdb = self.get_con_db()
        except ProviderNotReady as e:
            return e.message
        
        request = """
            SELECT metric_name
            FROM RAW_DATA
            INNER JOIN METRIC ON RAW_DATA.raw_data_id = METRIC.metric_raw_data_id
            WHERE raw_data_item_id=?
            GROUP BY metric_name;
        """
        request_param = [item_id]
        
        c = memdb.cursor()
        return {'metrics': [e[0] for e in c.execute(request, request_param)]}
    
    
    def ask_item_states(self, item_id, item_type, since):
        logger.debug("asking item states [%s-%s] since [%s]", item_type, item_id, since)
        try:
            memdb = self.get_con_db()
        except ProviderNotReady as e:
            return e.message
        
        KEY_ITEM_ID = 0
        return_val = {'items': {}}
        request = ""
        param_since = time.time() - since if since > 0 else -1
        request_param = [item_id, param_since]
        
        if item_type == ITEM_TYPE_CLUSTER:
            request = """
                SELECT raw_data_item_id, raw_data_state_id, raw_data_state_type_id, raw_data_context_id, raw_data_last_chk
                FROM LINK_CLUSTER_HOST INNER JOIN RAW_DATA ON LINK_CLUSTER_HOST.link_cluster_host_host_uuid = RAW_DATA.raw_data_item_id
                WHERE LINK_CLUSTER_HOST.link_cluster_host_cluster_uuid=? AND RAW_DATA.raw_data_last_chk>?
                ORDER BY raw_data_last_chk DESC
            """
        elif item_type == ITEM_TYPE_HOST or item_type == ITEM_TYPE_SERVICE:
            request = """
                SELECT raw_data_item_id, raw_data_state_id, raw_data_state_type_id, raw_data_context_id, raw_data_last_chk
                FROM RAW_DATA
                WHERE raw_data_item_id=? AND RAW_DATA.raw_data_last_chk>?
                ORDER BY raw_data_last_chk DESC
            """
        
        c = memdb.cursor()
        execute_r = list(c.execute(request, request_param))
        for e in execute_r:
            info = {'state_id': e[1], 'state_type_id': e[2], 'context': e[3], 'ts': e[4]}
            if e[KEY_ITEM_ID] in return_val['items']:
                return_val['items'][e[KEY_ITEM_ID]].append(info)
            else:
                return_val['items'][e[KEY_ITEM_ID]] = [info]
        return return_val
    
    
    def ask_item_metrics(self, item_id, item_type, since):
        logger.debug("asking item metric [%s-%s] since [%s]", item_type, item_id, since)
        try:
            memdb = self.get_con_db()
        except ProviderNotReady as e:
            return e.message
        
        KEY_ITEM_ID = 0
        return_val = {'items': {}}
        request = ""
        param_since = time.time() - since if since > 0 else -1
        request_param = [item_id, param_since]
        
        if item_type == ITEM_TYPE_CLUSTER:
            request = """
                SELECT raw_data_item_id, metric_name, raw_data_last_chk, metric_value, metric_uom, metric_warning, metric_critical, metric_min, metric_max
                FROM LINK_CLUSTER_HOST
                INNER JOIN RAW_DATA ON LINK_CLUSTER_HOST.link_cluster_host_host_uuid = RAW_DATA.raw_data_item_id
                INNER JOIN METRIC ON RAW_DATA.raw_data_id = METRIC.metric_raw_data_id
                WHERE LINK_CLUSTER_HOST.link_cluster_host_cluster_uuid=? AND RAW_DATA.raw_data_last_chk>?
                ORDER BY raw_data_last_chk DESC
            """
        elif item_type == ITEM_TYPE_HOST or item_type == ITEM_TYPE_SERVICE:
            request = """
                SELECT raw_data_item_id, metric_name, raw_data_last_chk, metric_value, metric_uom, metric_warning, metric_critical, metric_min, metric_max
                FROM RAW_DATA
                INNER JOIN METRIC ON RAW_DATA.raw_data_id = METRIC.metric_raw_data_id
                WHERE raw_data_item_id=?  AND RAW_DATA.raw_data_last_chk>?
                ORDER BY raw_data_last_chk DESC
            """
        
        c = memdb.cursor()
        
        execute_r = list(c.execute(request, request_param))
        for e in execute_r:
            metric_name = e[1]
            item_id = e[KEY_ITEM_ID]
            metric_info = {'ts': e[2], 'value': e[3], 'uom': e[4], 'warn': e[5], 'crit': e[6], 'min': e[7], 'max': e[8]}
            if not item_id in return_val['items']:
                return_val['items'][item_id] = {}
            metric = return_val['items'][item_id].get(metric_name, [])
            return_val['items'][item_id][metric_name] = metric
            metric.append(metric_info)
        return return_val
    
    
    def ask_item_metric(self, item_id, item_type, metric, since):
        logger.debug("asking item metric [%s-%s][%s] since [%s]", item_type, item_id, metric, since)
        try:
            memdb = self.get_con_db()
        except ProviderNotReady as e:
            return e.message
        
        KEY_ITEM_ID = 0
        return_val = {'items': {}}
        request = ""
        param_since = time.time() - since if since > 0 else -1
        request_param = [item_id, param_since, metric]
        
        if item_type == ITEM_TYPE_CLUSTER:
            request = """
                SELECT raw_data_item_id, metric_name, raw_data_last_chk, metric_value, metric_uom, metric_warning, metric_critical, metric_min, metric_max
                FROM LINK_CLUSTER_HOST
                INNER JOIN RAW_DATA ON LINK_CLUSTER_HOST.link_cluster_host_host_uuid = RAW_DATA.raw_data_item_id
                INNER JOIN METRIC ON RAW_DATA.raw_data_id = METRIC.metric_raw_data_id
                WHERE LINK_CLUSTER_HOST.link_cluster_host_cluster_uuid=? AND RAW_DATA.raw_data_last_chk>? AND METRIC.metric_name=?
                ORDER BY raw_data_last_chk DESC
            """
        elif item_type == ITEM_TYPE_HOST or item_type == ITEM_TYPE_SERVICE:
            request = """
                SELECT raw_data_item_id, metric_name, raw_data_last_chk, metric_value, metric_uom, metric_warning, metric_critical, metric_min, metric_max
                FROM RAW_DATA
                INNER JOIN METRIC ON RAW_DATA.raw_data_id = METRIC.metric_raw_data_id
                WHERE raw_data_item_id=?  AND RAW_DATA.raw_data_last_chk>? AND METRIC.metric_name=?
                ORDER BY raw_data_last_chk DESC
            """
        
        c = memdb.cursor()
        
        execute_r = list(c.execute(request, request_param))
        for e in execute_r:
            metric_name = e[1]
            item_id = e[KEY_ITEM_ID]
            metric_info = {'ts': e[2], 'value': e[3], 'uom': e[4], 'warn': e[5], 'crit': e[6], 'min': e[7], 'max': e[8]}
            if not item_id in return_val['items']:
                return_val['items'][item_id] = {}
            metric = return_val['items'][item_id].get(metric_name, [])
            return_val['items'][item_id][metric_name] = metric
            metric.append(metric_info)
        return return_val
