#!/usr/bin/python
# -*- coding: utf-8 -*-
#
# Copyright (C) 2013-2018:
# This file is part of Shinken Enterprise, all rights reserved.
import re
from index import Index
from shinken.log import logger


class IndexedCollection(object):
    
    def __init__(self, hash_key=None):
        super(IndexedCollection, self).__init__()
        self.counter = 0
        self.indexes = {}
        self.inverse = {}
        self.data = {}
        self.hash_key = hash_key
    
    
    def __iter__(self):
        return self.data.itervalues()
    
    
    def __len__(self):
        return len(self.data)
    
    
    def append(self, item):
        self._append(item)
    
    
    def extend(self, items):
        if items:
            for item in items:
                self._append(item)
    
    
    def remove(self, search):
        item = self._find_one(search)
        if not item:
            raise KeyError('Item not found [%s]' % search)
        item_key = self._get_item_key(item)
        key_index = self.inverse[item_key]
        for index in self.indexes.itervalues():
            index.unindex_item(key_index)
        del self.inverse[item_key]
        del self.data[key_index]


    def regenerate_index(self, item):
        key_index = self.inverse[self._get_item_key(item)]
        for index in self.indexes.itervalues():
            index.unindex_item(key_index)
            index.index_item(item, key_index)
    
    
    def add_index(self, property_name, getter=None, on_list=False):
        self.indexes[property_name] = Index(property_name, getter, on_list)
        self.indexes[property_name].add_index(self.data)
    
    
    def find_one(self, search):
        return self._find_one(search)
    
    
    def find(self, search, only_count=False):
        return self._find(search, only_count)
    
    
    def count(self, search):
        return self._find(search, only_count=True)
    
    
    def _get_item_key(self, item):
        key = id(item)
        if hasattr(item, 'get_key'):
            key = item.get_key()
        elif self.hash_key:
            key = item[self.hash_key]
        return key
    
    
    def _append(self, item):
        key_index = self.counter
        self.data[key_index] = item
        item_key = self._get_item_key(item)
        self.inverse[item_key] = key_index
        for index in self.indexes.itervalues():
            index.index_item(item, key_index)
        self.counter += 1
    
    
    def _find(self, search, only_count=False):
        if not search:
            return len(self.data.values()) if only_count else self.data.values()
        
        search = search.copy()
        prop_already_search = []
        found_key_indexes = set(self.data.keys())
        for prop_name, value_search in ((prop_name, value_search) for prop_name, value_search in search.iteritems() if prop_name in self.indexes):
            if isinstance(value_search, dict):
                for search_key, search_lookup in value_search.iteritems():
                    if search_key == "$in":
                        # The $in will search equality for each search_lookup
                        tmp = set()
                        prop_already_search.append(prop_name)
                        index = self.indexes[prop_name]
                        for sub_search_lookup in search_lookup:
                            key_indexes = index.find(sub_search_lookup)
                            tmp |= set(key_indexes)
                        found_key_indexes &= tmp
                    elif search_key == "$ne":
                        # The $ne will exclude the search lookup
                        prop_already_search.append(prop_name)
                        index = self.indexes[prop_name]
                        key_indexes_to_exclude = index.find(search_lookup)
                        found_key_indexes -= set(key_indexes_to_exclude)
                    elif search_key == "$nin":
                        # The $nin will exclude each the search lookup
                        tmp = set()
                        prop_already_search.append(prop_name)
                        index = self.indexes[prop_name]
                        for sub_search_lookup in search_lookup:
                            key_indexes = index.find(sub_search_lookup)
                            tmp |= set(key_indexes)
                        found_key_indexes -= tmp
            elif not isinstance(value_search, re._pattern_type):
                prop_already_search.append(prop_name)
                index = self.indexes[prop_name]
                key_indexes = index.find(value_search)
                found_key_indexes &= set(key_indexes)
        
        for search_index in prop_already_search:
            del search[search_index]
        
        if only_count and not search:
            return len(found_key_indexes)
        
        found_items = map(lambda i: self.data[i], found_key_indexes)
        for prop_name, value_search in search.iteritems():
            logger.info('[IndexedCollection] Using fallback for prop [%s] value_search[%s]' % (prop_name, value_search))
            for i in xrange(len(found_items) - 1, -1, -1):
                found_item = found_items[i]
                if not self._recursive_search(found_item, prop_name, value_search):
                    found_items.pop(i)
        
        return len(found_items) if only_count else found_items
    
    
    def _find_one(self, search):
        item = self._find(search)
        if len(item) == 1:
            return item[0]
        elif len(item) > 1:
            raise ValueError('Too many value found')
        return None
    
    
    def _recursive_search(self, sub_dict, full_prop_name, value_search):
        full_prop_name = full_prop_name.split('.', 1)
        prop = full_prop_name[0]
        if len(full_prop_name) > 1:
            sub_props = full_prop_name[1]
            if sub_dict.get(prop, None):
                return self._recursive_search(sub_dict[prop], sub_props, value_search)
            else:
                return False
        else:
            if isinstance(value_search, dict):
                for search_key, search_lookup in value_search.iteritems():
                    if search_key == "$ne":
                        return sub_dict.get(prop, '') != value_search
                    else:
                        raise NotImplementedError("'%s' filter is not implement in %s find method" % (value_search, self.__class__.__name__))
            elif isinstance(value_search, re._pattern_type):
                return bool(value_search.search(sub_dict.get(prop, '')))
            else:
                return sub_dict.get(prop, '') == value_search
