#!/usr/bin/python
# -*- coding: utf-8 -*-

# Copyright (C) 2013-2018:
# This file is part of Shinken Enterprise, all rights reserved.

import datetime
import re

from shinken.log import logger

from ...dao.def_items import DEF_ITEMS, ITEM_STATE, ITEM_TYPE

### Will be populated by the UI with it's own value
app = None


# Our page
def get_page():
    return user_login()


def user_login():
    user = app.get_user_auth()
    err = app.request.GET.get('error', None)
    r = {'app': app, 'error': err}
    
    for mod in app.modules_manager.get_internal_instances():
        if getattr(mod, 'login_placeholder', ''):
            r['login_placeholder'] = mod.login_placeholder
            break
    
    if user:
        if not user.is_admin() and not user.is_expert():
            err = app._('login.isnt_admin') % user.get('contact_name', 'unknown')
        else:
            app.redirect("/main")
    return r


def user_login_redirect():
    app.redirect("/user/login")
    return {}


def _clean_cookie():
    app.response.delete_cookie('user', secret=app.auth_secret, path='/')


def _set_valid_cookie(user_id):
    expires = datetime.datetime(2100, 1, 1)
    # expires = 'Fri, 01 Jan 2100 00:00:00 GMT'
    app.response.set_cookie('user', user_id, secret=app.auth_secret, path='/', expires=expires, httponly=True)
    

def user_logout():
    # To delete it, send the same, with different date
    user_id = app.request.get_cookie("user", secret=app.auth_secret)
    _clean_cookie()
    app.redirect("/user/login")
    return {}


def user_auth():
    logger.debug('[Auth] A user try to auth')
    
    login = app.request.forms.get('shinken_login', '').decode('utf-8', 'ignore')
    password = app.request.forms.get('password', '').decode('utf-8', 'ignore')
    c = app.check_auth(login, password)
    is_valid = c and c.get('enabled', '1') == '1'
    
    error_txt = ''
    is_bad = False
    
    if is_valid:
        contact_name = c.get_name()
        _set_valid_cookie(c['_id'])
        
        logger.debug('[Auth] user [%s] is logged in' % contact_name)
        
        if not c.is_admin() and not c.is_expert():
            _clean_cookie()
            error_txt = app._('login.isnt_admin') % login
            is_bad = True
    else:
        _clean_cookie()
        is_bad = True
        error_txt = app._('login.bad_auth')
    
    return {'app': app, 'is_bad': is_bad, 'error_txt':error_txt}


# manage the /. If the user is known, go to problems page.
# Should be /dashboard in the future. If not, go login :)
def get_root():
    user = app.request.get_cookie("user", secret=app.auth_secret)
    if user:
        app.redirect("/main")
    elif app.remote_user_variable in app.request.headers and app.remote_user_enable == '1':
        user_name = app.request.headers[app.remote_user_variable]
        
        if app.remote_user_case_sensitive:
            user_auth = app.datamanagerV2.find_item_by_name(user_name, ITEM_TYPE.CONTACTS, ITEM_STATE.STAGGING)
        else:
            where = {DEF_ITEMS[ITEM_TYPE.CONTACTS]['key_name']: re.compile(re.escape(user_name), re.IGNORECASE)}
            user_auth = app.datamanagerV2.find_items(ITEM_TYPE.CONTACTS, ITEM_STATE.STAGGING, where=where)
            user_auth = user_auth[0] if user_auth else None
        
        if not user_auth:
            print "Warning: You need to have a contact having the same name as your user %s" % user_name
            app.redirect("/user/login")
        else:
            _set_valid_cookie(user_auth.get_uuid())
            app.redirect("/main")
    else:
        app.redirect("/user/login")


def login_mobile():
    user = app.get_user_auth()
    if user:
        app.redirect("/main")
    
    err = app.request.GET.get('error', None)
    
    return {'error': err}


pages = {
    user_login         : {'routes': ['/user/login', '/user/login/'], 'view': 'login', 'static': True, 'wrappers': []},
    user_login_redirect: {'routes': ['/login'], 'static': True, 'wrappers': []},
    user_auth          : {'routes': ['/user/auth'], 'view': 'auth', 'method': 'POST', 'static': True, 'wrappers': []},
    user_logout        : {'routes': ['/user/logout', '/logout'], 'static': True, 'wrappers': []},
    get_root           : {'routes': ['/'], 'static': True, 'wrappers': []},
    login_mobile       : {'routes': ['/mobile', '/mobile/'], 'view': 'login_mobile', 'static': True, 'wrappers': []}
}
