#!/usr/bin/python
# -*- coding: utf-8 -*-
#
# Copyright (C) 2013-2020
# This file is part of Shinken Enterprise, all rights reserved.

import time
from collections import OrderedDict
from operator import itemgetter

import shinkensolutions.shinkenjson as json
from discovery import get_discovery_conf, new_discovery_conf, post_discovery_conf, delete_discovery_conf, get_ip_ranges
from discovery import set_app as discovery_set_app
from shinken.log import logger
from shinkensolutions.netaddr import iter_nmap_range, valid_nmap_range
from shinken.misc.type_hint import Optional
from ...business.source.source import Source
from ...dao.dataprovider.dataprovider_mongo import DataProviderMongo
from ...dao.def_items import ITEM_TYPE, ITEM_STATE, DEF_ITEMS, METADATA, SERVICE_OVERRIDE
from ...front_end.helper import natural_keys, display_service_override_value

app = None

NOT_TO_DISPLAY_IN_SOURCES = set(['_id', '@metadata', 'last_modification', 'overwrited_protected', 'work_area_info', 'presence_protection', 'to_merge', 'source_strong_overwrite', 'source_order', '__SYNC_IDX__'])


def set_app(_app):
    global app
    app = _app
    discovery_set_app(_app)


# ------------------
# Public
# ------------------
def get_source(source_name):
    # Index all element list to a specific source.
    
    # Check Source
    if not source_name:
        return app.abort(400, 'Missing source_name param')
    
    user = _get_user_auth()
    source = _get_source(source_name)
    
    if source is None:
        return app.abort(404, app._('source.no_source_with_this_name') % source_name)
    
    source_conf_form = _get_source_conf_form(source)
    
    template_mapping_form = app.source_controller.get_analyzer_template_mapping(source)
    source_list, total_elements = _get_source_elements(source_name)
    last_runs = _get_last_runs(source_name)
    last_runs = sorted(last_runs, key=lambda key: key['synchronization_time'], reverse=True)
    last_run_number = _get_last_run_number(source_name)
    
    filters = _get_params_filter()
    
    discovery_confs_cursor = get_ip_ranges(source_name)
    # add ip range as string list in each conf 
    discovery_confs = []
    for conf in discovery_confs_cursor:
        conf_string_ip_ranges = conf.get('iprange', '')
        ranges = set()
        for conf_string_ip_range in conf_string_ip_ranges.split(' '):
            if valid_nmap_range(conf_string_ip_range):
                ip_range = [str(s) for s in iter_nmap_range(conf_string_ip_range)]
                if '/' in conf_string_ip_range and len(ip_range) > 2:
                    # remove the first object and le last one (network and broadcast)
                    ip_range.pop(0)  # pop the first
                    ip_range.pop()  # pop the last
                
                ranges = ranges.union(set(ip_range))
        conf['string_iprange'] = list(ranges)
        discovery_confs.append(conf)
    
    # Get the data in json format for source data up to date
    sources_json = app.get_api_sources_from_backend()
    source_json = sources_json.get(source_name, {})
    nb_active_conf = 0
    nb_total_conf = 0
    if source_name in ('server-analyzer', 'discovery'):
        nb_active_conf, nb_total_conf = get_ratio_conf_enabled(source_name)
    source_list = sorted(source_list, key=itemgetter('@type', '@name'))
    
    display_item_types = [{'type': _type, 'display': app._('type.%s' % _type)} for _type in ITEM_TYPE.ALL_TYPES]
    display_item_types = sorted(display_item_types, key=itemgetter('display'))
    return {
        'app'                   : app,
        'user'                  : user,
        'source'                : source,
        'source_json'           : source_json,
        'list'                  : source_list,
        'total_elements'        : total_elements,
        'filters'               : filters,
        'last_runs'             : last_runs,
        'last_run_number'       : last_run_number,
        'discovery_confs'       : discovery_confs,
        'source_conf'           : source_conf_form,
        'template_mappping_form': template_mapping_form,
        'is_admin'              : json.dumps(user.is_admin()),
        'number_active_conf'    : nb_active_conf,
        'number_total_conf'     : nb_total_conf,
        'display_item_types'    : display_item_types
    }


def update_source_configuration(source_name):
    forms = app.request.forms
    source = _get_source(source_name)
    source_conf = app.source_controller.get_source_configuration(source)
    # get the form sent to the html to compare with the incoming values
    source_conf_form = _get_source_conf_form(source)
    
    for key, value in forms.iteritems():
        # split the keys with __ to find the config_type and fields
        config_type_and_field = key.split('__')
        config_type = config_type_and_field[0]
        field = config_type_and_field[1]
        
        # check if the field is protected, if yes we need to frontend uncipher it
        if value and source.get_configuration_fields()[config_type][field].get('protected', False):
            value = app.frontend_cipher._uncipher_value(value, source_conf[config_type][field])
        # Manage the 'on'/'' bool values as JS sent to us
        elif source.get_configuration_fields()[config_type][field].get('type', '') == 'checkbox':
            value = True if value == 'on' else False
        source_conf.get(config_type, {})[field] = value
        # mark the seen field so we will be able to find the fields that never come back (like unchecked checkbox)
        source_conf_form[config_type][field]['already_seen'] = True
    
    # find the fields that never come back and set a default value in source_conf depending on form type
    for config_type, fields in source_conf_form.iteritems():
        for field_name, properties in fields.iteritems():
            if not properties.get('already_seen', False):
                if properties.get('type', '') == 'checkbox':
                    # If a checkbox is unchecked, no value is sent, so we have to manually set it to false here
                    if not (source_name == 'listener-shinken' and field_name == 'authentication'):
                        source_conf[config_type][field_name] = False
    app.source_controller.set_source_configuration(source, source_conf)
    return app.redirect("/sources/%s?tab=tab-configuration" % source_name)


def update_template_mapping(source_name):
    forms = app.request.forms
    source = _get_source(source_name)
    template_mapping = app.source_controller.get_analyzer_template_mapping(source)
    for key, value in forms.iteritems():
        # split the keys with __ to find the config_type and fields
        config_type_and_field = key.split('__')
        group = config_type_and_field[0]
        template_name = config_type_and_field[1]
        group_in_template_mapping = template_mapping[group]
        if not group_in_template_mapping:
            template_mapping[group] = {'values': {}}
        group_in_template_mapping.pop('display_name', None)
        overload = True if value != "" else False
        template_mapping[group]['values'][template_name] = {'value': value, 'overload': overload}
    app.source_controller.set_template_mapping(source, template_mapping)
    return app.redirect("/sources/%s?tab=tab-analyzer-template-mapping" % source_name)


def post_analyzer_job_hosts(analyzer_name):
    form = app.request.forms
    hosts_to_analyze_state = form.get('hosts_to_analyze_state', '').decode('utf-8', 'ignore')
    
    r = get_source(analyzer_name)
    hosts_to_analyze = []
    if "hosts_to_analyze_uuids" in form:
        hosts_to_analyze_uuids = form.get('hosts_to_analyze_uuids').decode('utf-8', 'ignore').split(',')
        for host_uuid in hosts_to_analyze_uuids:
            # Find the real host from the datamanager, note: CAN be a None!
            host = app.datamanagerV2.find_item_by_id(host_uuid, ITEM_TYPE.HOSTS, hosts_to_analyze_state)
            entry = {
                'host_uuid': host_uuid,
                'name'     : '',
                'address'  : '',
                'state'    : 'stagging',
            }
            if host is not None:
                entry['name'] = host['host_name']
                entry['address'] = host.get('address', '')
                hosts_to_analyze.append(entry)
    
    elif 'ip_to_analyze' in form:
        ip_to_analyze = form.get('ip_to_analyze', "").decode('utf-8', 'ignore').split(',')
        discovery_name = form.get('discovery_name', '').decode('utf-8', 'ignore')
        if discovery_name:
            r['hosts_to_analyze'] = []
            dc = next((dc for dc in r['discovery_confs'] if dc['discovery_name'] == discovery_name), None)
            if not dc:
                return r
            if not dc['enabled']:
                if 'dc_disabled' not in r:
                    r['dc_disabled'] = []
                r['dc_disabled'].append(discovery_name)
                return r
        
        step = 100
        paginate_ip_list = [ip_to_analyze[i:i + step] for i in range(0, len(ip_to_analyze), step)]
        for page_ips in paginate_ip_list:
            
            # try now to find hosts from workarea and staging with this ip addresses
            where = {'address': {'$in': page_ips}}
            stagging_hosts = app.datamanagerV2.find_merge_state_items(ITEM_TYPE.HOSTS, item_states=[ITEM_STATE.WORKING_AREA, ITEM_STATE.STAGGING], where=where, lookup={'host_name': 1, 'address': 1, '_id': 1})
            for ip in page_ips:
                hosts_with_ip = [h for h in stagging_hosts if h['address'] == ip]
                if hosts_with_ip:
                    for stagging_host in hosts_with_ip:
                        entry = {
                            'host_uuid': stagging_host['_id'],
                            'name'     : stagging_host['host_name'],
                            'address'  : stagging_host['address'],
                            'state'    : 'stagging',
                        }
                        hosts_to_analyze.append(entry)
                else:
                    entry = {
                        'host_uuid': ip,
                        'name'     : ip,
                        'address'  : ip,
                        'state'    : 'discovery',
                    }
                    hosts_to_analyze.append(entry)
    
    hosts_to_analyze.sort(key=lambda x: natural_keys(x['address']))
    r['hosts_to_analyze'] = hosts_to_analyze
    return r


def _get_params_filter():
    req_filter = app.request.GET.get('filter', '').strip()
    filters = {}
    if req_filter:
        for filter_type in req_filter.split('~'):
            split = filter_type.split(':', 1)
            filter_tag = split[0].lower()
            filter_value = split[1].decode('utf8', 'ignore')
            filters[filter_tag] = filter_value
    return filters


def __cipher_source_element(item, item_type, keys, user):
    item = app.frontend_cipher.cipher(item, item_type=item_type, user=user)
    protected_class_dict = {}
    for item_property in keys:
        if app.frontend_cipher.match_protected_property(item_property, item_type, user):
            value = app._('element.password_protected')
            protected_class = "shinken-protected-password"
        elif item_property in ('update_date', 'import_date'):
            value = app.helper.print_date(item[item_property])
            protected_class = ""
        elif item_property == SERVICE_OVERRIDE:
            value, protected = display_service_override_value(item[item_property], user)
            protected_class = "shinken-protected-password" if protected else ""
        else:
            value = item[item_property]
            protected_class = ""
        item[item_property] = value
        protected_class_dict[item_property] = protected_class
    for key in keys:
        item[key] = {'value': item[key], 'protected_class': protected_class_dict[key]}
    return item


def _add_more_information_to_source_element(item, information):
    if information is None:
        return item
    for property_name, property_value in item.iteritems():
        property_value['more_informations'] = information.get(property_name, '')
    return item


def __pre_import_data_formatter(data):
    if not data:
        return None
    formatted_data = {}
    for key in data:
        formatted_data[key] = {'value': data[key], 'protected_class': ''}
    return formatted_data


def get_source_element(source_name, item_type, _id, last_run_number, _hash):
    user = app.get_user_auth()
    # Show one specific element.
    logger.debug("[sources][get_source_element] - _id = %s" % _id)
    logger.debug("[sources][get_source_element] - source_name = %s" % source_name)
    logger.debug("[sources][get_source_element] - item_type = %s" % item_type)
    logger.debug("[sources][get_source_element] - last_run_number from front = %s" % last_run_number)
    if not source_name or not item_type or not _id or not last_run_number:
        return app.abort(400, 'Missing source_name or last_run_id or item_type or _id parameter')
    if item_type not in DEF_ITEMS.keys():
        return app.abort(400, 'Invalid type')
    
    source = _get_source(source_name)
    last_run_number_from_db = _get_last_run_number(source_name)
    logger.debug("[sources][get_source_element] - last_run_number from db = %s" % last_run_number_from_db)
    provider_mongo = DataProviderMongo(app.mongodb_db, app.database_cipher)
    if int(last_run_number) != last_run_number_from_db:
        item = provider_mongo.find_items(item_type, item_state=ITEM_STATE.RAW_SOURCES, item_source=source.get_name(), where={'@metadata.hash': _hash})
        if item:
            # because find_items return a list but there only one element with this @metadata.hash.
            item = item[0]
    else:
        item = provider_mongo.find_item_by_id(_id, item_type, item_state=ITEM_STATE.RAW_SOURCES, item_source=source.get_name())
    
    if not item:
        has_error = True
        error_text = app._('source.refresh_list_error')
        return {
            'has_error' : has_error,
            'error_text': error_text,
            'item'      : item,
            'app'       : app
        }
    
    import_warnings = METADATA.get_metadata(item, METADATA.IMPORT_WARNINGS, [])
    import_errors = METADATA.get_metadata(item, METADATA.IMPORT_ERRORS, [])
    
    origin_item_info = METADATA.get_metadata(item, METADATA.ORIGIN_ITEM_INFO)
    origin_item_info = source.build_origin_item_info(origin_item_info)
    
    has_more_information_on_element = METADATA.get_metadata(item, 'more_informations_on_element', {})
    
    for key in NOT_TO_DISPLAY_IN_SOURCES:
        item.pop(key, None)
    keys = item.keys()
    keys.sort(key_sort)
    item = __cipher_source_element(item, item_type, keys, user)
    item = _add_more_information_to_source_element(item, has_more_information_on_element)
    res = {
        'has_error'                       : False,
        'app'                             : app,
        'user'                            : user,
        'item'                            : item,
        'item_type'                       : item_type,
        'import_warnings'                 : import_warnings,
        'import_errors'                   : import_errors,
        'has_more_informations_on_element': has_more_information_on_element,
        'origin_item_info'                : origin_item_info,
        'source_name'                     : source_name,
        'keys'                            : keys
    }
    
    return res


def get_source_last_run(source_name, source_time, source_type):
    result_title_info = ""
    logger.debug("[sources][get_source_last_run] - source_name = %s, source_time = %s" % (source_name, source_time))
    is_latest_run = True if app.request.GET.get('is-latest-run', None) == '1' else False
    user = _get_user_auth()
    if not source_name or not source_time:
        return app.abort(400, 'Missing sname or source_time parameter')
    if source_type == "listener":
        result_title_info = app._('source.request_received')
    if source_type == "analyzer":
        result_title_info = app._('source.analyzed_elements')
    source_time = int(source_time)
    run = app.last_synchronizations.find_one({'source_name': source_name, 'synchronization_time': source_time})
    if run is not None:
        run['is_latest_run'] = is_latest_run
    # we return values for the template (view). But beware, theses values are the
    # only one the template will have, so we must give it an app link and the
    # user we are logged with (it's a contact object in fact)
    return {'app': app, 'user': user, 'run': run, 'result_title_info': result_title_info}


def remove_source_element(source_name, item_type, _id):
    if not source_name or not item_type or not _id:
        return app.abort(400, 'Missing source_name or element_name or _id parameters')
    if item_type not in DEF_ITEMS.keys():
        return app.abort(400, 'Invalid type')
    
    app.api_remove_source_item_to_backend(source_name, [(item_type, _id)])
    return app.redirect("/sources/%s#table-source-js" % source_name)


def remove_source_elements(source_name):
    if not source_name:
        return app.abort(400, 'Missing source_name or element_name or _id parameters')
    
    items = json.loads(app.request.forms.get('items', ''))
    if not items:
        return app.abort(400, 'Missing items')
    
    info_items = [(item_type, item_id) for item_id, item_type in items]
    app.api_remove_source_item_to_backend(source_name, info_items)
    return app.redirect("/sources/%s#table-source-js" % source_name)


def get_configuration():
    pass


# ------------------
# Private
# ------------------
def _get_user_auth():
    user = app.get_user_auth()
    if not user:
        return app.abort(403, 'You need to be logued to call this!')
    return user


def _get_source(source_name):
    # type: (str) -> Optional[Source]
    source = next((s for s in app.sources if s.get_name() == source_name.strip()), None)  # type: Source
    if not source:
        return None
    source.update_from_last_synchronization(app.last_synchronization)
    return source


# get the source os_fields and add the values from db
# use the frontend cipher to hide the protected fields
def _get_source_conf_form(source):
    source_conf = app.source_controller.get_source_configuration(source)
    if not source_conf:
        return {}
    source_form = OrderedDict({})
    for config_type, fields in source.get_configuration_fields().iteritems():
        if config_type not in source_form:
            source_form[config_type] = OrderedDict({})
        for field_name, properties in fields.iteritems():
            properties = properties.copy()
            value = source_conf[config_type][field_name]
            if value and properties['protected']:
                value = app.frontend_cipher._cipher_value(value)
            properties['value'] = value
            source_form[config_type][field_name] = properties
    return source_form


def _get_source_elements(source_name):
    provider_mongo = DataProviderMongo(app.mongodb_db, app.database_cipher)
    source = _get_source(source_name)
    is_listener_or_analyzer = source.type in ('listener', 'analyzer')
    _list = []
    for item_type in DEF_ITEMS.iterkeys():
        key_name = DEF_ITEMS[item_type]['key_name']
        data = provider_mongo.find_items(
            item_type,
            item_state=ITEM_STATE.RAW_SOURCES,
            item_source=source_name,
            lookup={
                key_name                   : 1,
                'service_description'      : 1,
                '_SYNC_KEYS'               : 1,
                'name'                     : 1,
                'import_date'              : 1,
                '@metadata.update_date'    : 1,
                '@metadata.import_warnings': 1,
                '@metadata.import_errors'  : 1,
                '@metadata.hash'           : 1
            }
        )
        for e in data:
            e['@name'] = e.get(key_name, e.get('service_description', '--- Missing Name ----'))
            e['@type'] = item_type
            e['@update_date'] = METADATA.get_metadata(e, METADATA.UPDATE_DATE, e.get('import_date', time.time()))
            e['@import_warnings'] = METADATA.get_metadata(e, METADATA.IMPORT_WARNINGS, None)
            e['@import_errors'] = METADATA.get_metadata(e, METADATA.IMPORT_ERRORS, None)
            e['@hash'] = METADATA.get_metadata(e, METADATA.HASH, None)
            sync_keys = e.get('_SYNC_KEYS', set())
            if isinstance(sync_keys, basestring):
                sync_keys = sync_keys.split(',')
                e['_SYNC_KEYS'] = sync_keys
            e.pop('@metadata', None)
        if data:
            _list.extend(data)
    if is_listener_or_analyzer:
        _list.sort(key=lambda ki: ki.get('@update_date', ''), reverse=True)
    else:
        _list.sort(key=lambda ki: ki.get('@type', ''))
    total_elements = len(_list)
    _list = get_list_filtered(_list)
    return _list, total_elements


def _get_last_runs(source_name):
    return list(app.last_synchronizations.find({'source_name': source_name}, {'synchronization_time': 1, 'state': 1, 'run_detail': 1}))


def _get_last_run_number(source_name):
    current_import_nb = app.get_api_source_run_number(source_name)
    return current_import_nb


def _filter(_list, filters):
    logger.debug("[sources][filter]")
    filtered_list = []
    for item in _list:
        # logger.debug("[sources][filter] - item %s" % item)
        all_match = True
        for filter_type in filters.split('~'):
            split = filter_type.split(':', 1)
            filter_tag = split[0].lower()
            filter_value = split[1].lower().decode('utf8', 'ignore')
            
            # logger.debug("[sources][filter] %s %s" %(filter_tag, filter_value))
            # We filter by column
            match = False
            if filter_tag == 'class':
                col_value = item.get('@type')
                if (col_value != '') and (filter_value == col_value.lower()):
                    # logger.debug("[sources][filter]-class %s " % filter_value)
                    match = True
            elif filter_tag == 'name':
                col_value = item.get('@name')
                # logger.debug("[sources][filter]-name col_value =  %s" % col_value)
                # logger.debug("[sources][filter]-name filter_value = %s" % filter_value)
                # logger.debug("[sources][filter]-name cond == %s" % (filter_value in col_value))
                if (col_value != '') and (filter_value in col_value.lower()):
                    # logger.debug("[sources][filter]-name %s" % filter_value)
                    match = True
            elif filter_tag == 'sync':
                # logger.debug("[sources][filter]-sync %s %s" % (filter_tag, filter_value))
                col_value = ', '.join(item.get('_SYNC_KEYS', []))
                if (col_value != '') and (filter_value in col_value.lower()):
                    # logger.debug("[sources][filter]-sync %s" % filter_value)
                    match = True
            elif filter_tag == 'status':
                _warning = item.get('@import_warnings', None)
                _error = item.get('@import_errors', None)
                
                if not _warning and not _error and filter_value == "ok":
                    match = True
                elif _warning and filter_value == "warning":
                    match = True
                elif _error and filter_value == "error":
                    match = True
            
            all_match &= match
        
        if all_match:
            filtered_list.append(item)
    
    return filtered_list


def get_list_filtered(_list):
    req_filter = app.request.GET.get('filter', '').strip()
    if req_filter:
        _list = _filter(_list, req_filter)
    return _list


def key_sort(s1, s2):
    if s1.startswith('_') and s2.startswith('_'):
        return cmp(s1.lower(), s2.lower())
    if s1.startswith('_') and not s2.startswith('_'):
        return 1
    if not s1.startswith('_') and s2.startswith('_'):
        return -1
    return cmp(s1, s2)


def get_tagger(sname):
    # First we look for the user sid
    # so we bail out if it's a false one
    user = app.get_user_auth()
    if not user:
        app.redirect("/user/login")
        return
    taggers = [s for s in app.taggers]
    tagger = None
    for s in taggers:
        if s.get_name() == sname.strip():
            tagger = s
    return {'app': app, 'user': user, 'tagger': tagger}


def get_api_sources():
    return app.get_api_sources_from_backend()


def get_api_unique_source(sname):
    sources = app.get_api_sources_from_backend()
    return sources.get(sname, None)


def set_api_sources():
    user = app.get_user_auth()
    if not user:
        return app.abort(401, "not authorized")
    if not user.get('is_admin', '0') == '1':
        return app.abort(401, "not authorized")
    app.response.content_type = 'application/json'
    sname = app.request.GET.get('sname', '')
    enabled = app.request.GET.get('enabled', '')
    if not sname or not enabled:
        return app.abort(400, 'Missing parameter')
    return json.dumps(app.set_source_enabled_to_backend(sname, enabled))


def api_force_source(sname):
    user = app.get_user_auth()
    if not user:
        return app.abort(401, "not authorized")
    if not user.get('is_admin', '0') == '1':
        return app.abort(401, "not authorized")
    app.response.content_type = 'application/json'
    return json.dumps(app.api_force_source(sname))


def api_clean_source(sname):
    user = app.get_user_auth()
    if not user:
        return app.abort(401, "not authorized")
    if not user.get('is_admin', '0') == '1':
        return app.abort(401, "not authorized")
    app.response.content_type = 'application/json'
    return json.dumps(app.api_clean_source(sname))


def set_api_sources_order():
    app.response.content_type = 'application/json'
    order = app.request.GET.get('order', '')
    if not order:
        return app.abort(400, 'Missing parameter')
    return json.dumps(app.set_sources_order_to_backend(order))


def set_api_sources_conf():
    user = app.get_user_auth()
    if not user:
        return app.abort(401, "not authorized")
    if not user.get('is_admin', '0') == '1':
        return app.abort(401, "not authorized")
    app.response.content_type = 'application/json'
    _source_name = app.request.GET.get('source_name', '')
    _conf_id = app.request.GET.get('conf_id', '')
    _enabled = app.request.GET.get('enabled', '')
    
    if not _source_name or not _conf_id or not _enabled:
        return app.abort(400, 'Missing parameter')
    return json.dumps(app.set_source_conf_enabled_to_backend(_source_name, _conf_id, _enabled))


def get_ratio_conf_enabled(source_name):
    if source_name not in ('server-analyzer', 'discovery'):
        app.abort(400, "The source %s does't manage a switch button" % source_name)
    
    conf_name = 'discovery_confs'
    nb = app.mongodb_db[conf_name].find({"enabled": True, 'source_name': source_name}).count()
    nb_total = app.mongodb_db[conf_name].find({'source_name': source_name}).count()
    return nb, nb_total


# --------------
# DAO - Data Acces Object
# --------------

pages = {
    # Routes Views Page
    get_source                 : {'routes': ['/sources/:source_name'], 'view': 'source', 'static': True},
    update_source_configuration: {'routes': ['/sources/:source_name/configuration'], 'method': 'POST', 'static': True, 'wrappers': ['auth']},  # Analyzer is a POST to get the hosts to analyze, but in the end it's just a source page with one tab more
    update_template_mapping    : {'routes': ['/sources/:source_name/template-mapping'], 'method': 'POST', 'static': True, 'wrappers': ['auth']},
    
    # Analyser is a POST to get the hosts to analyse, but in the end it's just a source page with one tab more
    post_analyzer_job_hosts    : {'routes': ['/analyzers/:analyzer_name'], 'method': ('POST', 'GET'), 'view': 'source', 'static': True},
    
    # Routes Partials Views
    get_source_element         : {'routes': ['/sources/:source_name/:last_run_number/elements/:item_type/id/:_id/hash/:_hash'], 'view': '_last_import_element_detail', 'static': True},
    get_source_last_run        : {'routes': ['/sources/:source_type/:source_name/times/:source_time'], 'view': '_show_old_run', 'static': True},
    remove_source_element      : {'routes': ['/sources/:source_name/elements/:item_type/id/:_id/delete'], 'static': True, 'wrappers': ['auth']},
    remove_source_elements     : {'routes': ['/sources/:source_name/elements/delete'], 'method': 'DELETE', 'wrappers': ['auth', 'json']},
    
    # get_source_last_run: {'routes': ['/show_old_run'], 'view': 'sync_run_results', 'static': True},
    
    # Routes Actions / no template return
    
    # Old Routes
    get_tagger                 : {'routes': ['/tagger/:sname'], 'view': 'tagger', 'static': True},
    
    get_api_sources            : {'routes': ['/api/sources'], 'static': True, 'wrappers': ['auth']},
    get_api_unique_source      : {'routes': ['/api/sources/unique_source/:sname'], 'static': True, 'wrappers': ['auth']},
    set_api_sources            : {'routes': ['/api/sources/setenabled'], 'static': True, 'wrappers': ['auth', 'transaction']},
    api_force_source           : {'routes': ['/api/sources/forceimport/:sname'], 'static': True},
    api_clean_source           : {'routes': ['/api/sources/cleanimport/:sname'], 'static': True, 'wrappers': ['auth', 'transaction']},
    
    set_api_sources_order      : {'routes': ['/api/sources/setorder'], 'static': True, 'wrappers': ['auth', 'transaction']},
    set_api_sources_conf       : {'routes': ['/api/sources/conf/setenabled'], 'static': True, 'wrappers': ['auth', 'transaction']},
    
    # discovery conf independent from source
    get_discovery_conf         : {'routes': ['/sources/:sname/discovery/:id'], 'view': 'source_discovery_conf', 'static': True},
    new_discovery_conf         : {'routes': ['/sources/:sname/discovery/add/'], 'view': 'source_discovery_conf', 'static': True},
    post_discovery_conf        : {'routes': ['/sources/:sname/discovery/save/'], 'method': 'POST', 'view': None, 'static': True, 'wrappers': ['auth', 'transaction']},
    delete_discovery_conf      : {'routes': ['/sources/:sname/discovery/delete/'], 'method': 'POST', 'view': None, 'static': True, 'wrappers': ['auth', 'transaction']},
    
}
