#!/usr/bin/python
# Copyright (C) 2013:
#    Gabes Jean, j.gabes@shinken-solutions.com
#
# This file is part of Shinken Enterprise, all rights reserved.

import base64
import imp
import json
import logging
import optparse
import os
import glob

try:
    import pwd
    import grp
except ImportError:
    # don't expect to have this on windows :)
    pwd = grp = None

import re
import shutil
import subprocess
import sys
import tarfile
import time
import traceback
import uuid
from datetime import datetime
from cStringIO import StringIO

import pymongo
from pymongo.connection import Connection
from pymongo.errors import BulkWriteError
from shinken.objects.config import Config
from shinken.objects.module import Module as ShinkenModuleDefinition
from shinkensolutions.crypto import AESCipher
from shinkensolutions.localinstall import POSSIBLE_DAEMONS, get_local_instances_for_type

try:
    import shinken
except ImportError:
    print 'Cannot import shinken lib, please install it before launching this tool'
    sys.exit(2)

from shinkensolutions.localinstall import do_import_data_history_sanatize
from shinken.log import logger

formatter = logging.Formatter('[%(asctime)s] [%(name)s] %(message)s')
DEFAULT_LOG_FILENAME = '/var/log/shinken/sanitize.log'

orig_stdout = sys.stdout


class LogFile(object):
    def __init__(self, name=None):
        self.console_handler = logging.StreamHandler(orig_stdout)
        self.file_handler = None
        
        self.logger = logging.getLogger(name)
        self.logger.setLevel(logging.DEBUG)
        
        self.logger.addHandler(self.console_handler)
        self.encoding = None
    
    
    def set_console_level(self, level):
        self.console_handler.setLevel(level)
    
    
    def set_file_handler(self, path):
        self.file_handler = logging.handlers.RotatingFileHandler(path, maxBytes=1024 * 1024, backupCount=5)
        self.file_handler.setFormatter(formatter)
        self.logger.addHandler(self.file_handler)
    
    
    def write(self, msg, level=logging.DEBUG):
        # orig_stdout.write(msg)
        # msg = msg.strip()
        # s_time = datetime.datetime.now().strftime('[%Y-%b-%d %H:%M]')
        # s = '%s %s' % (s_time, msg)
        self.logger.log(level, msg.strip())
    
    
    def flush(self):
        for handler in self.logger.handlers:
            handler.flush()


sanitize_log = LogFile('Sanitize')
sys.stdout = sanitize_log
sys.stderr = sanitize_log
log = sanitize_log.logger

CONTEXT_PATH = '/var/lib/shinken/context.json'
VERSION = '2.03.02'

if os.path.exists(CONTEXT_PATH):
    context = json.loads(open(CONTEXT_PATH, 'r').read())
    VERSION = context.get('current_version', VERSION).split('-')[0]

if not os.getuid() == 0:
    print "ERROR: this script must be run as root"
    sys.exit(2)

# ENV variable to force unset at all cost
unset_variables = ['http_proxy', 'https_proxy', 'ftp_proxy']
for v in os.environ:
    if v.lower() in unset_variables:
        os.environ[v] = ''

success = {}
warnings = {}
errors = {}

parts = set()  # list of all parts possible

licence_data = None

# Please put them in the GOOD order for application if need!
ALL_FIX = []
MAX_FIX_NAME_LENGTH = 0


def system_call(command):
    p = subprocess.Popen([command], stdout=subprocess.PIPE, shell=True)
    p.wait()
    return p.returncode


def is_shinken_running():
    for daemon in POSSIBLE_DAEMONS:
        _activated_daemons = [id for id, enable in get_local_instances_for_type(daemon) if enable]
        if _activated_daemons:
            code = system_call('/etc/init.d/shinken-%s status' % daemon)
            if code == 0:
                return True
    return False


def add_success(s, part='unknown'):
    parts.add(part)
    if part not in success:
        success[part] = []
    success[part].append(s)


def add_warning(s, part='unknown'):
    parts.add(part)
    if part not in warnings:
        warnings[part] = []
    warnings[part].append(s)


def get_local_daemons():
    if not os.path.exists('/var/lib/shinken/context.json'):
        return ['central', 'poller', 'synchronizer', 'scheduler', 'reactionner', 'receiver', 'broker']
    f = open('/var/lib/shinken/context.json', 'r')
    context = json.loads(f.read())
    local_daemons = [k for (k, installed) in context['daemons'].iteritems() if installed]
    return local_daemons


def is_daemon_node(dtype):
    local_daemons = get_local_daemons()
    if 'central' in local_daemons or dtype in local_daemons:
        return True
    return False


def cut_line(s):
    size = 80
    lines = []
    words = s.split(' ')
    cline = ''
    for word in words:
        # Maybe this is too large already, put the line
        if len(cline) + len(word) > size:
            lines.append(cline)
            cline = ''
        cline += ' ' + word
    lines.append(cline)
    return lines


def print_line(title, color, s):
    lines = cut_line(s)
    title = '%-10s' % title
    print ' ' * 4 + '\033[%dm%s\033[0m %s' % (color, title, lines[0])
    if len(lines) > 1:
        for line in lines[1:]:
            print ' ' * 5 + ' ' * len(title) + line


def show_warnings(part):
    for (p, l) in warnings.iteritems():
        if p != part:
            continue
        title = 'AT RISK: '
        color = 35
        for s in l:
            print_line(title, color, s)


def add_error(s, part='unknown'):
    parts.add(part)
    if part not in errors:
        errors[part] = []
    errors[part].append(s)


def show_errors(part):
    for (p, l) in errors.iteritems():
        if p != part:
            continue
        title = 'ERROR: '
        color = 31
        for s in l:
            print_line(title, color, s)


def show_success(part):
    for (p, l) in success.iteritems():
        if p != part:
            continue
        title = 'OK: '
        color = 32
        for s in l:
            print_line(title, color, s)


def compute_diff(obj1, obj2):
    NOT_TO_LOOK = ['__SYNC_IDX__', 'import_date', 'sources', 'source_order',
                   '_SYNC_KEYS', 'overwrited_protected', '_id', 'source_strong_overwrite',
                   'imported_from', '_SE_UUID', '_SE_UUID_HASH', 'last_modification']
    changed = []
    for k in set(obj1.iterkeys()).union(obj2.iterkeys()):
        if k in NOT_TO_LOOK:
            continue
        vobj1 = obj1.get(k, None)
        vobj2 = obj2.get(k, None)
        # catch the same alias for null==__DEFAULT_NO_TEMPLATE__
        if vobj1 in ('null', '__DEFAULT_NO_TEMPLATE__') and vobj2 in ('null', '__DEFAULT_NO_TEMPLATE__'):
            continue
        if vobj1 != vobj2:
            changed.append(k)
    return changed


def add_history_info(item, old_item):
    last_modif = dict()
    last_modif['contact'] = '-1'
    last_modif['contact_name'] = 'shinken-core'
    last_modif['action'] = 'ELEMENT_MODIFICATION'
    last_modif['date'] = time.time()
    last_modif['change'] = []
    if old_item:
        for k in compute_diff(item, old_item):
            if k != "last_modification":
                last_modif['change'].append({"prop": k, "old": old_item.get(k, ''), "new": item.get(k, '')})
    
    if last_modif['change']:
        item['last_modification'] = last_modif


# Utility function for printing
def show_avance(pct):
    print '\r[',
    print '.' * pct, ' ' * (100 - pct),
    print '] %d%%' % pct,
    sys.stdout.flush()
    if pct == 100:
        print ''


def write_support_output(data, p):
    data = json.dumps(data)
    out = tarfile.open(p, mode='w:gz')
    try:
        info = tarfile.TarInfo('heath.json')
        info.size = len(data)
        out.addfile(info, StringIO(data))
    finally:
        out.close()
        print '\n\033[32mSupport ouput file is available: %s\033[0m' % (p)


def natural_key(string_):
    l = []
    for s in re.split(r'(\d+)', string_):
        if s.isdigit():
            l.append(int(s))
        else:
            l.append(s.lower())
    return l


def natural_version(f):
    return natural_key(f.version)


# Look if a function is ok to be launch on this server
def do_match_daemons(f):
    l_daemons = get_local_daemons()
    f_daemons = f.context_daemons
    if not f_daemons:
        return True
    for fd in f_daemons:
        if fd in l_daemons:
            return True
    return False


# TODO: look at real db path
def get_synchronizer_db():
    con = Connection()
    db = con.synchronizer
    return db


# TODO: look at real db path
def get_broker_db():
    con = Connection()
    db = con.shinken
    return db


def add_doc(doc_):
    def decorated(func):
        func.doc = doc_
        return func
    
    
    return decorated


def add_context_daemons(lst):
    # protect again dev that don't read the doc...
    if isinstance(lst, basestring):
        lst = [lst]
    
    
    def decorated(func):
        func.context_daemons = lst
        return func
    
    
    return decorated


def add_version(v):
    def decorated(func):
        func.version = v
        return func
    
    
    return decorated


def auto_launch(value):
    def decorated(func):
        func.auto_launch = value
        return func
    
    
    return decorated


def need_shinken_stop(value):
    def decorated(func):
        func.need_shinken_stop = value
        return func
    
    
    return decorated


def add_data_type(value):
    def decorated(func):
        func.data_type = value
        return func
    
    
    return decorated


def add_fix(func):
    global MAX_FIX_NAME_LENGTH
    ALL_FIX.append(func)
    name_length = len(func.__name__)
    if name_length > MAX_FIX_NAME_LENGTH:
        MAX_FIX_NAME_LENGTH = name_length
    return func


def _set_default_values_in_cfg_files(daemon_type, key_to_check, daemon_cfg_dir, daemon_default_values):
    g_run = False
    for daemon_config_file in os.listdir(daemon_cfg_dir):
        in_define = False
        cfg_file = os.path.join(daemon_cfg_dir, daemon_config_file)
        with open(cfg_file) as sync:
            lines = sync.readlines()
            for line_no, line in enumerate(lines):
                orig_line = line.rstrip()
                line = line.strip()
                if line.startswith('#'):
                    continue
                
                if line.startswith('define') and daemon_type in line:
                    in_define = True
                
                if in_define and line.startswith(key_to_check):
                    values = set([src.strip() for src in line.replace(key_to_check, '').split(',')])
                    
                    for expected_value in daemon_default_values:
                        if expected_value not in values:
                            orig_line = "%s,%s" % (orig_line, expected_value)
                            g_run = True
                
                if line.startswith('}'):
                    in_define = False
                
                lines[line_no] = "%s\n" % orig_line
        
        with open(cfg_file, 'w') as sync:
            sync.writelines(lines)
    return g_run


########################### FIX Functions

# Will look at contact template without 'name' and remove it from database
@add_fix
@add_doc('Fix contact templates that are invalid, like no name or even duplicate generic-contact object')
@add_context_daemons(['synchronizer'])
@add_version('02.03.03')
@auto_launch(False)
@need_shinken_stop(True)
def fix_invalid_contact_templates():
    g_did_run = False
    db = get_synchronizer_db()
    if not db.collection_names():
        return g_did_run
    col = getattr(db, 'configuration-stagging-contact')
    contacts = col.find({})
    to_clean = []
    for c in contacts:
        register = c.get('register', '1')
        if register != '0':
            continue
        _id = c['_id']
        name = c.get('name', None)
        if name is None:
            to_clean.append(_id)
            g_did_run = True
    if len(to_clean) == 0:
        logger.debug("    No contact template to clean")
        return g_did_run
    logger.info("    %d contact template to clean" % len(to_clean))
    for _id in to_clean:
        col.remove({'_id': _id})
    return g_did_run


# Will look at contact template without 'name' and remove it from database
@add_doc('Fix contact template generic-template in duplicate')
@add_context_daemons(['synchronizer'])
@add_version('02.03.02')
@add_fix
@auto_launch(False)
@need_shinken_stop(True)
def fix_duplicate_generic_contact():
    g_did_run = False  # did we fix something?
    db = get_synchronizer_db()
    if not db.collection_names():
        return g_did_run
    
    col = getattr(db, 'configuration-stagging-contact')
    contacts = col.find({})
    generic_contact = []
    for c in contacts:
        if c.get('register', '1') == '0' and c.get('name', None) == 'generic-contact':
            generic_contact.append(c)
    if len(generic_contact) in [0, 1]:
        logger.debug("    No generic-contact duplication.")
        return False
    # oups we have a duplication here
    # keep the first one
    logger.info("    Removing %d duplicate element" % (len(generic_contact) - 1))
    for c in generic_contact[1:]:
        col.remove({'_id': c['_id']})
    return True


# Will look at contact template without 'name' and remove it from database
@add_doc('Fix invalid objects from the first 2.x versions, like "" values in configuration database')
@add_context_daemons(['synchronizer'])
@add_version('02.03.03')
@add_fix
@auto_launch(False)
@need_shinken_stop(True)
def fix_empty_value_from_synchronizer_data():
    g_did_run = False  # did we fix something?
    db = get_synchronizer_db()
    if not db.collection_names():
        return g_did_run
    
    try:
        from shinken.synchronizer.dao.dataprovider.dataprovider_mongo import DataProviderMongo
        from shinken.synchronizer.dao.def_items import DEF_ITEMS, ITEM_STATE
    except ImportError:
        logger.error("You cannot run this fix on your shinken version please upgrade it.")
        return g_did_run
    
    data_provider_mongo = DataProviderMongo(db)
    for item_state in [ITEM_STATE.NEW, ITEM_STATE.MERGE_SOURCES, ITEM_STATE.WORKING_AREA, ITEM_STATE.STAGGING, ITEM_STATE.PRODUCTION]:
        for item_type in DEF_ITEMS.iterkeys():
            nb_cleaned = 0
            items = data_provider_mongo.find_items(item_type, item_state)
            nb_total = len(items)
            for item in items:
                _id = item['_id']
                to_del = []
                for (k, v) in item.iteritems():
                    if v == '':
                        to_del.append(k)
                if len(to_del) != 0:
                    nb_cleaned += 1
                    for k in to_del:
                        del item[k]
                    data_provider_mongo.save_item(item, item_type, item_state)
            if nb_total != 0 and nb_cleaned != 0:
                if nb_cleaned != nb_total:
                    g_did_run = True
                    logger.info("   Did clean [%s-%s] objects [%d/%d] are now cleaned" % (item_state, item_type, nb_cleaned, nb_total))
    return g_did_run


@add_fix
@add_doc('Fix duplicate objects in the database.')
@add_context_daemons(['synchronizer'])
@add_version('02.03.03')
@auto_launch(False)
@need_shinken_stop(True)
def fix_duplicate_elements():
    g_did_run = False  # did we fix something?
    db = get_synchronizer_db()
    if not db.collection_names():
        return g_did_run
    
    DATA_KEYS = {'host'                    : 'host_name',
                 'command'                 : 'command_name',
                 'timeperiod'              : 'timeperiod_name',
                 'contact'                 : 'contact_name',
                 'contactgroup'            : 'contactgroup_name',
                 'hostgroup'               : 'hostgroup_name',
                 'businessimpactmodulation': 'business_impact_modulation_name',
                 'notificationway'         : 'notificationway_name',
                 'escalation'              : 'escalation_name',
                 'macromodulation'         : 'macromodulation_name',
                 'service'                 : ''
                 }
    
    # First loop over classic objects with real names
    for (t, k_name) in DATA_KEYS.iteritems():
        did_run = False
        for col in [getattr(db, 'configuration-stagging-%s' % t), getattr(db, 'configuration-production-%s' % t)]:
            objs = col.find({})
            # which objects do we will need to clean?
            classic_objs = {}
            template_objs = {}
            to_clean = []
            for o in objs:
                if k_name:
                    register = o.get('register', '1')
                    if register == '0':
                        name = o.get('name', None)
                        if name is None:  # unamed template? come one, what a crazy configuration
                            continue
                        if name not in template_objs:
                            template_objs[name] = []
                        template_objs[name].append(o)
                    else:  # classic objects
                        oname = o.get(k_name, None)
                        if oname is None:  # object without a name? ... get lost
                            continue
                        if oname not in classic_objs:
                            classic_objs[oname] = []
                        classic_objs[oname].append(o)
                else:  # service...
                    # they are a bit more triky. We can clean only the element that have the host_name&description set (template or not)
                    sdesc = o.get('service_description', None)
                    hname = o.get('host_name', None)
                    name = o.get('name', None)
                    if name is not None:  # classic templates
                        if sdesc is not None:  # mix? cannot clean this!
                            continue
                        if name not in template_objs:
                            template_objs[name] = []
                        template_objs[name].append(o)
                    else:
                        if sdesc is None or hname is None:  # cannot clean a unclear service, bailout
                            continue
                        skey = '%s--%s' % (sdesc, hname)
                        if skey not in classic_objs:
                            classic_objs[skey] = []
                        classic_objs[skey].append(o)
            
            # Now go clean them
            all_elements = []
            # Mix al, but without mix names
            for (oname, lst) in classic_objs.iteritems():
                all_elements.append((oname, lst))
            for (oname, lst) in template_objs.iteritems():
                all_elements.append((oname, lst))
            # Go clean
            for (oname, lst) in all_elements:
                # no problem here
                if len(lst) <= 1:
                    continue
                
                
                def sortkey(o):
                    try:
                        return uuid.UUID(hex=o['_id']).time
                    except:
                        return -1
                
                
                # sort elements based on their uuids, hope it can works
                lst = sorted(lst, key=sortkey)
                
                new_lst = []
                nb_valid = 0
                # Maybe there are more than 1, and there is a syncui one
                for o in lst:
                    if 'syncui' in o.get('sources', ''):  # fuck a manually changed element, don't modify it
                        nb_valid += 1
                        continue
                    new_lst.append(o)
                
                # Maybe there is no much element to clean, so bailout this name, we can't clean it, too risky
                if len(new_lst) == 0:
                    continue
                
                # ok we will have to fix something here...
                did_run = True
                g_did_run = True
                # maybe there was no valid element saved, so skip one
                if nb_valid == 0:
                    to_clean.extend(new_lst[1:])
                else:  # there was at least one element save, so clean all the others :)
                    to_clean.extend(new_lst)
            if did_run:
                total = col.find({}).count()
                nb_clean = len(to_clean)
                logger.info("   Did clean %6d duplicate %15ss [%d-%d=%d]" % (nb_clean, t, total, nb_clean, total - nb_clean))
                for o in to_clean:
                    # print o.get('sources', ''), o.get('pack', '')
                    logger.debug("  CLEAN: %s => %s" % (t, o.get('name', o.get(k_name, o.get('service_description', '')))))
                    col.remove({'_id': o['_id']})
    
    return g_did_run


# Fix share collection
@add_fix
@add_doc('Fix share collection.')
@add_context_daemons(['broker'])
@add_data_type('user')
@add_version('02.03.04')
@auto_launch(False)
@need_shinken_stop(True)
def fix_share_collection():
    g_did_run = False  # did we fix something?
    db = get_broker_db()
    if not db.collection_names():
        return g_did_run
    
    col_share = getattr(db, 'share')
    col_hive = getattr(db, 'hive')
    col_list = getattr(db, 'list')
    
    shares = col_share.find({})
    
    for share in shares:
        if 'screen' in share:
            screen_type = share['screen'].get('type', None)
            screen_uuid = share['screen']['uuid']
            screen_name = share['name']
            
            if not screen_type:
                hive_uuid = col_hive.find_one({'_id': screen_uuid}, {'_id': 1})
                list_uuid = col_list.find_one({'_id': screen_uuid}, {'_id': 1})
                
                if hive_uuid:
                    share['screen']['type'] = 'hive'
                elif list_uuid:
                    share['screen']['type'] = 'list'
                
                col_share.save(share)
                
                g_did_run = True
                print "Fix screen [%s] type found [%s]" % (screen_name, share['screen']['type'])
    
    return g_did_run


# Will clean if there are double SYNC_KEYS in stagging
@add_fix
@add_doc('Will clean if there are double SYNC_KEYS in stagging.')
@add_context_daemons(['synchronizer'])
@add_version('02.03.03-U1')
@auto_launch(True)
@need_shinken_stop(True)
def fix_double_sync_keys():
    g_did_run = False  # did we fix something?
    db = get_synchronizer_db()
    if not db.collection_names():
        return g_did_run
    
    try:
        from shinken.synchronizer.dao.datamanagerV2 import DataManagerV2, FALLBACK_USER
        from shinken.synchronizer.dao.dataprovider.dataprovider_metadata import DataProviderMetadata
        from shinken.synchronizer.dao.dataprovider.dataprovider_mongo import DataProviderMongo
        from shinken.synchronizer.dao.def_items import ITEM_TYPE, ITEM_STATE, DEF_ITEMS
        from shinken.synchronizer.dao.helpers import get_name_from_type
        from shinken.synchronizer.dao.transactions.transactions import DBTransaction
    except ImportError:
        return 'You cannot run this fix on your shinken version please upgrade it.'
    
    db = get_synchronizer_db()
    data_provider_mongo = DataProviderMongo(db)
    
    for item_type in ITEM_TYPE.ALL_TYPES:
        logger.info('Searching double sync-key for type : %s have started.' % item_type)
        to_update_states = (ITEM_STATE.STAGGING, ITEM_STATE.PRODUCTION, ITEM_STATE.WORKING_AREA) if DEF_ITEMS[item_type]['has_work_area'] else (ITEM_STATE.STAGGING, ITEM_STATE.PRODUCTION)
        for item_state in to_update_states:
            to_regenerates = {}
            keys = {}
            for item in data_provider_mongo.find_items(item_type, item_state):
                
                sync_keys = set([sync_key.lower() for sync_key in item.get('_SYNC_KEYS', [])])
                item_name = get_name_from_type(item_type, item)
                item_id = item['_id']
                minimal_sync_keys = compute_minimal_sync_keys(item, item_type)
                
                item['__MINIMAL_KEYS'] = minimal_sync_keys
                item['__NAME'] = item_name
                
                if not sync_keys:
                    logger.info('[%12s] %s [%s]-[%s] item don\'t have _SYNC_KEYS. We set minimal sync keys:[%s] to it.' % (item_state, item_type, item_id, item_name, ','.join(minimal_sync_keys)))
                    to_regenerates[item['_id']] = item
                    continue
                
                for minimal_sync_key in minimal_sync_keys:
                    if minimal_sync_key not in sync_keys:
                        logger.info('[%12s] %s [%s]-[%s] item missing minimal sync key:[%s]. We add it to the item. Current sync_keys: [%s]' % (item_state, item_type, item_id, item_name, minimal_sync_key, ','.join(sync_keys)))
                        to_regenerates[item['_id']] = item
                        sync_keys.add(minimal_sync_key)
                        item['_SYNC_KEYS'] = list(sync_keys)
                
                for sync_key in sync_keys:
                    if sync_key in keys:
                        keys[sync_key].append(item)
                    else:
                        keys[sync_key] = [item]
            
            for share_sync_key, items in keys.iteritems():
                if len(items) > 1:
                    for item in items:
                        item_name = get_name_from_type(item_type, item)
                        item_id = item['_id']
                        logger.info('[%12s] %s [%s]-[%s] item share the sync-key [%s] with an other item. We set minimal sync keys:[%s] to it.' % (item_state, item_type, item_id, item_name, share_sync_key, ','.join(item['__MINIMAL_KEYS'])))
                        to_regenerates[item['_id']] = item
                        item['_SYNC_KEYS'] = item['__MINIMAL_KEYS']
            
            if to_regenerates:
                with DBTransaction(data_provider_mongo):
                    for item in to_regenerates.itervalues():
                        g_did_run = True
                        
                        del item['__MINIMAL_KEYS']
                        del item['__NAME']
                        
                        data_provider_mongo.save_item(item, item_type, item_state)
    
    if g_did_run:
        col = getattr(db, 'synchronizer-info')
        database_info = col.find_one({'_id': 'database_info'})
        if not database_info:
            database_info = {'_id': 'database_info'}
        # 3 round : see SEF-5690
        database_info['fix_double_sync'] = 3
        database_info['must_remake_import'] = 1
        col.save(database_info)
    
    return g_did_run


def compute_minimal_sync_keys(item, item_type):
    from shinken.synchronizer.dao.def_items import ITEM_TYPE
    from shinken.synchronizer.dao.helpers import get_name_from_type
    is_template = ITEM_TYPE.is_template(item_type)
    new_sync_keys = []
    
    if item_type in ITEM_TYPE.ALL_DEDICATED_SERVICES:
        item_sync_keys_uuid = item.get('_SE_UUID', 'core-%s-%s' % (item_type, item['_id']))
        new_sync_keys.append(item_sync_keys_uuid.lower())
    else:
        item_name = get_name_from_type(item_type, item)
        item_sync_keys_name = ('%s-tpl' % item_name if is_template else item_name)
        new_sync_keys.append(item_sync_keys_name.lower())
        
        item_sync_keys_uuid = item.get('_SE_UUID', None)
        if item_sync_keys_uuid:
            new_sync_keys.append(item_sync_keys_uuid.lower())
    
    return new_sync_keys


@add_fix
@add_doc('Regenerate invalid or missing sync-keys for a item_type.')
@add_context_daemons(['synchronizer'])
@add_version('02.07.05')
@auto_launch(False)
def regenerate_invalid_or_missing_sync_keys():
    g_did_run = False  # did we fix something?
    db = get_synchronizer_db()
    if not db.collection_names():
        return g_did_run
    
    try:
        from shinken.synchronizer.dao.datamanagerV2 import DataManagerV2, FALLBACK_USER
        from shinken.synchronizer.dao.dataprovider.dataprovider_metadata import DataProviderMetadata
        from shinken.synchronizer.dao.dataprovider.dataprovider_mongo import DataProviderMongo
        from shinken.synchronizer.dao.def_items import ITEM_TYPE, ITEM_STATE, DEF_ITEMS
        from shinken.synchronizer.dao.helpers import get_name_from_type
        from shinken.synchronizer.dao.transactions.transactions import DBTransaction
    except ImportError:
        return 'You cannot run this fix on your shinken version please upgrade it.'
    
    item_type = opts.item_type
    
    if not item_type:
        return 'Please choose a item type with --item-type option.'
    if item_type not in ITEM_TYPE.ALL_TYPES:
        return 'Unknown item_type : [%s] must be one of following : \n* %s' % (item_type, '\n* '.join(ITEM_TYPE.ALL_TYPES))
    
    db = get_synchronizer_db()
    data_provider_mongo = DataProviderMongo(db)
    
    to_save_items = []
    
    logger.info('Sanitize sync-key for type : %s have started.' % item_type)
    to_update_states = (ITEM_STATE.STAGGING, ITEM_STATE.PRODUCTION, ITEM_STATE.WORKING_AREA) if DEF_ITEMS[item_type]['has_work_area'] else (ITEM_STATE.STAGGING, ITEM_STATE.PRODUCTION)
    for item_state in to_update_states:
        to_save_items = []
        for item in data_provider_mongo.find_items(item_type, item_state):
            old_sync_keys = set(item.get('_SYNC_KEYS', []))
            minimal_sync_keys = compute_minimal_sync_keys(item, item_type)
            item_name = get_name_from_type(item_type, item)
            item_id = item['_id']
            
            if not old_sync_keys:
                item['_SYNC_KEYS'] = list(minimal_sync_keys)
                to_save_items.append(item)
                logger.info('[%12s] %s [%s]-[%s] item don\'t have _SYNC_KEYS. We set minimal sync keys:[%s] to it.' % (item_state, item_type, item_id, item_name, ','.join(minimal_sync_keys)))
                continue
            
            valide_sync_keys = set((item_value.lower() for item_value in item.itervalues() if hasattr(item_value, 'lower') and item_value.lower() in old_sync_keys))
            
            for minimal_sync_key in minimal_sync_keys:
                if minimal_sync_key not in valide_sync_keys:
                    valide_sync_keys.add(minimal_sync_key)
            
            if old_sync_keys == valide_sync_keys:
                logger.info('[%12s] %s [%s]-[%s] item have valide _SYNC_KEYS.' % (item_state, item_type, item_id, item_name))
                continue
            
            logger.info('[%12s] %s [%s]-[%s] item have some invalide or missing _SYNC_KEYS. We set new sync keys to it. [%s]->[%s]' % (item_state, item_type, item_id, item_name, ','.join(old_sync_keys), ','.join(valide_sync_keys)))
            item['_SYNC_KEYS'] = list(valide_sync_keys)
            to_save_items.append(item)
        
        if to_save_items:
            g_did_run = bool(to_save_items)
            with DBTransaction(data_provider_mongo):
                for item in to_save_items:
                    data_provider_mongo.save_item(item, item_type, item_state)
    
    if g_did_run:
        col = getattr(db, 'synchronizer-info')
        database_info = col.find_one({'_id': 'database_info'})
        if not database_info:
            database_info = {'_id': 'database_info'}
        database_info['must_remake_import'] = 1
        col.save(database_info)
    
    return g_did_run


# Will make double link for host, host template, hostgroup, contact, contact template, contactgroup, in stagging and production
# If necessary, the remove duplicate fix should be run before this one.
@add_fix
@add_doc('Will update _id of default items with the good SE_UUID, in stagging and production.')
@add_context_daemons(['synchronizer'])
@add_version('02.03.03-U1')
@auto_launch(True)
@need_shinken_stop(True)
def fix_default_item_se_uuid():
    g_did_run = False  # did we fix something?
    db = get_synchronizer_db()
    if not db.collection_names():
        return g_did_run
    
    DATA_KEYS = {'host'                    : 'host_name',
                 'command'                 : 'command_name',
                 'timeperiod'              : 'timeperiod_name',
                 'contact'                 : 'contact_name',
                 'contactgroup'            : 'contactgroup_name',
                 'hostgroup'               : 'hostgroup_name',
                 'businessimpactmodulation': 'business_impact_modulation_name',
                 'notificationway'         : 'notificationway_name',
                 'escalation'              : 'escalation_name',
                 'macromodulation'         : 'macromodulation_name',
                 'service'                 : 'service_description'
                 }
    
    try:
        from shinken.synchronizer.dao.datamanagerV2 import DataManagerV2, FALLBACK_USER, get_type_item_from_class
        from shinken.synchronizer.dao.dataprovider.dataprovider_metadata import DataProviderMetadata
        from shinken.synchronizer.dao.dataprovider.dataprovider_mongo import DataProviderMongo
        from shinken.synchronizer.dao.def_items import DEF_ITEMS, ITEM_STATE
        from shinken.synchronizer.dao.transactions.transactions import DBTransaction
    except ImportError:
        logger.error("You cannot run this fix on your shinken version please upgrade it.")
        return g_did_run
    
    col = getattr(db, 'synchronizer-info')
    database_info = col.find_one({'_id': 'database_info'})
    has_new_link_format = database_info and database_info.get('new_link_format', 0) == 1
    
    if has_new_link_format:
        data_provider_mongo = DataProviderMongo(db)
        dataprovider = DataProviderMetadata(data_provider_mongo)
        dataprovider.load_from_data_provider()
        datamanagerV2 = DataManagerV2(dataprovider, compute_double_links=False, use_default_callbacks=False)
    
    cfg_path = '/etc/shinken/packs/'
    
    cfg_path_file = open('/tmp/shinken_pack_tmp.cfg', 'w')
    cfg_path_file.write('cfg_dir=%s\n' % cfg_path)
    cfg_path_file.close()
    
    old_level = logger.level
    logger.setLevel(logging.FATAL)
    
    conf = Config()
    buf = conf.read_config(['/tmp/shinken_pack_tmp.cfg'])
    if not conf.conf_is_correct:
        logger.error('The sanatize can\'t load default configuration at [%s].' % cfg_path)
        return g_did_run
    
    logger.setLevel(old_level)
    
    raw_objects = conf.read_config_buf(buf, [])
    conf.load_packs()
    
    os.remove('/tmp/shinken_pack_tmp.cfg')
    
    for item_class, items in raw_objects.iteritems():
        if item_class not in DATA_KEYS:
            continue
        
        col_stagging = getattr(db, 'configuration-stagging-%s' % item_class)
        col_production = getattr(db, 'configuration-production-%s' % item_class)
        for item in items:
            if '_SE_UUID' not in item:
                continue
            uuid = item['_SE_UUID'][0].split('-')[2]
            key = 'name' if item.get('name', '') else DATA_KEYS[item_class]
            name = item.get(key, [''])[0]
            where = {key: name}
            
            if item_class == 'service':
                if item.get('register', ['1'])[0] == '1':
                    continue
                if item.get('host_name', [''])[0]:
                    where['host_name'] = item.get('host_name', [''])[0]
            
            if item_class == 'host' and item.get('register', ['1'])[0] == '1':
                continue
            
            item_in_stagging = col_stagging.find_one(where)
            
            if not item_in_stagging:
                continue
            
            _id = item_in_stagging['_id']
            updated_item_in_stagging = item_in_stagging.copy()
            
            if uuid != _id or item_in_stagging.get('_SE_UUID', '') != item['_SE_UUID'][0]:
                item_type = get_type_item_from_class(item_class + 's', item_in_stagging)
                updated_item_in_stagging['_id'] = uuid
                updated_item_in_stagging['_SYNC_KEYS'] = [k if k != updated_item_in_stagging.get('_SE_UUID', '') else item['_SE_UUID'][0] for k in updated_item_in_stagging['_SYNC_KEYS']]
                updated_item_in_stagging['_SE_UUID'] = item['_SE_UUID'][0]
                
                # If we do have the new links we need to update links to updated items with the new UUID
                if has_new_link_format:
                    item_in_stagging = datamanagerV2.find_item_by_id(_id, item_type, ITEM_STATE.STAGGING)
                    _update_reversed_links(item_in_stagging, uuid, datamanagerV2, data_provider_mongo)
                
                col_stagging.save(updated_item_in_stagging)
                if uuid != _id:
                    col_stagging.remove(_id)
                
                logger.debug('Changed the id of the item [%s-%s] from [%s] to [%s] (in stagging).' % (item_class, name, _id, uuid))
                g_did_run = True
                
                # Special case handling, contacts uid were used for last modification
                if item_class == 'contact' and not has_new_link_format:
                    for to_clean in DATA_KEYS.keys():
                        col_to_clean = getattr(db, 'configuration-stagging-%s' % to_clean)
                        impacted_items = list(col_to_clean.find({'last_modification.contact': _id}))
                        for item in impacted_items:
                            item['last_modification']['contact'] = uuid
                            col_to_clean.update({'_id': item['_id']}, item)
            
            item_in_production = col_production.find_one(where)
            
            if not item_in_production:
                continue
            
            _id = item_in_production['_id']
            
            if uuid != _id:
                updated_item_in_production = item_in_production.copy()
                updated_item_in_production['_id'] = uuid
                item_type = get_type_item_from_class(item_class + 's', item_in_production)
                updated_item_in_production['_id'] = uuid
                updated_item_in_production['_SYNC_KEYS'] = [k if k != updated_item_in_production.get('_SE_UUID', '') else item['_SE_UUID'][0] for k in updated_item_in_production['_SYNC_KEYS']]
                updated_item_in_production['_SE_UUID'] = item['_SE_UUID'][0]
                
                if has_new_link_format:
                    item_in_production = datamanagerV2.find_item_by_id(_id, item_type, ITEM_STATE.PRODUCTION)
                    _update_reversed_links(item_in_production, uuid, datamanagerV2, data_provider_mongo)
                
                col_production.save(updated_item_in_production)
                col_production.remove(_id)
                
                logger.debug('Changed the id of the item [%s-%s] from [%s] to [%s] (in production).' % (item_class, name, _id, uuid))
                # Special case handling, checks uid were used in dashboards
                if item_class in ('service', 'host'):
                    col_dashboard = getattr(get_broker_db(), 'tiles')
                    search = {'items': {'$elemMatch': {'dashboard.widgets': {'$elemMatch': {'element.uuid': re.compile(_id)}}}}}
                    impacted_dashboard = list(col_dashboard.find(search))
                    for tile in impacted_dashboard:
                        for item in tile['items']:
                            if not 'dashboard' in item or not 'widgets' in item['dashboard']:
                                continue
                            for widget in item['dashboard']['widgets']:
                                if not 'element' in widget or not 'uuid' in widget['element']:
                                    continue
                                widget['element']['uuid'] = widget['element']['uuid'].replace(_id, uuid)
                        col_dashboard.update({'_id': tile['_id']}, tile)
                
                g_did_run = True
    
    if not g_did_run:
        return g_did_run
    
    col = getattr(db, 'synchronizer-info')
    database_info = col.find_one({'_id': 'database_info'})
    if not database_info:
        database_info = {'_id': 'database_info'}
    
    database_info['fix_fix_default_uuid_v2'] = 1
    database_info['must_remake_import'] = 1
    col.save(database_info)
    return g_did_run


def _update_reversed_links(item, new_id, datamanagerV2, dataprovider_mongo):
    # Update links to item
    from shinken.synchronizer.dao.items import get_item_instance
    
    reverse_links = item.get_reverse_links().copy()
    for reversed_link in reverse_links:
        linked_item = datamanagerV2.find_item_by_id(item_id=reversed_link.item_id, item_type=reversed_link.item_type, item_state=reversed_link.item_state)
        raw_linked_item = linked_item.get_raw_item(flatten_links=False, keep_metadata=True)
        mock_baseitem = get_item_instance(reversed_link.item_type, raw_linked_item)
        for link in mock_baseitem.get_links(reversed_link.key):
            if '_id' in link and link['_id'] == item['_id']:
                link['_id'] = new_id
        raw_linked_item.pop('@metadata', None)
        dataprovider_mongo.save_item(raw_linked_item, item_type=linked_item.get_type(), item_state=linked_item.get_state())


@add_fix
@add_doc('Will remove shinken-core contact from configuration.')
@add_context_daemons(['synchronizer'])
@add_version('02.03.03-U1')
@auto_launch(True)
@need_shinken_stop(True)
def fix_remove_shinken_core():
    g_did_run = False  # did we fix something?
    db = get_synchronizer_db()
    if not db.collection_names():
        return g_did_run
    
    data_tables = ['host',
                   'command',
                   'timeperiod',
                   'contact',
                   'contactgroup',
                   'hostgroup',
                   'businessimpactmodulation',
                   'notificationway',
                   'escalation',
                   'macromodulation',
                   'service'
                   ]
    
    for col in [getattr(db, 'configuration-stagging-contact'), getattr(db, 'configuration-production-contact')]:
        item = col.find_one({'contact_name': 'shinken-core'})
        if item:
            col.remove({'_id': item['_id']})
            g_did_run = True
            
            # Remove all references as well
            for col in [getattr(db, 'configuration-stagging-host'), getattr(db, 'configuration-production-host'),
                        getattr(db, 'configuration-stagging-service'), getattr(db, 'configuration-production-service'),
                        getattr(db, 'configuration-stagging-escalation'), getattr(db, 'configuration-production-escalation')]:
                items = col.find({'contacts': re.compile('shinken-core')})
                for item in items:
                    new_item = item.copy()
                    new_item['contacts'] = ','.join([c for c in item['contacts'].split(',') if c and c.strip() != 'shinken-core'])
                    add_history_info(new_item, item)
                    col.update({'_id': new_item['_id']}, new_item)
            for col in [getattr(db, 'configuration-stagging-contactgroup'), getattr(db, 'configuration-production-contactgroup')]:
                items = col.find({'members': re.compile('shinken-core')})
                for item in items:
                    new_item = item.copy()
                    new_item['members'] = ','.join([c for c in item['members'].split(',') if c and c.strip() != 'shinken-core'])
                    add_history_info(new_item, item)
                    col.update({'_id': item['_id']}, new_item)
            
            for table in data_tables:
                col = getattr(db, 'configuration-stagging-%s' % table)
                items = col.find({'last_modification.contact': '16f36b30185f11e69025f8bc126497d6'})
                for item in items:
                    item['last_modification']['contact'] = '-1'
                    item['last_modification']['contact_name'] = 'shinken-core'
                    col.update({'_id': item['_id']}, item)
    
    return g_did_run


# Will clean deprecated check after poller update see SEF-1053 / SEF-1143
@add_fix
@add_doc('Will clean deprecated check after poller update.')
@add_context_daemons(['synchronizer'])
@add_version('02.03.03-U2')
@auto_launch(True)
@need_shinken_stop(True)
def fix_remove_deprecated_check():
    g_did_run = False  # did we fix something?
    db = get_synchronizer_db()
    if not db.collection_names():
        return g_did_run
    
    col_stagging = getattr(db, 'configuration-stagging-service')
    col_production = getattr(db, 'configuration-production-service')
    
    service_to_del = ['Scheduler - Performance API Connection',
                      'Scheduler - Performance Late Checks',
                      'Scheduler - Performance Latency',
                      'Poller - Performance API Connection',
                      'Poller - Performance CPU Load',
                      'Reactionner - Performance CPU Load',
                      'Reactionner - Performance API Connection']
    
    to_del = list(col_stagging.find({'service_description': {'$in': service_to_del}}, {'_id': 1, 'service_description': 1}))
    if to_del:
        for todel_item in to_del:
            col_stagging.remove({'_id': todel_item['_id']})
        g_did_run = True
    
    to_del = list(col_production.find({'service_description': {'$in': service_to_del}}, {'_id': 1, 'service_description': 1}))
    if to_del:
        logger.debug("We will deleted deprecated check.")
        for todel_item in to_del:
            logger.debug("Deleting check [%s]." % todel_item['service_description'])
            col_production.remove({'_id': todel_item['_id']})
        g_did_run = True
    
    return g_did_run


@add_fix
@add_doc('Will clean undefined templates used by aix templates.')
@add_context_daemons(['synchronizer'])
@add_version('02.03.03-U2')
@auto_launch(True)
@need_shinken_stop(True)
def fix_remove_undefined_aix_templates():
    g_did_run = False  # did we fix something?
    db = get_synchronizer_db()
    if not db.collection_names():
        return g_did_run
    
    try:
        from shinken.synchronizer.dao.datamanagerV2 import DataManagerV2
        from shinken.synchronizer.dao.dataprovider.dataprovider_metadata import DataProviderMetadata
        from shinken.synchronizer.dao.dataprovider.dataprovider_mongo import DataProviderMongo
        from shinken.synchronizer.dao.def_items import DEF_ITEMS, ITEM_STATE, ITEM_TYPE
        from shinken.synchronizer.dao.transactions.transactions import DBTransaction
    except ImportError:
        logger.error("You cannot run this fix on your shinken version please upgrade it.")
        return g_did_run
    
    user = {
        'contact_name': 'shinken-core',
        'is_admin'    : '1',
        '_id'         : '-1',
    }
    
    col = getattr(db, 'synchronizer-info')
    database_info = col.find_one({'_id': 'database_info'})
    if not database_info or database_info.get('new_link_format', 0) != 1:
        logger.error("You cannot run this fix on your shinken version please upgrade it or run the sanatize 'migrate_links_name_into_id'.")
        return g_did_run
    
    to_remove_use = ("10min_long", "20min_long", "12hours_short", "20min_medium")
    
    dataprovider = DataProviderMongo(db)
    datamanagerV2 = DataManagerV2(dataprovider, compute_double_links=False, use_default_callbacks=False)
    
    nb_items = 0
    with DBTransaction(datamanagerV2=datamanagerV2):
        for item_state in [ITEM_STATE.STAGGING, ITEM_STATE.PRODUCTION]:
            for item_type in ITEM_TYPE.ALL_SERVICES:
                items = list(datamanagerV2.find_items(item_type, item_state=item_state, where={'pack': 'aix'}))
                for item in items:
                    to_del = []
                    links = item['use']['links']
                    for link in links:
                        if link['exists']:
                            continue
                        elif link['name'] in to_remove_use:
                            to_del.append(link)
                    for link_to_del in to_del:
                        links.remove(link_to_del)
                    
                    if to_del:
                        g_did_run = True
                        nb_items += 1
                        datamanagerV2.save_item(item, user=user, item_state=item_state, item_type=item_type)
    
    if g_did_run:
        logger.info('[fix_remove_undefined_aix_templates] Removed AIX templates for [%d] items.' % (nb_items))
    else:
        return g_did_run
    
    col = getattr(db, 'synchronizer-info')
    database_info = col.find_one({'_id': 'database_info'})
    if not database_info:
        database_info = {'_id': 'database_info'}
    
    database_info['all_double_link'] = 1
    database_info['must_remake_import'] = 1
    col.save(database_info)
    
    return g_did_run


# Will clean old oracle checks which bloat the pack
@add_fix
@add_doc("""Will remove following oracle checks:
Oracle-$KEY$-enqueue-contention
Oracle-$KEY$-enqueue-waiting
Oracle-$KEY$-event-waits
Oracle-$KEY$-event-waiting
Oracle-$KEY$-latch-contention
Oracle-$KEY$-latch-waiting
Oracle-$KEY$-seg-top10-buffer-busy-waits
Oracle-$KEY$-seg-top10-logical-reads
Oracle-$KEY$-seg-top10-physical-reads
Oracle-$KEY$-seg-top10-row-lock-waits
Oracle-$KEY$-sysstat""")
@add_context_daemons(['synchronizer'])
@add_version('02.03.03-U2')
@auto_launch(False)
@need_shinken_stop(True)
def fix_remove_old_oracle_checks():
    g_did_run = False  # did we fix something?
    db = get_synchronizer_db()
    if not db.collection_names():
        return g_did_run
    
    col_stagging = getattr(db, 'configuration-stagging-service')
    col_production = getattr(db, 'configuration-production-service')
    
    service_to_del = [
        'Oracle-$KEY$-enqueue-contention',
        'Oracle-$KEY$-enqueue-waiting',
        'Oracle-$KEY$-event-waits',
        'Oracle-$KEY$-event-waiting',
        'Oracle-$KEY$-latch-contention',
        'Oracle-$KEY$-latch-waiting',
        'Oracle-$KEY$-seg-top10-buffer-busy-waits',
        'Oracle-$KEY$-seg-top10-logical-reads',
        'Oracle-$KEY$-seg-top10-physical-reads',
        'Oracle-$KEY$-seg-top10-row-lock-waits',
        'Oracle-$KEY$-sysstats'
    ]
    
    to_del = list(col_stagging.find({'service_description': {'$in': service_to_del}}, {'_id': 1, 'service_description': 1}))
    if to_del:
        for todel_item in to_del:
            col_stagging.remove({'_id': todel_item['_id']})
        g_did_run = True
    
    to_del = list(col_production.find({'service_description': {'$in': service_to_del}}, {'_id': 1, 'service_description': 1}))
    if to_del:
        logger.debug("We will delete deprecated check.")
        for todel_item in to_del:
            logger.debug("Deleting check [%s]." % todel_item['service_description'])
            col_production.remove({'_id': todel_item['_id']})
        g_did_run = True
    
    return g_did_run


# Will transform non-uppercase custom data keys and duplicate_foreach field
@add_fix
@add_doc("""Will force custom data keys and duplicate foreach to be uppercase""")
@add_context_daemons(['synchronizer'])
@add_version('02.03.03-U2')
@auto_launch(False)
@need_shinken_stop(True)
def fix_custom_data_uppercase():
    g_did_run = False  # did we fix something?
    db = get_synchronizer_db()
    if not db.collection_names():
        return g_did_run
    
    to_fix = [getattr(db, 'configuration-stagging-service'), getattr(db, 'configuration-production-service')]
    
    for col in to_fix:
        for service in col.find({'duplicate_foreach': {'$exists': True}}):
            if service['duplicate_foreach'] != service['duplicate_foreach'].upper():
                g_did_run = True
                service['duplicate_foreach'] = service['duplicate_foreach'].upper()
                logger.debug('Fixed %s' % service.get('name', service.get('service_description', '-- Missing name -- %s' % service)))
                col.update({'_id': service['_id']}, service)
    
    to_fix = [getattr(db, 'configuration-stagging-service'), getattr(db, 'configuration-production-service'),
              getattr(db, 'configuration-stagging-host'), getattr(db, 'configuration-production-host'),
              getattr(db, 'configuration-stagging-contact'), getattr(db, 'configuration-production-contact')]
    
    for col in to_fix:
        # Checks all items
        for item in col.find({}):
            keys_to_del = []
            for k in item.iterkeys():
                if k[0] == '_' and k != '_id' and k != k.upper():
                    keys_to_del.append(k)
                    g_did_run = True
            for k in keys_to_del:
                item[k.upper()] = item[k]
                del item[k]
            if keys_to_del:
                col.update({'_id': item['_id']}, item)
                logger.debug('Fixed %s' % item.get('name', item.get('contact_name', item.get('host_name', item.get('service_description', '-- Missing name -- %s' % item)))))
    
    return g_did_run


@add_fix
@add_doc("""Will fix checks and hosts with incoherent flapping thresholds""")
@add_context_daemons(['synchronizer'])
@add_version('02.03.03-U2')
@auto_launch(True)
@need_shinken_stop(True)
def fix_flapping_thresholds():
    g_did_run = False  # did we fix something?
    db = get_synchronizer_db()
    if not db.collection_names():
        return g_did_run
    
    to_fix = [getattr(db, 'configuration-stagging-service'), getattr(db, 'configuration-production-service'),
              getattr(db, 'configuration-stagging-host'), getattr(db, 'configuration-production-host')]
    
    for col in to_fix:
        for elem in col.find({'low_flap_threshold': {"$exists": 1}, 'high_flap_threshold': {"$exists": 1}}):
            if elem['low_flap_threshold'] > elem['high_flap_threshold']:
                g_did_run = True
                elem['low_flap_threshold'] = elem['high_flap_threshold']
                logger.debug('Fixed %s' % elem.get('name', elem.get('service_description', elem.get('host_name', '-- Missing name -- %s' % elem))))
                col.update({'_id': elem['_id']}, elem)
    
    return g_did_run


@add_fix
@add_doc("""Will remove host_name field inheritance from check templates""")
@add_context_daemons(['synchronizer'])
@add_version('02.05.00')
@auto_launch(True)
@need_shinken_stop(True)
def fix_host_name_inheritance():
    g_did_run = False  # did we fix something?
    db = get_synchronizer_db()
    if not db.collection_names():
        return g_did_run
    
    to_fix = [getattr(db, 'configuration-stagging-service'), getattr(db, 'configuration-production-service')]
    
    for col in to_fix:
        found_to_fix = list(col.find({'name': {'$exists': True}, 'host_name': {'$exists': True}}))
        
        for service in found_to_fix:
            g_did_run = True
            for son in col.find({'use': re.compile('(^|[&|,])\s*%s\s*([&|,]|$)' % service['name']), 'host_name': {'$exists': False}}):
                logger.debug('Applying host_name inheritance %s -> %s' % (service['name'], son.get('service_description', son.get('name', '-- Missing name --'))))
                son['host_name'] = service['host_name']
                col.update({'_id': son['_id']}, son)
                if son.get('name', ''):
                    found_to_fix.append(son)  # Python supports extending a list that is currently iterated. Saves some recursion tricks
            del service['host_name']
            logger.debug('Removing host_name from %s' % (service['name']))
            col.update({'_id': service['_id']}, service)
    
    return g_did_run


@add_fix
@add_doc("""Will move bp_rule arguments to a new configuration key called bp_rule""")
@add_context_daemons(['synchronizer'])
@add_version('02.05.00')
@auto_launch(True)
@need_shinken_stop(True)
def fix_bp_rule_args():
    g_did_run = False  # did we fix something?
    db = get_synchronizer_db()
    if not db.collection_names():
        return g_did_run
    
    to_fix = ['stagging', 'production']
    
    for db_name in to_fix:
        col = getattr(db, 'configuration-%s-host' % db_name)
        found_to_fix = list(col.find({'check_command': re.compile(r'^bp_rule.*')}))
        for cluster in found_to_fix:
            g_did_run = True
            cluster['is_cluster'] = '1'
            if '!' in cluster['check_command']:
                cluster['bp_rule'] = '!'.join(cluster['check_command'].split('!')[1:])
            del cluster['check_command']
            col.update({'_id': cluster['_id']}, cluster)
            logger.debug('Moving cluster %s rules to bp_rule key (in %s)' % (cluster.get('host_name', cluster.get('name', '-- Missing name --')), db_name))
    
    return g_did_run


# It will archive sla older than last week.
@add_fix
@add_doc('Will archive sla older than last week.')
@add_context_daemons(['broker'])
@add_data_type('sla')
@add_version('02.03.03-U1')
@auto_launch(False)
@need_shinken_stop(True)
def fix_no_archive_sla():
    g_did_run = False
    database_name = 'shinken'
    
    db = get_broker_db()
    if not db.collection_names():
        return g_did_run
    now = time.time()
    
    sla_module = None
    try:
        _mod_definition = ShinkenModuleDefinition({'module_name': 'sla', 'module_type': 'sla', 'uri': 'localhost', 'database': database_name, 'keep_raw_sla_day': '7'})
        sla_source = imp.load_source('sla', '/var/lib/shinken/modules/sla/module.py')
        sla_module = sla_source.get_instance(_mod_definition)
        sla_module.init(logger.level <= 10)
    except Exception as e:
        logger.error('The sla module load fail [%s]' % e)
        logger.error(traceback.format_exc())
    
    exclude_collection = []
    for day_count in range(0, 1):
        tm = time.localtime(now - day_count * 86400)
        exclude_collection.append("%s_%s" % (tm.tm_yday, tm.tm_year))
    to_archives = []
    logger.debug("Exclude collection [%s]" % exclude_collection)
    for collection_names in db.collection_names():
        if collection_names not in exclude_collection:
            try:
                yday = int(collection_names.split('_')[0])
                year = int(collection_names.split('_')[1])
                if year > 2000:
                    to_archives.append((yday, year))
                    logger.debug("to_archive [%s-%s]" % (yday, year))
            except:
                pass
    
    for to_archive in to_archives:
        _g_did_run = sla_module.archive_collection(to_archive[0], to_archive[1])
        todel = getattr(db, 'has_been_archive_%s_%s' % (to_archive[0], to_archive[1]), None)
        if _g_did_run and todel:
            todel.drop()
        g_did_run = _g_did_run or g_did_run
    
    return g_did_run


@add_fix
@add_doc('Will remove unknow key from configuration')
@add_context_daemons(['synchronizer'])
@add_version('02.04.00')
@auto_launch(False)
@need_shinken_stop(True)
def fix_remove_unknown_keys():
    g_did_run = False  # did we fix something?
    db = get_synchronizer_db()
    if not db.collection_names():
        return g_did_run
    
    try:
        from shinken.synchronizer.dao.datamanagerV2 import DataManagerV2
        from shinken.synchronizer.dao.helpers import get_name_from_type
        from shinken.synchronizer.dao.dataprovider.dataprovider_mongo import DataProviderMongo
        from shinken.synchronizer.dao.def_items import DEF_ITEMS, ITEM_STATE, ITEM_TYPE, NOT_TO_LOOK
    except ImportError:
        logger.error("You cannot run this fix on your shinken version please upgrade it.")
        return g_did_run
    
    data_provider_mongo = DataProviderMongo(db)
    
    for item_type in DEF_ITEMS:
        if item_type == ITEM_TYPE.TIMEPERIODS:
            continue
        _class = DEF_ITEMS[item_type]['class']
        prop_dict = getattr(_class, 'properties', {})
        prop_passthrough_dict = getattr(_class, 'passthrough', {})
        prop_not_inherited_passthrough = getattr(_class, 'not_inherited_passthrough', {})
        # blacklist prop are not allowed
        # prop_blacklist_dict = getattr(_class, 'blacklist', {})
        
        allowed_properties = NOT_TO_LOOK + prop_dict.keys() + prop_passthrough_dict.keys() + prop_not_inherited_passthrough.keys()
        
        for item_state in [ITEM_STATE.STAGGING, ITEM_STATE.PRODUCTION, ITEM_STATE.WORKING_AREA]:
            all_items = data_provider_mongo.find_items(item_type, item_state)
            for item in all_items:
                to_del_properties = [property for property in item if (property not in allowed_properties and not property.startswith('_'))]
                
                for property in to_del_properties:
                    print 'remove property [%s] in item [%s-%s]' % (property, item_type, get_name_from_type(item_type, item))
                    del item[property]
                    g_did_run = True
                
                if to_del_properties:
                    data_provider_mongo.save_item(item, item_type, item_state)
    
    if not g_did_run:
        return g_did_run
    
    col = getattr(db, 'synchronizer-info')
    database_info = col.find_one({'_id': 'database_info'})
    if not database_info:
        database_info = {'_id': 'database_info'}
    
    database_info['remove_unknown_key'] = 1
    database_info['must_remake_import'] = 1
    col.save(database_info)
    
    return g_did_run


@add_fix
@add_doc('Will delete non 0-5 business impact from configuration')
@add_context_daemons(['synchronizer'])
@add_version('02.04.00')
@auto_launch(True)
@need_shinken_stop(True)
def fix_business_impact():
    g_did_run = False  # did we fix something?
    db = get_synchronizer_db()
    if not db.collection_names():
        return g_did_run
    
    to_fix = [getattr(db, 'configuration-stagging-service'), getattr(db, 'configuration-production-service'),
              getattr(db, 'configuration-stagging-host'), getattr(db, 'configuration-production-host'),
              getattr(db, 'configuration-stagging-businessimpactmodulation'), getattr(db, 'configuration-production-businessimpactmodulation')]
    for col in to_fix:
        for item in col.find({'business_impact': {'$exists': True}}):
            if item['business_impact'] not in ('0', '1', '2', '3', '4', '5'):
                g_did_run = True
                del item['business_impact']
                col.update({'_id': item['_id']}, item)
    to_fix = [getattr(db, 'configuration-stagging-contact'), getattr(db, 'configuration-production-contact'),
              getattr(db, 'configuration-stagging-notificationway'), getattr(db, 'configuration-production-notificationway')]
    for col in to_fix:
        for item in col.find({'min_business_impact': {'$exists': True}}):
            if item['min_business_impact'] not in ('0', '1', '2', '3', '4', '5'):
                g_did_run = True
                del item['min_business_impact']
                col.update({'_id': item['_id']}, item)
    return g_did_run


@add_fix
@add_doc('Fix wrong uuid in sla info')
@add_context_daemons(['broker'])
@add_data_type('sla')
@add_version('02.05.00')
@auto_launch(True)
@need_shinken_stop(True)
def fix_uuid_in_sla_info():
    g_did_run = False
    db = get_broker_db()
    if not db.collection_names():
        return g_did_run
    
    col_sla_info = getattr(db, 'sla_info')
    to_fixs = list(col_sla_info.find({'_id': re.compile(r'.*-.*-.*')}))
    
    if not to_fixs:
        return g_did_run
    
    bulk_ops = col_sla_info.initialize_unordered_bulk_op()
    bulk_ops.find({'_id': re.compile(r'.*-.*-.*')}).remove()
    
    g_did_run = True
    for to_fix in to_fixs:
        to_fix['_id'] = to_fix['_id'][0:to_fix['_id'].rfind('-')]
        bulk_ops.insert(to_fix)
    
    try:
        bulk_ops.execute()
    except BulkWriteError as bwe:
        have_only_dup_key_error = True
        write_errors = ""
        for errmsg in bwe.details['writeErrors']:
            err_msg = errmsg.get("errmsg", "errmsg not found")
            if 'dup key' not in err_msg:
                write_errors += err_msg
                have_only_dup_key_error = False
        
        if not have_only_dup_key_error:
            sanitize_log.write("[SLA][bulk-%s] update_sla bulk fail : %s" % ('sla_info', write_errors), logging.ERROR)
    
    return g_did_run


@add_fix
@add_doc('Will rename keys from configuration')
@add_context_daemons(['synchronizer'])
@add_version('02.04.00')
@auto_launch(False)
@need_shinken_stop(True)
def fix_rename_key(_old, _new):
    g_did_run = False  # did we fix something?
    db = get_synchronizer_db()
    if not db.collection_names():
        return g_did_run
    
    if not _old:
        sanitize_log.write("Please specify a property to rename with --old option", logging.ERROR)
        return False
    if not _new:
        sanitize_log.write("Please specify a property to rename with --new option", logging.ERROR)
        return False
    
    try:
        from shinken.synchronizer.dao.datamanagerV2 import DataManagerV2, get_name_from_type
        from shinken.synchronizer.dao.dataprovider.dataprovider_mongo import DataProviderMongo
        from shinken.synchronizer.dao.def_items import DEF_ITEMS, ITEM_STATE, ITEM_TYPE, NOT_TO_LOOK
    except ImportError:
        logger.error("You cannot run this fix on your shinken version please upgrade it.")
        return False
    
    data_provider_mongo = DataProviderMongo(db)
    
    asking = False
    for item_type in DEF_ITEMS:
        _class = DEF_ITEMS[item_type]['class']
        prop_dict = getattr(_class, 'properties', {})
        prop_passthrough_dict = getattr(_class, 'passthrough', {})
        prop_not_inherited_passthrough = getattr(_class, 'not_inherited_passthrough', {})
        # blacklist prop are not allowed
        # prop_blacklist_dict = getattr(_class, 'blacklist', {})
        
        allowed_properties = NOT_TO_LOOK + prop_dict.keys() + prop_passthrough_dict.keys() + prop_not_inherited_passthrough.keys()
        
        if _old in allowed_properties:
            asking = True
            break
    
    if asking:
        logger.warn('Warning, the key you attempt to rename is a key used by Shinken, are you sure about renaming this key ? Enter YES to continue.')
        input_value = raw_input()
        if input_value != 'YES':
            logger.warn('Abort')
            return False
    
    for item_type in DEF_ITEMS:
        for item_state in [ITEM_STATE.STAGGING, ITEM_STATE.PRODUCTION, ITEM_STATE.WORKING_AREA]:
            all_items = data_provider_mongo.find_items(item_type, item_state)
            for item in all_items:
                if _old in item:
                    item[_new] = item[_old]
                    del item[_old]
                    print 'rename property [%s]->[%s] in item [%s-%s]' % (_old, _new, item_type, get_name_from_type(item_type, item))
                    data_provider_mongo.save_item(item, item_type, item_state)
                    g_did_run = True
    
    return g_did_run


@add_fix
@add_doc('Add missing add-ons to synchronizer configuration and make sure they are disabled when their status states "disabled"')
@add_context_daemons(['synchronizer'])
@add_version('02.05.00')
@auto_launch(True)
@need_shinken_stop(True)
def fix_synchronizer_install_missing_addons():
    g_run = False
    SHINKEN_ADDONS = ["nagvis", "nagvis-shinken-architecture"]
    
    for addon in SHINKEN_ADDONS:
        try:
            subprocess.check_call(['shinken-addons-has', addon], stdout=subprocess.PIPE)
        except subprocess.CalledProcessError as e:
            try:
                p = subprocess.Popen(["shinken-addons-disable", "--force", addon], stdout=subprocess.PIPE, stderr=subprocess.STDOUT)
                (output, _) = p.communicate()
                if "ALREADY DISABLED" not in output:
                    g_run = True
            except subprocess.CalledProcessError as e:
                print "ERROR: Unable to disable Shinken Nagvis addons\n"
    
    return g_run


@add_fix
@add_doc('Add missing configuration files')
@add_context_daemons(['synchronizer'])
@add_version('02.05.00')
@auto_launch(True)
@need_shinken_stop(True)
def fix_missing_skeletons():
    SHINKEN_CFG_DIR = "/etc/shinken"
    SYNCHRONIZER_CFG_FILE = "/etc/shinken/synchronizer.cfg"
    g_run = False
    SKELETONS_DIR = "/etc/shinken-skeletons"
    SKELETONS_TO_COPY = ["listeners", "analyzers"]
    
    with open(SYNCHRONIZER_CFG_FILE) as fp:
        synchronizer_cfg = fp.readlines()
    
    cfg_dirs = []
    for cfg_line in synchronizer_cfg:
        cfg_line = cfg_line.strip()
        if cfg_line.startswith("#"):
            continue
        
        if cfg_line.startswith('cfg_dir'):
            cfg_dirs.append(cfg_line.split('=')[1].strip())
    
    lines_to_add = []
    for skel in SKELETONS_TO_COPY:
        skel_dir = os.path.join(SKELETONS_DIR, skel)
        cfg_dir = os.path.join(SHINKEN_CFG_DIR, skel)
        if not os.path.exists(cfg_dir):
            shutil.copytree(skel_dir, cfg_dir)
            g_run = True
            if skel not in cfg_dirs:
                lines_to_add.append('cfg_dir=%s\n' % skel)
    
    if lines_to_add:
        synchronizer_cfg.extend(lines_to_add)
        with open(SYNCHRONIZER_CFG_FILE, 'w') as fp:
            fp.writelines(synchronizer_cfg)
    
    return g_run


@add_fix
@add_doc('Add missing sources for Synchronizer')
@add_context_daemons(['synchronizer'])
@add_version('02.05.00')
@auto_launch(True)
@need_shinken_stop(True)
def fix_missing_sources():
    return _set_default_values_in_cfg_files('synchronizer', 'source', '/etc/shinken/synchronizers', set(('listener-rest', 'server-analyzer')))


@add_fix
@add_doc('Make sure protected fields parameters are all present and up to date')
@add_context_daemons(['synchronizer'])
@add_version('02.05.00')
@auto_launch(True)
@need_shinken_stop(True)
def fix_update_protected_fields_parameters():
    SYNCHRONIZER_CFG_FILE = "/etc/shinken/synchronizer.cfg"
    
    g_run = False
    
    with open(SYNCHRONIZER_CFG_FILE) as fp:
        lines = fp.readlines()
    
    params_to_update = {
        'protect_fields__activate_database_encryption' : {
            'found'     : False,
            'updated'   : False,
            'prev_value': None,
            'next_name' : 'protect_fields__activate_encryption',
            'next_value': None,
            'default'   : '0'},
        'protect_fields__activate_encryption'          : {
            'found'     : False,
            'updated'   : False,
            'prev_value': None,
            'next_name' : 'protect_fields__activate_encryption',
            'next_value': None,
            'default'   : '0'},
        'protect_fields__activate_interface_encryption': {
            'found'     : False,
            'updated'   : False,
            'prev_value': None,
            'next_name' : 'protect_fields__activate_encryption',
            'next_value': None,
            'default'   : '0'},
        'protect_fields__encryption_keyfile'           : {
            'found'     : False,
            'updated'   : False,
            'prev_value': None,
            'next_name' : 'protect_fields__encryption_keyfile',
            'next_value': None,
            'default'   : '/etc/shinken/secrets/protected_fields_key'},
        'protect_fields__substrings_matching_fields'   : {
            'found'     : False,
            'updated'   : False,
            'prev_value': None,
            'next_name' : 'protect_fields__substrings_matching_fields',
            'next_value': None,
            'default'   : 'PASSWORD,PASSPHRASE,PASSE,DOMAINUSER,MSSQLUSER,MYSQLUSER,ORACLE_USER,SSH_USER,LOGIN'},
    }
    
    param_count = {}
    index_of_protected_fields_section = None
    
    # Find all versions of each of the parameters to migrate, and keep the last one
    for (index, line) in enumerate(lines):
        line = line.strip()
        if line.startswith("#"):
            continue
        if not line.startswith("protect_fields__"):
            continue
        
        (param, value) = line.split("=")
        
        if param in params_to_update:
            param_count[param] = param_count.get(param, 0) + 1
            
            if not index_of_protected_fields_section:
                index_of_protected_fields_section = index
            params_to_update[param]['prev_value'] = value
            params_to_update[param]['next_value'] = value
            params_to_update[param]['found'] = True
            params_to_update[params_to_update[param]['next_name']]['prev_value'] = value
            params_to_update[params_to_update[param]['next_name']]['next_value'] = value
            params_to_update[params_to_update[param]['next_name']]['found'] = True
    
    # If the parameters don't appear in the file, their default values will be inserted at top
    if not index_of_protected_fields_section:
        index_of_protected_fields_section = 1
    
    # Undefined parameters get their default values
    for key, param in params_to_update.iteritems():
        if not param['found']:
            param['next_value'] = param['default']
    
    # Prepare new config file to be rewritten with the new version of each parameter
    # Keep the list of lines to delete (those containing duplicate versions of the parameters)
    index_to_delete = []
    for (index, line) in enumerate(lines):
        line = line.strip()
        previous_line = lines[index].strip()
        if line.startswith("#"):
            continue
        if not line.startswith("protect_fields__"):
            continue
        
        (param, value) = line.split("=")
        
        if param in params_to_update:
            current_param = params_to_update[param]
            if current_param['updated'] == False and current_param['next_value'] and not params_to_update[current_param['next_name']]['updated']:
                lines[index] = "%s=%s\n" % (current_param['next_name'], current_param['next_value'])
                current_param['updated'] = True
                params_to_update[current_param['next_name']]['updated'] = True
                if lines[index].strip() != previous_line:
                    g_run = True
            else:
                current_param['updated'] = True
                index_to_delete.append(index)
    
    error_messages = []
    for param, count in param_count.iteritems():
        if count > 1:
            error_messages.append("Param [%s] defined multiple times in synchronizer.cfg " % param)
    
    if error_messages:
        error_messages.insert(0, "SANATIZE ABORTED TO PREVENT DATA CORRUPTION")
        return "\n".join(error_messages)
    
    # Remove multiple params definitions
    for index in sorted(index_to_delete, reverse=True):
        g_run = True
        del lines[index]
    
    # Add parameters missing from file
    for param in [params_to_update[current_param] for current_param in params_to_update if not params_to_update[current_param]['updated']]:
        if not params_to_update[param['next_name']]['updated']:
            lines.insert(index_of_protected_fields_section, "%s=%s\n" % (param['next_name'], param['next_value']))
            g_run = True
    
    with open(SYNCHRONIZER_CFG_FILE, "w") as fp:
        fp.writelines(lines)
    
    return g_run


@add_fix
@add_doc('Make sure overload configuration files and main configuration files respect the new configuration files tree')
@add_context_daemons(['synchronizer', 'arbiter'])
@add_version('02.05.00')
@auto_launch(True)
@need_shinken_stop(True)
def fix_all_move_overload_files():
    g_run = False
    ROOT_DIR = "/etc/shinken-user/configuration"
    DAEMONS = {
        'arbiters'     : '/etc/shinken/shinken.cfg',
        'synchronizers': '/etc/shinken/synchronizer.cfg',
        # In 2.4, OVERLOAD file for broker is not moved
    }
    shinken_uid, shinken_gid = pwd.getpwnam("shinken")[2:4]
    
    for daemon, daemon_cfg_file in DAEMONS.iteritems():
        
        overload_file = "%s_cfg_overload.cfg" % daemon[:-1]
        daemon_dir = os.path.join(ROOT_DIR, 'daemons', daemon)
        if not os.path.isdir(daemon_dir):
            os.mkdir(daemon_dir)
        
        src_overload_file = os.path.join(ROOT_DIR, overload_file)
        dst_overload_file = os.path.join(ROOT_DIR, 'daemons', daemon, overload_file)
        if os.path.exists(src_overload_file) and not os.path.exists(dst_overload_file):
            shutil.move(src_overload_file, dst_overload_file)
            g_run = True
        
        if not os.path.exists(dst_overload_file):
            # Create empty overload file
            with open(dst_overload_file, "w"):
                pass
            os.chown(dst_overload_file, shinken_uid, shinken_gid)
            g_run = True
        
        with open(daemon_cfg_file) as main_cfg_file_fp:
            daemon_config = main_cfg_file_fp.readlines()
        
        overload_path_found = False
        for lineno, config in enumerate(daemon_config):
            config = config.strip()
            if config.startswith('cfg_file=') and overload_file in config:
                daemon_config[lineno] = "cfg_file=%s\n" % dst_overload_file
                overload_path_found = True
                g_run = g_run or config.split('=')[1].strip() != dst_overload_file
        
        # If not entry for the overload file was found, we add it
        if not overload_path_found:
            daemon_config.append("cfg_file=%s\n" % dst_overload_file)
            g_run = True
        
        with open(daemon_cfg_file, "w") as fp:
            fp.writelines(daemon_config)
    
    return g_run


@add_fix
@add_doc('Remove undesired hostgroups in working area')
@add_context_daemons(['synchronizer'])
@add_version('02.05.00')
@auto_launch(True)
@need_shinken_stop(True)
def remove_hostgroups_in_working_area():
    g_did_run = False
    db = get_synchronizer_db()
    if not db.collection_names():
        return g_did_run
    
    col = getattr(db, 'configuration-working-area-hostgroup')
    if col.count() > 0:
        col.remove({})
        g_did_run = True
    return g_did_run


@add_fix
@add_doc("Make index for sla module")
@add_context_daemons(['broker'])
@add_version('02.06.00')
@auto_launch(True)
@need_shinken_stop(True)
def make_index_for_sla_module():
    g_did_run = False
    shinken_database = get_broker_db()
    collection_names = shinken_database.collection_names()
    if not collection_names:
        return g_did_run
    
    if 'sla_archive' not in collection_names or 'sla_info' not in collection_names:
        return g_did_run
    
    col_archive = getattr(shinken_database, 'sla_archive')
    col_sla_info = getattr(shinken_database, 'sla_info')
    
    col_sla_info_index_information = col_sla_info.index_information()
    col_archive_index_information = col_archive.index_information()
    
    # Add index for the collections
    if 'names' not in col_sla_info_index_information:
        col_sla_info.ensure_index([('host_name', pymongo.ASCENDING), ('service_description', pymongo.ASCENDING)], name='names')
        g_did_run = True
    if 'monitoring_start_time' not in col_sla_info_index_information:
        col_sla_info.ensure_index([('monitoring_start_time', pymongo.ASCENDING)], name='monitoring_start_time')
        g_did_run = True
    
    if 'hname_type_year_yday' not in col_archive_index_information:
        col_archive.ensure_index([('hname', pymongo.ASCENDING), ('type', pymongo.ASCENDING), ('year', pymongo.ASCENDING), ('yday', pymongo.ASCENDING), ], name='hname_type_year_yday')
        g_did_run = True
    if 'hname_sdesc_type_year_yday' not in col_archive_index_information:
        col_archive.ensure_index([('hname', pymongo.ASCENDING), ('type', pymongo.ASCENDING), ('sdesc', pymongo.ASCENDING), ('year', pymongo.ASCENDING), ('yday', pymongo.ASCENDING), ], name='hname_sdesc_type_year_yday')
        g_did_run = True
    if 'uuid_year_yday_idx' not in col_archive_index_information:
        col_archive.ensure_index([('uuid', pymongo.ASCENDING), ('year', pymongo.ASCENDING), ('yday', pymongo.ASCENDING), ], name='uuid_year_yday_idx')
        g_did_run = True
    if 'uuid_idx' not in col_archive_index_information:
        col_archive.ensure_index([('uuid', pymongo.ASCENDING)], name='uuid_idx')
        g_did_run = True
    if 'version_idx' not in col_archive_index_information:
        col_archive.ensure_index([('version', pymongo.ASCENDING)], name='version_idx')
        g_did_run = True
    
    return g_did_run


@add_fix
@add_doc("Migrate synchronizer's link from name to ids")
@add_context_daemons(['synchronizer'])
@add_version('02.06.00')
@auto_launch(True)
@need_shinken_stop(True)
def migrate_links_name_into_id():
    g_did_run = False  # did we fix something?
    db = get_synchronizer_db()
    if not db.collection_names():
        return g_did_run
    
    try:
        from shinken.synchronizer.dao.def_items import ITEM_TYPE
        from shinken.synchronizer.dao.datamanagerV2 import DataManagerV2
        from shinken.synchronizer.dao.dataprovider.dataprovider_mongo import DataProviderMongo
        from shinken.synchronizer.dao.dataprovider.dataprovider_memory import DataProviderMemory
        from shinken.synchronizer.dao.def_items import DEF_ITEMS, ITEM_STATE
        from shinken.synchronizer.dao.item_saving_formatter import build_link
        from shinken.synchronizer.dao.transactions.transactions import DBTransaction
    except ImportError:
        logger.error("You cannot run this fix on your shinken version please upgrade it.")
        return g_did_run
    
    col = getattr(db, 'synchronizer-info')
    database_info = col.find_one({'_id': 'database_info'}) or {}
    if database_info.get('new_link_format', 0) == 1:
        return g_did_run
    
    data_provider_mongo = DataProviderMongo(db)
    
    managed_states = (ITEM_STATE.STAGGING, ITEM_STATE.NEW, ITEM_STATE.WORKING_AREA, ITEM_STATE.PRODUCTION)
    all_items = {}
    to_save_items = {}
    for item_state in managed_states:
        all_items[item_state] = {}
        to_save_items[item_state] = {}
        for item_type in DEF_ITEMS.iterkeys():
            list_item = data_provider_mongo.find_items(item_type, item_state)
            all_items[item_state][item_type] = list_item
            to_save_items[item_state][item_type] = []
    
    data_provider_memory = DataProviderMemory(all_items)
    datamanagerV2 = DataManagerV2(data_provider_memory, compute_double_links=False, use_default_callbacks=False)
    
    for item_state in managed_states:
        # We need to build links for services first because their links (esp. "host_name") are needed to build links
        # for service_overrides
        all_types_except_services = set(ITEM_TYPE.ALL_TYPES).difference(ITEM_TYPE.ALL_SERVICES)
        for item_type in list(ITEM_TYPE.ALL_SERVICES) + list(all_types_except_services):
            for item in all_items[item_state][item_type]:
                edited = build_link(item, item_type, item_state, datamanagerV2)
                if edited:
                    to_save_items[item_state][item_type].append(item)
    
    for item_state, item_types in to_save_items.iteritems():
        for item_type, to_save_items in item_types.iteritems():
            for to_save_item in to_save_items:
                data_provider_mongo.save_item(to_save_item, item_type=item_type, item_state=item_state)
                g_did_run = True
    
    # The format of merge items change and we don't want to keep any of this items in this collections.
    # They will be rebuild with the must_remake_import = 1 option.
    for item_type in DEF_ITEMS.iterkeys():
        data_provider_mongo.get_collection(item_type, ITEM_STATE.MERGE_SOURCES).drop()
    
    if not g_did_run:
        return g_did_run
    
    col = getattr(db, 'synchronizer-info')
    database_info = col.find_one({'_id': 'database_info'})
    if not database_info:
        database_info = {'_id': 'database_info'}
    database_info['new_link_format'] = 1
    database_info['must_remake_import'] = 1
    col.save(database_info)
    return g_did_run


def _cipher_override(data_provider_mongo, db_cipher, item, item_state, item_type, source=''):
    effective = False
    if 'service_overrides' in item:
        item['service_overrides'], effective = db_cipher._read_service_override(item['service_overrides'], db_cipher._cipher_value, item_state, item, item_type, None)
        data_provider_mongo.save_item(item, item_type=item_type, item_state=item_state, item_source=source)
    return effective


# Will make double link for host, host template, hostgroup, contact, contact template, contactgroup, in stagging and production
@add_fix
@add_doc('Will make double link for host, host template, hostgroup, contact, contact template, contactgroup, in stagging and production. (update for 02.06.00)')
@add_context_daemons(['synchronizer'])
@add_version('02.06.00')
@auto_launch(True)
@need_shinken_stop(True)
def fix_double_link():
    g_did_run = False  # did we fix something?
    db = get_synchronizer_db()
    if not db.collection_names():
        return g_did_run
    
    try:
        from shinken.synchronizer.dao.datamanagerV2 import DataManagerV2, FALLBACK_USER
        from shinken.synchronizer.dao.dataprovider.dataprovider_metadata import DataProviderMetadata
        from shinken.synchronizer.dao.dataprovider.dataprovider_mongo import DataProviderMongo
        from shinken.synchronizer.dao.def_items import DEF_ITEMS, ITEM_STATE
        from shinken.synchronizer.dao.transactions.transactions import DBTransaction
    except ImportError:
        logger.error("You cannot run this fix on your shinken version please upgrade it.")
        return g_did_run
    
    col = getattr(db, 'synchronizer-info')
    database_info = col.find_one({'_id': 'database_info'})
    if not database_info or database_info.get('new_link_format', 0) != 1:
        logger.error("You cannot run this fix on your shinken version please upgrade it or run the sanatize 'migrate_links_name_into_id'.")
        return g_did_run
    
    data_provider_mongo = DataProviderMongo(db)
    dataprovider = DataProviderMetadata(data_provider_mongo)
    dataprovider.load_from_data_provider()
    datamanagerV2 = DataManagerV2(dataprovider, compute_double_links=False, use_default_callbacks=False)
    
    nb_items = 0
    with DBTransaction(datamanagerV2=datamanagerV2):
        for item_state in [ITEM_STATE.STAGGING, ITEM_STATE.PRODUCTION]:
            for item_type in DEF_ITEMS:
                items = list(datamanagerV2.find_items(item_type, item_state=item_state))
                for item in items:
                    if datamanagerV2._make_double_link(item, item_type, FALLBACK_USER, item)[2]:
                        g_did_run = True
                        nb_items += 1
    
    if not g_did_run:
        return g_did_run
    
    logger.info('[fix_double_link] make double link for [%d] items.' % nb_items)
    
    col = getattr(db, 'synchronizer-info')
    database_info = col.find_one({'_id': 'database_info'})
    if not database_info:
        database_info = {'_id': 'database_info'}
    
    database_info['all_double_link'] = 1
    database_info['must_remake_import'] = 1
    col.save(database_info)
    
    return g_did_run


@add_fix
@add_doc("Encrypt service overrides properties which are protected fields")
@add_context_daemons(['synchronizer'])
@add_version('02.06.00')
@auto_launch(True)
@need_shinken_stop(True)
def fix_service_overrides_encryption():
    g_did_run = False  # did we fix something?
    db = get_synchronizer_db()
    if not db.collection_names():
        return g_did_run
    
    try:
        from hashlib import sha256
        from shinken.synchronizer.dao.datamanagerV2 import DataManagerV2
        from shinken.synchronizer.dao.dataprovider.dataprovider_mongo import DataProviderMongo
        from shinken.synchronizer.dao.dataprovider.dataprovider_memory import DataProviderMemory
        from shinken.synchronizer.dao.def_items import DEF_ITEMS, ITEM_STATE, ITEM_TYPE
        from shinken.synchronizer.dao.item_saving_formatter import build_link
        from shinken.synchronizer.dao.transactions.transactions import DBTransaction
        from shinken.synchronizer.dao.crypto import DatabaseCipher
    except ImportError:
        logger.error("You cannot run this fix on your shinken version please upgrade it.")
        return g_did_run
    
    col = getattr(db, 'synchronizer-info')
    database_info = col.find_one({'_id': 'database_info'}) or {}
    if database_info.get('new_link_format', 0) == 0:
        logger.warning("The database must have the new link format for this sanatize to run.")
        return g_did_run
    if database_info.get("overrides_encrypted", 0) == 1:
        logger.info("Overrides already encrypted")
        return g_did_run
    
    synchronizer_info = col.find_one({'_id': 'protected_fields_info'}) or {}
    enabled = synchronizer_info.get('protect_fields__activate_database_encryption')
    
    if not enabled:
        logger.info("Encryption not enabled ; this sanatize is not necessary")
        return g_did_run
    
    db_key_name = synchronizer_info.get('protect_fields__encryption_key_name')
    db_key_hash = synchronizer_info.get('protect_fields__encryption_keyfile_hash')
    properties = ",".join(synchronizer_info.get('protect_fields__substrings_matching_fields'))
    keyfile_name = None
    for line in open('/etc/shinken/synchronizer.cfg'):
        if line.startswith('protect_fields__encryption_keyfile'):
            keyfile_name = line.split('=')[1].strip()
    
    if enabled and not keyfile_name:
        logger.critical("Encryption is enabled but the keyfile is not defined in your configuration.")
        return g_did_run
    
    try:
        complete_key = open(keyfile_name).read().strip()
        key_value = complete_key[complete_key.index("|") + 1:]
        key_name = complete_key[:complete_key.index("|")].strip().decode('utf8', 'ignore')
    except (OSError, IOError) as e:
        logger.critical("[protected_fields]  Make sure the file exists and that the user 'shinken' is allowed to read it")
        logger.critical("[protected_fields]  The synchronizer will not start because it will not be able to process encrypted data.")
        logger.critical("[protected_fields]  Cannot read the protected fields secret file '%s' : %s " % (keyfile_name, str(e)))
        return g_did_run
    except ValueError:
        logger.critical("[protected_fields]  The keyfile seems to be corrupted. If you have an export available, you can use it to restore the correct key")
        logger.critical("[protected_fields]  using the command shinken-protected-fields-keyfile-restore and restart the synchronizer.")
        logger.critical("[protected_fields]  The synchronizer will not start in order not to corrupt data.")
        logger.critical("[protected_fields]  The key contained in the keyfile does not have the right structure.")
        return g_did_run
    
    if db_key_hash != sha256(key_value).hexdigest() or db_key_name != key_name:
        logger.critical("protected_fields] The keyfile defined in the synchronizer configuration doesn't match the key present in your database")
        return g_did_run
    
    db_cipher = DatabaseCipher(enabled, properties, keyfile_name)
    
    data_provider_mongo = DataProviderMongo(db)
    
    # Source name is the collection name with the prefix "data-" and the suffix -<type> removed
    data_sources = set([name.replace('data-', '').rsplit('-', 1)[0] for name in db.collection_names() if name.startswith('data-')])
    managed_states = [ITEM_STATE.STAGGING, ITEM_STATE.WORKING_AREA, ITEM_STATE.PRODUCTION, ITEM_STATE.MERGE_SOURCES, ITEM_STATE.RAW_SOURCES]
    
    for item_state in managed_states:
        for item_type in ITEM_TYPE.ALL_HOST_CLASS:
            if item_state == ITEM_STATE.RAW_SOURCES:
                for source in data_sources:
                    for item in data_provider_mongo.find_items(item_type, item_state, item_source=source):
                        g_did_run |= _cipher_override(data_provider_mongo, db_cipher, item, item_state, item_type, source=source)
            else:
                for item in data_provider_mongo.find_items(item_type, item_state):
                    g_did_run |= _cipher_override(data_provider_mongo, db_cipher, item, item_state, item_type)
    
    col = getattr(db, 'synchronizer-info')
    database_info = col.find_one({'_id': 'database_info'})
    if not database_info:
        database_info = {'_id': 'database_info'}
    database_info['must_remake_import'] = 1
    database_info['overrides_encrypted'] = 1
    col.save(database_info)
    return g_did_run


@add_fix
@add_doc("Will enable synchronizer-module-database-backup for version < 02.06.00")
@add_context_daemons(['synchronizer'])
@add_version('02.06.00')
@auto_launch(True)
@need_shinken_stop(True)
def fix_enable_synchronizer_database_module():
    daemon_type = 'synchronizer'
    daemon_cfg_dir = '/etc/shinken/synchronizers'
    modules_dir = '/etc/shinken/modules/'
    skeleton_module_dir = '/etc/shinken-skeletons/modules/'
    module_file = 'synchronizer-module-database-backup.cfg'
    
    g_run = _set_default_values_in_cfg_files(daemon_type, 'modules', daemon_cfg_dir, set(('synchronizer-module-database-backup',)))
    
    if not os.path.exists(os.path.join(modules_dir, module_file)):
        shutil.copy(os.path.join(skeleton_module_dir, module_file), os.path.join(modules_dir, module_file))
        try:
            subprocess.check_call(['chown', 'shinken:shinken', os.path.join(modules_dir, module_file)], stdout=subprocess.PIPE)
        except subprocess.CalledProcessError as e:
            print 'ERROR: Unable to set the rights on file "%s"\n' % os.path.join(modules_dir, module_file)
        g_run = True
    
    return g_run


@add_fix
@add_doc("Will clean DATA in items for escape XSS char")
@add_context_daemons(['synchronizer'])
@add_version('02.06.00')
@auto_launch(True)
@need_shinken_stop(True)
def fix_escape_properties_for_xss():
    g_did_run = False  # did we fix something?
    db = get_synchronizer_db()
    if not db.collection_names():
        return g_did_run
    
    try:
        from shinken.synchronizer.dao.dataprovider.dataprovider_mongo import DataProviderMongo
        from shinken.synchronizer.dao.def_items import ITEM_STATE, ITEM_TYPE
        from shinken.synchronizer.dao.helpers import escape_XSS
        from shinken.synchronizer.dao.crypto import DatabaseCipher
    except ImportError:
        logger.error("You cannot run this fix on your shinken version please upgrade it.")
        return g_did_run
    
    synchronizer_info = getattr(db, 'synchronizer-info').find_one({'_id': 'protected_fields_info'}) or {}
    encryption_enabled = synchronizer_info.get('protect_fields__activate_database_encryption', False)
    encryption_properties = ','.join(synchronizer_info.get('protect_fields__substrings_matching_fields', ''))
    encryption_key_file_name = next((line.split('=')[1].strip() for line in open('/etc/shinken/synchronizer.cfg') if line.startswith('protect_fields__encryption_keyfile')), '')
    
    if encryption_enabled and not encryption_key_file_name:
        logger.critical("Encryption is enabled but the keyfile is not defined in your configuration.")
        return g_did_run
    
    if encryption_enabled:
        db_cipher = DatabaseCipher(encryption_enabled, encryption_properties, encryption_key_file_name)
    else:
        db_cipher = None
    data_provider_mongo = DataProviderMongo(db, db_cipher)
    
    for item_state in (ITEM_STATE.STAGGING, ITEM_STATE.WORKING_AREA, ITEM_STATE.PRODUCTION):
        for item_type in ITEM_TYPE.ALL_TYPES:
            for item in data_provider_mongo.find_items(item_type, item_state):
                save_item = False
                for property_name, property_value in item.iteritems():
                    if ((item_type in (ITEM_TYPE.CONTACTS, ITEM_TYPE.HOSTS) and property_name == 'display_name') or property_name.startswith('_')) and \
                            property_value and isinstance(property_value, basestring) and \
                            re.search(r'[<>"\'/&]', property_value):
                        property_value_initial = property_value
                        property_value = escape_XSS(property_value)
                        # We unescape & as it not escape anymore
                        property_value = property_value.replace('&amp;', '&')
                        
                        item[property_name] = property_value
                        g_did_run = property_value_initial != property_value
                        save_item = property_value_initial != property_value
                if save_item:
                    data_provider_mongo.save_item(item, item_type, item_state)
    
    return g_did_run


@add_fix
@add_doc("Will add internal options to retention mongo cfg")
@add_context_daemons(['scheduler'])
@add_version('02.06.03')
@auto_launch(True)
@need_shinken_stop(True)
def add_internal_option_to_retention_mongo_cfg():
    g_did_run = False
    cfg_file = "/etc/shinken/modules/retention-mongodb.cfg"
    
    updated_file = []
    f = open(cfg_file, 'r')
    line_index = 1
    last_line = 0
    for line in f.readlines():
        updated_file.append(line)
        if "INTERNAL : DO NOT EDIT FOLLOWING PARAMETER WITHOUT YOUR DEDICATED SUPPORT" in line:
            return g_did_run
        if '}' in line:
            last_line = line_index
        line_index += 1
    f.close()
    if last_line == 0:
        return g_did_run
    to_add = "\n" \
             "    #======== INTERNAL options =========\n" \
             "    #INTERNAL : DO NOT EDIT FOLLOWING PARAMETER WITHOUT YOUR DEDICATED SUPPORT\n" \
             "    # == Number of day we conserve retention data, after this time, we will delete data. By default it is 7 days ==\n"\
             "    #nb_of_max_retention_day	    7\n" \
             "    # == maximum number of elements load in one bulk pass ==\n" \
             "    #size_chunk_to_load	        1000\n" \
             "    # == maximum number of elements delete in one bulk pass ==\n" \
             "    #size_chunk_to_load	        1000\n"
    updated_file[last_line - 1:last_line - 1] = to_add
    f = open(cfg_file, 'w')
    f.write("".join(updated_file))
    f.close()
    g_did_run = True
    return g_did_run


@add_fix
@add_doc("Will add ssl parameters to the architecture export module")
@add_context_daemons(['arbiter'])
@add_version('02.07.00')
@auto_launch(True)
@need_shinken_stop(True)
def add_ssl_architecture_export():
    g_did_run = False
    cfg_file_path = "/etc/shinken/modules/architecture-export.cfg"
    
    if not os.path.exists(cfg_file_path):
        logger.info('The file %s doesn\'t exists, skipped' % cfg_file_path)
        return g_did_run
    
    updated_file = []
    with open(cfg_file_path, 'r') as cfg_file:
        line_index = 1
        last_line = 0
        for line in cfg_file.readlines():
            updated_file.append(line)
            if "listener_use_ssl" in line:
                return g_did_run
            if '}' in line:
                last_line = line_index
            line_index += 1
    
    if last_line == 0:
        return g_did_run
    
    to_add = ["\n    # Connection to the shinken-listener\n", "    #   if the connexion parameter to the shinken-listener has been changed, set the new value here.\n", "    #listener_use_ssl                     0\n",
              "    #listener_login                       Shinken\n", "    #listener_password                    OFU2SE4zOU1FMDdaQlJENFtljgwTcWn2hQ5ocksBWS0=\n"]
    updated_file[last_line - 1:last_line - 1] = to_add
    
    with open(cfg_file_path, 'w') as cfg_file:
        cfg_file.write("".join(updated_file))
    
    g_did_run = True
    return g_did_run


@add_fix
@add_doc("Will add ssl parameters to the shinken listener")
@add_context_daemons(['synchronizer'])
@add_version('02.07.00')
@auto_launch(True)
@need_shinken_stop(True)
def add_ssl_shinken_listener():
    g_did_run = False  # did we fix something?
    db = get_synchronizer_db()
    if not db.collection_names():
        return g_did_run
    
    try:
        from shinken.synchronizer.dao.crypto import DatabaseCipher
    except ImportError:
        logger.error("You cannot run this fix on your shinken version please upgrade it.")
        return g_did_run
    
    synchronizer_info = getattr(db, 'synchronizer-info').find_one({'_id': 'protected_fields_info'}) or {}
    encryption_enabled = True if synchronizer_info else False
    encryption_enabled = False if not encryption_enabled or not synchronizer_info['protect_fields__activate_database_encryption'] else True
    encryption_key_file_name = next((line.split('=')[1].strip() for line in open('/etc/shinken/synchronizer.cfg') if line.startswith('protect_fields__encryption_keyfile')), '')
    
    if encryption_enabled:
        if not encryption_key_file_name:
            logger.critical("Encryption is enabled but the keyfile is not defined in your configuration.")
            return g_did_run
        try:
            with open(encryption_key_file_name) as fd:
                data = fd.read()
            key = base64.decodestring(data[data.index("|") + 1:])
            cipher = AESCipher(key)
            password = cipher.encrypt('OFU2SE4zOU1FMDdaQlJENFtljgwTcWn2hQ5ocksBWS0=')
        except:
            logger.critical("Encryption is enabled but we can't take the key from your keyfile %s." % encryption_key_file_name)
            return g_did_run
    else:
        password = 'OFU2SE4zOU1FMDdaQlJENFtljgwTcWn2hQ5ocksBWS0='
    
    source_conf = getattr(db, 'sources-configuration')
    listener_shinken_conf = source_conf.find_one({'_id': "listener-shinken"})
    conf_to_apply = {'_id': "listener-shinken", 'configuration': {'authentication': True, 'login': 'Shinken', 'password': password, 'use_ssl': False, 'ssl_key': '', 'ssl_cert': ''}}
    
    if not listener_shinken_conf:
        source_conf.save(conf_to_apply)
        g_did_run = True
    else:
        _keys = listener_shinken_conf.get('configuration', {}).keys()
        if "authentication" in _keys and "ssl_key" in _keys and "ssl_cert" in _keys and listener_shinken_conf.get('configuration', {}).get('authentication') is True:
            return g_did_run
        source_conf.update_one({'_id': "listener-shinken"}, {"$set": conf_to_apply})
        g_did_run = True
    return g_did_run


@add_fix
@add_doc('Will normalize _SE_UUIDs so that they all use types instead of classes')
@add_context_daemons(['synchronizer'])
@add_version('02.07.00')
@auto_launch(False)
@need_shinken_stop(True)
def normalize_se_uuids():
    g_did_run = False  # did we fix something?
    db = get_synchronizer_db()
    if not db.collection_names():
        return g_did_run
    
    try:
        from shinken.synchronizer.dao.datamanagerV2 import DataManagerV2
        from shinken.synchronizer.dao.dataprovider.dataprovider_mongo import DataProviderMongo
        from shinken.synchronizer.dao.def_items import DEF_ITEMS, ITEM_STATE
    except ImportError:
        logger.error("You cannot run this fix on your shinken version please upgrade it.")
        return g_did_run
    
    data_provider_mongo = DataProviderMongo(db)
    
    last_synchronization_col = getattr(db, 'last_synchronization')
    if last_synchronization_col:
        sources = [source['_id'].rsplit('-', 1)[0] for source in last_synchronization_col.find({}, {'_id': 1})]
    else:
        sources = []
    
    for item_type in DEF_ITEMS:
        _class = DEF_ITEMS[item_type]['class']
        
        for source in sources:
            all_items = data_provider_mongo.find_items(item_type, item_state=ITEM_STATE.RAW_SOURCES, item_source=source)
            g_did_run |= _update_items_se_uuids_and_sync_keys(all_items, data_provider_mongo, ITEM_STATE.RAW_SOURCES, item_type, item_source=source)
        
        for item_state in [ITEM_STATE.STAGGING, ITEM_STATE.PRODUCTION, ITEM_STATE.WORKING_AREA, ITEM_STATE.MERGE_SOURCES, ITEM_STATE.NEW]:
            all_items = data_provider_mongo.find_items(item_type, item_state)
            g_did_run |= _update_items_se_uuids_and_sync_keys(all_items, data_provider_mongo, item_state, item_type)
    
    return g_did_run


def _update_items_se_uuids_and_sync_keys(all_items, data_provider_mongo, item_state, item_type, item_source=''):
    import hashlib
    g_did_run = False
    for item in all_items:
        item_se_uuid = item.get('_SE_UUID')
        if not item_se_uuid:
            continue
        
        (core, se_uuid_class_or_type, se_uuid_id) = item_se_uuid.split('-', 2)
        if se_uuid_class_or_type != item_type:
            new_se_uuid = "%s-%s-%s" % (core, item_type, se_uuid_id)
            se_uuid_hash = hashlib.md5(new_se_uuid).hexdigest()
            item['_SE_UUID'] = new_se_uuid
            item['_SE_UUID_HASH'] = se_uuid_hash
            
            if item_se_uuid in item.get('_SYNC_KEYS'):
                item['_SYNC_KEYS'].remove(item_se_uuid)
                item['_SYNC_KEYS'].append(new_se_uuid)
            g_did_run = True
            data_provider_mongo.save_item(item, item_type, item_state, item_source=item_source)
    return g_did_run


@add_fix
@add_doc("Will add rules_path to the discovery module config")
@add_context_daemons(['synchronizer'])
@add_version('02.07.00')
@auto_launch(True)
@need_shinken_stop(True)
def add_rules_path_discovery():
    g_did_run = False
    cfg_file = "/etc/shinken/sources/discovery.cfg"
    
    updated_file = []
    f = open(cfg_file, 'r')
    line_index = 1
    last_line = 0
    for line in f.readlines():
        updated_file.append(line)
        if "rules_path" in line:
            return g_did_run
        if '}' in line:
            last_line = line_index
        line_index += 1
    f.close()
    if last_line == 0:
        return g_did_run
    to_add = [
        "    rules_path              /etc/shinken-user/configuration/daemons/synchronizers/sources/discovery/discovery_rules.json\n"]
    updated_file[last_line - 1:last_line - 1] = to_add
    f = open(cfg_file, 'w')
    f.write("".join(updated_file))
    f.close()
    g_did_run = True
    return g_did_run


@add_fix
@add_doc("Will add nmap_mac_prefixes_path to the discovery module config")
@add_context_daemons(['synchronizer'])
@add_version('02.07.00')
@auto_launch(True)
@need_shinken_stop(True)
def add_nmap_mac_prefixes_path_discovery():
    g_did_run = False
    cfg_file = "/etc/shinken/sources/discovery.cfg"
    
    updated_file = []
    f = open(cfg_file, 'r')
    line_index = 1
    last_line = 0
    for line in f.readlines():
        updated_file.append(line)
        if "nmap_mac_prefixes_path" in line:
            return g_did_run
        if '}' in line:
            last_line = line_index
        line_index += 1
    f.close()
    if last_line == 0:
        return g_did_run
    to_add = [
        "    nmap_mac_prefixes_path  /etc/shinken-user/configuration/daemons/synchronizers/sources/discovery/nmap/nmap-mac-prefixes\n"]
    updated_file[last_line - 1:last_line - 1] = to_add
    f = open(cfg_file, 'w')
    f.write("".join(updated_file))
    f.close()
    g_did_run = True
    return g_did_run


@add_fix
@add_doc('Will update notes_multi_url to the new format')
@add_context_daemons(['synchronizer'])
@add_version('02.07.00')
@auto_launch(True)
@need_shinken_stop(True)
def update_notes_multi_url_syntax():
    g_did_run = False  # did we fix something?
    db = get_synchronizer_db()
    if not db.collection_names():
        return g_did_run
    
    # No need to run on source datas because this will be done by the merge validator
    to_fix = [getattr(db, 'configuration-stagging-service'), getattr(db, 'configuration-production-service'),
              getattr(db, 'newelements-service'), getattr(db, 'newelements-host'),
              getattr(db, 'merge_from_source-service'), getattr(db, 'merge_from_source-host'),
              getattr(db, 'configuration-stagging-host'), getattr(db, 'configuration-production-host'), getattr(db, 'configuration-working-area-host')]
    
    for col in to_fix:
        for item in col.find({'notes_multi_url': {'$exists': True}}):
            updated_urls = _compute_updated_notes_multi_url(item['notes_multi_url'])
            if updated_urls:
                item['notes_multi_url'] = updated_urls
                col.update({'_id': item['_id']}, item)
                g_did_run = True
        
        for item in col.find({'service_overrides': {'$exists': True}}):
            override_updated = False
            for override in item['service_overrides']['links']:
                if override['key'] != 'notes_multi_url':
                    continue
                updated_overrides = _compute_updated_notes_multi_url(override['value'])
                if updated_overrides:
                    override['value'] = updated_overrides
                    override_updated = True
            
            if override_updated:
                col.update({'_id': item['_id']}, item)
                g_did_run = True
    
    return g_did_run


@add_fix
@add_doc('Will enable cipher/uncipher transformation on your synchronizer collector linker sources')
@add_context_daemons(['synchronizer'])
@add_version('02.07.02')
@auto_launch(True)
@need_shinken_stop(True)
def update_synchronizer_collector_linker_collection_for_cipher_transformation():
    g_did_run = False
    db = get_synchronizer_db()
    if not db.collection_names():
        return g_did_run
    
    info = getattr(db, 'synchronizer-info')
    protected_fields_info = info.find_one({'_id': 'protected_fields_info'}) or {}
    encryption = protected_fields_info.get('protect_fields__activate_database_encryption', '0') in ('1', True)
    for collection_name in db.collection_names(include_system_collections=False):
        if not collection_name.endswith('_confs'):
            continue
        collection = getattr(db, collection_name)
        items = collection.find()
        for item in items:
            if not item.get('sync_name', None):
                break
            if not '@metadata' in item or ('crypted' not in item['@metadata'] and 'uncrypted' not in item['@metadata']):
                item['@metadata'] = {'uncrypted': True} if not encryption else {'crypted': True}
                g_did_run = True
                collection.save(item)
    return g_did_run


def _compute_updated_notes_multi_url(property_value):
    list_of_urls = property_value.split('~#~')
    updated_urls = []
    for url_spec in list_of_urls:
        url_components = url_spec.split('~=')
        if len(url_components) == 4:
            continue
        elif len(url_components) == 1:  # Simple URL
            url_name = ''
            url_value = url_spec
            url_tag = "tag"
            url_display = "NO"
        elif len(url_components) == 2:  # name~=URL
            (url_name, url_value) = url_components
            url_tag = "tag"
            url_display = "NO"
        else:
            continue
        
        updated_urls.append("%s~=%s~=%s~=%s" % (url_name, url_value, url_tag, url_display))
    if updated_urls:
        return "~#~".join(updated_urls)
    return None


@add_fix
@add_doc("Removes comment on the pidfile lines in mongodb configuration files (mongod, mongos and mongo-configsrv)")
@add_version('02.07.03')
@add_context_daemons(['synchronizer', 'broker', 'scheduler'])
@auto_launch(True)
@need_shinken_stop(False)
def remove_pidfile_comment_in_mongodb_config_files():
    g_did_run = False
    cfg_files = [
        "/etc/mongod.conf",
        "/etc/mongos.conf",
        "/etc/mongo-configsrv.conf",
    ]
    
    for cfg_file in cfg_files:
        if not os.path.exists(cfg_file):
            continue
        
        tmp_cfg_file = "%s.shinken_sanitize" % cfg_file
        
        with open(cfg_file, 'r') as f:
            with open(tmp_cfg_file, "w") as tmp_file:
                for line in f.readlines():
                    if "pidfilepath" in line.lower():
                        if "#" in line:
                            tmp_file.write("%s\n" % line.split("#")[0].rstrip())
                            g_did_run = True
                            continue
                    tmp_file.write(line)
        
        if g_did_run:
            shutil.move(cfg_file, "%s.shinken_sanitize_backup" % cfg_file)
            shutil.move(tmp_cfg_file, cfg_file)
        else:
            os.remove(tmp_cfg_file)
    
    return g_did_run


def _add_new_options_to_config_file(config_file, bypass_if_line_containing, add_before_line_containing, lines_to_add):
    g_did_run = False
    
    updated_file = []
    with open(config_file, 'r') as f:
        line_index = 1
        line_to_add_before = 0
        for line in f.readlines():
            updated_file.append(line)
            if bypass_if_line_containing in line:
                return g_did_run
            if add_before_line_containing in line:
                line_to_add_before = line_index
            line_index += 1
    
    if line_to_add_before == 0:
        return g_did_run
    updated_file[line_to_add_before - 1:line_to_add_before - 1] = lines_to_add
    with open(config_file, 'w') as f:
        f.write("".join(updated_file))
    g_did_run = True
    return g_did_run


@add_fix
@add_doc("Will add the sla output storing and worker sections to it's config")
@add_context_daemons(['broker'])
@add_version('02.08.00')
@auto_launch(True)
@need_shinken_stop(True)
def add_sla_output_store_and_worker_options():
    g_did_run = False
    config_file = "/etc/shinken/modules/sla.cfg"
    bypass_if_line_containing = 'store_output'
    add_before_line_containing = 'INTERNAL options'
    to_add = [
        '    # ======== SLA stored output ========\n',
        '    # This option enables or disables storing sla outputs.\n',
        '    # If 1, the output will be stored (Default value)\n',
        '    # If 0, the output and long output will not be stored (downtime and acknowledge will still be stored)\n',
        '    # store_output      1\n',
        '\n',
        '    # This option enables or disables storing sla long outputs.\n',
        '    # If 1, the long output will be stored (Default value)\n',
        '    # If 0, the long output will not be stored (output, downtime and acknowledge will still be stored)\n',
        '    # store_long_output     1\n',
        '\n',
        '    # This option will be used to filter which outputs and long outputs to store depending on the status of the sla.\n',
        '    # Possible values are : OK, WARNING, CRITICAL, UNKNOWN\n',
        '    # Seperator is : ,\n',
        '    # Default value : empty (all output states are stored)\n',
        '    # Example to only store OK and UNKNOWN outputs :  list_of_stored_output_status    OK,UNKNOWN\n',
        '    # list_of_stored_output_status\n',
        '    #======== Workers in the broker ========\n'
        '    # This module will use workers in the broker, each worker will manage a shard of all hosts/checks.\n'
        '    # This parameter is used by the broker to set the number of workers. Each worker will use one CPU, which will balance the sla processing load among CPUs.\n'
        '    # default: 1\n'
        '    broker_module_nb_workers     1\n'
        '\n',
    ]
    return _add_new_options_to_config_file(config_file, bypass_if_line_containing, add_before_line_containing, to_add)


@add_fix
@add_doc("Will add the retry options to the synchronize-import config file")
@add_context_daemons(['arbiter'])
@add_version('02.08.00')
@auto_launch(True)
@need_shinken_stop(True)
def add_retry_options_to_synchronizer_import():
    config_file = "/etc/shinken/modules/synchronizer-import.cfg"
    bypass_if_line_containing = 'max_try'
    add_before_line_containing = '}'
    to_add = [
        '\n',
        '    # Maximum number of tries for loading the synchronizer configuration before reporting a failure\n',
        '    #max_try     90\n',
        '\n',
        '    # Sleep time (in seconds) between two successive tries of loading the synchronizer configuration\n',
        '    #sleep_time   2\n',
        '\n'
    ]
    return _add_new_options_to_config_file(config_file, bypass_if_line_containing, add_before_line_containing, to_add)


@add_fix
@add_doc("Will add the map_realm_layout option to the architecture-export config file")
@add_context_daemons(['arbiter'])
@add_version('02.08.00')
@auto_launch(True)
@need_shinken_stop(True)
def add_map_realm_layout_to_architecture_export_module():
    config_file = "/etc/shinken/modules/architecture-export.cfg"
    bypass_if_line_containing = 'map_realm_layout'
    add_before_line_containing = '}'
    to_add = [
        '\n',
        '    # Sort order for realms in the NagVis maps\n',
        '    # - sort_by_name\n',
        '    # - sort_by_size (default)\n',
        '    #map_realm_layout sort_by_size\n',
        '\n'
    ]
    return _add_new_options_to_config_file(config_file, bypass_if_line_containing, add_before_line_containing, to_add)


@add_fix
@add_doc("Cleans up the monitoring packs by overwriting modified files with files from the Shinken installation")
@add_version('02.07.04')
@add_context_daemons(['synchronizer'])
@auto_launch(True)
@need_shinken_stop(True)
def cleanup_monitoring_pack():
    import os
    
    g_did_run = False
    root_path = "/etc/shinken/packs"
    
    for dir, _, files in os.walk(root_path):
        for file in files:
            if file.endswith('.rpmnew'):
                g_did_run = True
                source_file = os.path.join(dir, file)
                dest_file = source_file.replace('.rpmnew', '')
                
                os.unlink(dest_file)
                shutil.move(source_file, dest_file)
    return g_did_run


@add_fix
@add_doc('Make sure carbon-cache daemon (graphite writer) options are up to date')
@add_context_daemons(['broker'])
@add_version('02.07.04')
@auto_launch(False)
@need_shinken_stop(True)
def fix_carbon_cache_parameters():
    CARBON_CACHE_CONF = '/opt/graphite/conf/carbon.conf'
    
    did_run = False
    
    with open(CARBON_CACHE_CONF) as fp:
        lines = fp.readlines()
    forced_parameters = {'USER': 'apache', 'MAX_UPDATES_PER_SECOND': 'inf', 'MAX_CREATES_PER_MINUTE': 'inf'}
    
    new_lines = []
    
    for line in lines:
        line = line.strip()
        line = line + '\n'  # be sure to be in unix format
        if line.startswith("#"):
            new_lines.append(line)
            continue
        if '=' not in line:
            new_lines.append(line)
            continue
        (param, value) = line.split("=", 1)
        param = param.strip()
        value = value.strip()
        if param not in forced_parameters.keys():
            new_lines.append(line)
            continue
        forced_value = forced_parameters[param]
        if value != forced_value:
            logger.info('Updating the carbon-cache parameter %s:%s=>%s' % (param, value, forced_value))
            did_run = True
            new_lines.append('%s = %s\n' % (param, forced_value))
        else:
            logger.debug('The carbon-cache parameter %s is already at %s' % (param, value))
            new_lines.append(line)
    
    if did_run:
        logger.info('Writing the carbon cache configuration file %s with updated parameters' % CARBON_CACHE_CONF)
        with open(CARBON_CACHE_CONF, "w") as fp:
            fp.writelines(new_lines)
        try:
            subprocess.check_call(['/etc/init.d/carbon-cache', 'restart'], stdout=subprocess.PIPE)
        except subprocess.CalledProcessError as exp:
            print 'ERROR: Unable to restart carbon-cache"\n' % exp
    
    return did_run


@add_fix
@add_doc('Clean old Nagvis-shinken .tmp file')
@add_context_daemons(['arbiter'])
@add_version('02.08.01')
@auto_launch(True)
@need_shinken_stop(True)
def clean_old_nagvis_shinken_tmp_files():
    g_did_run = False
    
    nagvis_path = '/etc/shinken/external/nagvis/'
    paths_to_check = ['etc/maps', 'etc/conf.d/', 'share/userfiles/images/maps/']
    
    for tmp_path in paths_to_check:
        full_path = os.path.join(nagvis_path, tmp_path, '*.tmp')
        file_list = glob.glob(full_path)
        
        if file_list:
            g_did_run = True
            for file_path in file_list:
                logger.debug('File to delete %s' % file_path)
                try:
                    os.unlink(file_path)
                except OSError:
                    logger.error('The file %s can not be deleted' % file_path)
                    return None
    
    return g_did_run


@add_fix
@add_doc('Remove presence_protection from elements that are not shinken-core')
@add_context_daemons(['synchronizer'])
@add_version('02.07.05')
@auto_launch(True)
@need_shinken_stop(True)
def remove_presence_protection():
    g_did_run = False
    
    try:
        from shinken.synchronizer.dao.def_items import NAGIOS_TABLE_KEYS, ITEM_STATE
        from shinken.synchronizer.business.core_item import CORE_ITEMS
    except ImportError:
        logger.error("You cannot run this fix on your shinken version please upgrade it.")
        return g_did_run
    
    db = get_synchronizer_db()
    if not db.collection_names():
        return g_did_run
    
    ids_to_skip = []
    for item_type, items in CORE_ITEMS.iteritems():
        ids_to_skip.extend([e['_id'] for e in items])
    
    for item_state in (ITEM_STATE.MERGE_SOURCES, ITEM_STATE.WORKING_AREA, ITEM_STATE.STAGGING, ITEM_STATE.PRODUCTION):
        for (item_class, key_name) in NAGIOS_TABLE_KEYS.iteritems():
            col = getattr(db, 'configuration-%s-%s' % (item_state, item_class))
            items = col.find({'presence_protection': '1'})
            
            for item in items:
                if item.get('_id', '') in ids_to_skip:
                    continue
                
                did_run = item.pop('presence_protection', None)
                
                if did_run:
                    g_did_run = True
                    logger.info('Remove presence_protection from item %s:%s in state %s ' % (item_class, item.get('name', item.get(key_name, item.get('service_description'))), item_state))
                    col.save(item)
    
    return g_did_run


if __name__ == '__main__':
    logger.handlers = []
    logger_handler = logging.StreamHandler()
    logger.addHandler(logger_handler)
    logger_handler.setFormatter(logging.Formatter('%(levelname)s: %(message)s'))
    logger.setLevel('INFO')
    
    parser = optparse.OptionParser("%prog ", version="%prog: " + VERSION, description='This tool  is used to check the state of your Shinken Enterprise installation and configuration')
    parser.add_option('-l', '--list', dest='list_only', action='store_true', help="List available fixes")
    parser.add_option('-r', '--run', dest='run_one', help="Run only one specific fixe")
    parser.add_option('-a', '--auto', dest='run_auto', action='store_true', help="Run automatic fixes at once")
    parser.add_option('', '--all', dest='run_all', action='store_true', help="Run ALL fixes at once")
    parser.add_option('-v', '--verbose', dest='verbose', action='store_true', help="Show verbose output")
    parser.add_option('-q', '--quiet', dest='quiet', action='store_true', help="Do not output when no fix has been done")
    parser.add_option('-n', '--no-hint', dest='no_hint', action='store_true', help="Do not display hints for users")
    parser.add_option('-L', '--log_file', dest='log_file', default=DEFAULT_LOG_FILENAME, help="The log file path")
    parser.add_option('', '--old', dest='old', default='', help="Key to rename")
    parser.add_option('', '--new', dest='new', default='', help="New name of the key")
    parser.add_option('', '--item-type', dest='item_type', default='', help="use by fix 'fix_regenerate_sync_keys' to choose item type to regenerate")
    
    opts, args = parser.parse_args()
    
    # Look if the user ask for local or global, and if not, guess
    list_only = opts.list_only
    run_all = opts.run_all
    run_one = opts.run_one
    run_auto = opts.run_auto
    
    sanitize_log.set_console_level(logging.INFO)
    if opts.verbose or list_only:
        logger.setLevel('DEBUG')
        sanitize_log.set_console_level(logging.DEBUG)
    sanitize_log.set_file_handler(os.path.expanduser(opts.log_file))
    
    log.debug("------------------------------------------------------------------------------")
    log.debug("------------------------------------------------------------------------------")
    log.debug("Running sanatize at %s" % datetime.now().strftime('%H:%M:%S %d-%m-%Y'))
    log.debug("------------------------------------------------------------------------------")
    log.debug("------------------------------------------------------------------------------")
    
    local_daemons = get_local_daemons()
    
    # First sort ALL_FIX based on their versions
    ALL_FIX = sorted(ALL_FIX, key=natural_version)
    
    if list_only:
        cur_version = ''
        for f in ALL_FIX:
            if not hasattr(f, 'auto_launch'):
                raise Exception('Please specify auto launch parameter on fix [%s].' % f.__name__)
            v = f.version
            if v != cur_version:
                cur_version = v
                log.info("\n################ Version \033[35m%s\033[0m ##########" % v)
            log.info("-------------------")
            log.info(" \033[32m%-40s\033[0m:\n\tDoc: %s" % (f.__name__, f.doc))
        sys.exit(0)
    
    if [run_all, run_one, run_auto].count(True) > 1:
        sys.exit("Error: You cannot ask for a run one and run all and run auto option. Please choose")
    if not run_all and not run_one and not run_auto:
        parser.print_help()
        sys.exit(0)
    
    _need_shinken_stop = False
    
    if run_one:
        try:
            selected_fix = [f for f in ALL_FIX if f.__name__ == run_one][0]
            _need_shinken_stop = selected_fix.need_shinken_stop
        except IndexError:
            log.error('\n\033[31m ERROR: The fix "%s" does not exists\033[0m\n' % run_one)
            sys.exit(1)
    
    if _need_shinken_stop or run_all or run_auto:
        if is_shinken_running():
            log.warning("\033[33m###################### Warning #######################\033[0m")
            log.warning("\033[33m# Shinken is curently running. We will stop it now.  #\033[0m")
            log.warning("\033[33m# (We cannot sanatize data if shinken is running)    #\033[0m")
            log.warning("\033[33m######################################################\033[0m\n")
            subprocess.call(["/etc/init.d/shinken stop > /dev/null 2>&1"], shell=True)
    
    cur_version = ''
    errors_count = 0
    message_template = "   \033[%%dm%%-%ds\033[0m:  \033[%%dm %%s \033[0m" % (MAX_FIX_NAME_LENGTH + 2)
    
    for f in ALL_FIX:
        fname = f.__name__
        if run_all or fname == run_one or (run_auto and f.auto_launch):
            # If the function is not launchable on this type of server, skip it
            
            v = f.version
            if v != cur_version and not opts.quiet:
                cur_version = v
                log.debug('')
                log.debug("################ Version \033[35m%s\033[0m ##########" % v)
            
            log.debug('-----')
            log.debug('Executing: %s' % fname)
            try:
                has_error = False
                error_message = None
                if fname == 'fix_rename_key':
                    r = f(opts.old, opts.new)
                else:
                    r = f()
                    if not isinstance(r, bool):
                        error_message = r
                        has_error = True
                        errors_count += 1
            except Exception, exp:
                has_error = True
                errors_count += 1
                print '\n\033[31m ERROR: %s => %s\033[0m\n' % (fname, traceback.format_exc())
                r = None
            
            if has_error:
                state = "executed [Failure]"
                color = 31  # red
            elif r:
                state = "executed [OK]"
                color = 32  # green
                data_type = getattr(f, "data_type", 'configuration')
                # We did execute a sanatize, save it in the context
                do_import_data_history_sanatize(fname, data_type=data_type)
            else:
                state = "skip (unecessary)"
                color = 36  # ligh blue
            
            if r or not opts.quiet:
                log.info(message_template % (35, fname, color, state))
                if error_message:
                    print '\n\033[31m ERROR: %s => %s\033[0m\n' % (fname, error_message)
                    for message_line in error_message.split('\n'):
                        log.error("        \033[%dm %s" % (color, message_line))
    
    if errors_count:
        if not opts.no_hint:
            log_file = opts.log_file if opts.log_file else DEFAULT_LOG_FILENAME
            log.error("\n\n\033[31mSome errors occurred while running the fixes."
                      " Please check the log file for more information (\033[36m%s\033[31m)."
                      "Send this file to your Shinken support if needed.\033[0m" % log_file)
        sys.exit(1)
