#!/usr/bin/python
# -*- coding: utf-8 -*-

# Copyright (C) 2013:
# This file is part of Shinken Enterprise, all rights reserved.

try:
    import pyexcel as pe
except ImportError:
    pe = None
# import it to handle xls file
from shinken.basemodule import BaseModule, SOURCE_STATE
from shinken.log import logger
from shinken.objects.config import Config

COLUMN_NAMES = {
    'name'       : 'host_name',
    'address'    : 'address',
    'template'   : 'use',
    'description': 'display_name',
    'realm'      : 'realm',
    'contacts'   : 'contacts',
}

SYNC_KEY = ['host_name', 'address']

properties = {
    'daemons': ['synchronizer'],
    'type'   : 'file-excel-import',
}


# called by the plugin manager to get a module
def get_instance(plugin):
    logger.info("[File-Excel Import] Get a excel import module for plugin %s" % plugin.get_name())
    
    # Catch errors ??
    excel_path = getattr(plugin, 'excel_path', '')
    return FileExcelImport(plugin, excel_path)


def column_name_2_keys(name):
    return COLUMN_NAMES.get(name, "_" + name.upper())


def parse_hosts(records):
    hosts = []
    for record in records:
        host = {}
        # keys = sorted(record.keys())
        for key in record.keys():
            host[column_name_2_keys(key)] = record[key]
            logger.debug("[File-Excel Import] [%s]:[%s]" % (key, record[key]))
        
        sync_keys = [host[k] for k in SYNC_KEY if host.get(k, '')]
        host['_SYNC_KEYS'] = (','.join(sync_keys)).lower()
        hosts.append(host)
        logger.debug("[File-Excel Import] host [%s]" % host)
    return hosts


class FileExcelImport(BaseModule):
    def __init__(self, mod_conf, path):
        BaseModule.__init__(self, mod_conf)
        self.excel_path = path
        self.file = None
        self.conf = Config()
    
    
    # Called by Arbiter to say 'let's prepare yourself guy'
    def init(self):
        logger.info("[File-Excel Import] Initialization of the cfg file import module")
    
    
    def get_objects(self):
        err = 'This module is not release currently, and should not be used.'
        return {'output': err, 'state': SOURCE_STATE.CRITICAL, 'objects': {}, 'errors': [err], 'warnings': []}
        
        if pe is None:
            err = 'Critical : you need to install the pyexcel librairy, please install it.'
            return {'output': err, 'state': 'CRITICAL', 'objects': {}, 'errors': [err], 'warnings': []}
        
        logger.debug("[File-Excel Import] read %s" % self.excel_path)
        res = None
        try:
            sheet = pe.get_sheet(file_name=self.excel_path, name_columns_by_row=0)
            logger.debug("[File-Excel Import] do not pass here motherfucker")
            records = sheet.to_records()
            hosts = parse_hosts(records)
            logger.debug("[File-Excel Import] host found %s" % hosts)
            objects = {'host': hosts}
            res = {'state'  : 'OK', 'output': 'OK: load successful from excel file %s' % self.excel_path,
                   'objects': objects, 'errors': [], 'warnings': []}
            logger.debug("[File-Excel Import] object found %s" % objects)
        except IOError as e:
            logger.debug("[File-Excel Import] exp %s message %s attr %s", e.__class__.__name__, str(e), e.__dict__)
            if e.errno == 13:
                return {'output': 'Critical : Insufficient permission for : ' + self.excel_path, 'state': 'CRITICAL', 'objects': {}, 'errors': [str(e)], 'warnings': []}
            if e.errno == 2:
                return {'output': 'Critical : Excel file not found : ' + self.excel_path, 'state': 'CRITICAL', 'objects': {}, 'errors': [str(e)], 'warnings': []}
            raise
        except Exception as e:
            logger.debug("[File-Excel Import] exp %s message %s", e.__class__.__name__, str(e))
            raise
        
        return res
