#!/usr/bin/python
# -*- coding: utf-8 -*-

# Copyright (C) 2009-2012:
#    Gabes Jean, naparuba@gmail.com
#    Gerhard Lausser, Gerhard.Lausser@consol.de
#    Gregory Starck, g.starck@gmail.com
#    Hartmut Goebel, h.goebel@goebel-consult.de
#
# This file is part of Shinken.
#
# Shinken is free software: you can redistribute it and/or modify
# it under the terms of the GNU Affero General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# Shinken is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Affero General Public License for more details.
#
# You should have received a copy of the GNU Affero General Public License
# along with Shinken.  If not, see <http://www.gnu.org/licenses/>.


import json
import time

from shinken.log import logger
from shinken.misc.filter import is_authorized

app = None
default_options = [{'name': 'name', 'value': 'localhost', 'type': 'hst_OR_srv', 'label': 'Name', 'required': True}]
_useless_properties = (
    'duration_sec',
    'last_problem_id',
    'acknowledgement_type',
    'bp_state',
    'broks',
    'business_rule',
    'business_rule_downtime_as_ack',
    'business_rule_host_notification_options',
    'business_rule_output_template',
    'business_rule_service_notification_options',
    'business_rule_smart_notifications',
    'act_depend_of',
    'act_depend_of_me',
    'actions',
    'check_flapping_recovery_notification',
    'check_freshness',
    'check_interval',
    'checkmodulations',
    'checks_in_progress',
    'childs',
    'chk_depend_of',
    'chk_depend_of_me',
    'configuration_errors',
    'configuration_warnings',
    'consecutive_ok',
    'current_event_id',
    'current_notification_id',
    'current_problem_id',
    'customs',
    'dep_tree',
    'early_timeout',
    'edition_contact_groups',
    'edition_contacts',
    'escalations',
    'event_handler',
    'event_handler_enabled',
    'execution_time',
    'first_notification_delay',
    'flap_detection_enabled',
    'flap_detection_options',
    'flapping_changes',
    'flapping_comment_id',
    'freshness_threshold',
    'got_default_realm',
    'has_been_checked',
    'hash',
    'high_flap_threshold',
    'impacts',
    'in_checking',
    'in_hard_unknown_reach_phase',
    'in_maintenance',
    'in_partial_downtime',
    'in_scheduled_downtime',
    'in_scheduled_downtime_during_last_check',
    'initial_state',
    'is_impact',
    'last_event_id',
    'last_hard_state',
    'last_hard_state_change',
    'last_normal_chk',
    'last_notification',
    'last_perf_data',
    'last_state',
    'last_state_type',
    'last_state_update',
    'last_time_down',
    'last_time_unreachable',
    'last_time_up',
    'latency',
    'low_flap_threshold',
    'max_check_attempts',
    'modified_attributes',
    'my_own_business_impact',
    'nb_executions',
    'notification_interval',
    'notification_options',
    'notifications_in_progress',
    'obsess_over_host',
    'pack_id',
    'parent_dependencies',
    'passive_checks_enabled',
    'pending_flex_downtime',
    'process_perf_data',
    'processed_business_rule',
    'retry_interval',
    'return_code',
    's_time',
    's_time_sum',
    'scheduled_downtime_depth',
    'services',
    'services_counts',
    'should_be_scheduled',
    'source_problems',
    'stalking_options',
    'state',
    'state_before_hard_unknown_reach_phase',
    'state_before_impact',
    'state_changed_since_impact',
    'state_id_before_impact',
    'state_type',
    'state_type_id',
    'time_to_orphanage',
    'topology_change',
    'triggers',
    'u_time',
    'u_time_sum',
    'was_in_hard_unknown_reach_phase',
    'host_uuid',
    'instance_uuid'
)
useless_properties = set(_useless_properties)


def api_get_elt_widget_basic(uuid):
    return app.widget_service.api_get_elt_widget_basic(uuid)


def api_get_elt_widget_output(uuid):
    return app.widget_service.api_get_elt_widget_output(uuid)


def api_get_elt_widget_checklist(uuid):
    return app.widget_service.api_get_elt_widget_checklist(uuid)


def api_get_elt_widget_full(uuid):
    return app.widget_service.api_get_elt_widget_full(uuid)


# Return a tree of {'elt': Host, 'fathers': [{}, {}]}
def get_json_business_parents(obj, levels=5):
    d = fill_custom_states_uniq(obj)
    res = d
    d['fathers'] = []
    
    # If too low, get out!
    if levels == 0:
        return res
    
    for i in obj.parent_dependencies:
        # We want to get the levels deep for all elements, but
        # go as far as we should for bad elements
        if i.state_id != 0:
            par_elts = get_json_business_parents(i, levels=levels - 1)
            res['fathers'].append(par_elts)
    
    return res


def fill_custom_states(lst):
    r = []
    for e in lst:
        d = fill_custom_states_uniq(e)
        r.append(d)
    return r


def fill_custom_states_uniq(e):
    d = {'business_impact': e.business_impact,
         'state_id'       : e.state_id,
         'last_chk'       : e.last_chk,
         }
    if e.__class__.my_type == 'host':
        d['type'] = 'host'
        d['host_name'] = e.host_name
        d['uuid'] = e.uuid
    else:
        d['type'] = 'service'
        d['host_name'] = e.host.host_name
        d['service_description'] = e.service_description
        d['uuid'] = "%s-%s" % (e.host.uuid, e.uuid)
        d['uuid_parent'] = e.host.uuid
    return d


def __unobjfy_aggregation_tree(tree):
    # First ask to our sons to compute their states
    for s in tree['sons']:
        __unobjfy_aggregation_tree(s)
    new_services = []
    for s in tree['services']:
        unobj_s = app.widget_service._widget_status_basic_infos(s)
        new_services.append(unobj_s)
    tree['services'] = new_services


def get_element_host(uuid):
    user = app.get_user_auth()
    elt = app.datamgr.get_host_by_uuid(uuid)
    if elt is None:
        elt = {
            'uuid'      : uuid,
            'is_unknown': True
        }
        return app.abort(404, elt, True)
    if not is_authorized(elt, user, app):
        logger.info('get_element_host User is not authorized to view this element')
        return json.dumps(app.helper.get_unauthorized_item(elt))
    
    cls = elt.__class__
    couples = [(k, v) for (k, v) in cls.properties.iteritems()] + [(k, v) for (k, v) in cls.running_properties.iteritems()]
    d = {}
    for (k, v) in couples:
        try:
            if k in useless_properties:
                continue
            if k == 'services':
                value = [fill_custom_states_uniq(s) for s in elt.services]
            elif k == 'parents':
                value = [h.get_name() for h in elt.parents]
            elif k == 'hostgroups':
                value = [hg.get_name() for hg in elt.hostgroups]
            elif k == 'contacts':
                value = [contact.get_name() for contact in elt.contacts]
            elif k == 'contact_groups':
                value = [grp.get_name() for grp in elt.contact_groups]
            elif k == 'host':
                value = elt.host.host_name
            elif k == 'comments':
                value = app.helper.get_elt_dict(elt.comments)
            elif k == 'tags':
                value = [{'name': t, 'tag_icon': t in app.img_tags} for t in [t2 for t2 in elt.tags if t2 not in ('shinken-host', 'shinken-cluster')]]
            elif k in ('notification_period', 'check_period', 'maintenance_period'):
                if hasattr(elt, k):
                    value = getattr(elt, k).get_name()
                else:
                    value = 'always'
            else:
                value = getattr(elt, k)
            json.dumps(value)
            d[k] = value
        except Exception:  # if cannot json, pass
            pass
            # print 'CANNOT DUMP PROP/VALUE',str(exp), k, value
    
    if elt.got_business_rule:
        if 'address' in d:
            del d['address']
        
        if 'next_chk' in d:
            del d['next_chk']
        
        if 'last_check' in d:
            del d['last_check']
        
        if 'last_chk' in d:
            del d['last_chk']
        d['type'] = 'cluster'
    else:
        d['type'] = 'host'
        # Hook last_hard_state_id and last_state_id, state_id so 1=>2 (critical/bad)
        for prop in ['last_hard_state_id', 'last_state_id', 'state_id']:
            if d[prop] == 1:
                d[prop] = 2
    
    tree = get_json_business_parents(elt)
    d['dep_tree'] = tree
    
    tag_icon = app.get_first_tag(elt.tags)
    if not tag_icon:
        if elt.got_business_rule:
            tag_icon = 'cluster'
        else:
            tag_icon = 'hardware'
    d['tag_icon'] = tag_icon
    
    aggregation_tree = app.helper.get_host_service_aggregation_tree(elt)
    __unobjfy_aggregation_tree(aggregation_tree)
    
    for service in aggregation_tree.get('services', []):
        if service.get('got_business_rule', None):
            if 'next_check' in service:
                del service['next_check']
            
            if 'next_chk' in service:
                del service['next_chk']
            
            if 'last_check' in service:
                del service['last_check']
            
            if 'last_chk' in service:
                del service['last_chk']
    
    d['aggregation_tree'] = aggregation_tree
    
    # status and summary thing too
    e = app.helper.get_summary(elt)
    d['status'] = e['status']
    d['context'] = e['summary']
    
    if elt.got_business_rule:
        d['last_status'] = elt.last_state_id
    else:
        d['last_status'] = app.helper.get_ui_status(d['last_state_as_string'])
    d['last_change'] = {"status": d['last_status'], "date": elt.last_state_change}
    
    d['backend_localtime_epoch'] = int(time.time())
    
    # Lie about new element
    if elt.state == 'PENDING':
        d['state_id'] = 3
        d['last_hard_state_id'] = 3
        d['last_state_id'] = 3
    
    # and about the first change
    # if elt.last_state == 'PENDING':
    #    d['last_state_id'] = 3
    
    d['services_counts'] = [0, 0, 0, 0]
    for i in range(4):  # 0 => 3 included
        d['services_counts'][i] = len([s for s in elt.services if ((s.state != 'PENDING' and s.state_id == i) or (s.state == 'PENDING' and i == 3))])
    
    app.helper.get_ack_on_item(elt, d)
    app.helper.get_downtime_on_item(elt, d)
    
    d['is_unknown'] = False
    return json.dumps(d)


def get_element_service(uuid):
    user = app.get_user_auth()
    if not app.is_check_uuid(uuid):
        return app.abort(500, 'uuid is not a service')
    elt = app.datamgr.get_service_by_uuid(uuid)
    if elt is None:
        elt = {
            'uuid'      : uuid,
            'is_unknown': True
        }
        return app.abort(404, elt)
    
    if not is_authorized(elt.host, user, app):
        logger.info('get_element_service User is not authorized to view this element')
        return json.dumps(app.helper.get_unauthorized_item(elt))
    
    cls = elt.__class__
    couples = [(k, v) for (k, v) in cls.properties.iteritems()] + [(k, v) for (k, v) in cls.running_properties.iteritems()]
    d = {}
    for (k, v) in couples:
        try:
            if k in useless_properties:
                continue
            if k == 'services':
                value = [s.get_name() for s in elt.services]
            elif k == 'parents':
                value = [h.get_name() for h in elt.parents]
            elif k == 'hostgroups':
                value = [hg.get_name() for hg in elt.hostgroups]
            elif k == 'contact_groups':
                value = [grp.get_name() for grp in elt.contact_groups]
            elif k == 'contacts':
                value = [contact.get_name() for contact in elt.contacts]
            elif k == 'source_problems':
                value = fill_custom_states(elt.source_problems)
            elif k == 'impacts':
                value = fill_custom_states(elt.impacts)
            elif k == 'host':
                value = elt.host.host_name
            elif k in ('notification_period', 'check_period', 'maintenance_period'):
                if hasattr(elt, k):
                    value = getattr(elt, k).get_name()
                else:
                    value = 'always'
            elif k == 'uuid':
                value = '%s-%s' % (elt.host.uuid, elt.uuid)
            else:
                value = getattr(elt, k)
            json.dumps(value)
            d[k] = value
        except Exception, exp:  # if cannot json, pass
            pass
    
    d['last_change'] = {"status": elt.last_state_id, "date": elt.last_state_change}
    tree = get_json_business_parents(elt)
    d['dep_tree'] = tree
    
    tag_icon = app.get_first_tag(elt.tags)
    if not tag_icon:
        tag_icon = 'hardware'
    d['tag_icon'] = tag_icon
    
    # Aggregation tree is a non sense for a service, but I prefer to put the key in case of
    d['aggregation_tree'] = {}
    
    # status and summary thing too
    e = app.helper.get_summary(elt)
    d['status'] = e['status']
    d['context'] = e['summary']
    
    if elt.got_business_rule:
        d['last_status'] = elt.last_state_id
        if 'next_check' in d:
            del d['next_check']
        
        if 'next_chk' in d:
            del d['next_chk']
        
        if 'last_check' in d:
            del d['last_check']
        
        if 'last_chk' in d:
            del d['last_chk']
    else:
        d['last_status'] = app.helper.get_ui_status(d['last_state_as_string'])
    
    d['last_change'] = {"status": d['last_status'], "date": elt.last_state_change}
    d['backend_localtime_epoch'] = int(time.time())
    d['is_unknown'] = False
    
    app.helper.get_ack_on_item(elt, d)
    app.helper.get_downtime_on_item(elt, d)
    
    return json.dumps(d)


def is_elements_rechecked_post():
    if hasattr(app.request.body, 'getvalue'):
        data = json.loads(app.request.body.getvalue())
    else:  # file?
        data = json.loads(app.request.body.read())
    
    return_values = []
    uuids = data['uuids']
    for uuid in uuids:
        if app.is_check_uuid(uuid):
            elt = app.datamgr.get_service_by_uuid(uuid)
        else:
            elt = app.datamgr.get_host_by_uuid(uuid)
            
        if not elt:
            continue
            
        if elt.next_chk > time.time():
            return_values.append(uuid)
    
    return json.dumps(return_values)


def list_element_status():
    logger.debug("call list_element_status")
    
    user_uuid = app.request.GET.get('user_uuid', None)
    user = app.get_user_auth()
    
    if user.uuid != user_uuid:
        return app.abort(403, "User logout")
    
    if hasattr(app.request.body, 'getvalue'):
        data = json.loads(app.request.body.getvalue())
    else:
        data = json.loads(app.request.body.read())
    
    list_elements_status = set(data['list_elements_status'])
    list_elements_status_all = set(data['list_elements_status_all'])
    list_elements_sla = set(data['list_elements_sla'])
    
    # We get 3 list of elements : Status, SLA and "All status".
    # For the third list wa have to
    # - get and return all the check foreach host/cluster include in the list
    # - add theses check in the list Status
    items = {'list_elements_children': {}, 'list_elements': {}}
    if list_elements_status_all:
        # First : get all the check for each host/cluster
        items['list_elements_children'] = _get_all_checks_uuid_for_items(list_elements_status_all)
        
        # Second : add all the check_list and host in the Status list
        for _item_uuid, list_check in items['list_elements_children'].iteritems():
            list_elements_status.add(_item_uuid)
            if list_check:
                list_elements_status = list_elements_status | set(list_check)
    
    if list_elements_status:
        _get_items_status_context(list_elements_status, items['list_elements'], user)
    if list_elements_sla:
        _get_items_sla(list_elements_sla, items['list_elements'], user)
    
    items['list_elements'] = items['list_elements'].values()
    return json.dumps(items)


def _get_items_status_context(list_items, final_list, user):
    if len(list_items) == 0:
        return
    
    for item_uuid in list_items:
        if app.is_check_uuid(item_uuid):
            item = app.datamgr.get_service_by_uuid(item_uuid)
        else:
            item = app.datamgr.get_host_by_uuid(item_uuid)
        
        data = {'uuid': item_uuid, 'is_unknown': False, 'unsight': not is_authorized(item, user, app)}
        if item:
            summary = app.helper.get_summary(item)
            data['status'] = summary['status']
            data['context'] = summary['summary']
        else:
            data['is_unknown'] = True
        
        item = final_list.get(item_uuid, None)
        if item is None:
            final_list[item_uuid] = data
        else:
            item.update(data)


def _get_items_sla(list_items, final_list, user):
    if len(list_items) == 0:
        return {}
    sla_module = next((mod for mod in app.modules_manager.get_all_instances() if mod.properties['type'] == 'sla'), None)
    
    for item_uuid in list_items:
        unsight = final_list.get(item_uuid, {}).get('unsight', None)
        if unsight is None:
            # TODO BUG -> item not define
            unsight = not is_authorized(item, user, app)
            final_list[item_uuid] = {'uuid': item_uuid, 'is_unknown': False, 'unsight': unsight}
        
        if unsight:
            continue
        if app.is_check_uuid(item_uuid):
            item = app.datamgr.get_service_by_uuid(item_uuid)
        else:
            item = app.datamgr.get_host_by_uuid(item_uuid)
        
        data = {'uuid': item_uuid, 'is_unknown': False}
        if item:
            if sla_module:
                now = time.time()
                is_growing = ((item.state_id == 0) & (not item.is_flapping))
                
                week_info = sla_module.compute_sla_week_info(now, item_uuid)
                if week_info:
                    del week_info['by_days']
                    del week_info['yesterday']
                    del week_info['today']
            else:
                logger.error("[SLA] The sla module wasn't found")
                return app.abort(500, 'The sla module wasn\'t found', True)
            
            data['sla'] = {
                'is_growing': is_growing,
                'week_info' : week_info
            }
        else:
            data['is_unknown'] = True
        
        item = final_list.get(item_uuid, None)
        if item is None:
            final_list[item_uuid] = data
        else:
            item.update(data)


def _get_all_checks_uuid_for_items(list_items):
    items = {}
    for item_uuid in list_items:
        item = app.datamgr.get_host_by_uuid(item_uuid)
        if item:
            items[item_uuid] = [check.get_instance_uuid() for check in item.services]
        else:
            items[item_uuid] = []
    
    return items


# Empty method use to declare widget
def get_elt_widget():
    pass


widget_display_name = 'Status'
widget_desc = '''Show one element status
'''

pages = {
    is_elements_rechecked_post  : {'routes': ['/api/elements_rechecked'], 'method': 'POST', 'wrappers': ['auth', 'json']},
    get_element_host            : {'routes': ['/api/host/:uuid'], 'view': None, 'static': True, 'wrappers': ['auth', 'json']},
    get_element_service         : {'routes': ['/api/service/:uuid'], 'view': None, 'static': True, 'wrappers': ['auth', 'json']},
    
    api_get_elt_widget_basic    : {'routes': ['/api/widget/detail/basic/:uuid'], 'static': True, 'wrappers': ['json']},
    api_get_elt_widget_output   : {'routes': ['/api/widget/detail/output/:uuid'], 'static': True, 'wrappers': ['json']},
    api_get_elt_widget_checklist: {'routes': ['/api/widget/detail/checklist/:uuid'], 'static': True, 'wrappers': ['json']},
    api_get_elt_widget_full     : {'routes': ['/api/widget/detail/full/:uuid'], 'static': True, 'wrappers': ['json']},
    
    list_element_status         : {'routes': ['/api/state/elements'], 'method': 'POST', 'wrappers': ['auth', 'json']},
    get_elt_widget              : {
        'routes'             : ['/widget/detail'],
        'static'             : True,
        'widget'             : ['dashboard'],
        'widget_desc'        : widget_desc,
        'widget_name'        : 'detail',
        'widget_picture'     : '/static/eltdetail/img/widget_detail.png',
        'widget_size'        : {'width': 2, 'height': 2},
        'widget_options'     : default_options,
        'widget_display_name': widget_display_name,
        'widget_favoritable' : True,
        'old_style'          : False,
        'wrappers'           : ['json']
    },
    
}
