#!/usr/bin/python
# -*- coding: utf-8 -*-
#
# Copyright (C) 2013-2019
# This file is part of Shinken Enterprise, all rights reserved.

import sys
import itertools
import time
import threading


def print_progress_bar(iteration, total, prefix='', suffix='', decimals=1, length=100, fill='#'):
    """
    Call in a loop to create terminal progress bar
    @params:
        iteration   - Required  : current iteration (Int)
        total       - Required  : total iterations (Int)
        prefix      - Optional  : prefix string (Str)
        suffix      - Optional  : suffix string (Str)
        decimals    - Optional  : positive number of decimals in percent complete (Int)
        length      - Optional  : character length of bar (Int)
        fill        - Optional  : bar fill character (Str)
    """
    
    if iteration > total:
        iteration = total
    
    percent = ('{0:.' + str(decimals) + 'f}').format(100 * (iteration / float(total)))
    filled_length = int(length * iteration // total)
    bar = fill * filled_length + '-' * (length - filled_length)
    sys.stdout.write('\r%s |%s| %s%% %s' % (prefix, bar, percent, suffix))
    sys.stdout.flush()


class Spinner:
    busy = False
    delay = 0.1
    enable_spinner = True
    
    
    @staticmethod
    def spinning_cursor():
        for cursor in itertools.cycle(['-', '/', '|', '\\']):
            yield cursor
    
    
    def __init__(self, enable_spinner=True, delay=None):
        self.enable_spinner = enable_spinner
        self.spinner_generator = self.spinning_cursor()
        if delay and float(delay):
            self.delay = delay
    
    
    def spinner_task(self):
        while self.busy:
            sys.stdout.write(next(self.spinner_generator))
            sys.stdout.flush()
            time.sleep(self.delay)
            sys.stdout.write('\b')
            sys.stdout.flush()
    
    
    def __enter__(self):
        if not self.enable_spinner:
            return
        self.busy = True
        threading.Thread(target=self.spinner_task).start()
    
    
    def __exit__(self, exception, value, tb):
        if not self.enable_spinner:
            return
        self.busy = False
        time.sleep(self.delay)
        if exception is not None:
            return False
