#!/usr/bin/python
# -*- coding: utf-8 -*-
#
# Copyright (C) 2013-2019
# This file is part of Shinken Enterprise, all rights reserved.


from shinkensolutions.lib_checks.common import HTMLList, HTMLTag, BREAK_LINE, EXIT_STATUS, RaiseOnExitOptionParser, ParseOptionError, Result
from shinkensolutions.lib_checks.graphite import CheckGraphite

VERSION = '0.1'

parser = RaiseOnExitOptionParser(
    '%prog [options] [--help]',
    version='%prog ' + VERSION,
    epilog='This script allows you to check Graphite server. '
           'It will connect to the server using SSH, check Graphite state, and then return an exit code, outputs, and perfdatas. '
           'Please specify -w and -c options if you want to add thresholds on the space used by Graphite. You have to use the option -m to select either the standalone or Shinken mode.'
)
parser.add_option('-H', '--hostname', dest='hostname', help='Hostname to connect to, where Graphite is hosted. Mandatory.')
parser.add_option('-p', '--port', dest='ssl_port', type='int', default=22, help='SSH port to connect to. Default : 22')
parser.add_option('-i', '--ssh-key', dest='ssh_key_file', default='~/.ssh/id_rsa', help='SSH key file to use. By default it will take ~/.ssh/id_rsa.')
parser.add_option('-u', '--user', dest='user', default='shinken', help='remote user to use. By default shinken.')
parser.add_option('-P', '--passphrase', dest='passphrase', default='', help='SSH key passphrase. By default will use void.')
parser.add_option('-s', '--graphite_port', dest='graphite_port', default=2003, help='Graphite Socket. Default : 2003')
parser.add_option('-d', '--graphite_data_location', dest='data_location', default='/opt/graphite/storage/whisper', help='Graphite Data Location. Default : /opt/graphite/storage/whisper')
parser.add_option('-U', '--graphite_user', dest='graphite_user', default='apache', help='Graphite user. Default : apache')
parser.add_option('-w', '--warning', dest='warning', default=85, help='Warning value for Graphite space usage. In percent. Default : 85%')
parser.add_option('-c', '--critical', dest='critical', default=95, help='Critical value for Graphite space usage. In percent. Default : 95%')
parser.add_option('-f', '--conf-file', dest='conf_file', default='/opt/graphite/conf/carbon.conf', help='Carbon configuration file path. Default : /opt/graphite/conf/carbon.conf')
parser.add_option('-n', '--graphite_process_name', dest='graphite_process_name', default='', help='Graphite writer process name. Default : carbon-cache or carbon-relay')


def _parse_args():
    opts = None
    try:
        opts, args = parser.parse_args()
        if args:
            parser.error('Does not accept any argument.')
        
        if not opts.hostname:
            parser.error('Missing parameter hostname (-H/--hostname)')
        
        if (opts.warning and not opts.critical) or (opts.critical and not opts.warning):
            parser.error('Warning parameter (-w) and Critical parameter (-c) must be used together')
    except ParseOptionError as e:
        result = Result()
        result.hard_exit(EXIT_STATUS.CRITICAL, 'Fail to parse commande argument : %s %s' % (BREAK_LINE, BREAK_LINE.join(e.msg.split('\n'))))
    
    return opts


def main():
    opts = _parse_args()
    
    # retrieve args
    hostname = opts.hostname
    
    check_graphite = CheckGraphite(
        graphite_hostname=opts.hostname,
        graphite_port=opts.graphite_port,
        ssh_port=opts.ssl_port,
        ssh_key_file=opts.ssh_key_file,
        passphrase=opts.passphrase,
        user=opts.user,
        graphite_location=opts.data_location,
        graphite_user=opts.graphite_user,
        storage_usage_warning=opts.warning,
        storage_usage_critical=opts.critical,
        graphite_process_name=opts.graphite_process_name,
        graphite_conf_file=opts.conf_file,
    )
    
    check_graphite.do_all_check()
    
    title = 'Graphite service as %s' % check_graphite.get_mode()
    if check_graphite.result.status == EXIT_STATUS.CRITICAL:
        title = '%s is critical :' % title
    elif check_graphite.result.status in (EXIT_STATUS.WARNING, EXIT_STATUS.UNKNOWN):
        title = '%s is not working fine :' % title
    elif check_graphite.result.status == EXIT_STATUS.OK:
        title = '%s is working fine.' % title
    
    summary_text = HTMLList.header_list('Graphite (Address: %s) Type: %s' % (HTMLTag.color_text(hostname), check_graphite.get_mode()), check_graphite.summary)
    check_graphite.result.add_check(long_output=summary_text)
    if check_graphite.is_relay:
        nodes_summary = HTMLList.header_list('Node information', check_graphite.nodes_summary)
        check_graphite.result.add_check(long_output=nodes_summary)
    check_graphite.result.add_title(title)
    check_graphite.result.exit(sorted_by_level=True)


if __name__ == '__main__':
    main()
