#!/usr/bin/python
# -*- coding: utf-8 -*-
#
# Copyright (C) 2013-2019:
# This file is part of Shinken Enterprise, all rights reserved.

import hashlib
import os
import shutil
import uuid

from shinken.basemodule import BaseModule, SOURCE_STATE
from shinken.daterange import Daterange
from shinken.log import logger
from shinken.objects.config import Config

from shinken.synchronizer.dao.def_items import NAGIOS_TABLES, ADVANCED_PERIODS, TIMEPERIOD_BASIC_PROPS, NAGIOS_TABLE_KEYS
from shinken.synchronizer.business.source.source import Sources
from shinken.synchronizer.dao.datamanagerV2 import get_type_item_from_class

from shinkensolutions.time_period_parser import TimePeriodParser, PERIOD_TYPE


class NoRight(Exception):
    res = None
    
    
    def __init__(self, res):
        self.res = res


TEMPLATE_SE_UUID = u'''
# Shinken Enterprise. Lines added by import core. Do not remove it, it's used by Shinken Enterprise to update your objects if you re-import them.
   _SE_UUID             %s
   _SE_UUID_HASH        %s
# End of Shinken Enterprise part
'''

EXCLUDE_TYPE = ('arbiter', 'broker', 'receiver', 'synchronizer', 'scheduler', 'reactionner', 'poller')

properties = {
    'daemons': ['synchronizer'],
    'type'   : 'cfg-file-import',
}


# called by the plugin manager to get a module
def get_instance(plugin):
    logger.info("[CFG-File Import] Get a cfg-file import module for plugin %s" % plugin.get_name())
    
    # Catch errors
    cfg_path = getattr(plugin, 'cfg_path', '')
    transform_into_packs = getattr(plugin, 'transform_into_packs', '0') == '1'
    copy_libexec = getattr(plugin, 'copy_libexec', '0') == '1'
    
    instance = CFGFile_Import(plugin, cfg_path, transform_into_packs, copy_libexec)
    return instance


class CFGFile_Import(BaseModule):
    def __init__(self, mod_conf, path, transform_into_packs, copy_libexec):
        BaseModule.__init__(self, mod_conf)
        self.copy_libexec = copy_libexec
        self.cfg_path = path
        self.file = None
        self.conf = None
        self.transform_into_packs = transform_into_packs
        self.syncdaemon = None
        self.services_to_edit = []
        self.se_uuid_mapping = {}
        self.create_and_write_SEUUID_in_file = getattr(mod_conf, 'create_and_write_SEUUID_in_file', '0') == '1'
        self.update_cfg_with_staging_se_uuid = getattr(mod_conf, 'update_cfg_with_staging_se_uuid', '0') == '1'
        
        _possible_keys = getattr(mod_conf, 'properties_used_as_synckey', 'address')
        self.possible_keys = set([key.strip() for key in _possible_keys.split(',')])
    
    
    # Called by Synchronizer to say 'let's prepare yourself guy'
    def init(self):
        logger.info("[CFG-File Import] Initialization of the cfg file import module")
    
    
    def load(self, daemon):
        self.syncdaemon = daemon
    
    
    # If asked by the user, we can hook the service=>hostgroup logic and
    # transform it into checks templates => host templates
    def do_transform_into_packs(self, raw_objects):
        logger.debug('[CFG-File Import] DO TRANSFORM INTO PACKS')
        # look which services will need to be transformed
        service_to_migrate = []
        hostgroups_called = set()
        for s_ in raw_objects['service']:
            # onl manage real services
            register = s_.get('register', None)
            if register and register[0] == '0':
                logger.debug('[CFG-File Import] Wont touch check %s' % s_)
                continue
            logger.debug('[CFG-File Import] TRANSFORM PACK FOR %s' % s_)
            hgs = s_.get('hostgroup_name', None)
            # Try to look in templates if possible
            if not hgs:
                continue
            
            hgnames = [hgname.strip() for hgname in hgs[0].split(',')]
            for hgname in hgnames:
                # first clean hostgroup name as it can be a complex one
                _exp_characters = ['&', '|', '(', ')', '!']
                hgname_clean = hgname
                if hgname_clean.startswith('+'):
                    hgname_clean = hgname_clean[1:]
                for c in _exp_characters:
                    hgname_clean = hgname_clean.replace(c, ',')
                for n_ in hgname_clean.split(','):
                    n_ = n_.strip()
                    if n_:
                        hostgroups_called.add(n_)
                s_['register'] = ['0']
                s_['host_name'] = hgs
            del s_['hostgroup_name']
        
        # Create host templates from called hostgroups
        for hgname in hostgroups_called:
            h = {'name': [hgname], 'register': ['0']}
            raw_objects['host'].append(h)
        
        # And set host template the same as hostgroups
        for h in raw_objects['host']:
            if 'hostgroups' in h:
                h['use'] = h['hostgroups']
        
        # Also transform bp_rule service into hosts
        # WARNING: it can maybe have duplicate, if so... no luck really
        bp_rule_services = [s_ for s_ in raw_objects['service'] if s_.get('check_command', None) and s_['check_command'][0].split('!')[0] == 'bp_rule']
        real_services = [s_ for s_ in raw_objects['service'] if not (s_.get('check_command', None) and s_['check_command'][0].split('!')[0] == 'bp_rule')]
        
        # keep in services only real services
        raw_objects['service'] = real_services
        
        # and transform bp_rule services into hosts
        for s_ in bp_rule_services:
            sdesc = s_.get('service_description', None)
            if not sdesc:
                continue
            s_['host_name'] = sdesc
            del s_['service_description']
            # also set check every time
            s_['check_period'] = ['24x7']
            raw_objects['host'].append(s_)
    
    
    def copy_folder_global_data_to(self, destination_dir_name='/etc/shinken/resource.d'):
        is_find = False
        for cfg_dir_name in self.conf.packs_dirs:
            if not cfg_dir_name == destination_dir_name:
                # if it is the root folder => cfg_dir=.
                if os.path.basename(cfg_dir_name) == '.':
                    origin_dir_name = os.path.join(cfg_dir_name, 'global-data')
                    if os.path.exists(origin_dir_name):
                        logger.debug('[CFG-File Import] **** find the root folder %s' % origin_dir_name)
                        is_find = True
                        break
                
                # if the global-data is define => cfg_dir=global-data
                if os.path.basename(cfg_dir_name) == 'global-data':
                    logger.debug('[CFG-File Import] **** find the custom folder %s' % cfg_dir_name)
                    origin_dir_name = cfg_dir_name
                    is_find = True
                    break  # We have find, the global-data so we stop the loop.
        
        # If we find the global-data, so copy it.
        if is_find:
            destination_path = os.path.join(destination_dir_name, self.get_name(), 'global-data')
            logger.debug('[CFG-File Import] -- We will copy this folder: %s to %s', origin_dir_name, destination_path)
            if not os.path.exists(destination_path):
                # Create all directories if necessary.
                os.makedirs(destination_path)
                logger.debug('[CFG-File Import] -- create: %s' % destination_path)
            
            # Destroy the destination directories.
            shutil.rmtree(destination_path)
            logger.debug('[CFG-File Import] -- rmtree: %s' % destination_path)
            
            # Copy the global-data folder to the destination directories.
            shutil.copytree(origin_dir_name, destination_path)
            logger.debug('[CFG-File Import] -- copytree: %s' % destination_path)
        else:
            logger.info('[CFG-File Import] [%s] No global-data folder found' % self.get_name().replace('module-', ''))
    
    
    def copy_folder_libexec_to(self, destination_dir_name='/var/lib/shinken-user/libexec/'):
        origin_dir_name = None
        for cfg_dir_name in self.conf.packs_dirs:
            if not cfg_dir_name == destination_dir_name:
                # if it is the root folder => cfg_dir=.
                if os.path.basename(cfg_dir_name) == '.':
                    origin_dir_name = os.path.join(cfg_dir_name, 'libexec')
                    if os.path.exists(origin_dir_name):
                        logger.debug('[CFG-File Import] **** find the root folder %s' % origin_dir_name)
                        break
                
                # if the global-data is define => cfg_dir=libexec
                if os.path.basename(cfg_dir_name) == 'libexec':
                    logger.debug('[CFG-File Import] **** find the custom folder %s' % cfg_dir_name)
                    origin_dir_name = cfg_dir_name
                    break  # We have find, the global-data so we stop the loop.
        
        # If we find the global-data, so copy it.
        if origin_dir_name:
            destination_path = os.path.join(destination_dir_name, self.get_name().replace('module-', ''))
            logger.debug('[CFG-File Import] -- We will copy this folder: %s to %s', origin_dir_name, destination_path)
            
            if not os.path.exists(destination_path):
                # Create all directories if necessary.
                logger.debug('[CFG-File Import] -- create: %s' % destination_path)
                os.makedirs(destination_path)
            
            # Destroy the destination directories.
            shutil.rmtree(destination_path)
            logger.debug('[CFG-File Import] -- rmtree: %s' % destination_path)
            
            # Copy the global-data folder to the destination directories.
            shutil.copytree(origin_dir_name, destination_path)
            
            logger.debug('[CFG-File Import] -- copytree: %s' % destination_path)
        else:
            logger.info('[CFG-File Import] [%s] No libexec folder found' % self.get_name().replace('module-', ''))
    
    
    def get_objects(self):
        # A new objects each time!
        self.conf = Config()
        
        self.services_to_edit = []
        self.se_uuid_mapping = {}  # path of element ordered by their se_uuid
        
        if not self.cfg_path:
            res = {'state': SOURCE_STATE.NOT_CONFIGURED, 'output': self.syncdaemon._('import-cfg-file.output_nc_source_nc_skipped'), 'objects': {}, 'errors': [], 'warnings': []}
            return res
        
        if isinstance(self.cfg_path, list):
            res = {
                'state'   : SOURCE_STATE.CRITICAL,
                'output'  : self.syncdaemon._('import-cfg-file.output_critical'),
                'objects' : {},
                'errors'  : [self.syncdaemon._('import-cfg-file.err_multiple_cfg_path')],
                'warnings': []
            }
            return res
        
        if not os.path.exists(self.cfg_path):
            logger.error('[CFG-File Import] The file [%s] is missing' % self.cfg_path)
            msg = self.syncdaemon._('import-cfg-file.output_critical')
            res = {
                'state'   : SOURCE_STATE.CRITICAL,
                'output'  : msg,
                'objects' : {},
                'errors'  : [self.syncdaemon._('import-cfg-file.cr_missing_file') % self.cfg_path],
                'warnings': []
            }
            return res
        
        self.conf.read_config_silent = 1
        buf = self.conf.read_config([self.cfg_path])
        if self.conf.conf_is_empty:
            logger.error('[CFG-File Import] The source is not configured. Please set up the configuration path in [%s]' % self.cfg_path)
            res = {
                'state'   : SOURCE_STATE.NOT_CONFIGURED,
                'output'  : self.syncdaemon._('import-cfg-file.output_nc_source_nc_in') % self.cfg_path,
                'objects' : {},
                'errors'  : [],
                'warnings': []
            }
            return res
        if self.conf.bad_encoding_files:
            logger.error('[CFG-File Import] The source is not configured. Please set up the configuration path in [%s]' % self.cfg_path)
            res = {
                'state'   : SOURCE_STATE.CRITICAL,
                'output'  : self.syncdaemon._('import-cfg-file.output_cr_incorrect_encoding_file'),
                'objects' : {},
                'errors'  : self.conf.bad_encoding_files,
                'warnings': []
            }
            return res
        
        raw_objects = self.conf.read_config_buf(buf, EXCLUDE_TYPE)
        if not self.conf.conf_is_correct:
            logger.error('[CFG-File Import] The configuration file [%s] is incorrect' % self.cfg_path)
            res = {
                'state'   : SOURCE_STATE.CRITICAL,
                'output'  : self.syncdaemon._('import-cfg-file.output_critical'),
                'objects' : {},
                'errors'  : [self.syncdaemon._('import-cfg-file.cr_file_loading_fail') % '<br> * '.join(self.conf.configuration_errors)],
                'warnings': []
            }
            return res
        self.conf.load_packs()
        
        warnings = []
        
        # Copy the '[source_name]/global-data.cfg' file to '/etc/shinken/resource.d/[source_name]/global-data.cfg'
        self.copy_folder_global_data_to()
        
        if self.copy_libexec:
            self.copy_folder_libexec_to()
        
        # Incomplete configuration catch: we want the creation of missing hostgroups/contactgroups from hosts/services
        # if set on the elements, so if missing, create them in a void mode if need
        called_hostgroups = {}
        called_contactgroups = {}
        # loop over hosts and catch the hostgroups property
        # try to see which hostgroups are called, and by which hosts
        for item in raw_objects['host']:
            hname = item.get('host_name', '')
            if hname and isinstance(hname, list):
                hname = hname[0]
            
            hostgroups_ = item.get('hostgroups', '')
            if hostgroups_:
                if isinstance(hostgroups_, list):
                    hostgroups_ = ','.join(hostgroups_)
                
                hgs = self._parse_list(hostgroups_)
                for hg in hgs:
                    if hg not in called_hostgroups:
                        called_hostgroups[hg] = {'hostgroup_name': [hg], 'members': []}
                    if hname:
                        called_hostgroups[hg]['members'].append(hname)
            
            # same for contact groups, but do not need members as it's not hosts
            contactgroups_ = item.get('contact_groups', '')
            if contactgroups_:
                if isinstance(contactgroups_, list):
                    contactgroups_ = ','.join(contactgroups_)
                cgs = self._parse_list(contactgroups_)
                for cg in cgs:
                    if cg not in called_contactgroups:
                        called_contactgroups[cg] = {'contactgroup_name': [cg], 'members': []}
        
        # also loop over contacts to know in which contactgroups they are
        for item in raw_objects['contact']:
            cname = item.get('contact_name', '')
            if cname:
                if isinstance(cname, list):
                    cname = cname[0]
            # same for contact groups, but do not need members as it's not hosts
            contactgroups_ = item.get('contactgroups', item.get('contact_groups', ''))
            if contactgroups_:
                if isinstance(contactgroups_, list):
                    contactgroups_ = ','.join(contactgroups_)
                cgs = self._parse_list(contactgroups_)
                for cg in cgs:
                    if cg not in called_contactgroups:
                        called_contactgroups[cg] = {'contactgroup_name': [cg], 'members': []}
                    if cname:
                        called_contactgroups[cg]['members'].append(cname)
        
        already_created_hostgroups = set()
        already_created_contactgroups = set()
        for item in raw_objects['hostgroup']:
            hgname_ = item.get('hostgroup_name', '')
            if hgname_:
                if isinstance(hgname_, list):
                    hgname_ = ','.join(hgname_)
                already_created_hostgroups.add(hgname_)
        for item in raw_objects['contactgroup']:
            cgname_ = item.get('contactgroup_name', '')
            if cgname_:
                if isinstance(cgname_, list):
                    cgname_ = ','.join(cgname_)
                already_created_contactgroups.add(cgname_)
        
        # we want the hostgroups that are not already created
        for hgname_ in set(called_hostgroups.keys()) - already_created_hostgroups:
            if hgname_ != 'null':
                hg = called_hostgroups[hgname_]
                # members will be add with double link at import
                if 'members' in hg:
                    del hg['members']
                raw_objects['hostgroup'].append(hg)
                logger.debug("[CFG-File Import] Creating 'on the fly' host group [%s]" % hgname_)
        
        # Fill the contactgroups we need to create, with their members
        for cgname_ in set(called_contactgroups.keys()) - already_created_contactgroups:
            if cgname_ != 'null':
                cg = called_contactgroups[cgname_]
                # members will be add with double link at import
                if 'members' in cg:
                    del cg['members']
                raw_objects['contactgroup'].append(cg)
        
        # Try to fill holes in services, like service_description missing but present in the service_templates
        # used by the service
        # first save all service templates, and the one with names into it
        service_templates_ = {}
        services_without_description = []
        for s_ in raw_objects['service']:
            register = s_.get('register', None)
            if register and register[0] == '0':
                name_ = s_.get('name', [])
                if not name_:
                    continue
                name_ = name_[0]
                service_templates_[name_] = s_
            else:  # nor a template
                sdesc_ = s_.get('service_description', [])
                # maybe we have a valid service with a name, skip this, there is no hole here
                if sdesc_:
                    continue
                services_without_description.append(s_)
        
        for s_ in services_without_description:
            use = s_.get('use', None)
            # Try to look in templates if possible
            if not use:
                continue
            tpl_names_ = [_template.strip() for _template in use[0].split(',')]
            for tname_ in tpl_names_:
                if tname_ in service_templates_:
                    tpl = service_templates_[tname_]
                    sdesc_ = tpl.get('service_description', None)
                    if sdesc_:
                        s_['service_description'] = sdesc_
                        break
        
        # If ask, try to transform all checks to hostgroups into checks into host templates
        if self.transform_into_packs:
            logger.debug('[CFG-File Import] Will transform checks to hostgroups => checks to host templates [%s]')
            self.do_transform_into_packs(raw_objects)
        
        # Clean pass
        for (item_class, items) in raw_objects.iteritems():
            if item_class not in NAGIOS_TABLES:
                continue
            item_class_python = NAGIOS_TABLES.get(item_class, (None,))[0]
            item_class_s = "%ss" % item_class
            
            # Clean objects from list value, we just want string
            for item in items:
                if item_class == 'timeperiod':
                    # First be sure we are setting the valid value types
                    advanced_periods = []
                    for (property_name, property_value) in item.items():
                        if property_name == ADVANCED_PERIODS:
                            item.pop(ADVANCED_PERIODS, None)
                            _name = item.get('timeperiod_name', [self.syncdaemon._('element.unknown')])[0]
                            _from = item.get('imported_from', [self.syncdaemon._('element.unknown')])[0]
                            warnings.append(self.syncdaemon._('import-cfg-file.warn_advanced_timeperiod_skipped') % (_name, _from))
                            continue
                        if not property_name.startswith('_') and property_name not in TIMEPERIOD_BASIC_PROPS:
                            advanced_periods.extend(self.update_timeperiod_from_property(item, property_name, property_value))
                
                # Clean objects from list value, we just want string
                self.align_values(item, item_class_python)
                if item_class == 'timeperiod' and advanced_periods:
                    item[ADVANCED_PERIODS] = TimePeriodParser.SEPARATOR_ADVANCED_DEF.join(advanced_periods)
                
                # Populate imported_from property
                _orig_imported_from = self.populate_imported_from(item)
                
                if item_class == 'service' and 'name' not in item:
                    if '_SERVICE_ID' in item:
                        item['_SYNC_KEYS'] = (item.get('_SERVICE_ID')).lower()
                    else:
                        item['_SYNC_KEYS'] = None
                else:
                    keys = set()
                    for key in self.possible_keys:
                        if key in item:
                            keys.add(item[key].lower())
                    if item.get('register', '1') == '0':
                        if 'name' in item:
                            keys.add(item['name'] + '-tpl')
                    item['_SYNC_KEYS'] = (','.join(keys)).lower()
                
                # Populate pack property
                for p in self.conf.packs:
                    dir_path = p.dir_path + os.sep
                    if _orig_imported_from.startswith(dir_path):
                        item['pack'] = p.pack_name
                
                # Remapping alias -> display_name
                if item_class == 'host':
                    alias = item.get('alias', None)
                    if alias is not None and 'display_name' not in item:
                        item['display_name'] = alias
                        del item['alias']
                
                # Remapping active_check_enabled -> active_check_enabled
                if item_class == 'host':
                    active_check_enabled = item.get('active_check_enabled', None)
                    if active_check_enabled is not None:
                        item['active_checks_enabled'] = active_check_enabled
                        del item['active_check_enabled']
                
                # Populate is_cluster property
                if item_class == 'host':
                    check_command = item.get('check_command', '')
                    if check_command.startswith('bp_rule!'):
                        item['is_cluster'] = '1'
                
                # Mark all services that do not have _SE_UUID to be mark for file edition
                if (self.create_and_write_SEUUID_in_file or (item_class == 'service' and 'name' not in item)) and '_SE_UUID' not in item:
                    item_path, line_nb = self._get_item_definition_location(item)
                    item_type = get_type_item_from_class(item_class_s, item)
                    
                    s_uuid = uuid.uuid1().hex
                    _se_uuid = 'core-%s-%s' % (item_type, s_uuid)
                    _se_uuid_hash = hashlib.md5(_se_uuid).hexdigest()
                    item['_SE_UUID'] = _se_uuid
                    item['_SE_UUID_HASH'] = _se_uuid_hash
                    self.services_to_edit.append((item_path, line_nb, _se_uuid, _se_uuid_hash))
                elif '_SE_UUID' in item:
                    # Make sure the _SE_UUID has a type and not a class
                    item_type = get_type_item_from_class(item_class_s, item)
                    original_item_se_uuid = item['_SE_UUID']
                    split_original_item_se_uuid = original_item_se_uuid.split('-', 2)
                    if len(split_original_item_se_uuid) == 3:
                        (core, se_uuid_type, se_uuid_id) = split_original_item_se_uuid
                        
                        if item_type != se_uuid_type:
                            item_path, line_nb = self._get_item_definition_location(item)
                            
                            _se_uuid = 'core-%s-%s' % (item_type, se_uuid_id)
                            _se_uuid_hash = hashlib.md5(_se_uuid).hexdigest()
                            item['_SE_UUID'] = _se_uuid
                            item['_SE_UUID_HASH'] = _se_uuid_hash
                            self.services_to_edit.append((item_path, line_nb, _se_uuid, _se_uuid_hash))
                
                # Populate SYNC_KEYS property
                self.populate_sync_keys(item, item_class)
                
                if '_SE_UUID' not in item and self.update_cfg_with_staging_se_uuid:
                    existing_item = next(iter(Sources.find_existing_item_match(self.syncdaemon, item, item_class)), None)
                    if existing_item:
                        existing_item_uuid = existing_item['_id']
                        item_path, line_nb = self._get_item_definition_location(item)
                        
                        item_type = get_type_item_from_class(item_class_s, item)
                        _se_uuid = 'core-%s-%s' % (item_type, existing_item_uuid)
                        _se_uuid_hash = hashlib.md5(_se_uuid).hexdigest()
                        item['_SE_UUID'] = _se_uuid
                        item['_SE_UUID_HASH'] = _se_uuid_hash
                        item['_SYNC_KEYS'] = '%s,%s' % (item['_SYNC_KEYS'], _se_uuid) if item['_SYNC_KEYS'] else _se_uuid
                        item['_SYNC_KEYS'] = item['_SYNC_KEYS'].lower()
                        
                        self.services_to_edit.append((item_path, line_nb, _se_uuid, _se_uuid_hash))
                
                # Populate _SE_UUID mapping for found duplicate
                if '_SE_UUID' in item:
                    _se_uuid = item['_SE_UUID']
                    if _se_uuid not in self.se_uuid_mapping:
                        self.se_uuid_mapping[_se_uuid] = []
                    self.se_uuid_mapping[_se_uuid].append(_orig_imported_from)
        
        # Before return, warn about duplicated se_uuid
        se_uuid_errors = [lst for lst in self.se_uuid_mapping.itervalues() if len(lst) >= 2]
        if len(se_uuid_errors) != 0:
            err = self.syncdaemon._('import-cfg-file.err_elements_duplicated') % ' '.join([' '.join(['<li>%s</li>' % i for i in l]) for l in se_uuid_errors])
            res = {
                'state'   : SOURCE_STATE.CRITICAL,
                'output'  : self.syncdaemon._('import-cfg-file.output_critical'),
                'objects' : {},
                'errors'  : [err],
                'warnings': []
            }
            logger.error('[CFG-File Import] %s' % err)
            return res
        
        logger.debug('[CFG-File Import] services to edit: [%s]' % self.services_to_edit)
        
        try:
            self.update_se_uuid_in_file(self.services_to_edit)
        except NoRight as e:
            return e.res
        
        raw_objects = dict([(k, v) for k, v in raw_objects.iteritems() if v and k in NAGIOS_TABLES])
        res = {
            'state'   : SOURCE_STATE.WARNING if warnings else SOURCE_STATE.OK,
            'output'  : self.syncdaemon._('import-cfg-file.output_warn_property_skipped') % self.cfg_path if warnings else self.syncdaemon._('import-cfg-file.output_ok_load_successful') % self.cfg_path,
            'objects' : raw_objects,
            'errors'  : [],
            'warnings': warnings
        }
        return res
    
    
    def _get_item_definition_location(self, item):
        # Here we split first the two values and then the two other with a rsplit. Why ? Because while running TU on windows you have a C: in path  and we need to skip it !
        _, tmp = item['imported_from'].split(':', 1)
        item_path, line_nb = tmp.rsplit(':', 1)
        return item_path, int(line_nb)
    
    
    def update_timeperiod_from_property(self, item, property_name, property_value):
        basic_period = ""
        periods = []
        advanced_periods = []
        if isinstance(property_value, list):
            for single_value in property_value:
                basic_period = self.dispatch_period_def(advanced_periods, periods, basic_period, property_name, single_value)
        else:
            basic_period = self.dispatch_period_def(advanced_periods, periods, basic_period, property_name, property_value)
        
        if property_name in Daterange.weekdays and (periods or basic_period):
            periods = [basic_period] + periods
            
            # Method align_values() expects lists for properties not defined for the item class (Timeperiod does not define properties 'monday' through 'sunday')
            item[property_name] = [TimePeriodParser.SEPARATOR_DAY_DEF.join(periods)]
        else:
            if basic_period:
                basic_period = "%s%s" % (TimePeriodParser.SEPARATOR_DAY_OPT, basic_period)
                periods = [basic_period] + periods
            
            advanced_periods.extend(["%s %s" % (property_name, period) for period in periods])
            item.pop(property_name, None)
        return advanced_periods
    
    
    def dispatch_period_def(self, advanced_periods, periods, basic_period, property_name, value):
        res = TimePeriodParser.parse_time_periods_from_cfg(property_name, value)
        if res['type'] == PERIOD_TYPE.BASIC_DAY and basic_period == "":
            basic_period = res['value']
        elif res['type'] == PERIOD_TYPE.ADVANCED:
            advanced_periods.append("%s %s" % (property_name, res['value']))
        else:
            periods.append(res['value'])
        return basic_period
    
    
    def update_se_uuid_in_file(self, items_to_edit):
        # Check we have wright to write file with our item to edit
        cannot_write_files = [item_path for (item_path, line_nb, _, _) in items_to_edit if line_nb != 0 and not os.access(item_path, os.W_OK)]
        if len(cannot_write_files) > 0:
            start_tags = '<ul class="shinken-status-list"><li>'
            end_tags = '</li></ul>'
            files = start_tags + '</li><li>'.join(cannot_write_files) + end_tags
            err = [self.syncdaemon._('import-cfg-file.err_cannot_write_configuration') + files]
            res = {
                'state'   : SOURCE_STATE.CRITICAL,
                'output'  : self.syncdaemon._('import-cfg-file.output_critical'),
                'objects' : {},
                'errors'  : err,
                'warnings': []
            }
            logger.error('[CFG-File Import] %s' % err)
            raise NoRight(res)
        
        files_to_edit = {}
        for (item_path, line_nb, _se_uuid, _se_uuid_hash) in items_to_edit:
            if line_nb == 0:
                continue
            if item_path not in files_to_edit:
                files_to_edit[item_path] = []
            files_to_edit[item_path].append((line_nb, _se_uuid, _se_uuid_hash))
        
        for item_path, data in files_to_edit.iteritems():
            data.sort(key=lambda d: d[0], reverse=True)
            
            with open(item_path, 'r') as cfg_file:
                cfg_file_lines = cfg_file.read().decode("utf8").splitlines()
            
            for (line_nb, _se_uuid, _se_uuid_hash) in data:
                to_add = TEMPLATE_SE_UUID % (_se_uuid, _se_uuid_hash)
                # Find existing SE_UUID in object ; if found, replace, if not found insert template
                line_index = line_nb
                line_se_uuid = line_nb
                found_hash = False
                found_uuid = False
                while line_index < len(cfg_file_lines) and not cfg_file_lines[line_index].strip().startswith('}'):
                    
                    line = cfg_file_lines[line_index].strip()
                    
                    if '_SE_UUID_HASH' in line:
                        
                        found_hash = True
                        lines_modified = 0
                        
                        cfg_file_lines[line_index] = u'    _SE_UUID_HASH     %s' % _se_uuid_hash
                        
                        if cfg_file_lines[line_index - 1].strip().startswith(u'# Shinken Enterprise. Line'):
                            cfg_file_lines.pop(line_index - 1)
                            lines_modified += -1
                        
                        if not cfg_file_lines[line_index + lines_modified - 1].strip().startswith('_SE_UUID') and not cfg_file_lines[line_index - 1].strip().startswith(u'# Shinken Enterprise. Line'):
                            cfg_file_lines.insert(line_index + lines_modified, u'# Shinken Enterprise. Line updated by import core. Do not change it, it\'s used by Shinken Enterprise to update your objects if you re-import them.')
                            lines_modified += 1
                        
                        if not cfg_file_lines[line_index + lines_modified + 1].strip().startswith('_SE_UUID') and not cfg_file_lines[line_index + lines_modified + 1].strip() == u'# End of Shinken Enterprise part':
                            cfg_file_lines.insert(line_index + lines_modified + 1, u'# End of Shinken Enterprise part')
                            lines_modified += 1
                        
                        line_index += lines_modified
                    
                    
                    elif '_SE_UUID' in line:
                        found_uuid = True
                        lines_modified = 0
                        
                        cfg_file_lines[line_index] = u'    _SE_UUID          %s' % _se_uuid
                        
                        if cfg_file_lines[line_index - 1].strip().startswith(u'# Shinken Enterprise. Line'):
                            cfg_file_lines.pop(line_index - 1)
                            lines_modified += -1
                        
                        if not cfg_file_lines[line_index + lines_modified - 1].strip().startswith('_SE_UUID_HASH'):
                            cfg_file_lines.insert(line_index + lines_modified, u'# Shinken Enterprise. Line updated by import core. Do not change it, it\'s used by Shinken Enterprise to update your objects if you re-import them.')
                            lines_modified += 1
                        
                        line_se_uuid = line_index + lines_modified
                        
                        if not cfg_file_lines[line_index + lines_modified + 1].strip().startswith('_SE_UUID_HASH') and not cfg_file_lines[line_index + lines_modified + 1].strip().startswith(u'# End of Shinken Enterprise part'):
                            cfg_file_lines.insert(line_index + lines_modified + 1, u'# End of Shinken Enterprise part')
                            lines_modified += 1
                        
                        line_index += lines_modified
                    
                    line_index += 1
                
                if not found_hash and not found_uuid:
                    cfg_file_lines.insert(line_nb, to_add)
                elif not found_hash:
                    cfg_file_lines.insert(line_se_uuid + 1, u'    _SE_UUID_HASH     %s' % _se_uuid_hash)
            
            logger.debug('[CFG-File Import] Writing update cfg file [%s]' % item_path)
            
            # Fix the path with os.path to be available to launch TU on windows
            new_path = os.path.join(os.sep, 'tmp', '%s.se' % os.path.basename(item_path))
            with open(new_path, 'w') as cfg_file:
                cfg_file.write('\n'.join(cfg_file_lines).encode("utf8"))
            shutil.move(new_path, item_path)
    
    
    def populate_imported_from(self, item):
        # Now mark objects with from file source data
        _orig_imported_from = item.get('imported_from', '')
        clean_name = self.get_name().replace('module-', '')
        if _orig_imported_from:
            item['imported_from'] = '%s:%s' % (clean_name, os.path.abspath(_orig_imported_from))
        else:
            item['imported_from'] = '%s:0' % clean_name
            _orig_imported_from = ''
        
        return _orig_imported_from
    
    
    def populate_sync_keys(self, item, item_class):
        keys = set()
        is_check_on_element = item_class == 'service' and 'name' not in item
        
        if is_check_on_element and '_SERVICE_ID' in item:
            keys.add(item['_SERVICE_ID'])
        
        if '_SE_UUID' in item:
            keys.add(item['_SE_UUID'])
        
        if not is_check_on_element:
            keys.update([item[key].lower() for key in self.possible_keys if key in item])
            
            key_name = NAGIOS_TABLE_KEYS[item_class]
            if item.get('register', '1') == '0' and 'name' in item:
                keys.add(item['name'] + '-tpl')
            elif item.get('register', '1') == '1' and key_name in item:
                keys.add(item[key_name])
        
        item['_SYNC_KEYS'] = (','.join(keys)).lower()
    
    
    def align_values(self, item, item_class_python):
        for property_name, property_value in item.iteritems():
            if len(property_value) == 0:
                item[property_name] = ''
            elif property_name in getattr(item_class_python, 'properties', {}):
                # Properties allowing several instances in definition are merged together
                _property = item_class_python.properties[property_name]
                if _property.merging == 'duplicate':
                    item[property_name] = _property.separator.join(property_value)
                else:
                    item[property_name] = property_value[0]
            else:
                item[property_name] = property_value[0]
            
            # Clean bad property_name format if need, mongo do not allow . in key
            if '.' in property_name:
                nk = property_name.replace('.', '')
                item[nk] = item[property_name]
                del item[property_name]
    
    
    def _parse_list(self, _list):
        hgs = []
        for hg in _list.split(','):
            hg = hg.strip()
            if hg.startswith('+'):
                hg = hg[1:]
                hg = hg.strip()
            if hg:
                hgs.append(hg)
        return hgs
