#!/usr/bin/python
# -*- coding: utf-8 -*-
#
# Copyright (C) 2013-2018
# This file is part of Shinken Enterprise, all rights reserved.

import json
import os
import shutil
import sys
import time
import uuid

from pymongo.connection import Connection

from shinken.basemodule import BaseModule, SOURCE_STATE
from shinken.discovery.discoverymanager import DiscoveryManager
from shinken.log import logger
from shinken.objects import Command
from shinken.property import StringProp
from shinken.discovery.discoveryrules import DiscoveryRulesManager, INVALID_STATE, WARNING_RULES_STATES

properties = {
    'daemons': ['synchronizer'],
    'type'   : 'discovery-import',
    'tabs'   : True
}

app = None


# called by the plugin manager to get a module
def get_instance(plugin):
    logger.info("[Discovery Import] Get a Discovery import module for plugin %s" % plugin.get_name())
    
    # Catch errors
    mongodb_uri = plugin.mongodb_uri
    mongodb_database = plugin.mongodb_database
    
    instance = DiscoveryImport(plugin, mongodb_uri, mongodb_database)
    return instance


class DiscoveryImport(BaseModule):
    SHINKEN_NMAP_MAC_PREFIXES_PATH = "/etc/shinken/_default/sources/discovery/nmap/nmap-mac-prefixes"
    
    
    def __init__(self, mod_conf, mongodb_uri, database):
        BaseModule.__init__(self, mod_conf)
        self.mongodb_uri = mongodb_uri
        self.mongodb_database = database
        self.mongodb_con = self.mongodb_db = None
        self.syncdaemon = None
        self.source_name = mod_conf.get_name()[7:]
        self.discovery_rules_manager = None
        self.rules_path = getattr(mod_conf, 'rules_path', '')
        self.nmap_mac_prefixes_user_path = getattr(mod_conf, 'nmap_mac_prefixes_path', '')
        self.discovery_confs_db = None
        self.discovery_rules_db = None
        self.global_lang = ""
        self.langs_path = ""
        self.langs = {}
        self.tabs = []
        self.known_hosts_address = []
        self.nmap_user_file_state = "ok"
    
    
    # DON'T CHANGE THIS FUNCTION
    def _(self, s):
        lang = self.global_lang
        
        d = self.langs.get(lang, None)
        if d is None:
            pth = os.path.join(self.langs_path, lang + '.js')
            if not os.path.exists(pth):
                logger.error('Cannot load the lang file %s: no such file' % pth)
                return ''
            f = open(pth, 'r')
            buf = f.read()
            f.close()
            lines = buf.splitlines()
            new_lines = []
            for line in lines:
                line = line.strip()
                if line.startswith('//'):
                    continue
                if line.startswith('var lang'):
                    line = '{'
                if line.startswith('};'):
                    line = '}'
                new_lines.append(line)
            buf = '\n'.join(new_lines)
            o = json.loads(buf)
            self.langs[lang] = o
        o = self.langs[lang]
        elts = [e.strip() for e in s.split('.') if e.strip()]
        for e in elts:
            o = o.get(e, None)
            if o is None:
                logger.error('Traduction: cannot find %s in the lang %s' % (s, lang))
                return 'TO_TRAD(%s)' % s
        return o
    
    
    # Called by Arbiter to say 'let's prepare yourself guy'
    def init(self):
        logger.info("[Discovery Import] Initialization of the discovery import module")
    
    
    def load(self, daemon):
        self.syncdaemon = daemon
        self.__load_data_backend()
        
        self.global_lang = self.syncdaemon.lang
        self.langs_path = os.path.join(self.syncdaemon.modules_dir, properties["type"], 'htdocs', 'js', 'traductions')
        self.langs = {'en': None, 'fr': None}
        self.tabs = [{'name': "rules", 'displayed_name': self._('rules.rules'), 'content': "rules_list", 'errors': 0, 'warnings': 0}]
        self.update_tabs()
        # Clean old discovery, so we will start new one
        all_previous_running_disco = [e for e in self.discovery_confs_db.find(
            {'state': 'RUNNING', 'source_name': self.get_source_name()})]
        for e in all_previous_running_disco:
            e['state'] = 'CRITICAL'
            self.discovery_confs_db.save(e)
    
    
    def update_tabs(self):
        warnings = 0
        errors = 0
        if self.discovery_rules_manager.status['user'] == 0:
            for rule in self.discovery_rules_manager.rules:
                if rule.state == INVALID_STATE:
                    errors += 1
                elif rule.state in WARNING_RULES_STATES:
                    warnings += 1
        if warnings == 0:
            warnings = ''
        if errors == 0:
            errors = ''
        self.tabs[0]['errors'] = errors
        self.tabs[0]['warnings'] = warnings
    
    
    def get_source_name(self):
        return self.get_name()[7:]
    
    
    def get_objects(self):
        self.known_hosts_address = []
        res = {'state': "", 'output': '', 'objects': {}, 'warnings': [], 'errors': []}
        if self.discovery_rules_manager.status['default'] == 1:
            res['errors'].append(self.syncdaemon._(
                'import-discovery.critical_default_rule_file') % self.discovery_rules_manager.SHINKEN_RULES_PATH)
        if not self.__nmap_prefix_override():
            res['errors'].append(self.syncdaemon._(
                'import-discovery.critical_default_nmap_file') % self.SHINKEN_NMAP_MAC_PREFIXES_PATH)
        if self.nmap_user_file_state != "ok":
            res['errors'].append(self.syncdaemon._(
                'nmap-user-file.%s' % self.nmap_user_file_state) % self.nmap_mac_prefixes_user_path)
        if res['errors']:
            res['state'] = SOURCE_STATE.CRITICAL
            res['output'] = self.syncdaemon._(
                'import-discovery.output_critical')
            return res
        self.discovery_rules_manager.update_user_rules()
        logger.info("[Discovery Import] Search for objects")
        logger.debug("[Discovery Import] discovery_confs [%s] " % self.discovery_confs_db)
        
        discovery_confs_found = False
        objects = {'host': []}
        
        for dc in self.discovery_confs_db.find({'enabled': True, 'source_name': self.get_source_name()}):
            discovery_confs_found = True
            _id = dc['_id']
            state = dc['state']
            discovery_name = dc['discovery_name']
            
            logger.info("[Discovery Import] discovery_name %s state %s", discovery_name, state)
            
            if state in ('PENDING', 'OK', 'CRITICAL'):
                self.__do_launch_scan(dc)
            
            # Already launch, will be give after
            if state == 'RUNNING':
                continue
            
            lst = list(self.discovery_hosts_data.find({'_DISCOVERY_ID': _id}))
            logger.debug('[Discovery Import] %d hosts from %s' % (len(lst), discovery_name))
            for o in lst:
                _SYNC_KEYS = o.get('_SYNC_KEYS', [])
                # If this element is missing or is void, do not export it...
                if not _SYNC_KEYS:
                    logger.debug(
                        '[Discovery Import]  (%s) skipping element %s as it is missing sync keys' % (discovery_name, o))
                    continue
                o['_SYNC_KEYS'] = (','.join(_SYNC_KEYS)).lower()
                # Clean SE UUID inherited from discovery rules, and useless discovery id
                for k in ['_SE_UUID', '_SE_UUID_HASH', '_DISCOVERY_ID']:
                    if k in o:
                        del o[k]
                objects['host'].append(o)
        
        if discovery_confs_found:
            res = {
                'state'   : SOURCE_STATE.OK,
                'output'  : self.syncdaemon._('import-discovery.output_ok_load_successful'),
                'objects' : objects,
                'errors'  : [],
                'warnings': []
            }
            if self.discovery_rules_manager.status['user'] != 0:
                res['state'] = SOURCE_STATE.WARNING
                res['warnings'].append(self.syncdaemon._('import-discovery.warning_user_rules_not_parsed'))
                res['output'] = self.syncdaemon._(
                    'import-discovery.output_ok_load_sucessful_but_warnings')
        else:
            # We send OK state because we want clean old results
            res = {
                'state'   : SOURCE_STATE.NOT_CONFIGURED,
                'output'  : self.syncdaemon._('import-discovery.output_nc_no_scan_range'),
                'objects' : {'host': []},
                'errors'  : [],
                'warnings': []
            }
        
        logger.info('[Discovery Import] import done load [%s] hosts' % len(objects['host']))
        self.update_tabs()
        return res
    
    
    def compute_state(self, source):
        # Don't change the state if the source is running to prevent simultaneous runs : SEF-5260
        if source.state in (SOURCE_STATE.RUNNING, SOURCE_STATE.DIFFERENCE_COMPUTING):
            return
        
        source.output = ''
        source.summary_output = ''
        if self.is_all_confs_disabled(source.get_name()):
            source.state = SOURCE_STATE.NOT_CONFIGURED_BEFORE_IMPORT
            source.output = self.get_trad_no_conf_enabled()
        else:
            source.state = SOURCE_STATE.READY_FOR_IMPORT
    
    
    def set_conf_enabled(self, conf_id, enabled):
        dc = self.discovery_confs_db.find_one({'_id': conf_id})
        dc['enabled'] = (enabled == '1')
        self.discovery_confs_db.save(dc)
    
    
    def delete_conf(self, conf_id):
        self.discovery_confs_db.remove({'_id': conf_id})
    
    
    def save_conf(self, conf_id, new_conf, sname):
        conf = self.discovery_confs_db.find_one({'_id': conf_id})
        
        if conf is None:
            confs = [c['discovery_name'] for c in self.discovery_confs_db.find({'source_name': sname})]
            if new_conf['discovery_name'] in confs:
                return "name_already_exist"
                # return self.syncdaemon._('validator.disco_already_exist') % new_conf['discovery_name']
            
            conf = {'_id'             : uuid.uuid4().hex,
                    'state'           : 'PENDING',
                    'last_scan'       : 0,
                    'synchronizer'    : '',
                    'synchronizer_tag': '',
                    'scan_number'     : 0,
                    'source_name'     : sname,
                    'last_heartbeat'  : 0}
        
        conf['discovery_name'] = new_conf['discovery_name']
        conf['iprange'] = new_conf['iprange']
        conf['scan_interval'] = new_conf['scan_interval']
        conf['notes'] = new_conf['notes']
        conf['enabled'] = new_conf['enabled']
        conf['port_range'] = new_conf['port_range']
        conf['extra_option'] = new_conf['extra_option']
        
        self.discovery_confs_db.save(conf)
        
        return "ok"
    
    
    def is_all_confs_disabled(self, sname):
        all_conf = self.discovery_confs_db.find({'source_name': sname, 'enabled': True})
        return all_conf.count() == 0
    
    
    def get_trad_no_conf_enabled(self):
        return self.syncdaemon._('import-discovery.output_nc_no_scan_range')
    
    
    def get_confs(self):
        confs = []
        for dc in self.discovery_confs_db.find({'source_name': self.get_source_name()}):
            confs.append(dc)
        return confs
    
    
    # Give try to load the mongodb and turn until we are killed or the monogo goes up
    def __load_data_backend(self):
        logger.info("[Discovery Import] Load data from database")
        start = time.time()
        while not self.interrupted:
            try:
                self.__load_data_backend_try()
                # connect was ok? so bail out :)
                return
            except Exception:
                now = time.time()
                time_out = getattr(self.syncdaemon, 'mongodb_retry_timeout', 60)
                if now > start + time_out:  # more than 1min without mongo? not good!
                    logger.error('[Discovery Import] Cannot contact the mongodb server for more than %ss, bailing out' % time_out)
                    logger.error('[Discovery Import] Make sure the MongoDB server is running correctly')
                    sys.exit(2)
                
                # hummer the connection if need
                time.sleep(1)
    
    
    def __load_data_backend_try(self):
        try:
            if self.mongodb_con is not None:
                try:
                    self.mongodb_con.close()
                except Exception, exp:
                    logger.error('[Discovery Import] Closing mongodb : %s' % exp)
            self.mongodb_con = Connection(self.mongodb_uri, fsync=True)
            # force a fsync connection, so all access are forced even if it's slower on writes
            self.mongodb_db = getattr(self.mongodb_con, self.mongodb_database)
            self.discovery_confs_db = self.mongodb_db.discovery_confs
            self.discovery_rules_db = self.mongodb_db[self.source_name + "_rules"]
            if not self.discovery_rules_manager:
                self.discovery_rules_manager = DiscoveryRulesManager(self.discovery_rules_db, self.rules_path)
            else:
                self.discovery_rules_manager.update_user_rules()
            self.discovery_hosts_data = self.mongodb_db.discovery_hosts_data
        except Exception, e:
            logger.error("[Discovery Import] Mongodb Module: Error %s:" % e)
            raise
        logger.debug('[Discovery Import] Connection OK for %s' % self.mongodb_db)
    
    
    def __nmap_prefix_override(self):
        default_path = self.SHINKEN_NMAP_MAC_PREFIXES_PATH
        user_path = self.nmap_mac_prefixes_user_path
        final_dest = "/usr/share/nmap/nmap-mac-prefixes"
        self.nmap_user_file_state = "ok"
        if not os.path.isfile(default_path):
            logger.error("[Discovery Import] Default file nmap-mac-prefixes was not found at %s." % default_path)
            return False
        if user_path and not os.path.isfile(user_path):
            if user_path:
                logger.error("[Discovery Import] User file nmap-mac-prefixes was not found at %s." % user_path)
            self.nmap_user_file_state = "not_found"
            shutil.copyfile(default_path, final_dest)
            return True
        shinken_file = open(default_path, "r").readlines()
        user_file = []
        if user_path:
            user_file = open(user_path, "r").readlines()
        
        to_write = []
        index = 0
        while index < len(shinken_file) and shinken_file[index].startswith('#'):
            to_write.append(shinken_file[index])
            index += 1
        for line in user_file:
            line = line.strip(' ')
            if not line.startswith('#') and line != '\n':
                if len(line.split(' ')[0]) != 6:
                    logger.error("[Discovery Import] Error in your nmap-mac-prefixes file, please verify it.")
                    shutil.copyfile(default_path, final_dest)
                    self.nmap_user_file_state = "problem_in_file"
                    return True
                to_write.append(line)
        while index < len(shinken_file):
            to_write.append(shinken_file[index])
            index += 1
        final_file = open(final_dest, "w+")
        final_file.write(''.join(to_write))
        return True
    
    
    def __do_launch_scan(self, dc):
        self.__load_data_backend()
        logger.info("[Discovery Import] do launch scan for %s", dc['discovery_name'])
        
        now = int(time.time())
        _id = dc['_id']
        
        dc['state'] = 'RUNNING'
        dc['last_heartbeat'] = now
        dc['last_scan'] = now
        dc['synchronizer'] = self.syncdaemon.me.get_name()
        port_range = dc.get('port_range', '')
        extra_option = dc.get('extra_option', '')
        
        self.discovery_confs_db.save(dc)
        
        backend = DiscoveryDummyBackend()
        macros = [('NMAPTARGETS', dc['iprange'])]
        
        setattr(self.syncdaemon.conf, '$NMAPTARGETS$', dc['iprange'])
        
        nmap_extra_option = ''
        if port_range:
            nmap_extra_option += ' -p %s' % port_range
        if extra_option:
            nmap_extra_option += ' %s' % extra_option
        
        nmapextraoption = 'NMAPEXTRAOPTION'
        # Add the macro value, you must set all this.
        self.syncdaemon.conf.__class__.properties['$' + nmapextraoption + '$'] = StringProp(default='')
        self.syncdaemon.conf.__class__.macros[nmapextraoption] = '$' + nmapextraoption + '$'
        self.syncdaemon.conf.resource_macros_names.append(nmapextraoption)
        setattr(self.syncdaemon.conf, '$' + nmapextraoption + '$', nmap_extra_option)
        macros.append((nmapextraoption, nmap_extra_option))
        
        # HACK: I don't understand why the nmap_dsicovery command is not linked here, it's a non sense,
        # so temporary fix it by linking in HARD way a new command defined on the way
        nmap_discovery = Command(
            {
                'command_name'   : 'nmap_discovery',
                'shell_execution': '1',
                'command_line'   : '''$PLUGINSDIR$/discovery/nmap_discovery_runner.py --min-rate $NMAPMINRATE$ --max-retries $NMAPMAXRETRIES$ -t "%s" -o "%s"''' % (
                    dc['iprange'], nmap_extra_option)
            }
        )
        
        # Hook the runner nmap and force the command
        for r in self.syncdaemon.conf.discoveryruns:
            r.discoveryrun_command.command = nmap_discovery
        
        d = DiscoveryManager(
            path=None,
            macros=macros,
            overwrite=True,
            runners=['nmap'],
            output_dir='',
            dbmod='',
            backend=backend,
            modules_path='',
            merge=False,
            conf=self.syncdaemon.conf,
            first_level_only=True,
            trad=self.syncdaemon._,
            rulemanager=self.discovery_rules_manager
        )
        
        # Ok, let start the plugins that will give us the data
        d.launch_runners()
        all_ok = False
        while not all_ok:
            all_ok = d.is_all_ok()
            # Maybe dc was updated by configuration, if so
            # get the last version
            dc = self.discovery_confs_db.find_one({'_id': _id})
            dc['last_heartbeat'] = now
            logger.debug("[Discovery Import] The discovery range scan %s is still in progress" % dc['discovery_name'])
            self.discovery_confs_db.save(dc)
            time.sleep(0.5)
        
        d.wait_for_runners_ends()
        
        # We get the results, now we can reap the data
        d.get_runners_outputs()
        
        # and parse them
        d.read_disco_buf()
        
        # Now look for rules
        d.match_rules()
        
        d.loop_discovery()
        
        d.write_config()
        
        hosts_index = 0
        for h in backend.hosts:
            h['_DISCOVERY_ID'] = _id
            hname = h.get('host_name', '')
            address = h.get('address', '')
            possible_keys = [hname, address]
            h['_SYNC_KEYS'] = [k.lower() for k in possible_keys if k]
            if address in self.known_hosts_address:
                backend.hosts.pop(hosts_index)
            else:
                self.known_hosts_address.append(address)
            hosts_index += 1
        
        # First remove in the discovery_data col the
        self.discovery_hosts_data.remove({'_DISCOVERY_ID': _id})
        logger.debug("[Discovery Import] the discovery range %s did discover %d hosts" % (dc['discovery_name'], len(backend.hosts)))
        if len(backend.hosts) >= 1:
            self.discovery_hosts_data.insert(backend.hosts)
        
        # And save the dicovery entry
        dc['state'] = 'OK'
        self.discovery_confs_db.save(dc)


class DiscoveryDummyBackend:
    def __init__(self):
        self.hosts = []
    
    
    def write_host_config_to_db(self, hname, h):
        print "write_host_config_to_db::", hname, h
        self.hosts.append(h)


def _get_own_module(sname):  # This function return your module class | MANDATORY
    return next((mod for mod in app.modules_manager.get_all_instances() if mod.name[7:] == sname), None)


def rules(sname):
    reload_rules(sname)
    mod = _get_own_module(sname)
    
    rules_db = mod.discovery_rules_db
    rules = rules_db.find({}).sort("order")
    
    display_rules = []
    for rule in rules:
        display_rules.append(rule)
    
    status = mod.discovery_rules_manager.status['user']
    status_trad = mod.discovery_rules_manager.status['user_error_trad']
    rules_path = mod.rules_path
    
    return {
        'sname'        : sname,  # This is the name of your source
        'app'          : app,  # This is the synchronizerdaemon
        'mod'          : mod,  # This is your  class for access to lang and whatever you want
        'display_rules': display_rules,  # These are the rules in a list ready to be displayed
        'status'       : status,  # This is the user file status which allows us to show an error if the status isn't 0
        'status_trad'  : status_trad,
        'rules_path'   : rules_path
    }


def reload_rules(sname):
    mod = _get_own_module(sname)
    mod.discovery_rules_manager.update_user_rules()
    mod.update_tabs()
    return {'warnings': mod.tabs[0]['warnings'], 'errors': mod.tabs[0]['errors']}


pages = {
    rules       : {'routes': ['/sources/:sname/rules/rules_list'], 'view': 'rules_list', 'static': True, 'wrappers': ['json']},
    reload_rules: {'routes': ['/sources/:sname/rules/rules_list/reload_rules/'], 'view': None, 'static': True, 'wrappers': ['json']}
}
