#!/usr/bin/python
# -*- coding: utf-8 -*-
#
# Copyright (C) 2013-2019
# This file is part of Shinken Enterprise, all rights reserved.

from ctypes import c_int64
from multiprocessing import Value

from shinkensolutions.date_helper import get_now


class STATE_TYPE(object):
    SOFT = 0
    HARD = 1


class DOWNTIME(object):
    NONE = 0
    ACTIVE = 1
    INHERITED = 2


class ACK(object):
    NONE = 0
    ACTIVE = 1
    INHERITED = 2


class ITEM_TYPE(object):
    UNSET = u'unset'
    HOST = u'host'
    CHECK = u'check'
    CLUSTER = u'cluster'


class STATE(object):
    OK = 0
    WARN = 1
    CRIT = 2
    UNKNOWN = 3
    MISSING_DATA = 4
    SHINKEN_INACTIVE = 5
    
    STATE_TO_STR = {
        OK              : u'STATE_OK',
        WARN            : u'STATE_WARN',
        CRIT            : u'STATE_CRIT',
        UNKNOWN         : u'STATE_UNKNOWN',
        MISSING_DATA    : u'STATE_MISSING_DATA',
        SHINKEN_INACTIVE: u'STATE_SHINKEN_INACTIVE',
    }


# ---------------- Shared Data ----------------
class SharedData(object):
    def __init__(self):
        self._last_start_time = Value(c_int64, 0)
        self._shinken_inactive_period = Value(c_int64, 0)
    
    
    def get_last_start_time(self):
        return self._last_start_time.value
    
    
    def set_last_start_time(self, value):
        self._last_start_time.value = value
    
    
    def get_shinken_inactive_period(self):
        return self._shinken_inactive_period.value
    
    
    def set_shinken_inactive_period(self, value):
        self._shinken_inactive_period.value = value


shared_data = SharedData()

NO_STATS = 'NO_STATS'
NO_DATA = 'NO_DATA'


def get_sample(sampling, asked_seconds):
    if not sampling or sampling[0] == -1:
        return NO_STATS, NO_STATS
    
    _start = sampling[0]
    _end = _start
    
    _start_limit = get_now() - asked_seconds
    if _start[0] == 0 or _start[0] < _start_limit:
        return NO_DATA, NO_DATA
    
    for i in sampling:
        if i == -1:
            break
        if (_start[0] - i[0]) >= asked_seconds:
            break
        _end = i
    
    if _start == _end:
        return NO_DATA, NO_DATA
    
    return _start, _end
