#!/usr/bin/python
# -*- coding: utf-8 -*-
#
# Copyright (C) 2013-2019
# This file is part of Shinken Enterprise, all rights reserved.
from ec_common import ACK, DOWNTIME, STATE_TYPE
from ec_database_connection import ECDatabaseConnection
from ec_event import Event
from ec_filter import EventContainerFilter
from ec_reader_stats import ECReaderStats
from shinken.basemodule import BaseModule
from shinken.misc.filter import check_user_item_access
from shinken.misc.type_hint import List, Dict, Any, Text
from shinken.objects.contact import Contact
from shinkensolutions.date_helper import timestamp_from_datetime
from shinkensolutions.lib_modules.exception import EventContainerFilterError


class EventContainerWebUIModule(BaseModule):
    
    def __init__(self, configuration):
        BaseModule.__init__(self, configuration)
        self.database_connection = ECDatabaseConnection(configuration, self.logger, ttl_index=False)
        self.reader_stats = ECReaderStats(self, self.database_connection)
        self.event_container_filter = EventContainerFilter(self.logger)
        Event.reader_stats = self.reader_stats
    
    
    def init(self):
        self.database_connection.init()
        self.reader_stats.start_thread()
    
    
    def get_events(self, user, filters, date_filter, first_ordering_uuid, page_index, page_size):
        # type: (Contact, List, str, str, int, int) -> (List[Dict], bool)
        try:
            filters = self.event_container_filter.from_visu_filters_to_mongo_filters(filters, date_filter)
        except EventContainerFilterError:
            filters = None
        
        self.logger.debug(u'get_events filters:[%s]' % filters)
        _events, have_next = self._get_events(user, filters, first_ordering_uuid, page_index, page_size)
        return map(self._from_event_to_visu_event, _events), have_next
    
    
    def find_downtimes(self, downtime_ids):
        return self.database_connection.find_downtimes(downtime_ids)
    
    
    def find_acknowledge(self, acknowledge_id):
        return self.database_connection.find_acknowledge(acknowledge_id)
    
    
    def _from_event_to_visu_event(self, event):
        # type: (Event) -> Dict[Text, Any]
        
        visu_event = event.__dict__.copy()
        visu_event['event_since'] = timestamp_from_datetime(event.event_since)
        visu_event['event_hard_since'] = timestamp_from_datetime(event.event_hard_since) if event.state_type == STATE_TYPE.HARD else None
        visu_event['context'] = self._get_context(visu_event)
        
        if event.active_downtime_uuids:
            dts = self.find_downtimes(event.active_downtime_uuids)
            if dts:
                visu_downtimes = [{
                    u'author'    : dt[u'author'],
                    u'comment'   : dt[u'comment'],
                    u'end_time'  : dt.get(u'end_time', None),
                    u'start_time': dt.get(u'start_time', None),
                } for dt in dts]
                
                visu_event['downtimes'] = visu_downtimes
                visu_event['downtimes_inherited'] = visu_downtimes
        if event.acknowledge_id:
            a = self.find_acknowledge(event.acknowledge_id)
            if a:
                visu_acknowledges = {
                    u'author'    : a[u'author'],
                    u'comment'   : a[u'comment'],
                    u'start_time': a[u'start_time'],
                }
                visu_event['acknowledgement'] = visu_acknowledges
                visu_event['acknowledgement_inherited'] = visu_acknowledges
                visu_event['partial_acknowledge'] = visu_acknowledges
        
        visu_event['status'] = event.state_id
        visu_event['is_status_confirmed'] = event.state_type == STATE_TYPE.HARD
        visu_event['type'] = event.item_type
        visu_event['service_description'] = event.check_name
        
        visu_event.pop('_id', None)
        visu_event.pop('state_id', None)
        visu_event.pop('state_type', None)
        visu_event.pop('item_type', None)
        visu_event.pop('name', None)
        visu_event.pop('flapping', None)
        visu_event.pop('acknowledged', None)
        visu_event.pop('downtime', None)
        visu_event.pop('partial_flapping', None)
        visu_event.pop('partial_ack', None)
        visu_event.pop('partial_dt', None)
        visu_event.pop('check_name', None)
        return visu_event
    
    
    @staticmethod
    def _get_context(visu_event):
        context = u'NOTHING'
        if visu_event[u'partial_flapping']:
            context = u'PARTIAL-FLAPPING'
        if visu_event[u'flapping']:
            context = u'FLAPPING'
        if visu_event[u'partial_ack']:
            context = u'PARTIAL-ACKNOWLEDGED'
        if visu_event[u'acknowledged'] == ACK.INHERITED:
            context = u'INHERITED-ACKNOWLEDGED'
        if visu_event[u'acknowledged'] == ACK.ACTIVE:
            context = u'ACKNOWLEDGED'
        if visu_event[u'partial_dt']:
            context = u'PARTIAL-DOWNTIME'
        if visu_event[u'downtime'] == DOWNTIME.INHERITED:
            context = u'INHERITED-DOWNTIME'
        if visu_event[u'downtime'] == DOWNTIME.ACTIVE:
            context = u'DOWNTIME'
        return context
    
    
    @staticmethod
    def _filter_events_by_user_visibility(user, events):
        # type: (Contact, List[Event]) -> List[Event]
        filtered_events = [e for e in events if check_user_item_access(user, e.item_uuid)]
        return filtered_events
    
    
    def _get_events(self, user, filters, first_ordering_uuid, page_index, page_size):
        # type: (Contact, Dict[Text,Any], str, int, int) -> (List[Event], bool)
        
        have_next = False
        if first_ordering_uuid:
            filters = self._set_first_ordering_uuid(filters, first_ordering_uuid)
        
        if user.is_admin:
            events = self.database_connection.find_events(filters, skip_value=(page_index * page_size), page_size=page_size + 1)
        else:
            events = self._get_events_for_non_admin(user, filters, first_ordering_uuid, page_index, page_size)
        
        if len(events) > page_size:
            have_next = True
            events.pop()
        
        return events, have_next
    
    
    def _get_events_for_non_admin(self, user, filters, first_ordering_uuid, page_index, page_size):
        # type: (Contact, List, str, int, int) -> List[Event]
        
        events = []
        mongo_page_size = int(page_size * 1.5)
        mongo_page_index = 0
        to_skip = page_index * page_size
        end_index = (to_skip + page_size + 1)  # +1 to known if there are next element
        while True:
            all_events_page = self.database_connection.find_events(filters, skip_value=mongo_page_index, page_size=mongo_page_size)
            
            if not all_events_page:
                return events[to_skip:end_index]
            
            all_events_page = self._filter_events_by_user_visibility(user, all_events_page)
            if first_ordering_uuid is None and all_events_page:
                first_ordering_uuid = all_events_page[0].ordering_uuid
                filters = self._set_first_ordering_uuid(filters, first_ordering_uuid)
            
            if len(events) + len(all_events_page) > end_index:
                events = events + all_events_page
                return events[to_skip:end_index]
            
            events.extend(all_events_page)
            mongo_page_index += mongo_page_size
    
    
    @staticmethod
    def _set_first_ordering_uuid(original_filters, first_ordering_uuid):
        filters = original_filters
        if first_ordering_uuid:
            filters = EventContainerFilter.append_to_filter(original_filters, {u'ordering_uuid': {u'$lte': first_ordering_uuid}})
        return filters
    
    
    def get_raw_stats(self, param=''):
        try:
            seconds_of_stats = int(param)
        except:
            seconds_of_stats = 60
        
        status = BaseModule.get_state(self)
        status['raw_stats'] = self.reader_stats.get_raw_stats(seconds_of_stats)
        
        return status
    
    
    def send_stat_to_reader_stats(self, stat_name, stat_endpoint, filters, stat_time):
        self.reader_stats.add_request_stat(stat_name, stat_endpoint, filters, stat_time)
