#!/usr/bin/python
# -*- coding: utf-8 -*-
#
# Copyright (C) 2013-2019
# This file is part of Shinken Enterprise, all rights reserved.

from ec_common import NO_DATA, NO_STATS, get_sample
from ec_database_connection import ECDatabaseConnection
from shinken.misc.type_hint import NoReturn
from shinken.thread_helper import Thread
from shinken.withworkersandinventorymodule import WithWorkersAndInventoryModule
from shinkensolutions.date_helper import timestamp_from_datetime, get_datetime_from_utc_to_local_time_zone

SAMPLING_SPEED = 5  # in sec
SAMPLING_SIZE = 60  # nb entry / time  = SAMPLING_SIZE * SAMPLING_SPEED


class ECWriterData:
    def __init__(self, has_data=True):
        self.has_data = has_data
        self.database_status = True
        
        self.total_event_in_db = 0
        self.total_size_of_event_in_db = 0
        self.oldest_event_in_db = 0
        self.nb_day_keep = 0
        
        self.have_perf_stats = False
        self.total_duration = 0
        self.brok_handle = 0
        self.event_write = 0
        self.work_time = 0
        
        self.item_count = {}
        self.workers = {}
    
    
    def export(self):
        return self.__dict__
    
    
    def _set_worker_info(self, worker_id, info_name, info_value):
        if not self.workers.get(worker_id, None):
            self.workers[worker_id] = {}
        self.workers[worker_id][info_name] = info_value
    
    
    def set_worker_item_count(self, worker_id, item_count):
        self._set_worker_info(worker_id, 'item_count', item_count)
    
    
    def set_worker_alive_info(self, worker_id, alive_info):
        self._set_worker_info(worker_id, 'alive', alive_info)
    
    
    def set_worker_perf_stats(self, worker_id, perf_stats):
        if perf_stats == NO_STATS or perf_stats == NO_DATA:
            self._set_worker_info(worker_id, 'have_perf_stats', perf_stats)
        else:
            self._set_worker_info(worker_id, 'have_perf_stats', True)
            self._set_worker_info(worker_id, 'total_duration', perf_stats['total_duration'])
            self._set_worker_info(worker_id, 'brok_handle', perf_stats['brok_handle'])
            self._set_worker_info(worker_id, 'event_write', perf_stats['event_write'])
            self._set_worker_info(worker_id, 'work_time', perf_stats['work_time'])
    
    
    def set_total_item_count(self, item_count):
        self.item_count = item_count
    
    
    def set_perf_stats(self, perf_stats):
        if perf_stats == NO_STATS or perf_stats == NO_DATA:
            self.have_perf_stats = perf_stats
        else:
            self.have_perf_stats = True
            self.total_duration = perf_stats['total_duration']
            self.brok_handle = perf_stats['brok_handle']
            self.event_write = perf_stats['event_write']
            self.work_time = perf_stats['work_time']


class ECWriterStats(Thread):
    def __init__(self, ec_module_broker, database_connection):
        # type: (WithWorkersAndInventoryModule, ECDatabaseConnection) -> NoReturn
        super(ECWriterStats, self).__init__(loop_speed=SAMPLING_SPEED)
        self.ec_module_broker = ec_module_broker
        self.database_connection = database_connection
        self.sampling = {}
    
    
    def get_thread_name(self):
        return 'ec-writer-stats'
    
    
    def loop_turn(self):
        data = self.ec_module_broker.send_command('get_raw_stats')
        workers = data.get('workers', {})
        for worker_id, worker_data in workers.iteritems():
            if not worker_data:
                return
            sampling = self.sampling.get(worker_id, None)
            # We not build sampling in default of the get because if you do that you will build a array each call
            if sampling is None:
                sampling = {
                    'work_time'  : [-1 for _ in xrange(SAMPLING_SIZE)],
                    'write_event': [-1 for _ in xrange(SAMPLING_SIZE)],
                }
            
            sampling['write_event'].pop()
            sampling['write_event'].insert(0, worker_data['write_event_cumulative'])
            sampling['work_time'].pop()
            sampling['work_time'].insert(0, worker_data['work_time_cumulative']['total'])
            self.sampling[worker_id] = sampling
    
    
    @staticmethod
    def _count_worker_item(worker):
        inventory = worker['inventories']
        item_count = {
            'items'   : 0,
            'hosts'   : 0,
            'clusters': 0,
            'checks'  : 0,
        }
        for inventory in inventory.values():
            item_count['items'] += (inventory.get_len() if inventory else 0)
            hosts = inventory.get_all_hosts()
            for host, host_summary in hosts.iteritems():
                nb_check = len(host_summary.get_checks())
                item_count['checks'] += nb_check
                item_count['items'] += nb_check
                if host_summary.is_cluster():
                    item_count['clusters'] += 1
                else:
                    item_count['hosts'] += 1
        return item_count
    
    
    def get_raw_stats(self, seconds_of_stats):
        data = ECWriterData()
        
        total_item_count = {
            'items'   : 0,
            'hosts'   : 0,
            'clusters': 0,
            'checks'  : 0,
        }
        total_perf_stats = NO_STATS
        for worker_id, worker in self.ec_module_broker._workers.iteritems():
            alive = worker['process'].is_alive()
            item_count = self._count_worker_item(worker)
            
            for k, v in item_count.iteritems():
                total_item_count[k] = total_item_count[k] + v
            
            _work_time_start, _work_time_end = get_sample(self.sampling.get(worker_id, {}).get('work_time', []), seconds_of_stats)
            _write_event_start, _write_event_end = get_sample(self.sampling.get(worker_id, {}).get('write_event', []), seconds_of_stats)
            
            if _work_time_start == NO_STATS or _write_event_start == NO_STATS:
                data.set_worker_perf_stats(worker_id, NO_STATS)
            elif _work_time_start == NO_DATA or _write_event_start == NO_DATA:
                data.set_worker_perf_stats(worker_id, NO_DATA)
                if total_perf_stats == NO_STATS:
                    total_perf_stats = NO_DATA
            else:
                worker_perf_stats = {
                    'total_duration': _work_time_start[0] - _work_time_end[0],
                    'brok_handle'   : _work_time_start[2] - _work_time_end[2],
                    'event_write'   : _write_event_start[1] - _write_event_end[1],
                    'work_time'     : _work_time_start[1] - _work_time_end[1],
                }
                if total_perf_stats == NO_STATS or total_perf_stats == NO_DATA:
                    total_perf_stats = worker_perf_stats.copy()
                else:
                    total_perf_stats = {
                        'total_duration': max(total_perf_stats['total_duration'], worker_perf_stats['total_duration']),
                        'brok_handle'   : total_perf_stats['brok_handle'] + worker_perf_stats['brok_handle'],
                        'event_write'   : total_perf_stats['event_write'] + worker_perf_stats['event_write'],
                        'work_time'     : total_perf_stats['work_time'] + worker_perf_stats['work_time'],
                    }
                data.set_worker_perf_stats(worker_id, worker_perf_stats)
            
            data.set_worker_item_count(worker_id, item_count)
            data.set_worker_alive_info(worker_id, alive)
        
        data.nb_day_keep = self.database_connection.get_day_keep_data()
        data.set_total_item_count(total_item_count)
        data.set_perf_stats(total_perf_stats)
        
        data.total_event_in_db = self.database_connection.count_event_in_db()
        data.total_size_of_event_in_db = self.database_connection.get_size_of_database()
        oldest_event_in_db = self.database_connection.find_oldest_event_in_db()
        if oldest_event_in_db:
            oldest_event_in_db = get_datetime_from_utc_to_local_time_zone(oldest_event_in_db['event_since'])
            data.oldest_event_in_db = timestamp_from_datetime(oldest_event_in_db)
        
        return data.export()
