#!/usr/bin/python
# -*- coding: utf-8 -*-
#
# Copyright (C) 2013-2019:
# This file is part of Shinken Enterprise, all rights reserved.

import json
import io
import urllib2
from shinken.basemodule import BaseModule
from shinken.log import logger

# Mandatory: this bloc allow the collector to be load by the synchronizer as a collector
# IMPORTANT: the type is the module_type that will be call in the .cfg of the collector to identify it
properties = {
    'daemons': ['synchronizer'],
    'type'   : 'ima-collector',  # IMPORTANT: this value MUST be changed with your own
}


# Mandatory: this function is called by the daemon to GET a collector instance
def get_instance(collector_definition):
    logger.info("[IMACollector] Get an instance for the collector definition %s" % collector_definition.get_name())
    
    # Here you can read and parse your collector cfg parameters.
    # IMPORTANT: the values are STRINGS and so it's up to this code to parse and transform them
    
    # Get your instance
    instance = IMACollector(collector_definition)
    
    # Give back to the synchronizer daemon
    return instance


# Your Collector main class
# IMPORTANT: you MUST rename it with your own name
class IMACollector(BaseModule):
    def __init__(self, mod_conf):
        # Mandatory: this must be called to be a valid collector module
        BaseModule.__init__(self, mod_conf)
        
        # Your code now
        # IMPORTANT: do NOT open any database connection or network connection
        # in this method, as the synchronizer daemon will fork() and close all
        # filedescriptor AFTER this method to be call
        # Always open your database/network connections in the dedicated init() method
        try:
            self.uri = getattr(mod_conf, 'uri', '')
            self.apikey = getattr(mod_conf, 'apikey', '')
            self.response = None
        except AttributeError:
            logger.error("[IMACollector] The module is missing a property, check module configuration in import-json.cfg")
            raise
    
    
    # Mandatory: do not edit
    def load(self, syncdaemon):
        self.syncdaemon = syncdaemon
    
    
    # Mandatory:  init:: You can open all you databse/network connections here
    def init(self):
        logger.info("[IMACollector] Initialization of the cfg file import module")
    
    
    def call_server(self):
        # self.return_queue = self.properties['from_queue']
        try:
            logger.info("[IMACollector] Try to connect to website " + self.uri)
            request = urllib2.Request(self.uri, headers={"X-ShinkenSynchronizerApiKey": self.apikey})
            self.response = urllib2.urlopen(request)
            if self.response.headers.getparam('charset') != 'utf-8':
                logger.error("[IMACollector] Encoding from the webserver is not utf-8")
            else:
                logger.debug("[IMACollector] Encoding from the webserver is set correctly to utf-8")
        except Exception as e:
            logger.error("[IMACollector] Une erreur s'est produite: " + str(e))
    
    
    def clean_json(self, json_data):
        # The json must not contain an empty object class. Ex. : If a "business_impacts" class contains no object, it must not be sent to the synchronizer
        item_class_to_del = []
        for (item_class, items) in json_data.iteritems():
            if not items:
                item_class_to_del.append(item_class)
        
        for class_to_del in item_class_to_del:
            json_data.pop(class_to_del, None)
    
    
    # Mandatory:  get_objects:: Call each time your collector will run
    # NOTE: it must return ALL the collector elements, NOT just the new/different ones
    def get_objects(self):
        ###########################
        # YOUR CODE GOES HERE
        ###########################
        
        # The result object is a dict with:
        # 'state': [string] allow the synchronizer daemon to know in which state this collector is
        #      * 'Not configured' : the collector need a specific configuration, and the admin did not edit it
        #      * 'CRITICAL'       : the collector is failed for a reason (like a missing connection or database)
        #      * 'OK'             : all is OK
        # 'output': [string] an html string that will be print on the main configuration UI interface
        # 'objects': [dict] all the objects this collector will returns
        # 'errors': [list of strings] when CRITICAL, theses errors will be print on the interface
        # 'warnings': [list of strings] when OK, if there are strings in warnings, they will be print on the interface
        result = {
            'state'   : 'OK',
            'output'  : '',
            'objects' : {},
            'errors'  : [],
            'warnings': [],
        }
        #############################
        # Connect to server
        #############################
        logger.info("[IMACollector] Ask me for objects to return")
        self.call_server()
        code = 0
        if self.response:
            code = self.response.getcode()
        if code != 200:
            result['state'] = 'CRITICAL'
            result['output'] = 'Serveur indisponible'
            result['errors'] = ['Serveur indisponible']
            logger.error("[IMACollector] Serveur indisponible")
            return result
        try:
            content = self.response.read().decode('utf8')
        except Exception as e:
            result['state'] = 'CRITICAL'
            result['output'] = 'Erreur lors de la lecture de la réponse du serveur : %s' % str(e)
            result['errors'] = [str(e)]
            logger.error("[IMACollector] Erreur lors de la lecture de la réponse du serveur : %s" % str(e))
        
        logger.info("[IMACollector] Serveur disponible, Recuperation des donnees")
        
        #############################
        # Read Json
        #############################
        try:
            json_data = json.loads(content)
            result['objects'] = json_data
        except Exception as e:
            logger.error("[IMACollector] Une erreur a été rencontré pendant la lecture du json")
            result['state'] = 'CRITICAL'
            result['output'] = "Une erreur s'est produite durant le traitement des donnees : %s" % str(e)
            result['errors'] = [str(e)]
        
        self.clean_json(json_data)
        
        logger.debug("[IMACollector] Contenu du JSON: %s" % json_data)
        
        #############################
        # Write json to tmp file
        #############################
        try:
            file = io.open('/tmp/shinken_import_ima_content.json', 'w', encoding='utf-8')
            file.write(json.dumps(json_data, ensure_ascii=False))
            file.close()
        except Exception as e:
            result['state'] = 'CRITICAL'
            result['output'] = "Une erreur s'est produite pendant la sauvegarde des données reçues : %s" % str(e)
            result['errors'] = [str(e)]
            logger.error("[IMACollector] Une erreur s'est produite pendant la sauvegarde des données reçues : %s" % str(e))
        
        #############################
        # Send result
        #############################
        return result
