#!/usr/bin/python
# -*- coding: utf-8 -*-
#
# Copyright (C) 2013-2019
# This file is part of Shinken Enterprise, all rights reserved.

import sla_common
from shinken.objects.module import Module as ShinkenModuleDefinition
from sla_abstract_component import AbstractComponent
from sla_common import LIST_STATUS
from sla_component_manager import ComponentManager
from sla_info import SLAInfo


class SLAMigration(AbstractComponent):
    
    def __init__(self, conf, component_manager, sla_info):
        # type: (ShinkenModuleDefinition, ComponentManager,  SLAInfo) -> None
        super(SLAMigration, self).__init__(conf, component_manager)
        self.sla_info = sla_info
    
    
    @staticmethod
    def must_migrate(archive):
        current_archive_version = archive.get('version', -1)
        return sla_common.CURRENT_ARCHIVE_VERSION != current_archive_version
    
    
    def migrate_archive(self, archive_day):
        archive_name = '%s (%s-%s)' % (archive_day.get('uuid', ''), archive_day.get('hname', ''), archive_day.get('sdesc', ''))
        self.logger.debug('[MIGRATION] ' ' -> %s ' 'start migrate archive' % archive_name)
        current_archive_version = archive_day.get('version', -1)
        
        if not self.must_migrate(archive_day):
            return False
        
        if current_archive_version < 1:
            self.logger.debug('[MIGRATION] ' ' -> %s ' 'migration to version 1' % archive_name)
            current_archive_version = 1
            new_archive_day = {
                '_id'    : archive_day['_id'],
                'ranges' : archive_day['ranges'],
                'type'   : archive_day['type'],
                'hname'  : archive_day['hname'],
                'year'   : archive_day['year'],
                'yday'   : archive_day['yday'],
                'version': current_archive_version,
            }
            if archive_day['type'] == 'service':
                new_archive_day['sdesc'] = archive_day['sdesc']
            if archive_day.get('missing', False):
                new_archive_day['missing'] = archive_day['missing']
            _total = 0
            for prefix in LIST_STATUS:
                _sum = archive_day.get('dt_sum_%s' % prefix, 0) + archive_day.get('%s_dt_nack_sum' % prefix, 0) + archive_day.get('%s_dt_ack_sum' % prefix, 0)
                if _sum:
                    new_archive_day['history_dt_%s' % prefix] = _sum
            
            for prefix in LIST_STATUS:
                _sum = archive_day.get('%s_sum' % prefix, 0)
                _total += _sum
                if _sum:
                    new_archive_day['archive_%s' % prefix] = _sum
                    new_archive_day['sla_%s' % prefix] = _sum
                    new_archive_day['history_%s' % prefix] = _sum
            
            new_archive_day['archive_total'] = _total
            new_archive_day['sla_total'] = _total
            new_archive_day['history_total'] = _total
            new_archive_day['archive_format'] = '0000'
            new_archive_day['sla_format'] = '0000'
            new_archive_day['history_format'] = '0000'
            
            archive_day.clear()
            archive_day.update(new_archive_day)
        
        if sla_common.CURRENT_ARCHIVE_VERSION == current_archive_version:
            return True
        
        if current_archive_version < 2:
            self.logger.debug('[MIGRATION] ' ' -> %s ' 'migration to version 2' % archive_name)
            item_uuid = self.sla_info.get_uuid(archive_day['hname'], archive_day.get('sdesc', ''))
            if item_uuid:
                current_archive_version = 2
                archive_day['version'] = current_archive_version
                archive_day['uuid'] = item_uuid
                del archive_day['type']
                del archive_day['hname']
                archive_day.pop('sdesc', None)
            else:
                self.logger.debug('[MIGRATION] ' ' -> %s ' 'migration to version 2 fail. Unable to found uuid for name %s-%s' % (archive_name, archive_day['hname'], archive_day.get('sdesc', '')))
                return False
        
        if sla_common.CURRENT_ARCHIVE_VERSION == current_archive_version:
            return True
        
        if current_archive_version < 3:
            self.logger.debug('[MIGRATION] ' ' -> %s ' 'migration to version 3' % archive_name)
            current_archive_version = 3
            archive_day['version'] = current_archive_version
            archive_day['thresholds'] = self.sla_info.get_sla_thresholds(archive_day['uuid'])
            archive_day['total_ranges'] = len(archive_day['ranges'])
        
        return True
    
    
    def init(self):
        pass
    
    
    def tick(self):
        pass
