#!/usr/bin/python
# -*- coding: utf-8 -*-

# Copyright (C) 2013:
#    Gabes Jean, naparuba@gmail.com
#
# This file is part of Shinken Enterprise, all rights reserved.
import json
import os
import time

from shinken.basemodule import BaseModule
from shinken.http_client import HTTPClient, HTTPExceptions
from shinken.log import logger
from shinken.synchronizer.dao.def_items import PROP_DEFAULT_VALUE

properties = {
    'daemons': ['arbiter'],
    'type'   : 'synchronizer-import',
    'phases' : ['configuration'],
}


def _show_error(error):
    bloc_error_size = 80
    logger.error('*' * bloc_error_size)
    logger.error('')
    logger.error(error)
    logger.error('')
    logger.error('*' * bloc_error_size)


# called by the plugin manager to get a module
def get_instance(plugin):
    logger.info("[Synchronizer Import] Get a Synchronizer import module for plugin %s" % plugin.get_name())
    
    # Catch errors
    url = getattr(plugin, 'url', None)
    if url is None:
        _show_error('The "url" parameter for the synchronizer-import module (in file %s) is mandatory. It must link to the synchronizer daemon private address (with the 7765 port).' % plugin.imported_from)
        return None
    master_key = getattr(plugin, 'master_key', None)
    if master_key is None:
        _show_error(
            'The "master_key" parameter for the synchronizer-import module (in file %s) is mandatory. It must be the same than the synchronizer one (in the synchronizer.cfg file) so only allowed arbiter can get the configuration.' % plugin.imported_from)
        return None
    
    instance = Synchronizer_Import(plugin, url, master_key)
    return instance


# Just print some stuff
class Synchronizer_Import(BaseModule):
    def __init__(self, mod_conf, url, master_key):
        BaseModule.__init__(self, mod_conf)
        self.url = url
        self.master_key = master_key
        self.client = None
        self.activated = True  # Not activated means we run on spare. If so we do nothing
        self.max_try = int(getattr(mod_conf, 'max_try', 90))
        self.sleep_time = int(getattr(mod_conf, 'sleep_time', 2))
    
    
    # Called by Arbiter to say 'let's prepare yourself guy'
    def init(self):
        logger.info("[Synchronizer Import] Initialization of the Synchronizer import module")
    
    
    def get_objects(self):
        # Arbiter is spare so module is not activate, do nothing
        if not self.activated:
            return {}
        
        logger.info("[Synchronizer Import] Requesting configuration from the synchronizer daemon")
        if self.client is None:
            self.client = HTTPClient(uri=self.url, timeout=60)
        logger.debug("[Synchronizer Import] client created")
        
        start_time = time.time()
        
        nb_try = 0
        data = {}
        fetch_data_success = False
        max_try = self.max_try
        while nb_try < max_try:  # do the try for max_try times
            nb_try += 1
            # Look if we just check or if we will look at the prod part
            env = os.environ
            
            to_load = 'production'
            if env.get('CHECK_STAGGING', '0') == '1':
                to_load = 'stagging'
            elif env.get('CHECK_PROPOSE', '0') == '1':
                to_load = 'propose'
            elif env.get('CHECK_PREPROD', '0') == '1':
                to_load = 'preprod'
            
            logger.info('Getting synchronizer [%s] configuration' % to_load)
            try:
                args = {'states': to_load, 'auth_key': self.master_key}
                response = json.loads(self.client.post('/get_whole_configuration', args))
                if response['rc'] == 504:
                    logger.debug('[Synchronizer Import] Synchronizer is UP but Shinken objects are not yet fully initialized')
                    time.sleep(self.sleep_time)
                    continue
                if response['rc'] == 403:
                    logger.error('[Synchronizer Import] Synchronizer is UP but the provided master key was rejected')
                    print "[Synchronizer Import] Synchronizer is UP but the provided master key was rejected"
                    break
                if response['rc'] != 200:
                    logger.debug('[Synchronizer Import] Synchronizer is UP but there is a problem to get the conf')
                    time.sleep(self.sleep_time)
                    continue
                data = response['data']
                fetch_data_success = True
                break
            except HTTPExceptions, exp:
                # but sleep a bit to not hammer the sycnhronizer, maybe it's just closed so sleep a bit until it's ready
                time.sleep(self.sleep_time)
                logger.error('[Synchronizer Import] cannot get the Synchronizer %s data: "%s" still %d try with %ss pause time between try' % (self.url, str(exp), max_try - nb_try, self.sleep_time))
                print '[Synchronizer Import] cannot get the Synchronizer %s data: "%s" still %d try' % (self.url, str(exp), max_try - nb_try)
        
        if fetch_data_success is False:
            raise Exception('[Synchronizer Import] cannot get the Synchronizer %s data' % self.url)
        
        if not data:
            return {}
        # Sync data are without 's' in the end, arbiter need it...
        new_data = {}
        
        for (t, lst) in data.iteritems():
            new_lst = []
            # Loop over objects and remove NON string values
            for o in lst:
                # map _id as it should, in upper case here
                if '_id' in o:
                    o['_ID'] = o['_id']
                    o['uuid'] = o['_id']
                    
                    del o['_id']
                to_del = ['id']
                if t == 'timeperiod':
                    to_del = ['sources', 'overwrited_protected']
                for (k, v) in o.iteritems():
                    if v == PROP_DEFAULT_VALUE:
                        o[k] = 'null'
                    elif not isinstance(v, basestring):
                        to_del.append(k)
                    elif v == '':
                        to_del.append(k)
                    elif k == 'service_overrides':
                        o[k] = v.replace(PROP_DEFAULT_VALUE, 'null')
                
                for k in to_del:
                    try:
                        del o[k]
                    except KeyError:
                        pass
                
                # Skip bp_rule as it's an inner command
                if t == 'command':
                    cname = o.get('command_name', '')
                    if cname == 'bp_rule':
                        continue
                
                enable = o.get('enabled', '1')
                if enable in ['1', '']:
                    new_lst.append(o)
                
                # Force a shinken-host to all hosts, and at the end
                # and manage *_contacts/groups matchings
                if t == 'host':
                    _to_del = ['edition_contacts', 'edition_contact_groups']
                    for k in _to_del:
                        if k in o:
                            del o[k]
                    # and map notification_contacts/groups into contacts
                    if 'notification_contacts' in o:
                        o['contacts'] = o['notification_contacts']
                        del o['notification_contacts']
                    if 'notification_contact_groups' in o:
                        o['contact_groups'] = o['notification_contact_groups']
                        del o['notification_contact_groups']
                    
                    # Cluster are now with 'is_cluster'==1 and 'bp_rule' instead of check_command
                    if o.get('is_cluster', '0') == '1':
                        o['check_command'] = 'bp_rule!' + o.get('bp_rule', '')
                        o['check_interval'] = '1'
                    
                    # force check_interval, retry_interval and max_check_attempts on clusters
                    ck = o.get('check_command', '')
                    if ck.startswith('bp_rule!'):
                        o['check_interval'] = '1'
                        o['retry_interval'] = '1'
                        o['max_check_attempts'] = '1'
                        o['active_checks_enabled'] = '0'
                
                if 'contact_groups' in o:
                    o['contact_groups'] = ','.join((cg.strip() for cg in o['contact_groups'].split(',') if cg.strip()))
                if 'contactgroups' in o:
                    o['contactgroups'] = ','.join((cg.strip() for cg in o['contactgroups'].split(',') if cg.strip()))
            
            new_data[t + 's'] = new_lst
        
        logger.info('[performance] Synchronizer elements are get in %.3fs' % (time.time() - start_time))
        return new_data
    
    
    # We have the configuration ans now who we are
    # if the current arbiter is spare, this module will do nothing, so user will not have to edit his spare conf file
    def hook_read_configuration(self, arb):
        self.activated = not arb.me.spare
        if not self.activated:
            logger.info("[Synchronizer Import] The Arbiter is configured as Spare, Synchronizer Import is disabled because we don't have synchronizer on this host")
    
    
    # This part is here to check that there is a (valid) key to
    # do NOT limit the hosts numbers by settings silly things into it
    def hook_early_configuration(self, arb):
        from shinkensolutions.crypto import are_keys_valid
        d = are_keys_valid()
        # (are_valid, nodes_limit, is_powerful, customer, customer_email, testing) = are_keys_valid()
        are_valid = d['are_valid']
        testing = d['testing']
        nodes_limit = d['nodes_limit']
        
        if not are_valid:
            logger.error('No val' + 'id ke' + 'y found, will li' + 'mit the nu' + 'mber of ho' + 'sts to %d' % nodes_limit)
            i = -1
            for h in arb.conf.hosts:
                if getattr(h, 'register', '1') == '0' or getattr(h, 'check_command', '').startswith('bp_rule!'):
                    continue
                i += 1
                if i > nodes_limit and nodes_limit != 0:
                    # skip templates
                    logger.warning('OVER' + 'LOAD' + ' LIC' + 'ENCE, dis' + 'abling host %s' % getattr(h, 'host_name', ''))
                    h.use = 'disabled,shinken-host'
                    h.address = 'disabled [ Nodes licence EXCEEDED ]'
                    h.check_command = '_internal_host_up'
                    h.hostgroups = ''
                    h.customs = {}
                    h.business_impact = '0'
        else:
            if testing:
                logger.info('Enterprise valid testing key founded')
            else:
                logger.info('Ent' + 'erpr' + 'ise va' + 'lid' + ' ke' + 'y fou' + 'nded')
